/*
 *    Copyright (C) 1996  Burkhard Kohl
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * 
 * pc537.c implements the character device driver for Feger+Co 
 * PC537-Addin boards.
 *
 * Create the device with:
 *
 * mknod /dev/hw c 63 minor
 * For more info, see the pc537 manpage in section 4.
 *
 * TODO:
 * 	Further error checking for ioctl cmds.
 *
 *	What i/o speed is required/save. Do we really need REALLY_SLOW_IO?
 *
 *	Should reading/writing be buffered?
 * 	
 *	Add more pc537_select() functionality?
 */

/*
 * 	$Log: pc537.c,v $
 * 	Revision 0.12  1996/12/07 22:59:26  buk
 * 	Different changes to ioctl.
 *
 * 	Revision 0.11  1996/10/26 16:49:00  buk
 * 	Added minor support.
 * 	Message of init_module changed.
 *
 *
 * 	Revision 0.10  1996/10/13 17:11:50  buk
 * 	Changed the naming of ioctl cmds.
 *
 * 	Revision 0.9  1996/10/06 00:10:05  buk
 * 	Added a select mechanism for the SEL_IN type.
 *
 * 	Revision 0.8  1996/10/05 15:14:32  buk
 * 	Changed PC537_WRCPORT from reading arg as pointer to reading arg as
 * 	unsigned char.
 *
 * 	Revision 0.7  1996/10/04 22:44:15  buk
 * 	Request/release of ioports included into pc537_open()/pc537_close().
 *
 * 	Revision 0.6  1996/10/04 19:04:58  buk
 * 	Added GNU GPL.
 * 	Changed K-Port to C-Port and renamed IOCTL cmds.
 *
 * 	Revision 0.5  1996/10/01 22:06:10  buk
 * 	Made pc537_ioctl fully functional including verify_area.
 * 	Rearranged pc537_interrupt buffering scheme.
 * 	Changed pc537_open to reflect new struct irqbuf definition.
 * 	Switch to put_user/get_user from Kernel 2.0.x.
 * 	A lot of minor format modifications.
 *
 * 	Revision 0.4  1996/09/22 22:20:42  buk
 * 	Modified pc537_open and pc537_interrupt to make the interrupt handler
 * 	read the C-Port and deposit that value for later access.
 *
 * 	Revision 0.3  1996/09/21 22:12:44  buk
 * 	PC537_RESET and PC_RESTART functional.
 * 	pc537_open modified to allocate and initialize irq buffer related stuff.
 * 	pc537_close modified to free allocated memory.
 * 	pc537_interrupt added. Not really functional yet.
 * 	PC537_DEBUG scheme modified a bit.
 *
 * 	Revision 0.2  1996/09/14 21:44:55  buk
 * 	False addressing scheme for mc ram corrected.
 * 	IOCTLs for reading/writing the C-port, resetting/restarting the mc
 * 	and for raising mc interrupt 0/1 added.
 * 	Some minor changes regarding variable types.
 *
 * 	Revision 0.1  1996/08/28 19:32:13  buk
 * 	Initial revision with reading, writing and seeking on the device.
 *
 */



/* Kernel includes */

#define REALLY_SLOW_IO

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/major.h>
#if KERNEL_VERSION < 2
#	include "segment.h"
#else
#	include <asm/segment.h>
#endif
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/malloc.h>

/*
 * we should include the kernel identification string in the module.
 */
#include <linux/version.h>
static char kernel_version[] = UTS_RELEASE;

#include "pc537.h"

#if KERNEL_VERSION < 2
#include "panic.c"
#endif


/* experimental major, but it is tunable: insmod pc537 major=63 */
static unsigned int major = 63;

static int pc537_irqs[] = { 7, 5, 4, 3 };

static int pc537_bases[]= { 0x160, 0x140, 0x120, 0x100 };

static struct pc537_struct* pc537_table[PC537_MAXBOARDS];

static struct readbuf* rptr;

/*
 * The driver.
 */

/*
 * The addr values for read and write operations
 */
static unsigned char low_addr;
static unsigned char high_addr;


/*
 * lseek - support seeks on the device. 
 */
static int 
pc537_lseek(struct inode * inode, struct file * file, off_t offset, int orig)
{
#if PC537_DEBUG > 2
	printk("pc537_lseek entered.\n");
#endif
	if (offset > pc537_table[0]->endaddr) {
		/* Error message? */
		return -EINVAL;
	}

	switch(orig) {
		case 0:	
			file->f_pos  = offset;
			break; 
		case 1:
			if (offset + file->f_pos > pc537_table[0]->endaddr) {
				/* Error message? */
				return -EINVAL;
			}
			file->f_pos += offset;
			break;
		case 2:
			file->f_pos = pc537_table[0]->endaddr - offset;
			break;
		default:
			return -EINVAL;
	}

#if PC537_DEBUG > 1
	printk("pc537_lseek finished.\n");
#endif
#if PC537_DEBUG > 3
	printk("pc537_lseek current pos %lu.\n", file->f_pos);
#endif
	return file->f_pos;
}


/*
 * pc537_read
 */
static int
pc537_read(struct inode * inode, struct file * file, char * buf, int count)
{
	unsigned long copied = 0;
	int err = 0;
	int read_count = count;
	char tmp;

#if PC537_DEBUG > 2
	printk("pc537_read entered, count %i, pos: %lu.\n", read_count, file->f_pos);
#endif
	
	if (read_count > (pc537_table[0]->endaddr - file->f_pos +1)) {
		read_count = pc537_table[0]->endaddr - file->f_pos +1;
	}

	/* Make sure buf is safe to write into */
	if (read_count) {
		err = verify_area(VERIFY_WRITE, buf, read_count);
		if (err) {
#if PC537_DEBUG > 1
			printk("pc537_read: verify_area failed.\n");
#endif
			return err;
		}
	} else if (count) {
		return -EINVAL;
	}

	while (copied < read_count) {
		low_addr  = (int)(file->f_pos +copied)% 0x100;
/*			    ^^^^^
 *	This cast is taken from insmod's drv_hello driver to avoid
 * 	the dreaded "__moddi3" messages. Should be ok for 
 *	f_pos < 2^32.
 */ 
		outb_p(low_addr, pc537_table[0]->base);
		high_addr = (int)(file->f_pos +copied)/ 0x100;
/*			    ^^^^^
 *	This cast is taken from insmod's drv_hello driver to avoid
 * 	the dreaded "__divdi3" messages. Should be ok for 
 *	f_pos < 2^32.
 */ 
		outb_p(high_addr,pc537_table[0]->base +1);
		tmp = inb_p(pc537_table[0]->base +2);
		put_fs_byte(tmp, buf +copied++);
	}
/*
 *	Note that reading from the device resets it.
 *	In order to start it, you have to call the PC537_RESTART ioctl.
 */
	
	file->f_pos += copied;
#if PC537_DEBUG > 2
	printk("pc537_read finished, copied: %lu, current pos: %lu.\n", copied, file->f_pos);
#endif
	return copied;
}


/*
 * pc537_write
 */
static int
pc537_write(struct inode * inode, struct file * file, char * buf, int count)
{
	unsigned long copied = 0;
	int write_count = count;
	off_t endaddr = pc537_table[0]->endaddr, startaddr= 0;
	int err = 0;
	char tmp;

#if PC537_DEBUG > 2
	printk("pc537_write entered, count: %i, pos: %lu.\n", write_count, file->f_pos);
#endif
	/* Prevent attempts to write to EPROM memory */
	if (endaddr > 0x3FFF) {
		switch (pc537_table[0]->memtype) {
			case 0:
				/* RAM low and high, do nothing */
				break;
			case 1:
				/* EPROM low, RAM high */
				startaddr = 0x9000;
				break;
			case 2:
				/* RAM low, EPROM high */
				endaddr = 0x7fff;
				break;
			case 3:
				/* EPROM Only */
				endaddr = 0;
				startaddr= 0x10000;
				break;
			default:
				printk("pc537_write: internal error.\n");
				return -EIO;
				break;
		}
	}
	
	if (write_count > endaddr - file->f_pos +1) {
		write_count = endaddr - file->f_pos +1;
	}

#if PC537_DEBUG > 4
	printk("f_pos: %lu, startaddr: %ld, endaddr: %ld, memtype: %d.\n",
		(long unsigned int)file->f_pos, startaddr, endaddr, pc537_table[0]->memtype);
#endif
	if (file->f_pos < startaddr) {
		return -EINVAL;
	}
		
	/* Make sure buf is safe to read from */
	if (write_count) {
		err = verify_area(VERIFY_READ, buf, write_count);
		if (err) {
#if PC537_DEBUG > 1
	printk("pc537_write: verify_area failed.\n");
#endif
			return err;
		}
	} else if (count) {
		return -EINVAL;
	}

	while (copied < write_count) {
		low_addr  = (int)(file->f_pos +copied)% 0x100;
/*			    ^^^^^
 *	This cast taken from insmod's drv_hello driver to avoid
 * 	the dreaded "__moddi3" messages. Should be ok for 
 *	f_pos < 2^32.
 */ 
		outb_p(low_addr, pc537_table[0]->base);
		high_addr = (int)(file->f_pos +copied)/ 0x100;
/*			    ^^^^^
 *	Same as above to avoid "__divdi3" message.
 */ 
		outb_p(high_addr,pc537_table[0]->base +1);
		tmp = get_fs_byte(buf +copied++);
		outb_p(tmp, pc537_table[0]->base +2);
	}
/*
 *	Note that writing to the device resets it.
 *	In order to start it you have to call the PC537_RESTART ioctl.
 */
	
	file->f_pos += copied;
#if PC537_DEBUG > 3
	printk("pc537_write finished, copied: %lu, current pos: %lu.\n", copied, file->f_pos);
#endif
	return copied;
}


/*
 * pc537_ioctl 
 * No error checking so far.
 */
static int
pc537_ioctl(struct inode *inode, struct file * file, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	unsigned char c = 0;
	struct irqbuf* iptr;

	switch(cmd) {
		case PC537_RDIBUF:
			ret = verify_area(VERIFY_READ, (void *) arg, sizeof(struct readbuf));
			if (ret) {
				return ret;
			}
			memcpy_fromfs((void *)arg, rptr, sizeof(struct readbuf));

			ret = verify_area(VERIFY_WRITE, (void *) arg, sizeof(struct readbuf));
			if (ret) {
				return ret;
			}

			iptr = &(pc537_table[0]->ibuf);
			/* The following has to be atomic. */
			cli();
			if (0 == rptr->icnt || iptr->icnt == rptr->icnt) {
				rptr->val = iptr->bptr[(iptr->icnt-1) % PC537_IBUFSIZE];
				rptr->icnt= iptr->icnt;
				pc537_table[0]->ready = FALSE;
			} else if (rptr->icnt < iptr->icnt) {
				/* val is no longer held in the circular intr buffer */
				if ((iptr->icnt - rptr->icnt) >= PC537_IBUFSIZE) {  
					sti();
					return -EIO;
				} else {
					rptr->val = iptr->bptr[(rptr->icnt-1) % PC537_IBUFSIZE];
				}
			} else {
				/* requested intr has not occured yet */
				sti();
				return -EIO;
			}
			sti();

			memcpy_tofs((void *)arg, rptr, sizeof(struct readbuf));
			break;
		case PC537_RDCPORT:
			ret = verify_area(VERIFY_WRITE, (void *) arg, sizeof(unsigned char));
			if (ret) {
				return ret;
			}
			c = inb_p(pc537_table[0]->base +3);
#if PC537_DEBUG > 3
	printk("pc537_ioctl read from C-port: %u.\n", c);
#endif
			put_user(c, (unsigned char *)arg);
			pc537_table[0]->ready = FALSE;
			break;
		case PC537_WRCPORT:
#if PC537_DEBUG > 3
	printk("pc537_ioctl write to C-port: %u.\n", c);
#endif
			c = (unsigned char)arg;
			outb_p(c, pc537_table[0]->base +3);
			break;
		case PC537_RESTART:
			inb_p(pc537_table[0]->base +3);
			break;
		case PC537_RESET:
			outb_p((unsigned char)0x0, pc537_table[0]->base);
			break;
		case PC537_MCINT0:
			inb_p(pc537_table[0]->base +0);
			break;
		case PC537_MCINT1:
			inb_p(pc537_table[0]->base +1);
			break;
		default:
			return -EINVAL;
	}
	return ret;
				
}


/*
 *	pc537_select
 */
static int pc537_select(struct inode *inode, struct file *file, int sel_type, select_table *wait)
{
	int ret = 0;
#if PC537_DEBUG > 2
	printk("pc537_select entered with sel_type %d and wait %x.\n", sel_type, (unsigned int) wait);
#endif
	if (SEL_IN == sel_type) {
#ifdef 0
		PC537_INT_OFF();
#endif
		if (pc537_table[0]->ready) {
			ret = 1;
		} else  {
			select_wait(&(pc537_table[0]->wait), wait);
		}
#ifdef 0
		PC537_INT_ON();
#endif
	}
#if PC537_DEBUG > 2
	printk("pc537_select returning %d.\n", ret);
#endif
	return ret;
}


/*
 * 	pc537_interrupt
 *	PC537_IBUFSIZE interrupts will be written to a circular buffer.
 */
static void pc537_interrupt(int irq, struct pt_regs* regs)
{
	/*
	 * This handler assumes that open has requested only
	 * one IRQ.
	 */

	struct irqbuf* iptr = &(pc537_table[0]->ibuf);
	int bpos, res;
	pid_t pid;

	/*
	 * I should not printk from a fast interrupt handler.
	 */
#if PC537_DEBUG > 2
	printk("pc537_interrupt %lu entered with irq %d.\n", iptr->icnt +1, irq);
#endif
	if (pc537_table[0]->irq != irq) {
		printk("pc537_interrupt called with wrong irq %d.\n", irq);
		return;
	}

	bpos = (iptr->icnt) % PC537_IBUFSIZE;
	iptr->icnt++;
	iptr->bptr[bpos] = inb_p(pc537_table[0]->base +3);
	pid  = pc537_table[0]->pid;
	if (pid) {
		res = kill_proc(pid, SIGUSR1, 1);
#if PC537_DEBUG > 3
	printk("pc537_interrupt signalled process pid %d with result %d.\n", pid, res);
#endif
	} else {
		printk("pc537_interrupt: internal error - trying to send signal to init process.\n") ;
	}
	pc537_table[0]->ready = TRUE;
}


/*
 * MOD_INC_USE_COUNT makes sure that the driver memory is not freed
 * while the device is in use.
 */
static int
pc537_open( struct inode* ino, struct file* filep)
{
	int result = 0, minor = MINOR(ino->i_rdev);
	struct pc537_struct* pptr = NULL;
	char* bptr = NULL;

#if PC537_DEBUG > 3
	printk("pc537_open: opened by proc pid %d.\n", current->pid);
#endif

	/* 
	 *	Until we are set up for multiple opens, just 
   	 * 	refuse to open a second time.
  	 */
	if (MOD_IN_USE) {
		printk("pc537_open: Just one open on device allowed.\n");
		return -EBUSY;
	}

	MOD_INC_USE_COUNT;
	if (!(pptr = (void *)kmalloc(sizeof(struct pc537_struct), GFP_KERNEL))) {	
		printk("pc537_open: kmalloc() failed.\n");
		return -ENOMEM;
	}
	if (!(bptr = (void *)kmalloc(PC537_IBUFSIZE, GFP_KERNEL))) {	
		printk("pc537_open: kmalloc() failed.\n");
		return -ENOMEM;
	}
	if (!(rptr = (void *)kmalloc(sizeof(struct readbuf), GFP_KERNEL))) {	
		printk("pc537_open: kmalloc() failed.\n");
		return -ENOMEM;
	}

	/*
         * 	When we will deal with multiple cards,
	 * 	this scheme has to be replaced.
	 */
	pptr->irq	= pc537_irqs[minor % 4];
	pptr->base	= pc537_bases[(minor >> 2) % 4];
	pptr->endaddr   = (minor & 0x40) ? 0x3FFF : 0xFFFF;
	pptr->memtype	= (minor >> 4) % 8;
	pptr->pid 	= current->pid;
	pptr->ready	= FALSE;
	pptr->wait	= NULL;
	pptr->ibuf.bptr = bptr;
	/* this instr is obsolete now and might be removed */
	pptr->ibuf.bpos = PC537_IBUFSIZE -1;
	pptr->ibuf.icnt = 0;
	pc537_table[0] 	= pptr;
#if PC537_DEBUG > 3
	printk("pc537_open: endaddr %ld memtype: %d.\n",
		pc537_table[0]->endaddr, pc537_table[0]->memtype);
#endif

	if (check_region(pc537_table[0]->base, PC537_IOPORTS)) {
		printk("pc537_open: ioports %u through %u already in use.\n", pc537_table[0]->base, pc537_table[0]->base + PC537_IOPORTS -1);
		MOD_DEC_USE_COUNT;
		return -EBUSY;
	}
	request_region(pc537_table[0]->base, PC537_IOPORTS, "pc537");
	
	if ((result = request_irq(pc537_table[0]->irq, pc537_interrupt, SA_INTERRUPT, "pc537"))) {
		switch (result) {
			case -EBUSY:
				printk("pc537_open: IRQ %d already in use.\n", pc537_table[0]->irq);
				break;
			default:
				printk("pc537_open: internal error during IRQ allocation.\n");
				break;
		}
		return result;
	}

#if PC537_DEBUG > 2
	printk("pc537_open: opened by proc pid %d.\n", pc537_table[0]->pid);
#endif

	return 0;   
}


/*
 * Now decrement the use count.
 */
static void
pc537_close( struct inode* ino, struct file* filep)
{

#if PC537_DEBUG > 2
	printk("pc537_close: closing on behalf of proc pid %d.\n", pc537_table[0]->pid);
#endif
	free_irq(pc537_table[0]->irq);
	release_region(pc537_table[0]->base, PC537_IOPORTS);
	kfree((pc537_table[0]->ibuf).bptr);
	kfree(pc537_table[0]);
	MOD_DEC_USE_COUNT;
}

static struct file_operations pc537_fops = {
	pc537_lseek,	
	pc537_read,
	pc537_write,
	NULL,		/* readdir */
	pc537_select,
	pc537_ioctl,
	NULL,		/* mmap */
	pc537_open,
	pc537_close,
	NULL		/* fsync */
};


/*
 * And now the modules code and kernel interface.
 */



#ifdef __cplusplus
extern "C" {
#endif

#ifdef MODULE 
int
init_module( void)
{
	if (register_chrdev(major, "pc537", &pc537_fops)) {
		printk("pc537: unable to get major %d.\n", major);
		return -EIO;
	}
	printk("pc537: driver registered on major #%d.\n", major);
	return 0;
}

void
cleanup_module(void)
{

	if (MOD_IN_USE)
		printk("pc537: device busy, remove delayed.\n");

	if (unregister_chrdev(major, "pc537") != 0)
		printk("pc537: cleanup_module failed.\n");
	else
		printk("pc537: module released.\n");
}
#endif

#ifdef __cplusplus
}
#endif
