/******************************************************************************
 *
 *	i8255 Loadable Module for Linux
 *
 *
 * $Id: i8255.c,v 1.9 1998/10/07 21:26:10 sanne Exp $
 *
 * Copyright (c) 1998 by Sanne Graaf (sanne@mmm.xs4all.nl). 
 *   All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/
#ifndef __KERNEL__
   #define __KERNEL__
#endif
#ifndef MODULE
   #define MODULE
#endif

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/module.h>
#include <linux/major.h>
#include <linux/mm.h>

#include <asm/io.h>
#include <asm/segment.h>
#include <asm/system.h>
#include <asm/irq.h>

#include <linux/proc_fs.h>
#include <linux/stat.h>

#include "i8255.h"

static char CVS_ID[] = "$Id: i8255.c,v 1.9 1998/10/07 21:26:10 sanne Exp $";

// Command line parameters.

static int ioport[MAX_DEVICES];     	/* ioport commandline values	*/
static int irq[MAX_DEVICES];     	/* irq commandline values	*/
static int exclusive[MAX_DEVICES];  	/* single or multi usage values */
static int debug = -1;        		/* debug commandline setting	*/
static int slowio = -1;       		/* fast or slow IO syscalls	*/
static int proc = 1;			/* do /proc stuff or not,def=y	*/

// Global switches

static int num_devices = 0;      	/* total number of i8255's used */
static int debug8255 = 0;     		/* global driver debug value	*/
static int slow8255 = 0;      		/* global io speed value	*/

#define my_printk(level, format, args...) if (debug8255>=level) printk(DEV_NAME ": " format, ## args)

/******************************************************************************
 * IO read routine
 *
 * Reads one byte from ioport
 *
 * Args: ioport -> ioport to read from
 *
 * Return: byte read
 */

u_char ioread(int ioport)
{
   u_char byte_read;

   if (slow8255)
      byte_read = inb_p(ioport);
   else
      byte_read = inb(ioport);

   my_printk(3, "ioread-> I/O Port 0x%X -> 0x%X\n", ioport, byte_read);
   return byte_read;
}


/******************************************************************************
 * IO write routine
 *
 * Writes one byte to ioport
 *
 * Args: outbyte -> byte to send
 *       ioport  -> ioport to send to
 *
 * Return: always 0
 */

u_char iowrite(int ioport, u_char outbyte)
{
   my_printk(3, "iowrite-> I/O Port 0x%X -> 0x%X\n", ioport, outbyte);

   if (slow8255)
      outb_p(outbyte, ioport);
   else
      outb(outbyte, ioport);

   return (0);
}


/******************************************************************************
 * Set one bit
 *
 * Sets one bit within a byte
 *
 * Args: reg -> byte to set bit in
 *	 bit -> bitmask
 *	 value -> local copy of register
 *
 * Return: new register value
 *
 */

u_char set_onebit(int reg, u_char bit, u_char value)
{
   u_char val;

   val = value;
   my_printk(2, "set_onebit-> origional val: 0x%X (mask 0x%X)\n", val, bit);
   val |= ~bit;
   my_printk(2, "set_onebit-> new val: 0x%X\n", val);
   iowrite(reg, val);
   return (val);
}


/*****************************************************************************
 * Clear one bit
 *
 * Clear's a bit in a byte
 *
 * Args: reg -> byte to clear bit in
 *	 bit -> bitmask
 *	 value -> local copy of register
 *
 * Return: new register value
 *
 */

u_char clr_onebit(int reg, u_char bit, u_char value)
{
   u_char val;

   val = value;
   my_printk(2, "clr_onebit-> origional val: 0x%X (mask 0x%X)\n", val, bit);
   val &= bit;
   my_printk(2, "clr_onebit-> new val: 0x%X\n", val);
   iowrite(reg, val);
   return (val);
}


/******************************************************************************
 * Read from device
 *
 * routine to handle things like 'cat /dev/i8255x' ...
 *
 * Args: inode -> inode structure
 *	 file -> file structure
 *	 buffer -> buffer voor gelezen data
 *	 count -> aantal te lezen bytes
 *
 * Return: data from buffer in user-space allocated memory
 *
 */

static int i8255_read(struct inode *inode, struct file *file, char *buffer,
                      int count)
{
   int offset, i;

   my_printk(1, "i8255_read-> Request for %d bytes, minor: %d\n",
             count, MINOR(inode->i_rdev));
   if (count<=0) return -EINVAL;
   if (count > I8255_PORTNUM) count = I8255_PORTNUM;

   for (i=0; i < I8255_PORTNUM; i++)
      i8255_dev[MINOR(inode->i_rdev)].data_regs[i] =
      ioread(i8255_dev[MINOR(inode->i_rdev)].iobase+i);

   my_printk(2, "i8255_read-> Bytes: A:0x%X B:0x%X C:0x%X D:0x%X\n",
             i8255_dev[MINOR(inode->i_rdev)].data_regs[0],
             i8255_dev[MINOR(inode->i_rdev)].data_regs[1],
             i8255_dev[MINOR(inode->i_rdev)].data_regs[2],
             i8255_dev[MINOR(inode->i_rdev)].data_regs[3]);

   for (offset=0; offset<count; offset++)
      put_fs_byte(i8255_dev[MINOR(inode->i_rdev)].data_regs[offset],
                  buffer+offset);
   return offset;
}


/******************************************************************************
 * Write to device
 *
 * routine to handle things like 'cat [whatever] > /dev/i8255' ...
 *
 * Args: inode -> inode structure
 *	 file -> file structure
 *	 buffer -> buffer voor te schrijven data
 *	 count -> aantal te schrijven bytes
 *
 * Return: data from buffer in user-space allocated memory
 *
 */

static int i8255_write(struct inode *inode, struct file*file,
                       const char *buffer, int count)
{

   int orig_count, offset, i;

   my_printk(1, "i8255_write-> Request for %d bytes\n", count);
   orig_count = count;
   if (count > I8255_PORTNUM) count = I8255_PORTNUM;
   for (offset=0; offset<count; offset++) {
      i8255_dev[MINOR(inode->i_rdev)].data_regs[offset] =
      get_fs_byte(buffer+offset);
      iowrite(i8255_dev[MINOR(inode->i_rdev)].iobase+offset,
              i8255_dev[MINOR(inode->i_rdev)].data_regs[offset]);
   }
   my_printk(2, "i8255_write-> Bytes: A:0x%X B:0x%X C:0x%X D:0x%X\n",
             i8255_dev[MINOR(inode->i_rdev)].data_regs[0],
             i8255_dev[MINOR(inode->i_rdev)].data_regs[1],
             i8255_dev[MINOR(inode->i_rdev)].data_regs[2],
             i8255_dev[MINOR(inode->i_rdev)].data_regs[3]);
   return orig_count;
}

/******************************************************************************
 * Open and Lock device
 *
 * Orig: Routine to prevent more filehandles to be opened to the device
 * Now: only counts devices who use the device
 *
 * Args: inode -> inode structure
 * 	 file -> file structure
 *
 * Return: nothing (0).
 *
 */

static int i8255_open(struct inode *inode, struct file *file)
{
   my_printk(1, "i8255_open-> Request, minor: %d\n",
             MINOR(inode->i_rdev));
   if ((i8255_dev[MINOR(inode->i_rdev)].i8255stat & I8255_USED) &&
       (i8255_dev[MINOR(inode->i_rdev)].exclusive == 1)) {
      my_printk(1, "i8255_open-> Request on exclusive device, minor: %d\n",
                MINOR(inode->i_rdev));
      return -EBUSY;
   }
   MOD_INC_USE_COUNT;
   i8255_dev[MINOR(inode->i_rdev)].i8255stat |= I8255_USED;
   i8255_dev[MINOR(inode->i_rdev)].usenum++;
   my_printk(2, "i8255_open-> Opened %d times, minor: %d\n",
             i8255_dev[MINOR(inode->i_rdev)].usenum,
             MINOR(inode->i_rdev));
   return 0;
}


/******************************************************************************
 * Unlock device
 *
 * Routine to unlock device / decrement number of users
 *
 * Args: inode -> inode structure
 * 	 file -> file structure
 *
 * Return: nothing (0).
 *
 */

static void i8255_release(struct inode *inode, struct file *file)
{
   my_printk(1, "i8255_release-> Request, minor: %d\n",
             MINOR(inode->i_rdev));
   MOD_DEC_USE_COUNT;
   i8255_dev[MINOR(inode->i_rdev)].usenum--;
   if (i8255_dev[MINOR(inode->i_rdev)].usenum == 0)
      i8255_dev[MINOR(inode->i_rdev)].i8255stat &= ~I8255_USED;
}


/******************************************************************************
 * ioctl function
 *
 * Command based ioctl routine. This will do all the work to/from the 8255
 * 
 * Args: inode -> inode structure
 *	 file -> file structure
 *	 cmd -> command to execute (read/write/clear/set/reset)
 *	 arg -> argument, is used as pointer to i8255_struct
 *
 */

static int i8255_ioctl(struct inode *inode, struct file *file,
                       u_int cmd, u_long arg)
{
   int rc, i, temp, portnum;
   struct i8255_struct request;

   my_printk(1, "i8255_ioctl-> func=0x%X, arg=0x%X, minor=%d\n",
             cmd, (int) arg, MINOR(inode->i_rdev));

   if (cmd > I8255_TEST) {
      rc = verify_area(VERIFY_READ, (struct request*) arg, sizeof(request));
      if (rc)
         return rc;
      memcpy_fromfs(&request, (struct i8255_struct*) arg, sizeof(request));
      portnum = i8255_dev[MINOR(inode->i_rdev)].iobase + request.regnum;
      if ((portnum < i8255_dev[MINOR(inode->i_rdev)].iobase) ||
          (portnum > i8255_dev[MINOR(inode->i_rdev)].iobase+I8255_PORTNUM-1)) {
         my_printk(0, "i8255_ioctl -> ERROR: Wrong IOport address: 0x%X, minor: %d\n",
                portnum, MINOR(inode->i_rdev));
         return -EINVAL;
      }
   }

   switch (cmd) {
      case I8255_READ:            		/* Read register 	*/
	  i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum] =
	      ioread(portnum);
          return (i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum]);

      case I8255_WRITE:           		/* Write register 	*/
         iowrite(portnum, request.value);
         i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum] = 
         request.value;
         break;

      case I8255_SETBIT:			/* Sets one bit 	*/
         i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum] = 
         set_onebit(portnum, request.bitno,
                    i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum]);
         break;

      case I8255_CLRBIT:			/* Clears one bit 	*/
         i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum] = 
         clr_onebit(portnum, request.bitno,
                    i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum]);
         break;

      case I8255_READBIT:			/* Reads one bit 	*/
         temp = 0;
         i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum] =
         ioread(portnum);
         temp = i8255_dev[MINOR(inode->i_rdev)].data_regs[request.regnum]
                & ~request.bitno;     
         my_printk(2, "i8255_ioctl -> READBIT -> 0x%X\n", temp);
         if (temp)
            return (1);
         else
            return (0);
         break;

      case I8255_DEBUG:
               if ((arg > 0) && (arg < 4))
                  debug8255 = arg;
               else
                  debug8255 = 0;
	       my_printk(0, "ioctl -> new DEBUG level: %d\n", debug8255);          
         break; 

      case I8255_EXCL:
	       if (arg != 0)
    	           i8255_dev[MINOR(inode->i_rdev)].exclusive = 1;
	       else
	           i8255_dev[MINOR(inode->i_rdev)].exclusive = 0;
	 break;
	 
      case I8255_RESET:                 	/* Init 8255, all input	*/

#ifdef GENERIC_I8255
         iowrite(I8255_D0(i8255_dev[MINOR(inode->i_rdev)].iobase), 0x9B);
         i8255_dev[MINOR(inode->i_rdev)].data_regs[REG_D0] = 0x9B;
#endif
#ifdef ADVANTECH_PCL731
         iowrite(I8255_D0(i8255_dev[MINOR(inode->i_rdev)].iobase), 0x9B);
         i8255_dev[MINOR(inode->i_rdev)].data_regs[REG_D0] = 0x9B;
         iowrite(I8255_D1(i8255_dev[MINOR(inode->i_rdev)].iobase), 0x9B);
         i8255_dev[MINOR(inode->i_rdev)].data_regs[REG_D1] = 0x9B;
#endif         
#ifdef ADVANTECH_PCM3724
         iowrite(I8255_D0(i8255_dev[MINOR(inode->i_rdev)].iobase), 0x9B);
         i8255_dev[MINOR(inode->i_rdev)].data_regs[REG_D0] = 0x9B;
         iowrite(I8255_D1(i8255_dev[MINOR(inode->i_rdev)].iobase), 0x9B);
         i8255_dev[MINOR(inode->i_rdev)].data_regs[REG_D1] = 0x9B;
         iowrite(I8255_DIR(i8255_dev[MINOR(inode->i_rdev)].iobase), 0x00);
         i8255_dev[MINOR(inode->i_rdev)].data_regs[REG_DIR] = 0x00;
         iowrite(I8255_GATE(i8255_dev[MINOR(inode->i_rdev)].iobase), 0x00);
         i8255_dev[MINOR(inode->i_rdev)].data_regs[REG_GATE] = 0x00;
#endif         

         for (i=0; i < I8255_PORTNUM; i++)
            i8255_dev[MINOR(inode->i_rdev)].data_regs[i] =
		ioread(i8255_dev[MINOR(inode->i_rdev)].iobase+i);
	    
	 break;
      default:                          	/* invalid function number */
         return -EINVAL;
   }
   return 0;
}


/******************************************************************************
 * I8255 Interrupt routine
 *
 */

static void i8255_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
   struct i8255_dev *dev = (struct i8255_dev *) dev_id;
   int i;
   static int isr_active = 0;

   my_printk(1, "Interrupt: %d\n", irq);
   if (isr_active) {
      my_printk(2, "!!__Nested Interrupt__!!.\n");
      return;
   }
   isr_active++;      
   if (dev == NULL) {
      my_printk(0, "Interrupt from unknown device.\n");
   }
   for (i=0; i < num_devices; i++) {
      my_printk(1, "port: %X\n", i8255_dev[i].iobase);
      if (i8255_dev[i].irq == irq) {
         my_printk(1, "Got Interrupt from device %d, io 0x%X.\n", i, i8255_dev[i].iobase);
	 i8255_dev[i].data_regs[REG_C0] = ioread(i8255_dev[i].iobase+REG_C0);
      }
   }
   wake_up_interruptible(&i8255_wait);
   isr_active--;
}


/******************************************************************************
 * Static device init
 *
 * Init routine for kernel module, grabs I/O range and if used interrupt
 *
 */

int i8255_init(void)
{
   int i, j;

   if (register_chrdev(I8255_MAJOR, DEV_NAME, &i8255_fops)) {
      my_printk(0, "Cannot register to major device: %d\n",
             I8255_MAJOR);
      return -EIO;
   }
   my_printk(1, "Registered to major device %d.\n", I8255_MAJOR);

   if (proc > 0) {
       i8255_proc_entry.namelen = strlen(DEV_NAME);
       i8255_proc_entry.name = DEV_NAME;
       j = proc_register_dynamic(&proc_root, &i8255_proc_entry);
       my_printk(1, "Registered /proc/%s, ret: %d, num: %d\n", DEV_NAME,j, i8255_proc_entry.low_ino);
   }

   for (i = 0; i < num_devices; i++) {
      i8255_dev[i].iobase = ioport[i];
      i8255_dev[i].irq = irq[i];
      i8255_dev[i].i8255stat = 0;
      i8255_dev[i].usenum = 0;
      i8255_dev[i].exclusive = exclusive[i];

      if (check_region(i8255_dev[i].iobase, I8255_PORTNUM) < 0) {
         my_printk(0, "I/O PORT region (0x%X) is taken by other device\n",
                i8255_dev[i].iobase);
         continue;
      } else i8255_dev[i].i8255stat |= I8255_PORTOK;

      request_region(i8255_dev[i].iobase, I8255_PORTNUM, DEV_NAME);

      my_printk(2, "i8255_init-> Request I/O ports (0x%x - 0x%x), exclusive: %d.\n",
                i8255_dev[i].iobase, i8255_dev[i].iobase+I8255_PORTNUM-1,
                i8255_dev[i].exclusive);

      if (i8255_dev[i].irq > 0) {
         if (request_irq(i8255_dev[i].irq, i8255_interrupt, 0, DEV_NAME, i8255_dev)) {
            my_printk(0, "IRQ %d taken by other device.\n",
                   i8255_dev[i].irq);
            continue;
         } else i8255_dev[i].i8255stat |= I8255_IRQOK;

         my_printk(1, "i8255_init-> Request IRQ %d\n", i8255_dev[i].irq);
      }
   }
   return 0;
}


/******************************************************************************
 * Register and init dynamic module
 *
 */

int init_module(void)
{
    int errnum=0;

   if ((debug > 0) && (debug < 4))
      debug8255 = debug;
   else
      debug8255 = 0;

   if (slowio > 0)
      slow8255 = 1;

   my_printk(0, "Generic 8255 Kernel module v%s loaded.\n", I8255_VERSION);
   my_printk(1, "$Id: i8255.c,v 1.9 1998/10/07 21:26:10 sanne Exp $.\n");
   my_printk(0, "by Sanne Graaf, <sanne@mmm.xs4all.nl>.\n");
   my_printk(1, "Debug level: %d.\n", debug8255);
   if (slow8255)
      my_printk(0, "Slow IO mode.\n");

   while (ioport[num_devices] > 0) {
      if (num_devices >= MAX_DEVICES) {
         my_printk(0, "Init 8255, to many devices.\n");
	 break;
      }
      my_printk(1, "8255 num: %d, BasePort: 0x%X, IRQ: %d\n",
                num_devices, ioport[num_devices], irq[num_devices]);
      num_devices++;    
   }
   my_printk(1,"Start Init 8255's, total: %d\n", num_devices);
   
   errnum = i8255_init();
   if (errnum != 0) {
       my_printk(0,"Error(s) during Init 8255's: %d\n", errnum);
   }
   return 0;
}


/******************************************************************************
 * Unregister device dynamic module and cleanup.
 *
 */

void cleanup_module(void)
{
   int i = 0, j;

   unregister_chrdev(I8255_MAJOR, DEV_NAME);
   while (i < num_devices) {
      if (i8255_dev[i].i8255stat & I8255_PORTOK) {
          release_region(i8255_dev[i].iobase, I8255_PORTNUM);
	  my_printk(1, "Released IObase 0x%02X-0x%02X\n",
	       i8255_dev[i].iobase, i8255_dev[i].iobase+I8255_PORTNUM);
      }
      if ((irq[i] != 0) && (i8255_dev[i].i8255stat & I8255_IRQOK)) {
	my_printk(1, "Freeing IRQ %d.\n", i8255_dev[i].irq);
        free_irq(i8255_dev[i].irq, i8255_dev);
      }
      my_printk(1, "cleanup_module-> dev: %d, Released Baseport: 0x%02X, IRQ: %d\n",
                i, i8255_dev[i].iobase, i8255_dev[i].irq);
      i++;
   }

   if (proc > 0) {
       j = proc_unregister(&proc_root, i8255_proc_entry.low_ino);
       my_printk(1, "Unregistered /proc/%s, ret: %d, num: %d\n", DEV_NAME, j, i8255_proc_entry.low_ino);
   }

   my_printk(0, "Module unloaded.\n");
}


/******************************************************************************
 * /proc info routines.
 *
 */

int i8255_get_info(char *buf, char **start, off_t fpos, int length, int dummy)
{
    char *p;
    int i;

    p = buf;

    p += sprintf(p, "Type      : %s\nNumdevices: %d\nDebug     : %d\nSlow I/O  : %d\nMajor     : %d\n\n", 
		 DEV_NAME, num_devices, debug8255, slow8255, I8255_MAJOR);

#ifdef GENERIC_I8255
    p += sprintf(p, "Num IOBase IRQ X Use Stat [A0] [B0] [C0] [D0]\n");
    for (i=0; i < num_devices; i++) {
    p += sprintf(p, "%3d  0x%03X  %02d %d   %d 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X\n",i, i8255_dev[i].iobase, i8255_dev[i].irq,
		i8255_dev[i].exclusive, i8255_dev[i].usenum, i8255_dev[i].i8255stat,
		i8255_dev[i].data_regs[REG_A0],	i8255_dev[i].data_regs[REG_B0],
		i8255_dev[i].data_regs[REG_C0],	i8255_dev[i].data_regs[REG_D0]);
    }
#endif
#ifdef ADVANTECH_PCL731
    p += sprintf(p, "Num IOBase IRQ X Use Stat [A0] [B0] [C0] [D0] [A1] [B1] [C1] [D1]\n");
    for (i=0; i < num_devices; i++) {
    p += sprintf(p, "%3d  0x%03X  %02d %d   %d 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X\n",i, i8255_dev[i].iobase, i8255_dev[i].irq,
		i8255_dev[i].exclusive, i8255_dev[i].usenum, i8255_dev[i].i8255stat,
		i8255_dev[i].data_regs[REG_A0],	i8255_dev[i].data_regs[REG_B0],
		i8255_dev[i].data_regs[REG_C0],	i8255_dev[i].data_regs[REG_D0],
		i8255_dev[i].data_regs[REG_A1],	i8255_dev[i].data_regs[REG_B1],
		i8255_dev[i].data_regs[REG_C1],	i8255_dev[i].data_regs[REG_D1]);
    }
#endif
#ifdef ADVANTECH_PCM3724
    p += sprintf(p, "Num IOBase IRQ X Use Stat [A0] [B0] [C0] [D0] [A1] [B1] [C1] [D1] [DR] [GT]\n");
    for (i=0; i < num_devices; i++) {
    p += sprintf(p, "%3d  0x%03X  %02d %d   %d 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X 0x%02X\n",i, i8255_dev[i].iobase, i8255_dev[i].irq,
		i8255_dev[i].exclusive, i8255_dev[i].usenum, i8255_dev[i].i8255stat,
		i8255_dev[i].data_regs[REG_A0],	i8255_dev[i].data_regs[REG_B0],
		i8255_dev[i].data_regs[REG_C0],	i8255_dev[i].data_regs[REG_D0],
		i8255_dev[i].data_regs[REG_A1],	i8255_dev[i].data_regs[REG_B1],
		i8255_dev[i].data_regs[REG_C1],	i8255_dev[i].data_regs[REG_D1],
		i8255_dev[i].data_regs[REG_DIR], i8255_dev[i].data_regs[REG_GATE]);
    }
#endif

    p += sprintf(p, "\n");
    return p - buf;
};
