#ifndef __HEPC3_H__
#define __HEPC3_H__
/*
 * hepc3 --- a Linux kernel driver module for the Hunt Engineering HEPC3,
 *           HEPC4 and HECPCI1 [C]PCI TIM-40 motherboards
 *
 * This driver is
 *
 *                     Copyright (C) 1999-2000 by Albrecht Dre.
 *
 * This driver is free software, and you are welcome to redistribute it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * The driver is distributed in the hope that it will be useful and work with
 * your system's configuration.
 * Due to the fact that the driver is free, it is provided WITHOUT ANY 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. Please see the GNU General Public License (GPL)
 * for details. You should have received a copy of the GPL with this package;
 * if not, write to the Free Software Foundation, Inc., 675 Mass Ave., 
 * Cambride, MA 02139, USA
 *
 * Please report bugs or suggestions to ad@mpifr-bonn.mpg.de or send mail to:
 * Albrecht Dre, Max-Planck-Institut fr Radioastronomie, Box 2024, 
 * 53010 Bonn, GERMANY
 *
 * The development of this driver module was based upon the following
 * information:
 *
 * o Jim Partan's driver module "hep3_devel.[hc]; unfortunately this
 *   driver did not work with the HECPCI1, but LARGE parts are actually
 *   taken from it...
 *
 * o Hunt Engineering: HECPCI1 3U CompactPCI TIM-40 Motherboard: USER
 *   MANUAL. Hardware Rev A, Cocument Rev B, P.Warnes, 16/02/98. (see
 *   http://www.hunteng.co.uk)
 * 
 * o AMCC: PCI Products Data Book, Section 3: S5933 PCI Controller, 1998.
 *
 * o Alessandro Rubini: Linux Device Drivers. O'Reilly & Associates, 1998.
 *   (I used the german translation)
 *
 * o Ori Pomerantz: Linux Kernel Module Programming Guide, 1999. (for 2.0 to
 *   2.2 migration)
 *
 * Albrecht Dre, Jul 1, 1999.
 *
 *
 * Changes in Version 1.1.0, of Jan 23, 2001 (Albrecht Dre)
 * =========================================================
 * 
 * o fixed bug in port A/interrupt driven mode which could cause a hang under
 *   extreme conditions
 *
 * o rewritten port A dma/pci bus mastering stuff for more performance and
 *   stability
 */

#include <linux/ioctl.h>

/* request numbers for ioctl(2) calls */
#define HEPC3_IOC_MAGIC  0x42                         /* ;-)                  */
#define HEPC3_IO_B_RES   _IO(HEPC3_IOC_MAGIC,0)       /* reset board          */
#define HEPC3_IO_P_RES   _IO(HEPC3_IOC_MAGIC,1)       /* reset port           */
#define HEPC3_IO_G_ISTAT _IOR(HEPC3_IOC_MAGIC,2,int)  /* get input status     */
#define HEPC3_IO_G_OSTAT _IOR(HEPC3_IOC_MAGIC,3,int)  /* get output status    */
#define HEPC3_IO_G_CNFG  _IOR(HEPC3_IOC_MAGIC,4,int)  /* get config status    */
#define HEPC3_IO_S_SWAP  _IOW(HEPC3_IOC_MAGIC,5,int)  /* set swap mode        */
#define HEPC3_IO_G_SWAP  _IOR(HEPC3_IOC_MAGIC,6,int)  /* get swap mode        */

/* ----------- user programs will not see anything below this line ---------- */
#ifdef __KERNEL__

#include <linux/pci.h>

#define HEPC3_VENDOR     PCI_VENDOR_ID_AMCC                 /* AMCC           */
#define HEPC3_ID         PCI_DEVICE_ID_AMCC_S5933_HEPC3     /* HEPC3, HECPCI1 */

#define HEPC3_MAJOR      0            /* default: get "dynamic" major num.    */
#define HEPC3_NAME       "hepc3"      /* id string                            */

#define HEPC3_DRV_VER    "1.1.0"      /* driver version                       */

#define HEPC3_MAX_BOARD  8
#define HEPC3_MAX_NAME   8

#define HEPC3_REG_SIZE   64           /* size of io port regions 0 - 3        */
#define HEPC3_JTAG_SIZE  256          /* size of io port region 4 (JTAG)      */
#define HEPC3_BUF_SIZE0  1000         /* buffer size in dwords (Linux 2.0.x)  */
#define HEPC3_BUF_SIZE2  1024         /* buffer size in dwords (Linux 2.2.x)  */

#define HEPC3_RESET_MS   200          /* duration of reset in ms              */

/* --- registers of the AMCC S5933 MatchMaker ------------------------------- */
#define HEPC3_IMB4       0x1c         /* Incoming MailBox (fifo irq flags)    */
#define HEPC3_A_FIFO     0x20         /* comport A fifo                       */
#ifdef HEPC3_WITH_DMA
# define HEPC3_MWAR      0x24         /* Master Write Address Register        */
# define HEPC3_MWTC      0x28         /* Master Write Transfer Count          */
# define HEPC3_MRAR      0x2c         /* Master Read Address Register         */
# define HEPC3_MRTC      0x30         /* Master READ Transfer Count           */
#endif
#define HEPC3_INTCSR     0x38         /* Interrupt Control and Status Reg.    */
#define HEPC3_MCSR       0x3c         /* Master Control and Status Register   */

/* IMB4 status bits */
#define HEPC3_A_LGBIBIT  24           /* Latched-Gone-By IRQ on port A        */
#define HEPC3_A_LGBIRQ   (1 << HEPC3_A_LGBIBIT)
#define HEPC3_A_WRIBIT   25           /* write irq on port A                  */
#define HEPC3_A_WRIRQ    (1 << HEPC3_A_WRIBIT)
#define HEPC3_B_RDIBIT   28           /* 1+ word readable on port B IRQ       */
#define HEPC3_B_RDIRQ    (1 << HEPC3_B_RDIBIT)
#define HEPC3_B_RD60IBIT 27           /* 60+ words readable on port B IRQ     */
#define HEPC3_B_RD60IRQ  (1 << HEPC3_B_RD60IBIT)
#define HEPC3_B_WRIBIT   26           /* write irq on port B                  */
#define HEPC3_B_WRIRQ    (1 << HEPC3_B_WRIBIT)

/* INTCSR bits */
#define HEPC3_AMCC_INT   (1 << 23)    /* indicates an amcc irq                */
#define HEPC3_MBOX_IBIT  17           /* indicates or resets mailbox irq      */
#define HEPC3_INT_MBOX   (1 << HEPC3_MBOX_IBIT)
#define HEPC3_ENI_MBOX   0x1f00       /* enable mailbox interrupt             */
#ifdef HEPC3_WITH_DMA
# define HEPC3_RRDY_IBIT 19           /* indicates/resets read trf. complete  */
# define HEPC3_INT_RRDY  (1 << HEPC3_RRDY_IBIT)
# define HEPC3_WRDY_IBIT 18           /* indicates/resets write trf. complete */
# define HEPC3_INT_WRDY  (1 << HEPC3_WRDY_IBIT)
# define HEPC3_TABT_IBIT 21           /* indicates/resets target abort irq    */
# define HEPC3_INT_TABT  (1 << HEPC3_TABT_IBIT)
# define HEPC3_MABT_IBIT 20           /* indicates/resets master abort irq    */
# define HEPC3_INT_MABT  (1 << HEPC3_MABT_IBIT)
# define HEPC3_ENI_RRDY  (1 << 15)    /* enable read trf. complete interrupt  */
# define HEPC3_ENI_WRDY  (1 << 14)    /* enable write trf. complete interrupt */
#endif

/* MCSR status bits (read-only) */
#define HEPC3_AOUT_FULL  (1 << 0)     /* comport A out-fifo full              */
#define HEPC3_AOUT_4     (1 << 1)     /* comport A out-fifo 4+ words empty    */
#define HEPC3_AOUT_EMPTY (1 << 2)     /* comport A out-fifo empty             */
#define HEPC3_AIN_FULL   (1 << 3)     /* comport A in-fifo full               */
#define HEPC3_AIN_4      (1 << 4)     /* comport A in-fifo 4+ words full      */
#define HEPC3_AIN_EMPTY  (1 << 5)     /* comport A in-fifo empty              */
/* MCSR control bits (write-only) */
#define HEPC3_SYS_RESET  (1 << 24)    /* system reset line (active high)      */
#define HEPC3_AOUT_RES   (1 << 25)    /* comport A out-fifo reset             */
#define HEPC3_AIN_RES    (1 << 26)    /* comport A in-fifo reset              */
/* MCSR dma control bits (read/write) */
#ifdef HEPC3_WITH_DMA
# define HEPC3_DMA_READ  (1 << 14)    /* enable pci->comport dma              */
# define HEPC3_DMA_R4MIN (1 << 13)    /* DMA only if >= 4 empty FIFO elements */
# define HEPC3_DMA_RPRI  (1 << 12)    /* read has higher priority             */
# define HEPC3_DMA_WRITE (1 << 10)    /* enable comport->pci dma              */
# define HEPC3_DMA_W4MIN (1 << 9)     /* DMA only if >= 4 empty FIFO elements */
# define HEPC3_DMA_WPRI  (1 << 8)     /* write has higher priority            */
#endif

/* comport A control + status bits */
#define HEPC3_CPA_RESET  (1 << 0)     /* reset port A (active low)            */
#define HEPC3_CPA_LGBIRQ (1 << 1)     /* enable LGB IRQ (>= 1 word readable)  */
#define HEPC3_CPA_LGBCLR (1 << 2)     /* clear LGB                            */
#define HEPC3_CPA_OUTIRQ (1 << 3)     /* enable out-IRQ (>= 1 word writable)  */
#define HEPC3_CPA_ENDIAN (1 << 5)     /* 1 = big endian                       */
#define HEPC3_CPA_CONFIG (1 << 0)     /* status of config line (active low)   */

/* comport B control + status bits */
#define HEPC3_CPB_RESET  (1 << 0)     /* reset port A (active low)            */
#define HEPC3_CPB_OUTIRQ (1 << 1)     /* enable IRQ on >= 1 word writable     */
#define HEPC3_CPB_I60IRQ (1 << 2)     /* enable IRQ on >= 60 words readable   */
#define HEPC3_CPB_INIRQ  (1 << 3)     /* enable IRQ on >= 1 word readable     */
#define HEPC3_INTFREEZ   (1 << 4)     /* intfreez line                        */
#define HEPC3_CPB_ENDIAN (1 << 5)     /* 1 = big endian                       */
#define HEPC3_CPB_CONFIG (1 << 0)     /* status of config line (active low)   */
#define HEPC3_CPB_OUTAF  (1 << 1)     /* out-fifo almost full (active low)    */
#define HEPC3_CPB_OUTE   (1 << 2)     /* out-fifo empty (active low)          */
#define HEPC3_CPB_INAF   (1 << 3)     /* in-fifo almost full (active low)     */
#define HEPC3_CPB_INE    (1 << 4)     /* in-fifo  empty (active low)          */

/* data type to describe one of the two comports on a board */
typedef struct
  {
    unsigned int data;                /* address of the data port */
    unsigned int cont;                /* address of the control port */
    unsigned int fifo;                /* address of the fifo status port */
    unsigned char in_use;             /* indicates usage */
    u8 contbits;                      /* control bits (for IRQ control) */
    int swapdata;                     /* fifo endianess control setting... */
    u8 swapbits;                      /* ...and corresponding bit */
    u32 *inbuf, *inrd, *inwr;         /* input ring buffer */
    volatile int in_inbuf;            /* # dwords in inbuf */
    u32 *outbuf, *outrd, *outwr;      /* output ring buffer */
    volatile int in_outbuf;           /* # dwords in outbuf */
    struct wait_queue *inq, *outq;    /* queues of waiting processes */
  } hepc3_comport;

/* data type to describe one board */
typedef struct
  {
    char name [HEPC3_MAX_NAME];
    unsigned int baseaddr;            /* base address */
    unsigned int jtagaddr;            /* jtag address */
    unsigned char revid;              /* the boards revision */
    unsigned char irq;                /* pci interrupt line */
    hepc3_comport comport [2];        /* two comports (0 == A) */
    struct tq_struct isr_bh;          /* the isr's bottom half */
    u32 imb4_flags;                   /* irq flags for bottom half */
    u32 intcsr_flags;                 /* intcsr flags for bottom half */
#ifdef HEPC3_WITH_DMA
    unsigned int dma_size [2];        /* dma transfer size (0: pci->comport) */
#endif
  } hepc3_board;

/* 
 * The following macros are used for portability between kernel versions 2.0.x
 * and above -- taken from Rubini's book...
 */
#ifndef VERSION_CODE
#  define VERSION_CODE(vers,rel,seq) (((vers)<<16) | ((rel)<<8) | (seq))
#endif

/* this module does not support kernels < 2.0.0 */
#if LINUX_VERSION_CODE < VERSION_CODE(2,0,0)
#  error "Kernels < 2.0.0 are not supported --- upgrade NOW!"
#endif

#if LINUX_VERSION_CODE < VERSION_CODE(2,1,0)
#  define LINUX_20
#elif LINUX_VERSION_CODE < VERSION_CODE(2,2,0)
#  warning "kernel 2.1.x not supported --- trying to compile for 2.2.x"
#  define LINUX_22
#else
#  define LINUX_22
#endif

/*
 * "select" changed for Version 2.2
 */
#ifdef LINUX_22
#  include <linux/poll.h>
#endif

/* Other changes in the fops are solved using pseudo-types */
#ifdef LINUX_22
#  define lseek_t      loff_t
#  define lseek_off_t  loff_t
#else
#  define lseek_t      int
#  define lseek_off_t  off_t
#endif

/* changed the prototype of read/write */

#if defined(LINUX_22) || defined(__alpha__)
#  define count_t unsigned long
#else
#  define count_t int
#  define ssize_t int
#endif

/* changed the prototype of release */

#ifdef LINUX_20
#  define release_t void
#  define release_return(x) return
#else
#  define release_t int
#  define release_return(x) return (x)
#endif

/*
 * access to user space: use the 2.2 functions,
 * and implement them as macros for 2.0
 */

#ifdef LINUX_20
#  include <asm/segment.h>
#  define access_ok(t,a,sz)           (verify_area((t),(a),(sz)) ? 0 : 1)
#  define verify_area_20              verify_area
#  define copy_to_user(t,f,n)         (memcpy_tofs(t,f,n), 0)
#  define __copy_to_user(t,f,n)       copy_to_user((t),(f),(n))
#  define copy_to_user_ret(t,f,n,r)   copy_to_user((t),(f),(n))
#  define copy_from_user(t,f,n)       (memcpy_fromfs((t),(f),(n)), 0)
#  define __copy_from_user(t,f,n)     copy_from_user((t),(f),(n))
#  define copy_from_user_ret(t,f,n,r) copy_from_user((t),(f),(n))
#  define PUT_USER(val,add)           (put_user((val),(add)), 0)
#  define __PUT_USER(val,add)         PUT_USER((val),(add))
#  define PUT_USER_RET(val,add,ret)   PUT_USER((val),(add))
#  define GET_USER(dest,add)          ((dest)=get_user((add)), 0)
#  define __GET_USER(dest,add)        GET_USER((dest),(add))
#  define GET_USER_RET(dest,add,ret)  GET_USER((dest),(add))
#else
#  include <asm/uaccess.h>
#  include <asm/io.h>
#  define verify_area_20(t,a,sz) (0) /* == success */
#  define PUT_USER put_user
#  define __PUT_USER __put_user
#  define PUT_USER_RET put_user_ret
#  define GET_USER get_user
#  define __GET_USER __get_user
#  define GET_USER_RET get_user_ret
#endif

/* proc_register_dynamic no more existent for 2.2 */
#ifdef LINUX_22
#  define proc_register_dynamic proc_register
#endif

/* 2.2 removed these functions. Let's define them, just in case */
#ifdef LINUX_22
#  define queue_task_irq      queue_task
#  define queue_task_irq_off  queue_task
#endif

/* the method to check for queued signals - robust for 2.2 */
#ifdef LINUX_20
#  define SIGNAL_PENDING      (current->signal & ~current->blocked)
#else
#  define SIGNAL_PENDING      signal_pending (current)
#endif

/* 2.2 added test_and_{set|clear|change}_bit */
#ifdef LINUX_20
#  define test_and_set_bit(nr,addr)  test_bit((nr),(addr))
#  define test_and_clear_bit(nr,addr) clear_bit((nr),(addr))
#  define test_and_change_bit(nr,addr) change_bit((nr),(addr))
#endif

#endif  /* __KERNEL__ */

#endif  /* __HEPC3_H__  */
