/*
 * hepc3 --- a Linux kernel driver module for the Hunt Engineering HEPC3,
 *           HEPC4 and HECPCI1 [C]PCI TIM-40 motherboards
 *
 * This driver is
 *
 *                     Copyright (C) 1999-2001 by Albrecht Dre.
 *
 * This driver is free software, and you are welcome to redistribute it under
 * the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * The driver is distributed in the hope that it will be useful and work with
 * your system's configuration.
 * Due to the fact that the driver is free, it is provided WITHOUT ANY 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. Please see the GNU General Public License (GPL)
 * for details. You should have received a copy of the GPL with this package;
 * if not, write to the Free Software Foundation, Inc., 675 Mass Ave., 
 * Cambride, MA 02139, USA
 *
 * Please report bugs or suggestions to ad@mpifr-bonn.mpg.de or send mail to:
 * Albrecht Dre, Max-Planck-Institut fr Radioastronomie, Box 2024, 
 * 53010 Bonn, GERMANY
 *
 * The development of this driver module was based upon the following
 * information:
 *
 * o Jim Partan's driver module "hep3_devel.[hc]; unfortunately this
 *   driver did not work with the HECPCI1, but LARGE parts are actually
 *   taken from it...
 *
 * o Hunt Engineering: HECPCI1 3U CompactPCI TIM-40 Motherboard: USER
 *   MANUAL. Hardware Rev A, Cocument Rev B, P.Warnes, 16/02/98. (see
 *   http://www.hunteng.co.uk)
 * 
 * o AMCC: PCI Products Data Book, Section 3: S5933 PCI Controller, 1998.
 *
 * o Alessandro Rubini: Linux Device Drivers. O'Reilly & Associates, 1998.
 *   (I used the german translation)
 *
 * o Ori Pomerantz: Linux Kernel Module Programming Guide, 1999. (for 2.0 to
 *   2.2 migration)
 *
 * Albrecht Dre, Jul 1, 1999.
 *
 *
 * Changes in Version 1.1.0, of Jan 23, 2001 (Albrecht Dre)
 * =========================================================
 * 
 * o fixed bug in port A/interrupt driven mode which could cause a hang under
 *   extreme conditions
 *
 * o rewritten port A dma/pci bus mastering stuff for more performance and
 *   stability
 */
#ifndef __KERNEL__
#define __KERNEL__
#endif

#ifndef MODULE
#define MODULE
#endif

#include <linux/types.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/malloc.h>
#include <linux/pci.h>
#include <linux/ioport.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <asm/io.h>
#include <asm/segment.h>
#include <asm/bitops.h>
#include "hepc3.h"

#ifdef LINUX_20
#  include <linux/bios32.h>        /* obsolete for 2.2.x */
#endif

/* ==== global prototypes ================================================= */
#ifdef LINUX_20
static lseek_t hepc3_lseek (struct inode *inode, struct file *file, 
			    lseek_off_t offs, int org);
static ssize_t hepc3_read (struct inode *inode, struct file *file, 
			   char *buf, int count);
static ssize_t hepc3_write (struct inode *inode, struct file *file, 
			    const char *buf, int count);
static int hepc3_select (struct inode *inode, struct file *file, 
			 int mode, select_table *table);
#else
static lseek_t hepc3_lseek (struct file *file, lseek_off_t offs, int org);
static ssize_t hepc3_read (struct file *file, char *buf, size_t count, 
			   loff_t *offset);
static ssize_t hepc3_write (struct file *file, const char *buf, size_t count,
			    loff_t *offset);
static unsigned int hepc3_poll (struct file *file, poll_table *wait);
#endif
static int hepc3_ioctl (struct inode *inode, struct file *file, 
			unsigned int request, unsigned long arg);
static int hepc3_open (struct inode *inode, struct file *file);
static release_t hepc3_release (struct inode *inode, struct file *file);
static void hepc3_isr (int irq, void *dev_id, struct pt_regs *regs);
static void hepc3_isr_bh (void *board);
static int hepc3_read_procmem (char *buf, char **start, off_t offset,
			       int len, int unused);

/* ==== global variables ================================================== */
static hepc3_board *hepc3 [HEPC3_MAX_BOARD];

int hepc3_major = HEPC3_MAJOR;          /* may be overwritten during insmod */
#ifdef LINUX_20
int hepc3_buf_size = HEPC3_BUF_SIZE0;   /* may be overwritten during insmod */
#else
int hepc3_buf_size = HEPC3_BUF_SIZE2;   /* may be overwritten during insmod */
#endif
int hepc3_secure = 1;                   /* may be overwritten during insmod */
#ifdef HEPC3_WITH_DMA
int hepc3_latency = 32;                 /* may be overwritten during insmod */
#endif

#ifdef LINUX_22
MODULE_AUTHOR("Albrecht Dre <ad@mpifr-bonn.mpg.de>");
MODULE_DESCRIPTION("Hunt Engineering HEPC3/HEPC4/HECPCI1 driver");
MODULE_PARM (hepc3_major, "i");
MODULE_PARM (hepc3_buf_size, "i");
MODULE_PARM (hepc3_secure, "i");
#ifdef HEPC3_WITH_DMA
MODULE_PARM (hepc3_latency, "i");
#endif
#endif

static struct file_operations hepc3_fops =
  {
    hepc3_lseek,
    hepc3_read,
    hepc3_write,
    NULL,             /* hepc3_readdir: not implemented */
#ifdef LINUX_20
    hepc3_select,     /* Kernel 2.0.x */
#else
    hepc3_poll,       /* Kernel 2.2.x */
#endif
    hepc3_ioctl,
    NULL,             /* hepc3_mmap: not implemented */
    hepc3_open,
#ifdef LINUX_22
    NULL,             /* Kernel 2.2.x, hepc3_flush: not implemented */
#endif
    hepc3_release,    /* others are not defined... */
  };

static struct proc_dir_entry hepc3_proc_entry =
  {
    0,                    /* dynamic low_ino */
    5, HEPC3_NAME,        /* name field */
    S_IFREG | S_IRUGO,    /* mode */
    1, 0, 0,              /* nlinks, owner, group */
    0,                    /* size */
    NULL,                 /* default operations */
    &hepc3_read_procmem,
  };

/* ==== local routines ==================================================== */
#ifdef LINUX_20
static hepc3_board *hepc3_init_board (int board_num, unsigned char bus, 
				      unsigned char function);
#else
static hepc3_board *hepc3_init_board (int board_num, struct pci_dev *dev);
#endif
static int hepc3_get_ioports (hepc3_board *h);
static void hepc3_free_board (hepc3_board *h);
static void hepc3_sys_reset (hepc3_board *h);
#ifdef HEPC3_WITH_DMA
static inline void launch_write_dma (hepc3_board *h, u32 *data);
static inline void launch_read_dma (hepc3_board *h, u32 *data);
#else
static inline unsigned int hepc3_portAread (unsigned int status, 
			   unsigned int data, u32 *buf, unsigned int count);
static inline unsigned int hepc3_portAwrite (unsigned int status, 
			   unsigned int data, u32 *buf, unsigned int count);
#endif
static inline unsigned int hepc3_portBread (unsigned int status, 
			   unsigned int data, u32 *buf, unsigned int count);
static inline unsigned int hepc3_portBwrite (unsigned int status, 
			   unsigned int data, u32 *buf, unsigned int count);

/* the "high" nibble of the minor number contains the board number */
#define BOARD(dev)     (MINOR(dev) >> 4)

/* the "low" nibble of the minor number contains the port (see below) */
#define PORT(dev)      (MINOR(dev) & 0x0f)
#define HEPC3_PORTA    0
#define HEPC3_PORTB    1
#define HEPC3_JTAG     2

/* ==== global routines =================================================== */

/*
 * Module init routine (for insmod and friends)
 */
int init_module (void)
  {
#ifdef CONFIG_PCI
#ifdef LINUX_20
    if (pcibios_present ())
#else
    if (pci_present ())
#endif
      {
#ifdef LINUX_20
	unsigned char bus, function;
#else
	struct pci_dev *pdev = NULL;
#endif
	int result, index = 0;
	hepc3_board *h;

#ifdef LINUX_20
	while ((result = pcibios_find_device (HEPC3_VENDOR, HEPC3_ID, index, 
					      &bus, &function))
	       == PCIBIOS_SUCCESSFUL && index < HEPC3_MAX_BOARD)
#else
	while ((pdev = pci_find_device (HEPC3_VENDOR, HEPC3_ID, pdev)) 
	       && index < HEPC3_MAX_BOARD)
#endif
	  {
	    int portres;

	    /* found a board, try to init... */
#ifdef LINUX_20
	    if (!(h = hepc3 [index] = hepc3_init_board (index, bus, function)))
#else
	    if (!(h = hepc3 [index] = hepc3_init_board (index, pdev)))
#endif
	      { /* no memory: remove everything claimed up to now */
		while (index--)
		  hepc3_free_board (hepc3 [index]);
		return -ENODEV;
	      }
	    /* try to request the io ports */
	    if ((portres = hepc3_get_ioports (h)) != 0)
	      { /* error claiming io ports: remove everything */
		kfree (h);   /* free this board's memory; no regions claimed */
		while (index--)
		  hepc3_free_board (hepc3 [index]);		
		return portres;
	      }
	    printk (KERN_INFO "%s: Hunt Engineering HEPC3/HEPC4/HECPCI1 [C]PCI "
		    "TIM40 motherboard Rev. %d\n", h->name, h->revid);
	    printk (KERN_INFO "%s: IRQ=%d, IO at 0x%x, 0x%x, 0x%x, 0x%x, "
		    "0x%x\n", h->name, h->irq, h->baseaddr, h->comport [1].data,
		    h->comport [0].cont, h->comport [1].cont, h->jtagaddr);

	    /* set the values for the bottom half isr */
	    h->isr_bh.routine = hepc3_isr_bh;
	    h->isr_bh.data = h;
	    /* finally reset the board */
	    hepc3_sys_reset (h);
	    
	    ++index;
	  }

	if (index == 0)
	  {
	    printk (KERN_WARNING HEPC3_NAME ": no HEPC3/HEPC4/HECPCI1 found\n");
	    return -ENODEV;  /* need at least one board */
	  }
	
	/* install as char device; default: dynamic major number */
	result = register_chrdev (hepc3_major, HEPC3_NAME, &hepc3_fops);
	if (result < 0)               /* static major number failed */
	  {
	    printk (KERN_WARNING HEPC3_NAME ": can't get major number %d\n", 
		    hepc3_major);
	    for (index = 0; index < HEPC3_MAX_BOARD; ++index)
	      hepc3_free_board (hepc3 [index]);		
	    return result;
	  }
	if (hepc3_major == 0)
	  hepc3_major = result; /* dynamic major number */
	printk (KERN_INFO HEPC3_NAME 
		": character device has major number %d\n", hepc3_major);
	proc_register_dynamic (&proc_root, &hepc3_proc_entry);
	return 0;
      }
    else
      return -ENODEV;  /* no pci bios found */
#else   /* !CONFIG_PCI */
    return -ENODEV;
#endif  /* CONFIG_PCI */
  }

/*
 * Module cleanup routine (for rmmod)
 */
void cleanup_module (void)
  {
    int index;

    proc_unregister (&proc_root, hepc3_proc_entry.low_ino);
    unregister_chrdev (hepc3_major, HEPC3_NAME);
    
    for (index = 0; index < HEPC3_MAX_BOARD; ++index)
      hepc3_free_board (hepc3 [index]);
  }

/*
 * "lseek" method --- fails always; not seekable
 */
#ifdef LINUX_20
static lseek_t hepc3_lseek (struct inode *inode, struct file *file, 
			    lseek_off_t offs, int org)
#else
static lseek_t hepc3_lseek (struct file *file, lseek_off_t offs, int org)
#endif
  {
    return -ESPIPE;
  }

/*
 * "read" method
 */
#ifdef LINUX_20
static ssize_t hepc3_read (struct inode *inode, struct file *file, 
			   char *buf, int count)
#else
static ssize_t hepc3_read (struct file *file, char *buf, size_t count, 
			   loff_t *offset)
#endif
  {
    int rdev, port;
    hepc3_board *h;
    hepc3_comport *cp;
    int dwread, transfer;

#ifdef LINUX_22
    /* do not accept pread calls */
    if (*offset != file->f_pos)
      return -ESPIPE;
#endif
    /* count must be > 0 and a multiple of 4 */
    if (count <= 0 || count & 0x3)
      return -EINVAL;

#ifdef LINUX_20
    rdev = inode->i_rdev;
#else
    rdev = file->f_dentry->d_inode->i_rdev;
#endif
    port = PORT (rdev);
    h = hepc3 [BOARD (rdev)];
    cp = &(h->comport [port]);
    dwread = 0;                /* # of read dwords */
    count = count >> 2;        /* dwords left to read */

    do
      {
	cli ();
#ifndef HEPC3_WITH_DMA
	/* 
	 * On port A, a LGB interrupt might get lost, so check the port 
	 * register for data.
	 */
	if (port == 0 && cp->in_inbuf == 0 && 
	    !(inb (cp->fifo) & HEPC3_AIN_EMPTY))
	  {
#ifdef HEPC3_VERBOSE_ERRORS
	    printk (KERN_INFO "%s: LGB irq lost, re-toggle\n", h->name);
#endif
	    cp->inrd = cp->inwr = cp->inbuf;
	    cp->in_inbuf = hepc3_portAread (cp->fifo, cp->data, cp->inwr, 
					    hepc3_buf_size - 1);
	    cp->inwr += cp->in_inbuf;
	  }
#endif

	/* transfer max. the filled area of the in buffer */
	if (count > cp->in_inbuf)
	  transfer = cp->in_inbuf;
	else
	  transfer = count;

	/* copy the data */
	if (transfer)
	  {
	    if (cp->inrd - cp->inbuf + transfer > hepc3_buf_size)
	      {
		/* handle wrapping around the end of the buffer */
		int chunk = hepc3_buf_size - (cp->inrd - cp->inbuf);
		copy_to_user (buf, cp->inrd, chunk << 2);
		cp->inrd = cp->inbuf;
		buf += chunk << 2;
		chunk = transfer - chunk;
		copy_to_user (buf, cp->inrd, chunk << 2);
		buf += chunk << 2;
		cp->inrd += chunk;
	      }
	    else
	      {
		copy_to_user (buf, cp->inrd, transfer << 2);
		buf += transfer << 2;
		cp->inrd += transfer;
		if (cp->inrd >= cp->inbuf + hepc3_buf_size)
		  cp->inrd = cp->inbuf;
	      }
	    cp->in_inbuf -= transfer;
	    count -= transfer;
	    dwread += transfer;
	  }

	/* set IRQ level and enable interrupts */
	if (port == 0)
	  {
#ifdef HEPC3_WITH_DMA
	    /* start dma if currently not active and buffer empty */
	    if (count && h->dma_size [1] == 0 && cp->in_inbuf == 0)
	      {
		/* simplify pointers... */
		cp->inrd = cp->inwr = cp->inbuf;
		/* get max. count words, or (if count is zero) only one */
		if (count)
		  if (count > hepc3_buf_size)
		    h->dma_size [1] = hepc3_buf_size;
		  else
		    h->dma_size [1] = count;
		else
		  h->dma_size [1] = 1;
		launch_write_dma (h, cp->inwr);
	      }
#else
	    cp->contbits |= HEPC3_CPA_LGBIRQ;
	    outb (cp->contbits, cp->cont);
#endif
	  }
	else
	  {
	    cp->contbits |= HEPC3_CPB_INIRQ;
	    outb (cp->contbits, cp->cont);
	  }
#ifndef HEPC3_WITH_DMA	
	/* 
	 * to avoid deadlocks, do not sleep on port A if we possibly lost a 
	 * LGB irq 
	 */
	if (port == 0 && !(inb (cp->fifo) & HEPC3_AIN_EMPTY))
	  sti ();
	else
#endif
	  /* if we read blocking, wait for all data being transferred. */
	  if (count && !(file->f_flags & O_NONBLOCK) && cp->in_inbuf == 0)
	    {
	      interruptible_sleep_on (&cp->inq);
	      sti ();
	      if (SIGNAL_PENDING)
		return -ERESTARTSYS;
	    }
	  else
	    sti ();
      }
    while (count && !(file->f_flags & O_NONBLOCK));

    if (!count && port == 1)
      {
	cp->contbits &= ~HEPC3_CPB_INIRQ;
	cp->contbits |= HEPC3_CPB_I60IRQ;	
	outb (cp->contbits, cp->cont);
      }
    /* for 0 words read and non-blocking io, return EAGAIN */
    if (dwread == 0 && file->f_flags & O_NONBLOCK)
      return -EAGAIN;
    else
      return dwread << 2;    /* return # of transferred bytes */
  }

/*
 * "write" method
 */
#ifdef LINUX_20
static ssize_t hepc3_write (struct inode *inode, struct file *file, 
			    const char *buf, int count)
#else
static ssize_t hepc3_write (struct file *file, const char *buf, size_t count, 
			    loff_t *offset)
#endif
  {
    int rdev, port;
    hepc3_board *h;
    hepc3_comport *cp;
    int written, transfer;

#ifdef LINUX_22
    /* do not accept pwrite calls */
    if (*offset != file->f_pos)
      return -ESPIPE;
#endif
    /* count must be > 0 and a multiple of 4 */
    if (count <= 0 || count & 0x3)
      return -EINVAL;

#ifdef LINUX_20
    rdev = inode->i_rdev;
#else
    rdev = file->f_dentry->d_inode->i_rdev;
#endif
    port = PORT (rdev);
    h = hepc3 [BOARD (rdev)];
    cp = &(h->comport [port]);
    written = 0;               /* # of written dwords */
    count = count >> 2;        /* dwords left to write */

    /* non-blocking and out buffer full: return immediately */
    if (hepc3_buf_size == cp->in_outbuf && file->f_flags & O_NONBLOCK)
      return -EAGAIN;
    
    do
      {
	cli ();
	/* transfer max. the free area of out buffer */
	if (hepc3_buf_size - cp->in_outbuf > count)
	  transfer = count;
	else
	  transfer = hepc3_buf_size - cp->in_outbuf;
	/* copy the data */
	if (transfer)
	  {
	    if (cp->outwr - cp->outbuf + transfer > hepc3_buf_size)
	      {
		/* handle wrapping around the end of the buffer */
		int chunk = hepc3_buf_size - (cp->outwr - cp->outbuf);
		copy_from_user (cp->outwr, buf, chunk << 2);
		cp->outwr = cp->outbuf;
		buf += chunk << 2;
		chunk = transfer - chunk;
		copy_from_user (cp->outwr, buf, chunk << 2);
		buf += chunk << 2;
		cp->outwr += chunk;
	      }
	    else
	      {
		copy_from_user (cp->outwr, buf, transfer << 2);
		buf += transfer << 2;
		cp->outwr += transfer;
		if (cp->outwr >= cp->outbuf + hepc3_buf_size)
		  cp->outwr = cp->outbuf;
	      }
	    cp->in_outbuf += transfer;
	    count -= transfer;
	    written += transfer;
	  }

	/* set IRQ level and enable interrupts */
	if (port == 0)
	  {
#ifdef HEPC3_WITH_DMA
	    /* start dma immediately if currently not active */
	    if (h->dma_size [0] == 0 && cp->in_outbuf)
	      {
		if (cp->outwr > cp->outrd)
		  h->dma_size [0] = cp->outwr - cp->outrd;
		else
		  h->dma_size [0] = hepc3_buf_size - (cp->outrd - cp->outbuf);
		launch_read_dma (h, cp->outrd);
	      }
	    else
	      outw (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_ENI_RRDY,
		    h->baseaddr + HEPC3_INTCSR);
#else
	    cp->contbits |= HEPC3_CPA_OUTIRQ;
	    outb (cp->contbits, cp->cont);
#endif
	  }
	else
	  {
	    cp->contbits |= HEPC3_CPB_OUTIRQ;
	    outb (cp->contbits, cp->cont);
	  }
	/* if we write blocking, wait for all data being transferred. */
	if (count && !(file->f_flags & O_NONBLOCK) && 
	    cp->in_outbuf == hepc3_buf_size)
	  {
	    interruptible_sleep_on (&cp->outq);
	    sti ();
 	    if (SIGNAL_PENDING)
 	      return -ERESTARTSYS;
	  }
	else
	  sti ();
      }
    while (count && !(file->f_flags & O_NONBLOCK));
    return written << 2;    /* return # of transferred bytes */
  }

#ifdef LINUX_20
/*
 * "select" method
 */
static int hepc3_select (struct inode *inode, struct file *file, 
			 int mode, select_table *table)
  {
    hepc3_board *h = hepc3 [BOARD (inode->i_rdev)];
    hepc3_comport *cp;
    int port = PORT (inode->i_rdev);

    cp = &(h->comport [port]);

    switch (mode)
      {
      case SEL_IN:
	if (cp->in_inbuf)
	  return 1;
#ifdef HEPC3_WITH_DMA
	/* 
	 * on port A, if the buffer is empty and no DMA running, start DMA
	 * for one word so select will be waked up...
	 */
	if (port == 0 && h->dma_size [1] == 0)
	  {
	    /* simplify pointers... */
	    cp->inrd = cp->inwr = cp->inbuf;
	    h->dma_size [1] = 1;
	    launch_write_dma (h, cp->inwr);
	  }
#else
	/*
	 * reading from port A needs special handling: we might loose a LGB
	 * irq (see `read' and `isr_bh' for further description).
	 */
	if (port == 0 && !(inb (cp->fifo) & HEPC3_AIN_EMPTY))
	  return 1;
#endif
	select_wait (&cp->inq, table);
	return 0;
      case SEL_OUT:
	if (cp->in_outbuf < hepc3_buf_size)
	  return 1;
	select_wait (&cp->outq, table);
	return 0;
      case SEL_EX:
	return 0;
      }
    return 0;
  }

#else /* 2.2 kernel --- uses `poll' instead of `select' */

/*
 * "poll" method
 */
static unsigned int hepc3_poll (struct file *file, poll_table *wait)
  {
    int rdev = file->f_dentry->d_inode->i_rdev;
    hepc3_board *h;
    hepc3_comport *cp;
    int port;
    unsigned int mask = 0;

    h = hepc3 [BOARD (rdev)];
    port = PORT (rdev);
    cp = &(h->comport [port]);

    poll_wait (file, &cp->inq, wait);
    poll_wait (file, &cp->outq, wait);

    /* check for readable data */
    if (cp->in_inbuf)
      mask |= POLLIN | POLLRDNORM;
    else
#ifdef HEPC3_WITH_DMA
      /* 
       * on port A, if the buffer is empty and no DMA running, start DMA
       * for one word so poll will be waked up...
       */
      if (port == 0 && h->dma_size [1] == 0)
	{
	  /* simplify pointers... */
	  cp->inrd = cp->inwr = cp->inbuf;
	  h->dma_size [1] = 1;
	  launch_write_dma (h, cp->inwr);
	}
#else
      /*
       * reading from port A needs special handling: we might loose a LGB
       * irq (see `read' and `isr_bh' for further description).
       */
      if (port == 0 && !(inb (cp->fifo) & HEPC3_AIN_EMPTY))
	mask |= POLLIN | POLLRDNORM;
#endif

    /* check for writeable data */
    if (cp->in_outbuf < hepc3_buf_size)
      mask |= POLLOUT | POLLWRNORM;

    return mask;
  }
#endif /* select/poll */

/*
 * "ioctl" method
 */
static int hepc3_ioctl (struct inode *inode, struct file *file, 
			unsigned int request, unsigned long arg)
  {
    int board = BOARD (inode->i_rdev);
    int port = PORT (inode->i_rdev);
    hepc3_board *h;
    hepc3_comport *cp;
    int value;
    u8 statreg;
    unsigned long flags;

    h = hepc3 [board];
    cp = &(h->comport [port]);

    switch (request)
      {
      case HEPC3_IO_B_RES:    /* reset the whole board */
	/* if hepc3_secure is != 0, only root may do this */
	if (hepc3_secure && !suser ())
	  return -EPERM;
	hepc3_sys_reset (h);
	break;
      case HEPC3_IO_P_RES:    /* reset one channel */
	save_flags (flags);
	cli ();
	/* clear buffers (safe due to cli) */
	cp->inrd = cp->inwr = cp->inbuf;
	cp->in_inbuf = 0;
	cp->outrd = cp->outwr = cp->outbuf;
	cp->in_outbuf = 0;
	/* the ports need different approaches */
	switch (port)
	  {
	  case 0:           /* port A */
#ifdef HEPC3_WITH_DMA
	    /* avoid irq when aborting dma */
	    outw (inw (h->baseaddr + HEPC3_INTCSR) & 
		  ~(HEPC3_ENI_RRDY | HEPC3_ENI_WRDY),
		  h->baseaddr + HEPC3_INTCSR);
	    /* abort all dma jobs */
	    h->dma_size [0] = 0;
	    outw (0x0, h->baseaddr + HEPC3_MCSR);
	    outl (0x0, h->baseaddr + HEPC3_MWTC);
	    outl (0x0, h->baseaddr + HEPC3_MRTC);
#else
	    outw (inw (h->baseaddr + HEPC3_INTCSR) & ~HEPC3_ENI_MBOX,
		  h->baseaddr + HEPC3_INTCSR);
#endif
	    /* assert port reset, clear LGB, reset S5933 FIFO's */
	    outb (HEPC3_CPA_LGBCLR, cp->cont);
	    outl (HEPC3_AOUT_RES | HEPC3_AIN_RES, cp->fifo);
	    /* enable read irq, disable write irq */
#ifdef HEPC3_WITH_DMA
	    cp->contbits = HEPC3_CPA_RESET | cp->swapbits;
#else
	    cp->contbits = HEPC3_CPA_RESET | HEPC3_CPA_LGBIRQ | cp->swapbits;
#endif
	    /* clear port reset and set interrupts */
	    outb (cp->contbits, cp->cont);
#ifdef HEPC3_WITH_DMA
	    h->dma_size [1] = 0;
#else
	    outw (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_ENI_MBOX,
		  h->baseaddr + HEPC3_INTCSR);
#endif
	    break;
	  case 1:           /* port B */
	    outw (inw (h->baseaddr + HEPC3_INTCSR) & ~HEPC3_ENI_MBOX,
		  h->baseaddr + HEPC3_INTCSR);
	    /* assert port reset */
	    outb (0, h->comport [0].cont);
	    /* enable low-priority read irq, disable write irq */
	    cp->contbits = HEPC3_CPB_RESET | HEPC3_CPB_I60IRQ | cp->swapbits;
	    /* clear port reset */
	    outb (cp->contbits, cp->cont);
	    outw (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_ENI_MBOX,
		  h->baseaddr + HEPC3_INTCSR);
	    break;
	  case 2:           /* JTAG -- not (yet) supported */
	    break;
	  }
	restore_flags (flags);
	break;
      case HEPC3_IO_G_ISTAT:    /* get minimum number readable dwords */
	value = cp->in_inbuf;
	statreg = inb (cp->fifo);
	switch (port)
	  {
	  case 0:    /* port A */
	    if (statreg & HEPC3_AIN_FULL)
	      value += 8;
	    else
	      if (statreg & HEPC3_AIN_4)
		value += 4;
	      else
		if (!(statreg & HEPC3_AIN_EMPTY))
		  value += 1;
	    PUT_USER (value, (int *)arg);
	    break;
	  case 1:    /* port B */
	    if (!(statreg & HEPC3_CPB_INAF))
	      value += 60;
	    else
	      if (statreg & HEPC3_CPB_INE)
		value += 1;
	    PUT_USER (value, (int *)arg);
	    break;
	  /* case 2: --- no JTAG yet... */
	  }
	break;
      case HEPC3_IO_G_OSTAT:    /* get minimum number writable dwords */
	value = hepc3_buf_size - cp->in_outbuf;
	statreg = inb (cp->fifo);
	switch (port)
	  {
	  case 0:    /* port A */
	    if (statreg & HEPC3_AOUT_EMPTY)
	      value += 8;
	    else
	      if (statreg & HEPC3_AOUT_4)
		value += 4;
	      else
		if (!(statreg & HEPC3_AOUT_FULL))
		  value += 1;
	    PUT_USER (value, (int *)arg);
	    break;
	  case 1:    /* port B */
	    if (!(statreg & HEPC3_CPB_OUTE))
	      value += 64;
	    else
	      if (statreg & HEPC3_CPB_OUTAF)
		value += 1;
	    PUT_USER (value, (int *)arg);
	    break;
	  /* case 2: --- no JTAG yet... */
	  }
	break;
      case HEPC3_IO_G_CNFG:    /* get status of config line */
	switch (port)
	  {
	  case 0:
	    value = !(inb (cp->cont) & HEPC3_CPA_CONFIG); /* active low */
	    PUT_USER (value, (int *)arg);
	    break;
	  case 1:
	    value = !(inb (cp->cont) & HEPC3_CPB_CONFIG); /* active low */
	    PUT_USER (value, (int *)arg);
	    break;
	  default:
	    return -EINVAL;    /* JTAG doesn't have config... */
	  }
      case HEPC3_IO_S_SWAP:    /* set swap mode        */
	if (port == 0 || port == 1)
	  {
	    GET_USER (cp->swapdata, (int *)arg);
	    /* 
	     * Using PCI Bus master Mode reverts the meaning of the fifo endian
	     * bit on big endian machines (at least on the PowerMac...).
	     */
	    if (port)
	      cp->swapbits = cp->swapdata ? HEPC3_CPB_ENDIAN : 0;
	    else
#if defined(HEPC3_WITH_DMA) && defined(HEPC3_BIGENDIAN)
	      cp->swapbits = cp->swapdata ? 0 : HEPC3_CPA_ENDIAN;
#else  
	      cp->swapbits = cp->swapdata ? HEPC3_CPA_ENDIAN : 0;
#endif
	  }
	else
	  return -EINVAL;
	break;
      case HEPC3_IO_G_SWAP:    /* get swap mode        */
	if (port == 0 || port == 1)
	  {
	    value = cp->swapdata;
	    PUT_USER (value, (int *)arg);
	  }
	else
	  return -EINVAL;
	break;
      default:
	return -EINVAL;
      }
    return 0;
  }

/*
 * "open" method
 */
static int hepc3_open (struct inode *inode, struct file *file)
  {
    int board = BOARD (inode->i_rdev);
    int port = PORT (inode->i_rdev);
    hepc3_board *h;
    hepc3_comport *cp;
    int result;

    /* first check for legal board/port number and existing board */
    if (board >= HEPC3_MAX_BOARD || port > HEPC3_JTAG)
      return -EINVAL;
    if (!(h = hepc3 [board]))
      return -ENXIO;
    
    /* JTAG is not (yet) supported */
    if (port == HEPC3_JTAG)
      return -EINVAL;
    cp = &(h->comport [port]);
    /* reject if already in use */
    if (cp->in_use)
      return -EBUSY;

    /* allocate space for input and output ring buffers */
    if (!(cp->inbuf = kmalloc (hepc3_buf_size << 2, GFP_KERNEL)))
      {
	printk (KERN_WARNING "%s: could not allocate input buffer\n", h->name);
	return -ENOMEM;
      }
    if (!(cp->outbuf = kmalloc (hepc3_buf_size << 2, GFP_KERNEL)))
      {
	printk (KERN_WARNING "%s: could not allocate output buffer\n", h->name);
	kfree (cp->inbuf);
	return -ENOMEM;
      }
    /* ok, init the port... they need different approaches (reset, IRQ) */
    cli ();
    cp->in_use = 1;
    cp->inq = cp->outq = NULL;
    /* clear buffers */
    cp->inrd = cp->inwr = cp->inbuf;
    cp->in_inbuf = 0;
    cp->outrd = cp->outwr = cp->outbuf;
    cp->in_outbuf = 0;
    /*
     * do not double-install the irq handler; check wether the other port 
     * is already in use.
     */
    if (!h->comport [(port + 1) & 1].in_use)
      {
	h->imb4_flags = h->intcsr_flags = 0; /* clear irq routine flags */
	result = request_irq (h->irq, hepc3_isr, SA_SHIRQ, h->name, h);
	if (result)
	  {  /* request_irq failed for some reason... */
	    printk (KERN_WARNING "%s: could not get irq line %d\n", h->name,
		    h->irq);
	    cp->in_use = 0;
	    kfree (cp->inbuf);
	    kfree (cp->outbuf);
	    sti ();
	    return -EBUSY;
	  }
      }

    /*
     * By default, byte swapping is enabled on big endian machines and dis-
     * abled on little endians.
     */
#ifdef HEPC3_BIGENDIAN
    cp->swapdata = 1;
#else
    cp->swapdata = 0;
#endif
    /* 
     * Using PCI Bus master Mode reverts the meaning of the fifo endian bit on
     * big endian machines (at least on the PowerMac...).
     */
    if (port)
      cp->swapbits = cp->swapdata ? HEPC3_CPB_ENDIAN : 0;
    else
#if defined(HEPC3_WITH_DMA) && defined(HEPC3_BIGENDIAN)
      cp->swapbits = cp->swapdata ? 0 : HEPC3_CPA_ENDIAN;
#else  
      cp->swapbits = cp->swapdata ? HEPC3_CPA_ENDIAN : 0;
#endif

    /* reset port, enable the port's read irq's */
    if (port)
      {
	/* assert port reset */
	outb (0, cp->cont);
	/* enable low-priority read irq, disable write irq */
	cp->contbits = HEPC3_CPB_RESET | HEPC3_CPB_I60IRQ | cp->swapbits;
	/* enable interrupt in the AMCC S5933 */
	outw (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_ENI_MBOX,
	      h->baseaddr + HEPC3_INTCSR);
      }
    else
      {
	/* assert port reset, clear LGB, reset S5933 FIFO's */
	outb (HEPC3_CPA_LGBCLR, cp->cont);
	outl (HEPC3_AOUT_RES | HEPC3_AIN_RES, cp->fifo);
#ifdef HEPC3_WITH_DMA
	/* no fifo interrupts */
	cp->contbits = HEPC3_CPA_RESET | cp->swapbits;
	/* dma setup */
	outw (0x0, h->baseaddr + HEPC3_MCSR);
	outl (0x0, h->baseaddr + HEPC3_MWTC);
	outl (0x0, h->baseaddr + HEPC3_MRTC);
	h->dma_size [0] = h->dma_size [1] = 0;
#else
	/* enable read irq, disable write irq */
	cp->contbits = HEPC3_CPA_RESET | HEPC3_CPA_LGBIRQ | cp->swapbits;
	/* enable mailbox interrupt in the AMCC S5933 */
	outl (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_ENI_MBOX,
	      h->baseaddr + HEPC3_INTCSR);
#endif
      }
    outb (cp->contbits, cp->cont);
    sti ();
    
    MOD_INC_USE_COUNT;
    return 0;
  }

/*
 * "release" method
 */
static release_t hepc3_release (struct inode *inode, struct file *file)
  {
    int board = BOARD (inode->i_rdev);
    int port = PORT (inode->i_rdev);
    hepc3_board *h;
    hepc3_comport *cp;    

    h = hepc3 [board];
    cp = &(h->comport [port]);
    /* ensure that we will not be interrupted */
    cli ();
    /* clear the irq flags for the port */
    if (port)   
      cp->contbits = HEPC3_CPB_RESET;     /* comport B */
    else
      {
	cp->contbits = HEPC3_CPA_RESET;   /* comport A */
	outl (HEPC3_AOUT_RES | HEPC3_AIN_RES, h->baseaddr + HEPC3_MCSR);
#ifdef HEPC3_WITH_DMA
	outw (inw (h->baseaddr + HEPC3_INTCSR) & 
	      ~(HEPC3_ENI_RRDY | HEPC3_ENI_WRDY), h->baseaddr + HEPC3_INTCSR);
	outl (0x0, h->baseaddr + HEPC3_MWTC);
	outl (0x0, h->baseaddr + HEPC3_MRTC);
#endif
      }
    /* remove irq if the other port is not in use */
    if (!h->comport [(port + 1) & 1].in_use)
      {
	outw (inw (h->baseaddr + HEPC3_INTCSR) & ~HEPC3_ENI_MBOX,
	      h->baseaddr + HEPC3_INTCSR);
	free_irq (h->irq, h);
      }
    /* free allocated memory */
    cp->in_inbuf = cp->in_outbuf = 0;
    kfree (cp->inbuf);
    kfree (cp->outbuf);
    cp->in_use = 0;
    sti ();

    MOD_DEC_USE_COUNT;
    release_return (0);
  }

/* ==== local routines ==================================================== */

/*
 * init the structure for a board and read all its data from the pci bus
 */
#ifdef LINUX_20
static hepc3_board *hepc3_init_board (int board_num, unsigned char bus, 
				      unsigned char function)
#else
static hepc3_board *hepc3_init_board (int board_num, struct pci_dev *dev)
#endif
  {
    hepc3_board *h;
    unsigned short w;
#ifdef HEPC3_WITH_DMA
    unsigned char c;
#endif

    /* allocate memory for the board's decription and set some defaults */
    if (!(h = (hepc3_board *)kmalloc (sizeof (hepc3_board), GFP_KERNEL)))
      {
	printk (KERN_WARNING HEPC3_NAME ": can't allocate memory\n");
	return NULL;
      }
    sprintf (h->name, "hepc3%d", board_num);
    h->comport [0].in_use = h->comport [1].in_use = 0;

    /* 
     * identify the board's interrupt line and revision id 
     * In 2.2, get the irq from the dev structure to avoid problems with
     * re-mapping
     */
#ifdef LINUX_20
    pcibios_read_config_byte (bus, function, PCI_INTERRUPT_LINE, &(h->irq));
    pcibios_read_config_byte (bus, function, PCI_REVISION_ID, &(h->revid));
#else
    h->irq = dev->irq;
    pci_read_config_byte (dev, PCI_REVISION_ID, &(h->revid));
#endif
    
    /*
     * Get the base adresses for the following regions:
     * region 0: AMCC S5933 registers (offset 0x20: comport A fifo)
     * region 1: comport B fifo
     * region 2: comport A control
     * region 3: comport B control
     * region 4: JTAG
     * region 5: not used
     * In 2.2, get the addresses from the dev structure to avoid problems with
     * re-mapping
     */
    /* region 0 */
#ifdef LINUX_20
    pcibios_read_config_dword (bus, function, PCI_BASE_ADDRESS_0, 
			       &(h->baseaddr));
    h->baseaddr &= PCI_BASE_ADDRESS_IO_MASK;
#else
    h->baseaddr = dev->base_address [0] & PCI_BASE_ADDRESS_IO_MASK;
#endif
    h->comport [0].data = h->baseaddr + HEPC3_A_FIFO;
    h->comport [0].fifo = h->baseaddr + HEPC3_MCSR;

    /* region 1 */
#ifdef LINUX_20
    pcibios_read_config_dword (bus, function, PCI_BASE_ADDRESS_1, 
			       &(h->comport [1].data));
    h->comport [1].data &= PCI_BASE_ADDRESS_IO_MASK;
#else
    h->comport [1].data = dev->base_address [1] & PCI_BASE_ADDRESS_IO_MASK;
#endif
   
    /* region 2 */
#ifdef LINUX_20
    pcibios_read_config_dword (bus, function, PCI_BASE_ADDRESS_2, 
			       &(h->comport [0].cont));
    h->comport [0].cont &= PCI_BASE_ADDRESS_IO_MASK;
#else
    h->comport [0].cont = dev->base_address [2] & PCI_BASE_ADDRESS_IO_MASK;
#endif
   
    /* region 3 */
#ifdef LINUX_20
    pcibios_read_config_dword (bus, function, PCI_BASE_ADDRESS_3, 
			       &(h->comport [1].cont));
    h->comport [1].cont &= PCI_BASE_ADDRESS_IO_MASK;
#else
    h->comport [1].cont = dev->base_address [3] & PCI_BASE_ADDRESS_IO_MASK;
#endif
    h->comport [1].fifo = h->comport [1].cont;
   
    /* region 4 */
#ifdef LINUX_20
    pcibios_read_config_dword (bus, function, PCI_BASE_ADDRESS_4, 
			       &(h->jtagaddr));
    h->jtagaddr &= PCI_BASE_ADDRESS_IO_MASK;
#else
    h->jtagaddr = dev->base_address [4] & PCI_BASE_ADDRESS_IO_MASK;
#endif

    /* 
     * set S5933 Bus Master Enable and PCI latency if necessary; always 
     * enable IO and memory accesses (necessary on LinuxPPC)
     * use new pci interface functions with 2.2
     */
#ifdef HEPC3_WITH_DMA
#ifdef LINUX_20
    pcibios_read_config_word (bus, function, PCI_COMMAND, &w);
    pcibios_write_config_word (bus, function, PCI_COMMAND, 
			       w | PCI_COMMAND_MASTER | PCI_COMMAND_MEMORY |
			       PCI_COMMAND_IO);
    pcibios_write_config_byte (bus, function, PCI_LATENCY_TIMER, hepc3_latency);
    pcibios_read_config_byte (bus, function, PCI_LATENCY_TIMER, &c);
#else  /* LINUX_22 */
    pci_read_config_word (dev, PCI_COMMAND, &w);
    pci_write_config_word (dev, PCI_COMMAND, w | PCI_COMMAND_MASTER
			   | PCI_COMMAND_MEMORY | PCI_COMMAND_IO);
    pci_write_config_byte (dev, PCI_LATENCY_TIMER, hepc3_latency);
    pci_read_config_byte (dev, PCI_LATENCY_TIMER, &c);
#endif
    printk (KERN_INFO "%s: PCI_LATENCY_TIMER set to %d (requested %d)\n",
	    h->name, c, hepc3_latency);
#else /* !HEPC3_WITH_DMA */
#ifdef LINUX_20
    pcibios_read_config_word (bus, function, PCI_COMMAND, &w);
    pcibios_write_config_word (bus, function, PCI_COMMAND, 
			       (w | PCI_COMMAND_MEMORY | PCI_COMMAND_IO)
			       & ~PCI_COMMAND_MASTER);
#else  /* LINUX_22 */
    pci_read_config_word (dev, PCI_COMMAND, &w);
    pci_write_config_word (dev, PCI_COMMAND, 
			   (w | PCI_COMMAND_MEMORY | PCI_COMMAND_IO)
			   & ~PCI_COMMAND_MASTER);
#endif
#endif
    
    return h;
  }

/*
 * free all resources claimed by board h
 */
static void hepc3_free_board (hepc3_board *h)
  {
    /* don't free anything which was not allocated... */
    if (!h)
      return;
    /* free io ports */
    release_region (h->jtagaddr, HEPC3_JTAG_SIZE);
    release_region (h->comport [1].cont, HEPC3_REG_SIZE);
    release_region (h->comport [0].cont, HEPC3_REG_SIZE);
    release_region (h->comport [1].data, HEPC3_REG_SIZE);
    release_region (h->baseaddr, HEPC3_REG_SIZE);
    /* remove data structure */
    kfree (h);
  }

/*
 * tries to claim the io ports for board h and returns 0 on success.
 */
static int hepc3_get_ioports (hepc3_board *h)
  {
    int result;

    /* region 0 */
    if ((result = check_region (h->baseaddr, HEPC3_REG_SIZE)) < 0)
      {
	printk (KERN_WARNING "%s: can't get io at 0x%x\n", h->name, 
		h->baseaddr);
	return result;
      }
    request_region (h->baseaddr, HEPC3_REG_SIZE, h->name);
    
    /* region 1 */
    if ((result = check_region (h->comport [1].data, HEPC3_REG_SIZE)) < 0)
      {
	printk (KERN_WARNING "%s: can't get io at 0x%x\n", h->name, 
		h->comport [1].data);
	release_region (h->baseaddr, HEPC3_REG_SIZE);
	return result;
      }
    request_region (h->comport [1].data, HEPC3_REG_SIZE, h->name);
    
    /* region 2 */
    if ((result = check_region (h->comport [0].cont, HEPC3_REG_SIZE)) < 0)
      {
	printk (KERN_WARNING "%s: can't get io at 0x%x\n", h->name, 
		h->comport [0].cont);
	release_region (h->comport [1].data, HEPC3_REG_SIZE);
	release_region (h->baseaddr, HEPC3_REG_SIZE);
	return result;
      }
    request_region (h->comport [0].cont, HEPC3_REG_SIZE, h->name);
    
    /* region 3 */
    if ((result = check_region (h->comport [1].cont, HEPC3_REG_SIZE)) < 0)
      {
	printk (KERN_WARNING "%s: can't get io at 0x%x\n", h->name, 
		h->comport [1].cont);
	release_region (h->comport [0].cont, HEPC3_REG_SIZE);
	release_region (h->comport [1].data, HEPC3_REG_SIZE);
	release_region (h->baseaddr, HEPC3_REG_SIZE);
	return result;
      }
    request_region (h->comport [1].cont, HEPC3_REG_SIZE, h->name);
    
    /* region 4 */
    if ((result = check_region (h->jtagaddr, HEPC3_JTAG_SIZE)) < 0)
      {
	printk (KERN_WARNING "%s: can't get io at 0x%x\n", h->name, 
		h->jtagaddr);
	release_region (h->comport [1].cont, HEPC3_REG_SIZE);
	release_region (h->comport [0].cont, HEPC3_REG_SIZE);
	release_region (h->comport [1].data, HEPC3_REG_SIZE);
	release_region (h->baseaddr, HEPC3_REG_SIZE);
	return result;
      }
    request_region (h->jtagaddr, HEPC3_JTAG_SIZE, h->name);
    
    return 0;
  }

/*
 * clock the boards system reset
 */
static void hepc3_sys_reset (hepc3_board *h)
  {
#ifdef LINUX_20
    unsigned long j = jiffies + (HZ * HEPC3_RESET_MS) / 1000;
#endif
    unsigned long flags;
    hepc3_comport * cp;
    int k;

    save_flags (flags);
    cli ();
#ifndef HEPC3_WITH_DMA
    /* disable and clear all interrupts */
    outl (HEPC3_INT_MBOX, h->baseaddr + HEPC3_INTCSR);
#else
    outl (HEPC3_INT_MBOX | HEPC3_INT_RRDY | HEPC3_INT_WRDY,
	  h->baseaddr + HEPC3_INTCSR);
    /* abort all running dma transfers */
    outw (0x0, h->baseaddr + HEPC3_MCSR);
    outl (0x0, h->baseaddr + HEPC3_MWTC);
    outl (0x0, h->baseaddr + HEPC3_MRTC);
    h->dma_size [0] = 0;
#endif
    restore_flags (flags);
    outl (HEPC3_SYS_RESET, h->baseaddr + HEPC3_MCSR); /* set reset */
    /* now init all internal structures */
    for (k = 0; k < 2; ++k)
      {
	cp = &(h->comport [k]);
	cp->inrd = cp->inwr = cp->inbuf;
	cp->in_inbuf = 0;
	cp->outrd = cp->outwr = cp->outbuf;
	cp->in_outbuf = 0;
      }
    /* restart the ports */
    h->comport[0].contbits = HEPC3_CPA_RESET | h->comport[0].swapbits;
    flags = 0;
#ifndef HEPC3_WITH_DMA
    if (h->comport[0].in_use)
      {
	h->comport[0].contbits |= HEPC3_CPA_LGBIRQ;
	flags |= HEPC3_ENI_MBOX;
      }
#endif
    h->comport[1].contbits = HEPC3_CPB_RESET | h->comport[1].swapbits;
    if (h->comport[1].in_use)
      {
	h->comport[1].contbits |= HEPC3_CPB_I60IRQ;
	flags |= HEPC3_ENI_MBOX;
      }
    /* 
     * let the reset line be asserted for HEPC3_RESET_MS 
     * 2.2 introduced schedule_timeout; in 2.0 a "raw" method is used
     */
    current->state = TASK_INTERRUPTIBLE;
#ifdef LINUX_20
    current->timeout = j;
    schedule ();
    current->timeout = 0;
#else
    if (!signal_pending (current))
      schedule_timeout ((HZ * HEPC3_RESET_MS) / 1000);
#endif
    outl (0x0, h->baseaddr + HEPC3_MCSR); /* clear reset */
    /* re-initialize the comports and the INTCSR */
    outb (h->comport[0].contbits, h->comport[0].cont);
    outb (h->comport[1].contbits, h->comport[1].cont);
    outl (flags, h->baseaddr + HEPC3_INTCSR);
#ifdef HEPC3_WITH_DMA
    h->dma_size [1] = 0;
#endif
  }

#ifdef HEPC3_WITH_DMA
/*
 * launch a write dma (pci->cpu memory) transfer
 */
static inline void launch_write_dma (hepc3_board *h, u32 *data)
  {
    /* set the number of _bytes_ to be transferred */
    outl (h->dma_size [1] << 2, h->baseaddr + HEPC3_MWTC); 
    /* set the destination address */
    outl (virt_to_bus ((volatile void *)data), h->baseaddr + HEPC3_MWAR);
    /* start the dma ... */
    outw (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_ENI_WRDY,
	  h->baseaddr + HEPC3_INTCSR);
    outw (inw (h->baseaddr + HEPC3_MCSR) | HEPC3_DMA_WRITE | HEPC3_DMA_W4MIN | 
	  HEPC3_DMA_WPRI,  h->baseaddr + HEPC3_MCSR);
  }

/*
 * launch a read dma (cpu memory->pci) transfer
 */
static inline void launch_read_dma (hepc3_board *h, u32 *data)
  {
    /* set the number of _bytes_ to be transferred */
    outl (h->dma_size [0] << 2, h->baseaddr + HEPC3_MRTC);
    /* set the source address */
    outl (virt_to_bus ((volatile void *)data), h->baseaddr + HEPC3_MRAR);
    /* start the dma ... */
    outw (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_ENI_RRDY,
	  h->baseaddr + HEPC3_INTCSR);
    outw (inw (h->baseaddr + HEPC3_MCSR) | HEPC3_DMA_READ | HEPC3_DMA_R4MIN |
	  HEPC3_DMA_RPRI, h->baseaddr + HEPC3_MCSR);
  }

#else /* port A uses interrupts... */

/*
 * try to read count data elements from comport A to buf and return # of
 * read dwords
 */
static inline unsigned int hepc3_portAread (unsigned int status, 
			   unsigned int data, u32 *buf, unsigned int count)
  {
    unsigned int dwread = count;
    register u32 statflgs;

    /* try to read as much as possible to avoid interrupts */
    while (count && 
	   !((statflgs = inb (status) & 
	      (HEPC3_AIN_FULL | HEPC3_AIN_4 | HEPC3_AIN_EMPTY)) & 
	     HEPC3_AIN_EMPTY))                /* leave if fifo is empty */
      {
	register int n;

	if (statflgs & HEPC3_AIN_FULL)        /* read max. 8 elements */
	  n = (count > 8) ? 8 : count;
	else
	  if (statflgs & HEPC3_AIN_4)         /* read max. 4 elements */
	    n = (count > 4) ? 4 : count;
	  else
	    n = 1;                            /* read a single element */
	insl (data, buf, n);
	count -= n;
	buf += n;
      }
    return dwread - count;
  }

/*
 * try to write count data elements from buf to comport A and return # of
 * written dwords
 */
static inline unsigned int hepc3_portAwrite (unsigned int status, 
			   unsigned int data, u32 *buf, unsigned int count)
  {
    unsigned int written = count;
    register u32 statflgs;

    /* try to write as much as possible to avoid interrupts */
    while (count &&
	   !((statflgs = inb (status) & 
	      (HEPC3_AOUT_FULL | HEPC3_AOUT_4 | HEPC3_AOUT_EMPTY)) &
	     HEPC3_AOUT_FULL))            /* leave if fifo is full */
      {
	register int n;

	if (statflgs & HEPC3_AOUT_EMPTY)  /* write max. 8 elements */
	  n = (count > 8) ? 8 : count;
	else
	  if (statflgs & HEPC3_AOUT_4)    /* write max. 4 elements */
	    n = (count > 4) ? 4 : count;
	  else
	    n = 1;                        /* write a single element */
	outsl (data, buf, n);
	count -= n;
	buf += n;
      }
    return written - count;
  }

#endif

/*
 * try to read count data elements from comport B to buf and return # of 
 * read dwords
 */
static inline unsigned int hepc3_portBread (unsigned int status, 
			   unsigned int data, u32 *buf, unsigned int count)
  {
    unsigned int dwread = count;
    register u32 statflgs;

    /* 
     * try to read as much as possible to avoid interrupts 
     * note: port B out-fifo status flags are active low!
     */
    while (count &&
	   ((statflgs = inb (status) &
	     (HEPC3_CPB_INAF | HEPC3_CPB_INE)) & HEPC3_CPB_INE))
      {
	register int n;

	if (statflgs & HEPC3_CPB_INAF)      /* read 1 element */
	  n = 1;
	else                                /* read max. 60 elements */
	  n = (count > 60) ? 60 : count;
	insl (data, buf, n);
	count -= n;
	buf += n;
      }
    return dwread - count;
  }

/*
 * write count data elements from buf to comport B and return # of
 * written bytes
 */
static inline unsigned int hepc3_portBwrite (unsigned int status, 
                           unsigned int data, u32 *buf, unsigned int count)
  {
    unsigned int written = count;
    register u32 statflgs;

    /* 
     * try to read as much as possible to avoid interrupts 
     * note: port B out-fifo status flags are active low! 
     */
    while (count &&
	   ((statflgs = inb (status) &
	     (HEPC3_CPB_OUTAF | HEPC3_CPB_OUTE)) & HEPC3_CPB_OUTAF))
      {
	register int n;

	if (!(statflgs & HEPC3_CPB_OUTE))      /* write max. 64 elements */
	  n = (count > 64) ? 64 : count;
	else
	  n = (count > 4) ? 4 : count;         /* write max. 4 elements */ 
	outsl (data, buf, n);
	count -= n;
	buf += n;
      }
    return written - count;
  }

/*
 * interrupt service routine --- top half
 */
static void hepc3_isr (int irq, void *dev_id, struct pt_regs *regs)
  {
    hepc3_board *h = (hepc3_board *)dev_id;
    unsigned long irq_flags, clear_mask = 0;

    /* is this really my interrupt? */
#ifdef HEPC3_WITH_DMA
    if (!((irq_flags = inl (h->baseaddr + HEPC3_INTCSR)) & 
	  (HEPC3_AMCC_INT | HEPC3_INT_TABT | HEPC3_INT_MABT)))
      return;   /* no: return immediately */
#else
    if (!((irq_flags = inl (h->baseaddr + HEPC3_INTCSR)) & 
	  HEPC3_AMCC_INT))
      return;   /* no: return immediately */
#endif

    /* 
     * to handle the irq in the bottom half: remember irq bits, freeze 
     * mailbox interrupts (to avoid losses), clear irq flags and disable 
     * detected bus master irq's
     */
    h->intcsr_flags |= irq_flags;
    h->imb4_flags |= inl (h->baseaddr + HEPC3_IMB4);
    outb (h->comport [1].contbits | HEPC3_INTFREEZ, h->comport [1].cont);
#ifdef HEPC3_WITH_DMA
    if (irq_flags & HEPC3_INT_RRDY)
      clear_mask |= HEPC3_ENI_RRDY;
    if (irq_flags & HEPC3_INT_WRDY)
      clear_mask |= HEPC3_ENI_WRDY;
#endif
    outl (irq_flags & ~clear_mask, h->baseaddr + HEPC3_INTCSR);

    /* finally queue the bottom half */
    queue_task_irq_off (&(h->isr_bh), &tq_immediate);
    mark_bh (IMMEDIATE_BH);
  }

/*
 * interrupt service routine --- bottom half
 */
static void hepc3_isr_bh (void *board)
  {
    hepc3_board *h = (hepc3_board *)board;
    hepc3_comport *cp;
    int trfmax, res, bogus = 1;

#ifdef HEPC3_WITH_DMA
    /* check the abort signals */
    if (test_and_clear_bit (HEPC3_TABT_IBIT, &h->intcsr_flags))
      {
	bogus = 0;
#ifdef HEPC3_VERBOSE_ERRORS
	printk (KERN_INFO "%s: target abort\n", h->name);
#endif
      }
    if (test_and_clear_bit (HEPC3_MABT_IBIT, &h->intcsr_flags))
      {
	bogus = 0;
#ifdef HEPC3_VERBOSE_ERRORS
	printk (KERN_INFO "%s: master abort\n", h->name);
#endif
      }

    cp = &(h->comport [0]);
    /* check read transfer count is zero (from write call) */
    if (test_and_clear_bit (HEPC3_RRDY_IBIT, &h->intcsr_flags))
      {
	/* disable dma */
	outw (inw (h->baseaddr + HEPC3_MCSR) & 
	      ~(HEPC3_DMA_READ | HEPC3_DMA_R4MIN | HEPC3_DMA_RPRI),
	      h->baseaddr + HEPC3_MCSR);
	bogus = 0;

	/* adjust buffer pointers */
	if (h->dma_size [0])
	  {
	    cp->in_outbuf -= h->dma_size [0];
	    cp->outrd += h->dma_size [0];
	    if (cp->outrd >= cp->outbuf + hepc3_buf_size)
	      cp->outrd = cp->outbuf;
	  }
	else
	  {
#ifdef HEPC3_VERBOSE_ERRORS
	    printk (KERN_INFO "%s: RRDY with no data?\n", h->name);
#endif
	  }
		
	/* if there is data left in the buffer, restart immediately */
	if (cp->in_outbuf)
	  {
	    if (cp->outwr > cp->outrd)
	      h->dma_size [0] = cp->outwr - cp->outrd;
	    else
	      h->dma_size [0] = hepc3_buf_size - (cp->outrd - cp->outbuf);
	    /* set the number of _bytes_ to be transferred */
	    if (inl (h->baseaddr + HEPC3_MRTC))
	      printk (__FUNCTION__ ": MRTC not zero\n");
	    if (!h->dma_size [0])
	      printk (__FUNCTION__ ": dma_size == 0\n");	      
	    outl (h->dma_size [0] << 2, h->baseaddr + HEPC3_MRTC);
	    /* set the source address and launch dma job */
	    outl (virt_to_bus ((volatile void *)cp->outrd), 
		  h->baseaddr + HEPC3_MRAR);
	    outw (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_ENI_RRDY, 
		  h->baseaddr + HEPC3_INTCSR);
	    outw (inw (h->baseaddr + HEPC3_MCSR) | HEPC3_DMA_READ |
		  HEPC3_DMA_R4MIN | HEPC3_DMA_RPRI,
		  h->baseaddr + HEPC3_MCSR);
	  }
	else
	  {
	    h->dma_size [0] = 0;
	  }
	wake_up_interruptible (&(cp->outq));
      }

    /* check write transfer count is zero (from read call) */
    if (test_and_clear_bit (HEPC3_WRDY_IBIT, &h->intcsr_flags))
      {
	/* disable dma */
	outw (inw (h->baseaddr + HEPC3_MCSR) & 
	      ~(HEPC3_DMA_WRITE | HEPC3_DMA_W4MIN | HEPC3_DMA_WPRI),
	      h->baseaddr + HEPC3_MCSR);
	bogus = 0;

	/* adjust buffer pointers */
	if (h->dma_size [1])
	  {
	    cp->in_inbuf += h->dma_size [1];
	    cp->inwr += h->dma_size [1];
	    if (cp->inwr >= cp->inbuf + hepc3_buf_size)
	      cp->inwr = cp->inbuf;
	    h->dma_size [1] = 0;   /* indicate no dma in progress */
	  }
	else
	  {
#ifdef HEPC3_VERBOSE_ERRORS
	    printk (KERN_INFO "%s: WRDY with no data?\n", h->name);
#endif
	  }
	wake_up_interruptible (&(cp->inq));
      }
#endif

    /* check mailbox sources */
    if (test_and_clear_bit (HEPC3_MBOX_IBIT, &h->intcsr_flags))
      {
	outl (inw (h->baseaddr + HEPC3_INTCSR) | HEPC3_INT_MBOX, 
	      h->baseaddr + HEPC3_INTCSR);	
#ifndef HEPC3_WITH_DMA     /* dma irq's on port A are handled above */
	cp = &(h->comport [0]);
	/* check comport A "Latched-Gone-By" (data readble) */
	if (test_and_clear_bit (HEPC3_A_LGBIBIT, &h->imb4_flags))
	  {
	    bogus = 0;
	    if (cp->in_inbuf < hepc3_buf_size)
	      {
		if (cp->inrd > cp->inwr)
		  trfmax = cp->inrd - cp->inwr;
		else
		  trfmax = hepc3_buf_size - (cp->inwr - cp->inbuf);
		res = hepc3_portAread (cp->fifo, cp->data, cp->inwr, trfmax);
		/*
		 * Do not clear the LGB flag if there is data in the fifo.
		 * Even if we do a cli before, this does NOT guarantee that we
		 * don't lose a LGB event; a new dword might be transferred to
		 * the fifo between reading the status and de-asserting LGBCLR.
		 */
		if (inb (cp->fifo) & HEPC3_AIN_EMPTY)
		  {
		    unsigned long flags;

		    save_flags (flags);
		    cli ();
		    outb (cp->contbits | HEPC3_CPA_LGBCLR, cp->cont);
		    outb (cp->contbits, cp->cont);
		    restore_flags (flags);
		  }
		/* adjust pointers and counter */
		cp->in_inbuf += res;
		cp->inwr += res;
		if (cp->inwr >= cp->inbuf + hepc3_buf_size)
		  cp->inwr = cp->inbuf;
		
		if (cp->in_inbuf == hepc3_buf_size)
		  cp->contbits &= ~HEPC3_CPA_LGBIRQ;
	      }
	    else
	      cp->contbits &= ~HEPC3_CPA_LGBIRQ;
	    wake_up_interruptible (&(cp->inq));
	  }
	
	/* check comport A data writeable */
	if (test_and_clear_bit (HEPC3_A_WRIBIT, &h->imb4_flags))
	  {
	    bogus = 0;
	    if (cp->in_outbuf)
	      {
		if (cp->outwr > cp->outrd)
		  trfmax = cp->outwr - cp->outrd;
		else
		  trfmax = hepc3_buf_size - (cp->outrd - cp->outbuf);
		res = hepc3_portAwrite (cp->fifo, cp->data, cp->outrd, trfmax);
		cp->in_outbuf -= res;
		cp->outrd += res;
		if (cp->outrd >= cp->outbuf + hepc3_buf_size)
		  cp->outrd = cp->outbuf;
		
		if (cp->in_outbuf == 0)
		  cp->contbits &= ~HEPC3_CPA_OUTIRQ;
	      }
	    else
	      cp->contbits &= ~HEPC3_CPA_OUTIRQ;
	    wake_up_interruptible (&(cp->outq));
	  }
#endif
	
	/* check comport B data readable */
	cp = &(h->comport [1]);
	if (test_and_clear_bit (HEPC3_B_RDIBIT, &h->imb4_flags) + 
	    test_and_clear_bit (HEPC3_B_RD60IBIT, &h->imb4_flags))
	  {
	    bogus = 0;
	    if (cp->in_inbuf < hepc3_buf_size)
	      {
		if (cp->inrd > cp->inwr)
		  trfmax = cp->inrd - cp->inwr;
		else
		  trfmax = hepc3_buf_size - (cp->inwr - cp->inbuf);
		res = hepc3_portBread (cp->fifo, cp->data, cp->inwr, trfmax);
		cp->in_inbuf += res;
		cp->inwr += res;
		if (cp->inwr >= cp->inbuf + hepc3_buf_size)
		  cp->inwr = cp->inbuf;
		
		if (cp->in_inbuf == hepc3_buf_size)
		  cp->contbits &= ~(HEPC3_CPB_I60IRQ | HEPC3_CPB_INIRQ);
	      }
	    else
	      cp->contbits &= ~(HEPC3_CPB_I60IRQ | HEPC3_CPB_INIRQ);
	    wake_up_interruptible (&(cp->inq));
	  }
	
	/* check comport B data writable */
	if (h->imb4_flags & HEPC3_B_WRIRQ)
	  {
	    bogus = 0;
	    if (cp->in_outbuf)
	      {
		if (cp->outwr > cp->outrd)
		  trfmax = cp->outwr - cp->outrd;
		else
		  trfmax = hepc3_buf_size - (cp->outrd - cp->outbuf);
		res = hepc3_portBwrite (cp->fifo, cp->data, cp->outrd, trfmax);
		cp->in_outbuf -= res;
		cp->outrd += res;
		if (cp->outrd >= cp->outbuf + hepc3_buf_size)
		  cp->outrd = cp->outbuf;
		
		if (cp->in_outbuf == 0)
		  cp->contbits &= ~HEPC3_CPB_OUTIRQ;
	      }
	    else
	      cp->contbits &= ~HEPC3_CPB_OUTIRQ;
	    wake_up_interruptible (&(cp->outq));
	  }
	
	/* write port A status bits back */
	outb (h->comport [0].contbits, h->comport [0].cont);
      }
    /* write port B status bits back (will also clear intfreez) */
    outb (h->comport [1].contbits, h->comport [1].cont);

#ifdef HEPC3_VERBOSE_ERRORS
    if (bogus)
      {
	/*
	 * A (mailbox) irq occurred, but no flag was set. I hope I removed
	 * the source (bad order of commands)...
	 */
	printk (KERN_INFO "%s: bogus interrupt INTCSR = 0x%x IMB4 = 0x%x \n", 
		h->name, h->intcsr_flags, h->imb4_flags);
      }
#endif
  }

/*
 * write some info for proc
 */
#define LIMIT (PAGE_SIZE - 80)
static int hepc3_read_procmem (char *buf, char **start, off_t offset,
			       int len, int unused)
  {
    int b, p;
    hepc3_board *h;
    hepc3_comport *cp;

    for (b = 0; hepc3 [b]; ++b);
    len = sprintf (buf, "\n\tHunt Engineering HEPC3/HEPC4/HECPCI1 [C]PCI TIM40"
		   " motherboards\n\t%d board(s), device major number is %d\n"
		   "\tdriver version " HEPC3_DRV_VER "\n",
		   b, hepc3_major);
#ifdef HEPC3_WITH_DMA
    len += sprintf (buf + len, 
		    "\tport A uses DMA io, port B interrupt driven io\n\n");
#else
    len += sprintf (buf + len, "\tport A and B use interrupt driven io\n\n");
#endif
    for (b = 0; b < HEPC3_MAX_BOARD; ++b)
      if ((h = hepc3 [b]))
	{
	  len += sprintf (buf + len, "%s: Rev. %d, IRQ=%d, IO at 0x%x, 0x%x,"
			  " 0x%x, 0x%x, 0x%x\n", h->name, h->revid, h->irq,
			  h->baseaddr, h->comport [1].data, h->comport [0].cont,
			  h->comport [1].cont, h->jtagaddr);
	  if (len > LIMIT)
	    return len;
	  for (p = 0; p < 2; ++p)
	    {
	      cp = &(h->comport [p]);
	      len += sprintf (buf + len, "  %c: %s (0x%x), buffered (i/o) "
			      "%d/%d; byte swap %sabled\n", 'A' + p, 
			      cp->in_use ? "active" : "passive", 
			      cp->contbits, cp->in_inbuf, cp->in_outbuf,
			      cp->swapdata ? "en" : "dis");
	      if (len > LIMIT)
		return len;
	    }
#ifdef HEPC3_VERBOSE_ERRORS
#ifdef HEPC3_WITH_DMA
	  len += 
	    sprintf (buf + len, "  IMB4=%08x INTCSR=%08x MCSR=%08x\n"
		     "  MWAR=%08x MWTC=%08x MRAR=%08x MRTC=%08x\n"
		     "  dmasize[0]=%d [1]=%d\n",
	      inl (h->baseaddr + HEPC3_IMB4), inl (h->baseaddr + HEPC3_INTCSR),
              inl (h->baseaddr + HEPC3_MCSR), inl (h->baseaddr + HEPC3_MWAR),
	      inl (h->baseaddr + HEPC3_MWTC), inl (h->baseaddr + HEPC3_MRAR),
	      inl (h->baseaddr + HEPC3_MRTC), h->dma_size[0], h->dma_size[1]);
#else
	  len += 
	    sprintf (buf + len, "  IMB4=%08x INTCSR=%08x MCSR=%08x\n"
		     "  MWAR=%08x MWTC=%08x MRAR=%08x MRTC=%08x\n",
	      inl (h->baseaddr + HEPC3_IMB4), inl (h->baseaddr + HEPC3_INTCSR),
              inl (h->baseaddr + HEPC3_MCSR), inl (h->baseaddr + HEPC3_MWAR),
	      inl (h->baseaddr + HEPC3_MWTC), inl (h->baseaddr + HEPC3_MRAR),
	      inl (h->baseaddr + HEPC3_MRTC));
#endif
	  if (len > LIMIT)
	    return len;
#endif
	}
    return len;
  }
