/*
 * Simple device driver for DAS8AO A/D & D/A card (version 0.1).
 * Only basic operation is implemented (read, write and some settings).
 *
 * Written by eloranta@voima.jkl.fi
 *            eloranta@dirac.chem.jyu.fi
 *
 * TODO: implement all features of DAS8AO board. And use interrupts
 *       rather than polling. Should read/write go via read() and write()
 *       or ioctl()?
 *
 * Thanks to Glenn Moloney for instructive LabPC driver.
 * 
 */

#define __KERNEL__

#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/malloc.h>

#include <asm/io.h>
#include <asm/segment.h>
#include <asm/system.h>

#include <stdlib.h>

#include "kernel_version.h"

#include "das8.h"

static struct DAS8_t *board = NULL;

static struct semaphore mutex = MUTEX;

static int das8_open(struct inode *inode, struct file *file) {

/*  int minor = MINOR(inode->i_rdev); */

  if (board == NULL || board->open != 0)
    return -ENODEV;

  board->open = 1;
  return 0;
}

static void das8_close(struct inode *inode, struct file *file) {

  board->open = 0;
  return;
}

static int das8_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		      unsigned long arg) {

  int *uptr, stat, xl, xh;

  if(board == NULL)
    return -EINVAL; /* card not initialized */

  switch (cmd) {
  case DAS8_SETC_AD: /* set current A/D channel */
    if(arg > 7)
      return -EINVAL; /* no such channel */
    board->cur_ad = (int) arg;
    break;
  case DAS8_SETC_DA: /* set current D/A channel */
    if(arg < 1 && arg > 2)
      return -EINVAL;
    board->cur_da = (int) arg;
    break;
  case DAS8_SETR_AD: /* set A/D conversion resolution */
    if(arg != DAS8_8BIT && arg != DAS8_12BIT)
      return -EINVAL;
    board->resol = (int) arg;
    break;
  case DAS8_SETG_AD: /* set A/D input gain */
    if(arg != 0 && (arg < DAS8_BI_10V || arg > DAS8_UNI_002V))
      return -EINVAL;
    board->gain = (int) arg;
    outb_p((char) arg,board->base+3);
    break;
  case DAS8_GETV_AD: /* get data from current A/D channel */
    uptr = (int *) arg;
    if((stat = verify_area(VERIFY_WRITE, (char *) uptr, sizeof(int))) != 0)
      return stat;
    /* anding with 0x7 means: no digital outputs and no IRQ */
    outb_p((char) board->cur_ad & 0x7, board->base+2);
    down(&mutex);
    /* start conversion */
    if(board->resol == DAS8_8BIT)
      outb_p(0, board->base); /* 8 bit */
    else
      outb_p(0, board->base+1); /* 12 bit */
    while(inb_p(board->base+2) >= 128);
    xl = (int) inb_p(board->base);
    xh = (int) inb_p(board->base+1);
    put_fs_long((xh*16 + xl/16), uptr);
    up(&mutex);
    break;
  case DAS8_PUTV_DA:
    /* voltage range set by dip switches */
    if(arg < 0 || arg > 4095)
      return -EINVAL;
    xh = (arg & 0x0f00) >> 8;
    xl = arg & 0xff;
    stat = (board->cur_da?1:0) * 2;
    cli();
    outb_p(xl, board->base + 8 + stat);
    outb_p(xh, board->base + 9 + stat);
    sti();
    break;
  default:
    return -EINVAL;
  }

  return 0;
}

static struct file_operations das8_fops = {
  NULL,		/* seek */
  NULL,		/* read */
  NULL,		/* write */
  NULL,		/* readdir */
  NULL,		/* select */
  das8_ioctl,	/* ioctl (this does all the work!) */
  NULL,		/* mmap */
  das8_open,    /* open */
  das8_close,	/* release */
  NULL,
  NULL
};

#ifdef __cplusplus
extern "C" {
#endif

int init_module(void) {
  
  int i;

  /* we test the presence of the card by setting values to base+2
   * and check if we got the appropriate results.
   */
  
  /* loop through every channel */
  /* I don't think this is safe ? */
#if 0
  for (i = 0; i < 7; i++) {
    outb_p((unsigned char) i, DAS8_BASE+2);
    if((inb_p(DAS8_BASE+3) & 0x80) >> 4 != i) break;
  }
  if(i != 7) {
    printk("DAS8: board not found.\n");
    return -EIO;
  }
  printk("DAS8: Board found at base address %x.\n", DAS8_BASE);
#else
  printk("DAS8: Board assumed to be at base address %x.\n", DAS8_BASE);
#endif
  if((board = (struct DAS8_t *) kmalloc(sizeof(struct DAS8_t), GFP_KERNEL))
     == NULL)
    return -ENOMEM;
  
  board->major   = DAS8MAJOR;
  board->base    = DAS8_BASE;
  board->stat    = 0;
  board->open    = 0;
  board->present = 1;
  board->cur_ad  = 0;           /* current A/D channel */
  board->cur_da  = 0;           /* current D/A channel */
  board->gain    = DAS8_BI_10V; /* current gain setting */
  board->resol   = DAS8_8BIT;   /* 8 bit resolution */
  
  /* Set defaults for A/D channels */
  /* gain setting to +- 10 Volts */
  outb_p(8, DAS8_BASE+3);
  /* A/D channel 0, digital outputs to 0 V and no IRQ */
  outb_p(0, DAS8_BASE+2);
    
  /* register loadable module */
  if (register_chrdev(DAS8MAJOR, "das8", &das8_fops) < 0) {
    printk( "DAS8: Unable to get major %d for DAS8 device.\n", board->major);
    kfree(board);
    return -EINVAL;
  } else
    printk( "DAS8 Card detected. Driver installed.\n" );
  
  return 0;
}

void cleanup_module(void) {

  if (MOD_IN_USE) {
    printk("DAS8: device busy, remove delayed.\n");
    return;
  }

  if (unregister_chrdev(DAS8MAJOR, "das8") != 0) {
    printk("DAS8: device busy or other module error.\n");
    return;
  } else {
    printk("DAS8: succesfully removed.\n");
  } 
  
  kfree(board);
}

#ifdef __cplusplus
}
#endif
