/*
	irq - driver to delivery hardware interrupts to user space.

	(C) 1997 Dmitry A. Fedorov, fedorov@inp.nsk.su
	Copying-policy: GNU GPL

	$Id: irq.c,v 1.85 2003/06/12 11:30:46 fedorov Exp $
*/

#define DRIVER_NAME        "irq"
#define DRIVER_DESCRIPTION "IRQ for user space driver"
#define DRIVER_COPYRIGHT   "(C) 1997 Dmitry A. Fedorov, fedorov@inp.nsk.su"
#define DRIVER_LICENSE     "GPL"

#ifdef __cplusplus
# error This is a C code
#endif

#if !defined(__KERNEL__) || !defined(linux)
# error This is a Linux kernel module
#endif

#ifndef  IRQ_MAJOR
# define IRQ_MAJOR 60			// experimental
#endif


#include <linux/compat/module.h>

# if  LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
#  error Kernel versions >= 2.5.0 yet not supported
# endif

# if  LINUX_VERSION_CODE < KERNEL_VERSION(2,0,0)
#  error Kernel versions < 2.0.0 not supported
# endif


#include <linux/compat/init.h>
#include <linux/compat/atomic.h>
#include <linux/compat/bitops.h>
#include <linux/compat/uaccess.h>
#include <linux/compat/system.h>		// xchg_compat()
#include <linux/compat/wait.h>			// wait_queue_head
#include <linux/compat/spinlock.h>
#include <linux/compat/fs.h>			// file_operations
#include <linux/compat/poll.h>			// poll_wait()
#include <linux/compat/sched.h> 		// NEED_RESCHED
#include <linux/compat/errno.h> 		// ENOTSUP
#include <linux/compat/malloc.h>		// kmalloc(), kfree()
#include <linux/compat/smp_lock.h>		// LOCK_KERNEL_COMPAT()

#include <linux/types.h>
#include <asm/irq.h>				// NR_IRQS
#include <linux/major.h>			// MAX_CHRDEV

#include <linux/df/pr.h>			// pr_error(), pr_endl()
#include <linux/df/assert.h>			// assert_retvoid(), assert_retval()
#include <linux/df/save_flags_and_cli.h>
#include <linux/df/strdup.h>

#include <irq.h>				// API definitions
#include "./.release.h"				// PROJECT_RELEASE


#define REGPARM(n) __attribute__(( regparm(n) ))

#define UNUSED __attribute__(( unused ))


//----------------------------------------------------------------------
//+ local static variables

static const char        name[]             = DRIVER_NAME;
static const char description[] __init_data = DRIVER_DESCRIPTION;
static const char     version[] __init_data = PROJECT_RELEASE;
static const char   copyright[] __init_data = DRIVER_COPYRIGHT;

static int major = IRQ_MAJOR;	// 0 - auto - it works, but useless

//- local static variables
//----------------------------------------------------------------------
//+ module parameters (non-resident .modinfo section)

MODULE_AUTHOR          (DRIVER_COPYRIGHT);
MODULE_DESCRIPTION     (DRIVER_DESCRIPTION);
MODULE_SUPPORTED_DEVICE(DRIVER_NAME);
MODULE_LICENSE         (DRIVER_LICENSE);

MODULE_PARM     (major , "1i");
MODULE_PARM_DESC(major , "device major number");

//- module parameters (non-resident .modinfo section)
//----------------------------------------------------------------------
//+ declarations

typedef enum { false=0, true=1, Off=false, On=true } bool;
typedef struct __irq_request_arg __irq_request_arg;
typedef struct inode Inode;
typedef struct file File;
typedef struct task_struct task_struct;

//- declarations
//----------------------------------------------------------------------
//+ utilities

# include <linux/df/strdup.imp>

static int nomem(void)
{
	pr_error("%s: Out of memory!\n", name);
	return -ENOMEM;
}

extern inline bool has_readwrite_access(const File* file)
{
	return ( (file->f_flags&O_ACCMODE) != O_RDWR ) ? false : true;
}


//++ irq utilities

extern inline uint irq_minor(ushort minor)
{
	return minor;	// /dev/irqN has minor N
}

extern inline uint irq_inode(const Inode* inode)
{
	return irq_minor(INODE_MINOR(inode));
}

extern inline uint irq_file(const File* file)
{
	return irq_minor(FILE_MINOR(file));
}

//-- irq utilities

//- utilities
//----------------------------------------------------------------------
//+ declarations


//++ file_operations
static int              irq_open (Inode* inode, File* file);
static fops_close_ret_t irq_close(Inode* inode, File* file);

static FOPS_SEEK_PROTOTYPE(irq_seek,inode,file,offset,origin);

static int irq_ioctl(Inode* inode, File* file, uint cmd, ulong arg);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,1,23)
static int  irq_select(Inode* inode, File* file, int sel_type, select_table* wt);
#else
static uint irq_poll(struct file* file, poll_table* pt);
#endif
//-- file_operations


//++ data structures and methods

typedef struct  irq_channel  irq_chan;
typedef struct task_channel task_chan;



// interrupt delivery channel
struct irq_channel
{
#ifndef NDEBUG
	ulong signature;		// assertions
# define ICH_SIG 0xdeadbeef
#endif

	File* file;
	uint irq;				// access speed-up
	task_chan* tch;			// my container
	int sig;				// sig num for delivery, -1 for select(2) only
	int flags;				// enum irq_flags
	char* devname;			// for info in /proc/interrupt

	wait_queue_head_t wqh;

	size_t irq_count;		// received interrupts
	size_t sig_count;		// signals sent
	size_t sel_count;		// interrupt counter for irq_select()
	size_t skipping_count;	// local skipping mode counter
};

static irq_chan* ich_ctor(File* file, int sig, int flags);
static void ich_dtor(irq_chan* ich);
static void ich_attach_tch(irq_chan* ich, task_chan* tch);
static void ich_detach_tch(irq_chan* ich);




// user task's interrupt delivery channel
struct task_channel
{
#ifndef NDEBUG
	ulong signature;			// assertions
# define TCH_SIG 0xbeefdead
#endif

	task_chan* next;			// list
	task_chan* prev;			//

	irq_chan* ichs[NR_IRQS];	// interrupt delivery channels
	size_t nr_ich;

	task_struct* owner_task;	// interrupt requester/owner
	pid_t owner_pid;			//
};

static task_chan* tch_ctor(task_struct* task);
static void       tch_dtor(task_chan* tch);
static void       tch_link(task_chan* tch);
static void       tch_unlink(task_chan* tch);
static task_chan* tch_find(const task_struct* task);
static task_chan* tch_new_link(task_struct* task);
#if 0
static task_chan* tch_find_new_link(task_struct* task);
#endif
static void tch_attach_ich(task_chan* tch, irq_chan* ich);
static void tch_detach_ich(task_chan* tch, uint irq);


// are all of interrupt delivery channels empty?
extern inline bool tch_is_empty(const task_chan* tch)
{
	assert_retval(tch!=NULL, false);
	assert_retval(tch->signature==TCH_SIG, false);
	return tch->nr_ich==0;
}

// delete self if all of interrupt delivery channels are empty.
// returns empty flag.
extern inline bool tch_dtor_if_empty(task_chan* tch)
{
	return tch_is_empty(tch) ? (tch_dtor(tch), true) : false;
}


static bool check_ownership         (const irq_chan* ich) REGPARM(1);

static int  check_null_and_ownership(const irq_chan* ich) REGPARM(1);

//-- data structures and methods


//++
static int do_request(File* file, int sig, int flags);

static int register_irq_handler(irq_chan* ich)	REGPARM(1);
static bool    free_irq_ich_tch(irq_chan* ich)	REGPARM(1);

static void skipping_on        (irq_chan* ich)	REGPARM(1);
static int  skipping_off       (irq_chan* ich)	REGPARM(1);
static void skipping_sub       (irq_chan* ich)	REGPARM(1);

static void irq_simulate(uint irq);

static void irq_handler(int irq, void* dev_id, struct pt_regs* ptregs);
static void do_irq(irq_chan* ich) REGPARM(1);
//--


//- declarations
//----------------------------------------------------------------------
//+ common variables & tables


// file_operations switcher
static struct file_operations fops =
	COMPAT_FOPS_INIT(
		irq_seek,		// for error reporting only
		NULL,			// read
		NULL,			// write
		NULL,			// readdir
		irq_select,
		irq_poll,
		irq_ioctl,
		NULL,			// mmap
		irq_open,
		NULL,			// flush,
		irq_close,
		NULL,			// fsync
		NULL,			// fasync
		NULL,			// check_media_change
		NULL,			// revalidate
		NULL,			// lock
		NULL,			// readv
		NULL,			// writev
		NULL,			// sendpage
		NULL			// get_unmapped_area
	);


static size_t irq_channels_registered[NR_IRQS] = { [0 ... NR_IRQS-1] = 0 };

// list of tasks with opened interrupt delivery channels
static task_chan* tchlist=NULL;

// skipping mode counter
static size_t skipping_irq_counts[NR_IRQS] = { [0 ... NR_IRQS-1] = 0 };

static size_t disabled_irq_counts[NR_IRQS] = { [0 ... NR_IRQS-1] = 0 };


//- common variables & tables
//----------------------------------------------------------------------


#if LINUX_VERSION_CODE <= KERNEL_VERSION(2,1,0)

static void REGPARM(1) DISABLE_IRQ(uint irq)
{
	assert_retvoid(irq<NR_IRQS);
	disabled_irq_counts[irq] = 1;
	disable_irq(irq);
}

static void REGPARM(1) ENABLE_IRQ(uint irq)
{
	assert_retvoid(irq<NR_IRQS);
	disabled_irq_counts[irq] = 0;
	enable_irq(irq);
}

#else

static void REGPARM(1) DISABLE_IRQ(uint irq)
{
	assert_retvoid(irq<NR_IRQS);
	++disabled_irq_counts[irq];
	disable_irq(irq);
}

static void REGPARM(1) ENABLE_IRQ(uint irq)
{
	assert_retvoid(irq<NR_IRQS);
	if (disabled_irq_counts[irq] > 0)
	{
		--disabled_irq_counts[irq];
		enable_irq(irq);
	}
}

#endif


//----------------------------------------------------------------------
//+ debugging

static void _prefix(const char* file, int line, const char* func)
{
	printk("%s:%d %s(): ", file, line, func);
}

#define prefix() _prefix(__FILE__,__LINE__,__FUNCTION__)


#ifdef DEBUG

static size_t msg_counter = 0;

static void print_msg_counter(void)
{
	printk("%u: ", msg_counter++);
}

static void pfx(uint irq)
{
	printk("%s[%u]: ", name, irq);
}

# define PFX(irq)		pfx(irq)
# define PREFIX()		prefix()
# define PRDEBUG(fmt,arg...)	printk(fmt,##arg)
# define PRMSGCOUNTER()	print_msg_counter()
# define PRENDL() pr_endl()
#
#else
#
# define PFX(irq)
# define PREFIX()
# define PRDEBUG(fmt,arg...)
# define PRMSGCOUNTER()
# define PRENDL()
#
#endif



#ifdef DEBUG

static void ICH_PRINT(const irq_chan* ich)
{
	ulong flags;
	irq_chan tich;

	if (ich==NULL)
	{
		printk("!NULL!");
		return;
	}

	flags=save_flags_and_cli();
	memcpy(&tich, ich, sizeof(tich));
	restore_flags(flags);

	printk( "(devname=%s irq=%d tch=%p sig=%d flags=0x%08X "
			"irq_count=%u sig_count=%u)",
		tich.devname, tich.irq, tich.tch, tich.sig, tich.flags,
		tich.irq_count, tich.sig_count);
}


static void ICHLIST_PRINT(const task_chan* tch)
{
	int i;

	if (tch==NULL)
		printk("!NULL!");

	assert_retvoid(tch->signature==TCH_SIG);

	printk("[");
	for (i=0; i<NR_IRQS; i++)
	{
		assert_retvoid(tch->ichs[i]==NULL || tch->ichs[i]->signature==ICH_SIG);
		printk(" %p",tch->ichs[i]);
	}
	printk(" ]");
}


static void TCHLIST_PRINT(void)
{
	task_chan* tch;

	for(tch=tchlist; tch!=NULL; tch=tch->next)
	{
		assert_break(tch->signature==TCH_SIG);

		printk("%s %s(): tch=%p next=%p, ichlist: ",
			name, __FUNCTION__, tch, tch->next);
		ICHLIST_PRINT(tch);

		pr_endl();
	}
}

#else	// ndef DEBUG

# define ICHLIST_PRINT(tch)
# define TCHLIST_PRINT()
# define ICH_PRINT(ich)

#endif	// def DEBUG


//- debugging
//----------------------------------------------------------------------
//+ implementation

//++ data structures and methods

// allocate memory for interrupt delivery channel structure and initialize.
// returns NULL on memory exhausted.
static irq_chan* ich_ctor(File* file, int sig, int flags)
{
	register irq_chan* ich;

	assert_retval(irq_file(file) < NR_IRQS, NULL);

	ich=kmalloc(sizeof(irq_chan), GFP_KERNEL);

	if (ich!=NULL)
	{
#ifndef NDEBUG
		ich->signature=ICH_SIG;
#endif

		ich->file=file;
		file->private_data = ich;
		ich->irq=irq_file(file);
		ich->tch=NULL;
		ich->sig=sig;
		ich->flags=flags;
		ich->devname=NULL;

		init_waitqueue_head(&ich->wqh);

		ich->irq_count=0;
		ich->sig_count=0;
		ich->sel_count=0;
		ich->skipping_count=0;
	}

	PREFIX(); PRDEBUG("ich=%p ich=", ich); ICH_PRINT(ich); PRENDL();
	return ich;
}


static void REGPARM(1) ich_dtor(irq_chan* ich)
{
	assert_retvoid(ich!=NULL);
	assert_retvoid(ich->signature==ICH_SIG);
	assert_retvoid(ich->tch==NULL);
	PREFIX(); ICH_PRINT(ich); PRENDL();

	ich->file->private_data = NULL;

#ifndef NDEBUG
	ich->signature=0;
	ich->file=NULL;
	ich->tch=NULL;
#endif

	kfree(ich->devname);

#ifndef NDEBUG
	ich->devname=NULL;
#endif

	kfree(ich);
}


static void ich_attach_tch(irq_chan* ich, task_chan* tch)
{
	assert_retvoid(ich!=NULL);
	assert_retvoid(ich->signature==ICH_SIG);
	assert_retvoid(tch!=NULL);
	assert_retvoid(tch->signature==TCH_SIG);

	ich->tch=tch;
}


static void ich_detach_tch(irq_chan* ich)
{
	assert_retvoid(ich!=NULL);
	assert_retvoid(ich->signature==ICH_SIG);
	assert_retvoid(ich->tch!=NULL);
	assert_retvoid(ich->tch->signature==TCH_SIG);

	ich->tch=NULL;
}



// allocate memory for task channel structure and initialize.
// returns NULL on memory exhausted.
static task_chan* tch_ctor(task_struct* task)
{
	register task_chan* tch=kmalloc(sizeof(task_chan),GFP_KERNEL);

	if (tch!=NULL)
	{
#ifndef NDEBUG
		tch->signature=TCH_SIG;
#endif
		tch->next=NULL;
		tch->prev=NULL;
		memset(tch->ichs, 0, sizeof(tch->ichs));
		tch->nr_ich=0;
		tch->owner_task=task;
		tch->owner_pid=task->pid;
	}
	return tch;
}


static void tch_dtor(task_chan* tch)
{
	assert_retvoid(tch!=NULL);
	assert_retvoid(tch->signature==TCH_SIG);

	tch_unlink(tch);
	if (tch_is_empty(tch))
	{
		assert_retvoid(tch->next==NULL);
		assert_retvoid(tch->prev==NULL);
		assert_retvoid(tch->nr_ich==0);

#ifndef NDEBUG
		memset(&tch->ichs, 0, sizeof(tch->ichs));
		tch->owner_task=NULL;
		tch->owner_pid=-1;
#endif
		kfree(tch);
	}
	else
	{
		prefix(); printk("task_chan* %p is not empty!\n", tch);
	}
}


// insert new and cleared task channel at begin of list.
static void tch_link(task_chan* tch)
{
	assert_retvoid(tch!=NULL);
	assert_retvoid(tch->signature==TCH_SIG);
	assert_retvoid(tch->next==NULL);
	assert_retvoid(tch->prev==NULL);

	if (tchlist!=NULL)
	{
		assert_retvoid(tchlist->signature==TCH_SIG);

		tch->next=tchlist;
		tchlist->prev=tch;
	}
	tchlist=tch;
}


static void tch_unlink(task_chan* tch)
{
	assert_retvoid(tch!=NULL);
	assert_retvoid(tch->signature==TCH_SIG);

	assert_retvoid(tchlist==NULL || tchlist->signature==TCH_SIG);

	assert_retvoid(tch->next==NULL || tch->next->signature==TCH_SIG);
	assert_retvoid(tch->prev==NULL || tch->prev->signature==TCH_SIG);

	assert_retvoid(tch->next==NULL || tch->next->prev == tch);
	assert_retvoid(tch->prev==NULL || tch->prev->next == tch);

	if (tch->next!=NULL)		// I am not a last member in list
		tch->next->prev = tch->prev;

	if (tch->prev!=NULL)		// I am not a first member in list
		tch->prev->next = tch->next;
	else						// I am a first member in list
		tchlist=tch->next;

#ifndef NDEBUG
	tch->next = NULL;       /* 19.11.2001 (henry) */
	tch->prev = NULL;       /* tch_dtor check this with assert */
#endif
}


// find task channel.
static task_chan* tch_find(const task_struct* task)
{
	task_chan* tch;

	for(tch=tchlist; tch!=NULL; tch=tch->next)
	{
		assert_break(tch->signature==TCH_SIG);

		if (tch->owner_task==task && tch->owner_pid==task->pid)
			return tch;
	}

	return NULL;
}


// create new task channel and add it to list.
// returns NULL on ENOMEM.
static task_chan* tch_new_link(task_struct* task)
{
	task_chan* tch;

	if ((tch=tch_ctor(task))==NULL)
		return NULL;
	assert_retval(tch->signature==TCH_SIG, NULL);
	tch_link(tch);
	return tch;
}


#if 0
// find task channel, if isn't create new and add it to list.
// returns NULL on ENOMEM.
static task_chan* tch_find_new_link(task_struct* task)
{
	task_chan* tch=tch_find(task);
	if (tch!=NULL) return tch;

	return tch_new_link(task);
}
#endif


static void tch_attach_ich(task_chan* tch, irq_chan* ich)
{
	assert_retvoid(tch!=NULL);
	assert_retvoid(tch->signature==TCH_SIG);
	assert_retvoid(ich!=NULL);
	assert_retvoid(ich->signature==ICH_SIG);

	assert_retvoid(ich->irq<NR_IRQS);
	assert_retvoid(tch->ichs[ich->irq]==NULL);

	tch->ichs[ich->irq] = ich;
	tch->nr_ich++;
	ich_attach_tch(ich,tch);
}


static void tch_detach_ich(task_chan* tch, uint irq)
{
	assert_retvoid(tch!=NULL);
	assert_retvoid(tch->signature==TCH_SIG);
	assert_retvoid(irq<NR_IRQS);
	assert_retvoid(tch->ichs[irq] != NULL);
	assert_retvoid(tch==tch->ichs[irq]->tch);
	assert_retvoid(tch->nr_ich>0);

	ich_detach_tch(tch->ichs[irq]);
	tch->ichs[irq] = NULL;
	tch->nr_ich--;
}


//returns error flag
static bool check_ownership(const irq_chan* ich)
{
	assert_retval(ich!=NULL, true);
	assert_retval(ich->signature==ICH_SIG, true);
	assert_retval(ich->tch!=NULL, true);
	assert_retval(ich->tch->signature==TCH_SIG, true);

	return (ich->tch->owner_pid != current->pid);
}

//returns -errno or 0
static int check_null_and_ownership(const irq_chan* ich)
{
	if (ich==NULL)
		return -EPERM;

	if (check_ownership(ich))
		return -EACCES;

	return 0;
}

//-- data structures and methods


//- implementation
//----------------------------------------------------------------------
//+ init/cleanup


static int __init_text _driver_init(void)
{
	int rc;

	EXPORT_NO_SYMBOLS;

	pr_info("%s driver %s\n%s    %s\n",
		name,version, KERN_INFO,copyright);
	pr_info("%s: ", name);
	if ( (rc=register_chrdev(major,name,&fops)) <0 )
	{
		printk("registration failed: ");
		if (rc==-EBUSY)
		{
			if (major==0)
				printk("no free major slots");
			else
				printk("major %d is busy",major);
		}
		else if (rc==-EINVAL)
				printk("major number %d > %d", major, MAX_CHRDEV-1);
		else
			printk("rc = %d",rc);
		pr_endl();
		return rc;
	}


	if (major==0) major=rc;
	printk("registered with major %u\n", major);

	// static object constructors

	return 0;
}

static int __init_text  driver_init(void)
{
    int rc;
    LOCK_KERNEL_COMPAT();
    rc = _driver_init();
    UNLOCK_KERNEL_COMPAT();
    return rc;
}


#ifdef MODULE
static void __exit_text driver_exit(void)
{
	int rc;
	LOCK_KERNEL_COMPAT();

	rc=unregister_chrdev(major,name);
	pr_info("%s unloaded ",name);
	if (rc==0)
		printk("successfully");
	else
	{
		printk("with error: ");
		if (rc==-EINVAL)
			printk("major number %d is not registered"
					" with the matching name %s",
				major, name);
		else
			printk("%d",rc);
	}
	pr_endl();

	UNLOCK_KERNEL_COMPAT();
}
#endif


COMPAT_MODULE_INIT(driver_init, irq_init)
COMPAT_MODULE_EXIT(driver_exit)


//- init/cleanup
//----------------------------------------------------------------------
//+ file_operations routines


static int _irq_open(Inode* inode, File* file)
{
	PREFIX(); PFX(irq_inode(inode)); PRDEBUG("opening by %d...\n",
		current->pid);

	if (irq_inode(inode)>=NR_IRQS)
		return -ENXIO;
	if (file->f_flags & ~O_ACCMODE)
		return -EINVAL;

	file->private_data=NULL;

	MOD_INC_USE_COUNT;
	PREFIX(); PFX(irq_inode(inode)); PRDEBUG("opened by %d\n", current->pid);
	PREFIX(); TCHLIST_PRINT();
	return 0;
}

static int irq_open(Inode* inode, File* file)
{
    int rc;
    LOCK_KERNEL_COMPAT();
    rc = _irq_open(inode, file);
    UNLOCK_KERNEL_COMPAT();
    return rc;
}


static fops_close_ret_t irq_close(Inode* UNUSED inode, File* file)
{
	register irq_chan* ich=file->private_data;

	LOCK_KERNEL_COMPAT();

	PREFIX(); PFX(irq_inode(inode)); PRDEBUG("closing by %d...\n",
		current->pid);
	PREFIX(); TCHLIST_PRINT();

	if (ich!=NULL)
	{
		assert_goto(ich->signature==ICH_SIG, out);
		assert_goto(ich->tch!=NULL, out);
		assert_goto(ich->tch->signature==TCH_SIG, out);

		PREFIX(); PRDEBUG("%d found, tch=%p\n", current->pid, ich->tch);
		PREFIX(); TCHLIST_PRINT();

		free_irq_ich_tch(ich);

		PREFIX(); TCHLIST_PRINT();
	}

#ifndef NDEBUG /*(henry)*/
out:
#endif

	MOD_DEC_USE_COUNT;

	PREFIX(); PFX(irq_inode(inode)); PRDEBUG("closed by %d", current->pid);
	if (ich==NULL)		//no registered handler
	{
		PRDEBUG(" (w/o irq channels)");
		;
	}
	PRENDL();

	UNLOCK_KERNEL_COMPAT();

	return FOPS_CLOSE_VALUE(0);
}


static FOPS_SEEK_PROTOTYPE(irq_seek, UNUSED inode, UNUSED file,
	UNUSED offset, UNUSED origin)
{
	return -ESPIPE;
}


static int _irq_ioctl(Inode* inode, File* file, uint cmd, ulong arg)
{
	uint irq=irq_inode(inode);
	int rc=0;
	__irq_request_arg req_arg;
	irq_chan* ich=file->private_data;

	assert_retval(irq<NR_IRQS, -EBADFD);
	PREFIX(); PFX(irq); PRDEBUG("ioctl by %d, cmd: 0x%08X, arg: 0x%08lX\n",
		current->pid, cmd, arg);
	PREFIX(); TCHLIST_PRINT();

	switch (cmd)
	{
		case __IRQ_REQUEST_IOCTL:
			if ( COPY_FROM_USER(&req_arg, (const void*)arg, sizeof(req_arg)) )
				return -EFAULT;

			PREFIX(); PFX(irq);
				PRDEBUG("ioctl IRQ_REQUEST: pid=%d sig=%d flags=0x%08X\n",
					current->pid, req_arg.sig, req_arg.flags);

			if ( ich!=NULL && check_ownership(ich) )
				return -EACCES;

			return do_request(file, req_arg.sig, req_arg.flags);

		case __IRQ_FREE_IOCTL:
			PREFIX(); PFX(irq); PRDEBUG("ioctl IRQ_FREE: pid=%d sig=%ld\n",
				current->pid, arg);

			rc=check_null_and_ownership(ich);
			if (rc!=0)
				return rc;

			free_irq_ich_tch(ich);
			return 0;

		case __IRQ_ENABLE_IOCTL:
			PREFIX(); PFX(irq); PRDEBUG("ioctl IRQ_ENABLE: pid=%d\n",
				current->pid);

			rc=check_null_and_ownership(ich);
			if (rc!=0)
				return rc;
			if (ich->flags&IRQ_SHARED)
				return -EPERM;

			ENABLE_IRQ(irq);
			break;

		case __IRQ_DISABLE_IOCTL:
			PREFIX(); PFX(irq); PRDEBUG("ioctl IRQ_DISABLE: pid=%d\n",
				current->pid);

			rc=check_null_and_ownership(ich);
			if (rc!=0)
				return rc;
			if (ich->flags&IRQ_SHARED)
				return -EPERM;

			DISABLE_IRQ(irq);
			break;

		case __IRQ_SKIPPING_ON_IOCTL:
			PREFIX(); PFX(irq); PRDEBUG("ioctl IRQ_SKIPPING_ON: pid=%d\n",
				current->pid);

			rc=check_null_and_ownership(ich);
			if (rc!=0)
				return rc;

			skipping_on(ich);
			break;

		case __IRQ_SKIPPING_OFF_IOCTL:
			PREFIX(); PFX(irq); PRDEBUG("ioctl IRQ_SKIPPING_OFF: pid=%d\n",
				current->pid);

			rc=check_null_and_ownership(ich);
			if (rc!=0)
				return rc;

			return skipping_off(ich);

		case __IRQ_SKIPPING_CONSENT_ON_IOCTL:
			PREFIX(); PFX(irq);
				PRDEBUG("ioctl IRQ_SKIPPING_CONSENT_ON: pid=%d\n",
					current->pid);

			rc=check_null_and_ownership(ich);
			if (rc!=0)
				return rc;

			ich->flags|=IRQ_SKIPPING_CONSENT;
			break;

		case __IRQ_SKIPPING_CONSENT_OFF_IOCTL:
			PREFIX(); PFX(irq);
				PRDEBUG("ioctl IRQ_SKIPPING_CONSENT_OFF: pid=%d\n",
					current->pid);

			rc=check_null_and_ownership(ich);
			if (rc!=0)
				return rc;

			ich->flags &= ~IRQ_SKIPPING_CONSENT;
			break;

		case __IRQ_SIMULATE_IOCTL:
			PREFIX(); PFX(irq); PRDEBUG("ioctl IRQ_SIMULATE: pid=%d\n",
				current->pid);

			if (!has_readwrite_access(file))
				return -EACCES;

			irq_simulate(irq);
			break;

		case __IRQ_STAT_IOCTL:
			return -ENOTSUP;
		default:
			return -ENOSYS;
	}

	return 0;
}

static int irq_ioctl(Inode* inode, File* file, uint cmd, ulong arg)
{
    int rc;
    LOCK_KERNEL_COMPAT();
    rc = _irq_ioctl(inode, file, cmd, arg);
    UNLOCK_KERNEL_COMPAT();
    return rc;
}


# if LINUX_VERSION_CODE < KERNEL_VERSION(2,1,23)

// sel_type: SEL_IN || SEL_OUT || SEL_EX
// return: 0 if wait, 1 if ready
static int _irq_select(Inode* UNUSED inode, File* file,
	int sel_type, select_table* wt)
{
	irq_chan* ich=file->private_data;

	if (ich==NULL) return 1;	// unrequested irq

	assert_retval(ich->signature==ICH_SIG, 1);

	if (sel_type==SEL_IN)
	{
		if ( xchg_compat( &ich->sel_count, 0 ) == 0 )
			select_wait(&ich->wqh, wt);
		else
		{
			PREFIX(); PRMSGCOUNTER(); PFX(ich->irq);
				PRDEBUG("sel_count was %d\n", ich->sel_count);

			return 1;
		}
	}
	return 0;
}

static int irq_select(Inode* inode, File* file,
	int sel_type, select_table* wt)
{
    int rc;
    LOCK_KERNEL_COMPAT();
    rc = _irq_select(inode, file, sel_type, wt);
    UNLOCK_KERNEL_COMPAT();
    return rc;
}

#else	// LINUX_VERSION_CODE >= KERNEL_VERSION(2,1,23)

static uint _irq_poll(File* file, poll_table* pt)
{
	irq_chan* ich=file->private_data;

	if (ich==NULL)	// unrequested irq
		return POLLERR|POLLNVAL;

	assert_retval(ich->signature==ICH_SIG, POLLERR|POLLNVAL);

	// queue current process into any wait queue that
	// may awaken in the future (read and write)
	poll_wait(file, &ich->wqh, pt);

	// we have an input event only
	return xchg_compat( &ich->sel_count, 0 ) ? POLLIN|POLLRDNORM : 0;
}

static uint irq_poll(File* file, poll_table* pt)
{
    uint rc;
    LOCK_KERNEL_COMPAT();
    rc = _irq_poll(file, pt);
    UNLOCK_KERNEL_COMPAT();
    return rc;
}

#endif	// LINUX_VERSION_CODE < KERNEL_VERSION(2,1,23)


//- file_operations routines
//----------------------------------------------------------------------


static int do_request(File* file, int sig, int flags)
{
	task_chan* tch;
	irq_chan* ich;
	uint irq;
	int rc;

	if (!has_readwrite_access(file))
		return -EACCES;

	if ( sig != -1 && (sig<1 || sig>_NSIG) )
		return -EINVAL;

	if ( flags & ~(IRQ_SHARED|IRQ_SKIPPING|IRQ_SKIPPING_CONSENT) )
		return -EINVAL;

	irq=irq_file(file);
	assert_retval(irq<NR_IRQS, -EBADFD);

	if ( (tch=tch_find(current)) != NULL )
	{
		if ( tch->ichs[irq] != NULL )// this task already owned IRQ
			return -EBUSY;
	}
	else
		if ( (tch=tch_new_link(current)) == NULL )
			return nomem();

	if ((ich=ich_ctor(file, sig, flags))==NULL)
	{
		tch_dtor_if_empty(tch);		// clean up
		return nomem();
	}

	tch_attach_ich(tch,ich);

	PREFIX(); PFX(irq); PRDEBUG("ich=%p ich=", ich); ICH_PRINT(ich); PRENDL();
	PREFIX(); TCHLIST_PRINT();

	rc=register_irq_handler(ich);
	if (rc!=0)
	{	// clean up
		tch_detach_ich(tch,irq);
		ich_dtor(ich);
		tch_dtor_if_empty(tch);
	}

	return rc;
}


static int register_irq_handler(irq_chan* ich)
{
	int rc;
	char buf[128];
	char* devname;
	uint irq;

	assert_retval(ich!=NULL, -EBADFD);
	assert_retval(ich->signature==ICH_SIG, -EBADFD);

	irq=ich->irq;
	assert_retval(irq<NR_IRQS, -EBADFD);

	//+ info for /proc/interrupts
	if (ich->sig!=-1)		//signal
		sprintf(buf,"%s:%d.%d", name, current->pid, ich->sig);
	else					//select() only
		sprintf(buf,"%s:%d", name, current->pid);

	devname=strdup(buf);
	if (devname==NULL)	return nomem();
	ich->devname=devname;
	//- info for /proc/interrupts

	if (ich->flags&IRQ_SKIPPING)
		skipping_on(ich);

	rc=request_irq(irq, irq_handler,
		(ich->flags&IRQ_SHARED) ? SA_SHIRQ : 0, devname, ich);
	if (rc!=0)
	{	// clean up
		skipping_sub(ich);
		return rc;
	}
	++irq_channels_registered[irq];


	PREFIX(); PRDEBUG("%s: ich=%p ich=",name,ich); ICH_PRINT(ich); PRENDL();
	PREFIX(); TCHLIST_PRINT();
	return 0;
}


// free IRQ and its structures.
// returns destroyed tch flag.
static bool free_irq_ich_tch(irq_chan* ich)
{
	task_chan* tch;
	uint irq;

	if (ich==NULL)	return false;

	assert_retval(ich->signature==ICH_SIG, false);

	tch=ich->tch;

	assert_retval(tch!=NULL, false);
	assert_retval(tch->signature==TCH_SIG, false);

	irq=ich->irq;
	assert_retval(irq<NR_IRQS, false);
	assert_retval(tch->ichs[irq]==ich, false);

	free_irq(irq,ich); //no more interrupts on this channel

	--irq_channels_registered[irq];

	skipping_sub(ich);

	tch_detach_ich(tch, irq);
	ich_dtor(ich);
	return tch_dtor_if_empty(tch);
}


static void skipping_on(register irq_chan* ich)
{
	assert_retvoid(ich!=NULL);
	assert_retvoid(ich->signature==ICH_SIG);
	assert_retvoid(ich->tch!=NULL);
	assert_retvoid(ich->tch->signature==TCH_SIG);
	assert_retvoid(ich->irq<NR_IRQS);

	ich->skipping_count++;
	skipping_irq_counts[ich->irq]++;
}


static int skipping_off(register irq_chan* ich)
{
	assert_retval(ich!=NULL, -EBADFD);
	assert_retval(ich->signature==ICH_SIG, -EBADFD);
	assert_retval(ich->tch!=NULL, -EBADFD);
	assert_retval(ich->tch->signature==TCH_SIG, -EBADFD);
	assert_retval(ich->irq<NR_IRQS, -EBADFD);

	if (ich->skipping_count && skipping_irq_counts[ich->irq])
	{
		ich->skipping_count--;
		skipping_irq_counts[ich->irq]--;
		return 0;
	}
	else
		return -EPERM;
}


static void skipping_sub(register irq_chan* ich)
{
	assert_retvoid(ich!=NULL);
	assert_retvoid(ich->signature==ICH_SIG);
	assert_retvoid(ich->tch!=NULL);
	assert_retvoid(ich->tch->signature==TCH_SIG);
	assert_retvoid(ich->irq<NR_IRQS);

	skipping_irq_counts[ich->irq] -= ich->skipping_count;
	ich->skipping_count=0;
}


static void irq_simulate(uint irq)
{
	task_chan* tch;
	irq_chan* ich;

	assert_retvoid(irq<NR_IRQS);

	for(tch=tchlist; tch!=NULL; tch=tch->next)
	{
		assert_break(tch->signature==TCH_SIG);
		ich=tch->ichs[irq];

		if (ich!=NULL)
		{
			assert_break(ich->signature==ICH_SIG);

			if ( (ich->flags&IRQ_SKIPPING_CONSENT && skipping_irq_counts[irq])
				|| disabled_irq_counts[irq] )
				continue;

			do_irq(ich);
		}
	}
}


// "slow" interrupt handler
static void irq_handler(int irq, void* dev_id, struct pt_regs* UNUSED ptregs)
{
	register irq_chan* ich = (irq_chan*)dev_id;

	PREFIX(); PRDEBUG("IRQ %d interrupt handler\n", irq);

	assert_retvoid((uint)irq<NR_IRQS);
	assert_retvoid(ich!=NULL);
	assert_retvoid(ich->signature==ICH_SIG);
	assert_retvoid(ich==ich->file->private_data && (uint)irq==ich->irq);

	ich->irq_count++;

	if ( ich->flags&IRQ_SKIPPING_CONSENT && skipping_irq_counts[irq] )
		return;

	do_irq(ich);
}


static void do_irq(register irq_chan* ich)
{
	assert_retvoid(ich!=NULL);
	assert_retvoid(ich->signature==ICH_SIG);
	assert_retvoid(ich->tch!=NULL);
	assert_retvoid(ich->tch->signature==TCH_SIG);
	assert_retvoid(ich->irq<NR_IRQS);

	PREFIX(); PFX(ich->irq); PRDEBUG("sending "); ICH_PRINT(ich); PRENDL();

	ich->sel_count++;
	wake_up_interruptible_all(&ich->wqh);

	if (ich->sig != -1)
		if (send_sig(ich->sig, ich->tch->owner_task, 1)==0)
		{
			ich->sig_count++;
			PREFIX(); PFX(ich->irq); PRDEBUG("sent ");
				ICH_PRINT(ich); PRENDL();
		}

	xchg_compat( &NEED_RESCHED, 1 );
}

// end
//----------------------------------------------------------------------

