package Slinke::Serial;

use strict;
use warnings;
use Exporter;
use Carp;
use POSIX qw(CLOCAL CREAD CS8);
use Time::HiRes qw(setitimer ITIMER_REAL);
use vars qw(@ISA $VERSION @EXPORT);

$VERSION = 1.00;
@ISA = qw(Exporter);


my %termiosBaudValue = (
                        38400 => &POSIX::B38400,
                        19200 => &POSIX::B19200,
                         9600 => &POSIX::B9600,
                         4800 => &POSIX::B4800,
                         2400 => &POSIX::B2400,
                         1200 => &POSIX::B1200,
                        );

my %termiosBaudRate = reverse(%termiosBaudValue);


sub setBaud($$) {
    my ($deviceFh, $baud) = @_;

    my $termios = POSIX::Termios->new;

    $termios->getattr(fileno($deviceFh)) or die ("Can't get tty attributes");

    my $baudparm = $termiosBaudValue{$baud};
    if (!defined($baudparm)) {
        die("Illegal baud value $baud passed to setDteBaud()");
    }

    $termios->setospeed($baudparm);

    $termios->setattr(fileno($deviceFh), &POSIX::TCSANOW) 
        or die("Can't set tty attributes");
}



sub getBaud($) {
    my ($deviceFh) = @_;

    my $termios = POSIX::Termios->new;

    $termios->getattr(fileno($deviceFh)) or die ("Can't get tty attributes");
    
    my $baudparm = $termios->getospeed();

    my $baudrate;

    if (!defined($baudparm)) {
        die("POSIX::Termios::getospeed() failed.");
    } elsif ($baudparm == &POSIX::B0) {
        # Baud rate is unset.
        $baudrate = undef;
    } else {
        $baudrate = $termiosBaudRate{$baudparm};
        if (!defined($baudrate)) {
            die("Got garbage from POSIX::Termios::getospeed()");
        }
    }
    return $baudrate;
}



sub timedRead($$$) {
    my ($fh, $transferSize, $timeout) = @_;
#-----------------------------------------------------------------------------
#  Read $transferSize bytes from file with file handle $fh, but don't
#  wait more than $timeout seconds for the read to complete.
#
#  If the read fails, is short, or times out, abort the program with an
#  error message.
#
#  Assume VMIN is set to 1 in the termios for $fh.
#
#  Use the SIGALRM signal.
#-----------------------------------------------------------------------------

# Two other possibilities for implementing this: 
#
#   1) set VTIME and VMIN to 0 in termios and do a select()/sysread() loop
#      until either the timeout has passed or the requested number of bytes 
#      have been received.
#
#   2) set VTIME to the timeout and VMIN to 0 in termios and do a 
#      sysread() loop until either the timeout has passed or the requested
#      number of bytes have been received

    my $response;

    $response = '';  # initial value
    
    eval {
        local $SIG{ALRM} = sub { die "Signal ALRM received"; };
        setitimer(ITIMER_REAL, $timeout);
        my $charsLeft = $transferSize;   # initial value
        while ($charsLeft) {
            # Because VMIN is 1 in the termios, this returns as soon as at
            # least one character has arrived.
            my $bytesRead = sysread($fh, my $chars, $charsLeft);
            $response .= $chars;
            $charsLeft -= length($chars);
        }
        setitimer(ITIMER_REAL, 0);
    };
    if ($@) {
        # The eval block terminated abnormally.
        if ($@ =~ m{^Signal ALRM received}) {
            # It was terminated by our signal handler; That means the receive
            # timed out.  So we return success, with whatever data was
            # received before the signal.
        } else {
            # We were supposed to die; our eval accidentally saved us.
            die("Slinke::Control::timedRead died.  $@");
        }
    }
    if (length($response) > $transferSize) {
        die("Some how, I read more bytes than I asked for.");
    }
    return($response);
}



sub initSerialPort($) {

    my ($serialPortFh) = @_;

    my $termios = POSIX::Termios->new;

    $termios->getattr(fileno($serialPortFh)) 
        or die ("Can't get tty attributes");

    $termios->setospeed(&POSIX::B38400);
    $termios->setispeed(&POSIX::B38400);

    # CRTSCTS is a Linux extension, and not defined by POSIX::Termios.
    # This value is from asm/termbits.h
    my $CRTSCTS = 020000000000;
    
    $termios->setcflag(CLOCAL | $CRTSCTS | CREAD | CS8);
    $termios->setiflag(0);
    $termios->setlflag(0);
    $termios->setoflag(0);
    $termios->setcc(&POSIX::VTIME, 0);  # Reads don't time out
    $termios->setcc(&POSIX::VMIN, 1);   
        # A read returns as soon as at least 1 byte has been received

    $termios->setattr(fileno($serialPortFh), &POSIX::TCSANOW) 
        or die("Can't set tty attributes");
}

1;
