package Slinke::Receiver;

use strict;
use Exporter;
use vars qw( @ISA $VERSION @EXPORT );
use English;
use POSIX "SIGTERM";
use IPC::Msg;
use IPC::SysV qw(IPC_PRIVATE IPC_NOWAIT S_IRWXU);
use Data::Dumper;
use Slinke::Foundation;
use Slinke::Serial;

#print Data::Dumper->Dump([ \%args ], [ "args" ]);

$VERSION = 1.00;
@ISA = qw(Exporter);

# Sadly, IPC::SysV does not give us MSGMAX, the maximum size of an
# IPC::Msg message (IPC::Msg::snd fails if you try to send a message larger
# than this).  So we have to hardcode it here, according to what Bryan 
# sees in his Linux kernel.

my $MSGMAX = 4056;

my ($TRUE, $FALSE) = (1,0); 


my $debugLevel;

sub debug($$) {
    my ($level, $message) = @_;
    
    if ($debugLevel >= $level) {
        printf("Slinke::Receiver: $message\n");
    }
}



sub hexBits($) {
    my ($input) = @_;
#-----------------------------------------------------------------------------
#  Return a list of the bit positions of the 1's in the binary representation
#  of the hexadecimal number $input.
#
#  Return 0 for least significant bit, 1 for next least significant bit, etc.
#-----------------------------------------------------------------------------
    my $number = hex($_);

    my @retval;

    @retval = ();
    for (my $i = 0; $i < 32; ++$i) {
        if ($number & (0x80 >> $i) == 1) {
            push(@retval, $i);
        }
    } 
    return \@retval;
}



sub receiveControlReportData($$$$$) {

    my ($serialPortFh, $port, $action, $reportDataR, $errorR) = @_;

    debug(3, "Receiving remainder of $action report");

    my $reportType = $REPORT_TYPE{$action};

    if (!defined($reportType)) {
        die("Undefined action $action passed to receiveControlReport2()");
    }

    my $bytesToRead = $reportType->{DATALEN};

    my $reportData;

    if ($bytesToRead == 0) {
        $reportData = "";
    } else {
        my $chars = 
            Slinke::Serial::timedRead($serialPortFh, $bytesToRead, 1);
        my $bytesRead = length($chars);
        debug(5, "rest of report: 0x" . unpack("H*", $chars) .
              " ($bytesRead bytes)");
        if ($bytesRead != $bytesToRead) {
            $$errorR = "Read of $bytesToRead bytes to complete a $action " .
                "report for port $port returned only $bytesRead " .
                "bytes after 1 second";
        } else {
            $reportData = unpack("H*", $chars);
        }
    }
    $$reportDataR = $reportData;
}



sub receiveControlReport($$$$) {
    my ($serialPortFh, $port, $reportR, $errorR) = @_;
#-----------------------------------------------------------------------------
#  Receive from the serial port the remainder of a Control report, assuming
#  Byte 0 has already been read.
#
#  Return as $$reportR a hash reference $report such that:
#
#    $report->{PORT}        e.g. "PORT_IR"
#        The port on which it was received
#    $report->{REPORTTYPE}  e.g. "RPT_VERSIONIS"
#        The report type
#    $report->{DATA}        e.g. "30"
#        The data, if any, in the report (Bytes 2-END of the report),
#        in hexadecimal.  Guaranteed to be the correct number of bytes,
#        as determined by the action.
#-----------------------------------------------------------------------------

    debug(4, "Receiving remainder of Control report");
    
    my $receivedChar = 
        Slinke::Serial::timedRead($serialPortFh, 1, 0.060);
    if ($receivedChar eq '') {
        $$errorR = "read of action code (Byte 1) from serial port returned " .
            "nothing after 60 milliseconds";
    } else {
        debug(5, "Byte 1 of report: " . "0x" . unpack("H*", $receivedChar));

        my $actionCode = ord($receivedChar);
        my $action = reportTypeFromActionCode($actionCode, $port);
        if (defined($action)) {
            debug(4, "$action report received.");

            my %report;

            $report{PORT} = $port;
            $report{REPORTTYPE} = $action;

            receiveControlReportData($serialPortFh, $port, $action, 
                                     \$report{DATA}, $errorR);
            $$reportR = \%report;
        } else {
            $$errorR = sprintf("Unrecognized action code: 0x%02x");
        }
    }
}



sub receivePortMsgData($$$$) {

    my ($serialPortFh, $reportType, $portMsgDataR, $errorR) = @_;
    
    if ($reportType == $CONTROL_TYPE || 
        $reportType == $END_OF_PORT_MESSAGE_TYPE ||
        ($reportType & 0x1F) != $reportType) {
        die("Invalid report type (not Port Message Received) passed " .
            "to receivePortMsgData()");
    }
    my $datalen = $reportType;
    
    debug(4, "Receiving $datalen bytes of Port Message Received report");

    my $chars = 
        Slinke::Serial::timedRead($serialPortFh, $datalen, 1);

    my $bytesReceived = length($chars);
    if ($bytesReceived != $datalen) {
        $$errorR = "Read of $datalen bytes to complete a " .
                   "Port Message Received report received only " .
                   "$bytesReceived characters in one second.";
    } else {
        debug(5, "rest of report: 0x" . unpack("H*", $chars) .
              " ($bytesReceived bytes)");
        $$portMsgDataR = unpack("H*", $chars);
    }
}



sub sendControlReport($$) {
    my ($reportR, $msgQ) = @_;
#-----------------------------------------------------------------------------
#  Marshall a control report into IPC message queue $msgQ.
#
#  The contents of the report are described by $report, as would be
#  returned by receiveControlReport().
#-----------------------------------------------------------------------------
    debug(4, "Sending control report into message queue.");

    my $msg = "$reportR->{PORT} $reportR->{REPORTTYPE} $reportR->{DATA}";

    my $success = $msgQ->snd($IPCTYPE_SLINKE, $msg, IPC_NOWAIT);
    
    if (!$success) {
        warn("Failed to send control report to response message queue.  " .
             "errno=$ERRNO");
    } else {
        debug(5, "Control report has been sent into message queue.");
    }
}



sub sendUnsolControlReport($$) {
    my ($reportR, $msgQ) = @_;
#-----------------------------------------------------------------------------
#  Marshall an unsolicited control report into IPC message queue $msgQ.
#
#  The contents of the report are described by $report, as would be
#  returned by receiveControlReport().
#-----------------------------------------------------------------------------
    debug(4, "Sending control report into unsol msg queue.");
    
    my $msg = "CONTROL " .
        "$reportR->{PORT} " .
        "$reportR->{REPORTTYPE} " .
        "$reportR->{DATA}";

    my $success = $msgQ->snd($IPCTYPE_SLINKE, $msg, IPC_NOWAIT);
    
    if (!$success) {
        warn("Failed to send control report to unsol message queue");
    } else {
        debug(5, "Control report has been sent into message queue.");
    }
}



sub sendPortMessage($$$$) {
    my ($port, $portMessageData, $zone, $msgQ) = @_;
#-----------------------------------------------------------------------------
#  Send a port message into the IPC message queue $msgQ.
#
#  $portMessageData is the port message data in hexadecimal.
#
#  $zone is the IR zone.  If this port doesn't have a zone concept, it is
#  undef.
#-----------------------------------------------------------------------------
    debug(4, "Sending port message message into msg queue.");

    my $zoneInfo = $zone || "";
    
    my $msg = "PORTMSG $port $portMessageData $zoneInfo";

    if (length($msg) > $MSGMAX) {
        warn("Cannot send port message message.  " .
             "Port message data is too long (" .
             length($portMessageData) . " hex characters)");
    } else {
        my $success = $msgQ->snd($IPCTYPE_SLINKE, $msg, IPC_NOWAIT);
        
        if (!$success) {
            warn("Failed to send port message message '$msg'" .
                 "to message queue.  " .
                 "errno=$ERRNO");
        } else {
            debug(5, "Port Message message has been sent into message queue.");
        }
    }
}


sub processControlReport($$$$$$) {

    my ($serialPortFh, $port, $responseMsgQ, $receiveMsgQ,
        $currentPortMsgDataR, $errorR) = @_;

    receiveControlReport($serialPortFh, $port, 
                         \my $reportInfo, \my $error);
    if ($error) {
        $$errorR = "Failed to receive remainder of Control report.  " .
            "$error";
    } else {
        if ($REPORT_TYPE{$reportInfo->{REPORTTYPE}}->{RESPONSE}) {
            sendControlReport($reportInfo, $responseMsgQ);
        } elsif ($ERRORS{$reportInfo->{REPORTTYPE}}) {
            sendControlReport($reportInfo, $responseMsgQ);
        } elsif ($reportInfo->{REPORTTYPE} eq "RPT_LASTRXZONEIS") {
            if (!defined($currentPortMsgDataR->{$port})) {
                $$errorR = "Last Receive Zone Is report received with " .
                    "no preceding Port Message Received message";
            } else {
                debug(4, "Last Receive Zone Is data = $reportInfo->{DATA}");
                sendPortMessage($port,
                                $currentPortMsgDataR->{$port},
                                $reportInfo->{DATA},
                                $receiveMsgQ);
                $currentPortMsgDataR->{$port} = undef;
            }
        } else {
            sendUnsolControlReport($reportInfo, $receiveMsgQ);
        }
    }
}
    


sub processEndOfPortMessage($$$$$) {
    my ($port, $receiveMsgQ, $multiZone, $currentPortMsgDataR, $errorR) = @_;

    if (!defined($currentPortMsgDataR->{$port})) {
        $$errorR = "Port Message End Received report received without any " .
            "prior Port Message Received report";
    } else {
        if ($port eq "PORT_IR" &&  $multiZone) {
            # A "last receive zone" report will be coming next, and
            # we can't send the port message along until we get it.
        } else {
            # There's no more information coming for this port message,
            # so send it along.
            my $zone;
            if ($port eq "PORT_IR") {
                $zone = 0;
            }
            sendPortMessage($port, $currentPortMsgDataR->{$port}, 
                            $zone, $receiveMsgQ);
            $currentPortMsgDataR->{$port} = undef;
        }
    }
}



sub processPortMessageReceived($$$$$) {

    my ($serialPortFh, $reportType, $port, $currentPortMsgDataR, $errorR) = @_;
    receivePortMsgData($serialPortFh, $reportType, 
                       \my $newPortMsgData, \my $error);
    if ($error) {
        $$errorR = "Failed to receive remainder of " .
            "Port Message Received report.  $error";
        $currentPortMsgDataR->{$port} = undef;
    } else {
        $currentPortMsgDataR->{$port} .= $newPortMsgData;
    }
}



sub processReport($$$$$$$$) {
    my ($serialPortFh, $reportType, $port, 
        $responseMsgQ, $receiveMsgQ,
        $multiZone, $currentPortMsgDataR, $errorR) = @_;
#-----------------------------------------------------------------------------
#  Process the Slink-e report from port $port of which Caller has
#  received the first byte from the Slink-e.  Start by receiving the
#  remainder of the report from the Slink-e, via serial port on file handle
#  $serialPortFh.
#
#  - If it's a Port Message Received, just add
#    the port message data it contains to $$currentPortMsgDataR.  
#
#  - If it's a report that signals that the Slink-e is through
#    reporting on a port message it received, send a report of that 
#    port message (with the accumuated port message data 
#    $$currentPortMsgDataR) into the unsolicited report IPC message queue
#    $receiveMsgQ.
#
#    For a port without multiple zones (!$multiZone), an End Of Port
#    Message report is such a report.  For a multizone port, the Last
#    Received Zone Is report that follows Port Message End Received is it.
#
#  - If it's a response (solicited) control report, send it into the 
#    response IPC message queue $responseMsgQ.
# 
#  - If it's an unsolicited control report that reports a Slink-e error,
#    send it into the response IPC message queue $responseMsgQ.  Note
#    that this is _not_ a response, but the fact that Slink-e
#    generated this unsolicited error report means that a process is
#    probably sent a command and is waiting for a response and that
#    response will never come because Slink-e encountered an error in
#    the middle of the command and entered Needs Resume state.  That 
#    process will want to treat this unsolicited error as a response to
#    his command.
#
#  - If it's any other unsolicited control report, send it into the receive
#    IPC message queue $receiveMsgQ.
#-----------------------------------------------------------------------------
    debug(4, "Receiving report for Port $port");

    if ($reportType == $CONTROL_TYPE) {
        processControlReport($serialPortFh, $port,
                             $responseMsgQ, $receiveMsgQ,
                             $currentPortMsgDataR, $errorR);
    } elsif ($reportType == $END_OF_PORT_MESSAGE_TYPE) {
        processEndOfPortMessage($port, $receiveMsgQ, $multiZone, 
                                $currentPortMsgDataR, $errorR);
    } else {
        processPortMessageReceived($serialPortFh, $reportType, $port, 
                                   $currentPortMsgDataR, $errorR)
    }
}



sub receiver(%) {

# There is some weirdness here that I can't see a clear way around.
# This subroutine runs in a different process from the one that does
# the Slinke package calls.  It was forked by new(), and the fork
# created clones of every object that existed at the time.  It didn't
# call constructors for the objects -- it just copied them.  So
# $receiveMsgQ is a copy of the original IPC::Msg object.  When this
# process exits, Perl will call the destructor for these objects.
# Note that changes to the object by the main process will not affect
# our copy, and vice versa.

# Another way to go would be to exec().  If we did that, the clones
# of the objects would disappear without destructors being called.
# In that case, we couldn't use anonymous message queues.

# In any case, we also have all the file descriptors that the process that
# created us had, and exec() won't change that.  Those files cannot close
# until we exit.
    
    my %args = @_;

    my $serialPortFh = $args{SERIALPORTFH};
    my $receiveMsgQ  = $args{RECEIVEMSGQ};
    my $responseMsgQ = $args{RESPONSEMSGQ};
    my $multiZone    = $args{MULTIZONE};

    $debugLevel = $args{DEBUG} || 0;

    debug(1, "Receiver process running.");

    my %currentPortMsgData;
        # $currentPortMsgData{$port} is all the port message data we
        # have received so far for the port message we are currently
        # building for port $port.  It is a hexadecimal string,
        # e.g. "12FF".  It is undefined if we are not building a port
        # message for port $port.  Recall that the Slink-e delivers a port
        # message in multiple reports because each report has a
        # capacity of only 30 bytes.

    while ($TRUE) {
        debug(5, "Waiting for Byte 0 of next report or 12 hours...");

        my $receivedChar = 
            Slinke::Serial::timedRead($serialPortFh, 1, 12*60*60);

        if ($receivedChar eq '') {
            debug(5, "Read completed with nothing read.");
        } else {
            debug(5, "Byte 0 of report: 0x" . unpack("H*", $receivedChar));
            my $byte0 = ord($receivedChar);
        
            my $portCode = $byte0 >> 5;
            my $reportType = $byte0 & 0x1F;

            my $port;
            $port = $PORTNAME{$portCode};
            if (!defined($port)) {
                die("PORTNAME failed to translate port code $portCode");
            }
            processReport($serialPortFh, $reportType, $port,
                          $responseMsgQ, $receiveMsgQ,
                          $multiZone,
                          \%currentPortMsgData, \my $error);

            if ($error) {
                debug(1, "Error in receiving a report.  $error");
            }
        }
    }
}



1;
