#-----------------------------------------------------------------------------
#  This module provides functions to Slinke::Control
#-----------------------------------------------------------------------------
package Slinke::Receive;

use strict;
use warnings;
use Data::Dumper;
use Exporter;
use vars qw( @ISA $VERSION @EXPORT );
use English;
use Errno qw(:POSIX EIDRM);
use IPC::Msg;
use IPC::SysV qw(IPC_PRIVATE IPC_NOWAIT MSG_NOERROR S_IRWXU);
use Slinke::Foundation;

$VERSION = 1.00;
@ISA = qw(Exporter);


my ($TRUE, $FALSE) = (1,0); 

my $debugLevel;

sub debug {
    my ($level, $message) = @_;
    
    if ($debugLevel >= $level) {
        printf("Slinke::Receive: $message\n");
    }
}



sub processControlMsg($$$$) {
    my ($slinke, $msgText, $messageR, $errorR) = @_;

    if ($msgText =~ m{ ^(\S+) \s (\S+) \s (\S*) $ }x) {
        $messageR->{TYPE} = "CONTROL";
        $messageR->{PORT} = $1;
        $messageR->{REPORTTYPE} = $2;
        $messageR->{DATA} = $3;
    } else {
        $$errorR = "Invalid control message format.  (not PORT TYPE DATA)";
    }
}




sub processPortMsg($$$$) {
    my ($slinke, $msgText, $messageR, $errorR) = @_;

    if ($msgText =~ m{ ^(\S+) \s (\S+) \s (\S*) $ }x) {
        my ($port, $portMessageData, $zone) = ($1, $2, $3);
        $messageR->{TYPE} = "PORTMSG";
        $messageR->{PORT} = $port;
        $messageR->{CONTENTS} = $portMessageData;
        $messageR->{ZONE} = $zone;
    } else {
        $$errorR = "Invalid control message format.  (not PORT TYPE DATA)";
    }
}



sub receiveFromMsgQ($$$$) {
    my ($msgQ, $msgTypeR, $messageTextR, $errorR) = @_;

    my $maxMessageLength = 4096;
    
    my $complete;
    my $errorRet;

    while (!$complete && !$errorRet) {
        my $msgType = $msgQ->rcv($$messageTextR, $maxMessageLength);
        if (defined($msgType)) {
            $complete = $TRUE;
        } else {
            if ($ERRNO != EINTR) {
                # There's no modern handle for the message queue, so it can
                # disappear between when we get it and when we do the receive,
                # or during the receive.  If it disappears before the receive
                # starts, the errno is EINVAL.  If it disappears while we are
                # waiting, the errno is EIDRM.
                if ($ERRNO == E2BIG) {
                    $errorRet = "A message in the queue is longer than " .
                        "$maxMessageLength bytes.  We can't receive it.";
                } elsif ($ERRNO == EIDRM || $ERRNO == EINVAL) {
                    $errorRet = "Message queue has been removed.";
                } else {
                    $errorRet = "IPC::Msg::rcv() failed.  Errno=$ERRNO";
                }
            }
        }
    }
    $$errorR = $errorRet;
}



sub receiveFromSlinke(%) {
#-----------------------------------------------------------------------------
#  Receive a message from the Slink-e receive message queue (which is
#  fed by the Slinke::Receiver process, which continuously receives
#  from the Slink-e serial port, and passes them into the receive
#  message queue, formatted into friendly text).
#
#  We parse the message and return the results.
#
#  If there is no message in the queue, we wait until there is one.
#
#  Argument is a hash with the following keys:
#
#    MESSAGE => reference to a variable in which we return a reference
#               to a hash.  That hash's keys are:
#
#               TYPE       => Type of message.  "PORTMSG" or "CONTROL".
#
#               PORT       => Slink-e port from which the message came.  E.g.
#                             'PORT_IR'.
#
#               For a port message message (TYPE eq 'PORTMSG'):
#
#                 CONTENTS => The port message data in hexadecimal text.
#
#                 ZONE     => For an IR port (PORT eq 'PORT_IR'), this is the
#                             number of the zone in which the Slink-e received
#                             the port message.  Undefined for other ports.
#
#               For a control report (TYPE eq 'CONTROL'):
#
#                 REPORTTYPE => The Slink-e report type.  E.g. 
#                               'RPT_MSGDATA_UNDERRUN'.
#
#                 DATA       => Report data, in hexadecimal text.
#
#    ERROR   => reference to a variable in which we return an error message
#               if we fail, undef if we don't.    
#
#-----------------------------------------------------------------------------
    my $slinke = shift;
    my (%args) = @_;

    my %message;
    my $errorRet;

    $debugLevel = $slinke->{DEBUGLEVEL} || 0;
    my $msgQ = $slinke->{RECEIVEMSGQ};

    debug(3, "Receiving from receive message queue");

    receiveFromMsgQ($msgQ, \my $msgType, \my $messageText, \my $error);

    if ($error) {
        $errorRet = $error;
    } else {
        debug(2, "Received '$messageText' from receive message queue");
        
        if ($messageText =~ m{ ^(\S+) \s+ (.*) }x) {
            my ($type, $rest) = ($1, $2);
            if ($type eq "CONTROL") {
                processControlMsg($slinke, $rest, \%message, \my $error);
                if ($error) {
                    $errorRet = 
                        "Can't process control message '$rest'.  $error";
                }
            } elsif ($type eq "PORTMSG") {
                processPortMsg($slinke, $rest, \%message, \my $error);
                if ($error) {
                    $errorRet = 
                        "Can't process port message '$rest'.  $error";
                }
            } else {
                $errorRet = "Invalid type in message received from " .
                    "receive IPC message queue:  $type";
            }
        } else {
            $errorRet = "Invalid message received from " .
                "receive IPC message queue.  Does " .
                "not contain a type keyword";
        }
    }

    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
    
    if (defined($args{MESSAGE})) {
        $ {$args{MESSAGE}} = \%message;
    }
}

1;
