package Slinke::Messager;

use strict;
use warnings;

use Exporter;
use Carp;
use vars qw(@ISA $VERSION @EXPORT);
use Errno qw(:POSIX);
use English;
use POSIX qw(SIGTERM floor setsid);
use IPC::Msg;
use IPC::SysV qw(IPC_PRIVATE IPC_NOWAIT MSG_NOERROR S_IRWXU);
use Slinke::Receive;
use Slinke::Foundation qw(min max);
use Slinke::Control;

my $TRUE=1; my $FALSE = 0;


my $DEFAULT_TIMEOUT_PERIODS = 200;
my $DEFAULT_SAMPLE_PERIOD = 100;  # In microseconds


# A Slinke::Messager object has the following elements:
#
#   SLINKE => Slinke::Control object
#
#   IRSAMPLEPERIOD => Current IR sample period set in the Slink-e, in
#   microseconds.  This is the value that the Slinke should have.
#
#   TRANSMITZONEMASK => Current IR transmit zone mask set in the Slinke-e
#   (in the form returned by Slinke::Control::getTransmitZoneMask()).
#
#   RECEIVEZONEMASK => Current IR receive zone mask set in the Slinke-e,
#   unless it is temporarily 0x00 to prevent feedback from sending a 
#   message.  In the form returned by Slinke::Control::getIRReceiveZoneS().
#
#   IRTIMEOUT => Current IR timeout period in sample periods.  This is
#   the value that the Slinke should have.

my $debugLevel;

sub debug {
    my ($level, $message) = @_;
    
    if ($debugLevel >= $level) {
        printf("Slinke::Messager: $message\n");
    }
}



sub d2b($$) {
    my ($value, $bits) = @_;

    return unpack("B$bits", pack('I', $value));
}



sub resetSlinke($$) {
    my ($messager, $errorR) = @_;

    my $slinke = $messager->{SLINKE};

    # TODO: should do a $slinke->reset() and $slinke->setbaud() (or whatever)
    # here.

    $slinke->setIRCarrier(REQCARRIERFREQ => 40000);

    $slinke->setIRSamplingPeriod(PERIOD => $messager->{IRSAMPLEPERIOD}/1e6);

    $slinke->setIRMinimumLength(MINLENGTH => 6,
                                ERROR     => \my $error,
                                );
    if ($error) {
        $$errorR = "Unable to set the minimum message length.  $error";
    } else {
        $slinke->setIRTransmitZones(ZONEMASK => $messager->{TRANSMITZONEMASK},
                                    ERROR    => \my $error,
                                    );
        if ($error) {
            $$errorR = "Unable to set the transmit zone mask.  $error";
        } else {
            $slinke->setIRReceiveZones(
                ZONEMASK => $messager->{RECEIVEZONEMASK},
                ERROR    => \my $error,
                );
            if ($error) {
                $$errorR = "Unable to set the receive zone mask.  $error";
            } else {
                $slinke->setIRTimeoutPeriod(PERIOD => $messager->{IRTIMEOUT},
                                            ERROR  => \my $error); 
                
                if ($error) {
                    $$errorR = "setIRTimeoutPeriod() failed to " .
                        "set timeout to " .
                        "$messager->{IRTIMEOUT} sample periods.  $error";
                } else {
                    $$errorR = undef;
                }
            }
        }
    }
}



sub loadSlinkeSettings {
    my ($messager, $errorR) = @_;

    my $slinke = $messager->{SLINKE};

    $slinke->getBaud(BAUD  => \$messager->{BAUD},
                     ERROR => \my $error,
                     );
    if ($error) {
        $$errorR = "Unable to get current baud rate from the " .
            "Slinke::Control object.  $error";
    } else {
        $slinke->getIRSamplingPeriod(PERIOD => \my $samplePeriodSeconds,
                                     ERROR  => \my $error,
                                     );
        if ($error) {
            $$errorR = "Unable to get current sampling period from the " .
                "Slinke::Control object.  $error";
        } else {
            $messager->{IRSAMPLEPERIOD} = $samplePeriodSeconds * 1E6;

            $slinke->getIRTransmitZones
                (
                 ZONEMASK => \$messager->{TRANSMITZONEMASK},
                 ERROR    => \my $error,
                 );
            if ($error) {
                $$errorR = "Unable to get current transmit zone mask " .
                    "from the Slinke::Control object.  $error";
            } else {
                $slinke->getIRReceiveZones
                    (
                     ZONEMASK => \$messager->{RECEIVEZONEMASK},
                     ERROR    => \my $error,
                     );
                if ($error) {
                    $$errorR = "Unable to get current receive zone mask " .
                        "from the Slinke::Control object.  $error";
                } else {
                    $slinke->getIRTimeoutPeriod(
                        PERIOD => \$messager->{IRTIMEOUT},
                        ERROR  => \my $error); 
                
                    if ($error) {
                        $$errorR = "Unable to get current IR timeout from " .
                            "the Slinke::Control object.  $error";
                    } else {
                        $$errorR = undef;
                    }
                }
            }
        }
    }
}




=head2 $messager->setSamplePeriod(%ARGS);

This function sets the sample period that the Slink-e uses in sending
and receiving IR messages.

This does not affect any IR messages that have already been received 
by the Slink-e.

%ARGS is a hash with these elements:

    PERIOD => the sample period, in microseconds.

    ERROR  => a reference to the variable in which this function
              returns a description of the error if it fails.  If it
              doesn't fail, this function does not set the variable.


=cut

sub setSamplePeriod(%) {

    my ($messager, %args) = @_;

    my $slinke = $messager->{SLINKE};

    my $errorRet;

    if (!defined($args{PERIOD})) {
        $errorRet = "You must specify the PERIOD argument";
    } else {
        if ($args{PERIOD} > 1000) {
            print("Sample period is too long.  " .
                  "Maximum value is 1000 microseconds.\n");
        } elsif ($args{PERIOD} < 50) {
            print("Sample period is too short.  " .
                  "Minimum value is 50 microseconds.\n");
        } else {
            $slinke->setIRSamplingPeriod(PERIOD => $args{PERIOD}/1e6,
                                         ERROR  => \my $error,
                                         );
            
            if ($error) {
                $errorRet = "Failed.  $error";
            } else {
                $messager->{IRSAMPLEPERIOD} = $args{PERIOD};
            }
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub pulseTrainToPortMessageData($$) {
    my ($dataR, $samplePeriod) = @_;
        
    # $samplePeriod is in microseconds.

    my $outsum;
    my $truesum;
    my @bin;

    $outsum = 0;
    $truesum = 0;
        
    foreach my $dataElement (@$dataR) {
        my $duration = abs($dataElement);   # microseconds

        $truesum += $duration;
        # convert microseconds into the number of periods of the current
        # Slink-e sample clock.
        my $periodCount = int(($truesum-$outsum) / $samplePeriod + 0.5); 
        $outsum += $periodCount * $samplePeriod;

        my $signBit = $dataElement < 0 ? 0 : 0x80;

        # Generate as many bytes as it takes to add up to $periodCount periods.
        my $remainingPeriodCount;
        $remainingPeriodCount = $periodCount;
        while ($remainingPeriodCount > 0) {
            my $periodsThisByte = min($remainingPeriodCount, 127);
            push @bin, ($periodsThisByte | $signBit);
            $remainingPeriodCount -= $periodsThisByte;
        }
    }
    return @bin;
}



sub computePulseTrainFromPortMessageData($$$$) {
    my ($portMessageData, $samplePeriod, $pulseTrainR, $errorR) = @_;
#-----------------------------------------------------------------------------
#  $samplePeriod is in microseconds.
#
#  We return a pulse train @{$pulseTrainR}, in which each element represents
#  a single pulse.  Its value is a number; the magnitude of that number is
#  the duration of the pulse in microseconds.  The sign is positive for an
#  on pulse, negative for an off pulse.  Numbers are guaranteed to alternate
#  sign.  The pulse train we return ends with an ON pulse.  We do not include
#  the final ersatz off pulse that the Slink-e includes in the port message
#  data.
#-----------------------------------------------------------------------------

    # Note that the Slink-e reports pulse durations in number of samples, 
    # but the pulse train we return is in terms of microseconds.

    my @pulseTrain;

    my $byteCount = length($portMessageData) / 2;
    if ($byteCount != int($byteCount)) {
        $$errorR = "port message data is not even number of hex digits";
    } else {
        my $currentState;
        my $currentDuration;

        $currentState = "ON";
        $currentDuration = 0;
        
        for (my $i = 0; $i < $byteCount; ++$i) {
            my $byte = hex(substr($portMessageData, $i*2, 2));
            my $state = ($byte >> 7) ? "ON" : "OFF";
            my $duration = $byte & 0x7F;
            
            if ($state eq $currentState) {
                $currentDuration += $duration;
            } else {
                # We've flipped state, so the previous pulse is complete.
                # Add it to the pulse train.
                my $pulse = $currentState eq "ON" ? 
                    $currentDuration : - $currentDuration;
                push (@pulseTrain, $pulse * $samplePeriod);
                $currentState = $state;
                $currentDuration = $duration;
            }
        }
        # The last pulse in the pulse set is an off pulse that is quite
        # meaningless.  It is the first part of a real off pulse -- just
        # enough of it that the Slink-e timed out and thus considered the
        # pulse set complete.  We don't bother to return that, but we do
        # a sanity check of it.
        
        debug(4, "Dropping final pulse: $currentDuration sample periods");
        
        if ($currentState ne "OFF") {
            $$errorR = "pulse set returned by Slink-e does not end with " .
                "an OFF pulse.  Something's weird.";
        }
        $$pulseTrainR = \@pulseTrain;
    }
}



sub numberFromBitPositions(@) {
#-----------------------------------------------------------------------------
#  Given a list of bit positions, return the number whose pure binary 
#  representation has those and only those bits on.
#-----------------------------------------------------------------------------
    my $output;
    foreach(@_) {
        $output += 1 << $_;
    }
    return $output;
}



sub establishTransmitZones($$$) {
    my ($messager, $zonesArg, $errorR) = @_;

    my $zoneMask = numberFromBitPositions(@{$zonesArg});

    if ($zoneMask != $messager->{TRANSMITZONEMASK}) {
        debug(2, "Changing transmit zone mask from " .
              d2b($messager->{TRANSMITZONEMASK},8) . 
              " to " . d2b($zoneMask,8));

        $messager->{SLINKE}->setIRTransmitZones(ZONEMASK => $zoneMask,
                                                ERROR    => \my $error,
                                                );
        if ($error) {
            $$errorR = "Unable to set transmit zone mask to " .
                "$zoneMask.  $error";
        } else {
            $messager->{TRANSMITZONEMASK} = $zoneMask;
        }
    }
}



sub establishReceiveZones($$$) {
    my ($messager, $zonesArg, $errorR) = @_;

    my $zoneMask = numberFromBitPositions(@{$zonesArg});

    if ($zoneMask != $messager->{RECEIVEZONEMASK}) {
        debug(2, "Changing receive zone mask from " .
              d2b($messager->{RECEIVEZONEMASK},8) . 
              " to " . d2b($zoneMask,8));

        $messager->{SLINKE}->setIRReceiveZones(ZONEMASK => $zoneMask,
                                               ERROR    => \my $error,
                                               );
        if ($error) {
            $$errorR = "Unable to set receive zone mask to " .
                "$zoneMask.  $error";
        } else {
            $messager->{RECEIVEZONEMASK} = $zoneMask;
        }
    }
}



sub handleFeedback($$$) {
    my ($messager, $feedbackArg, $errorR) = @_;

    my $slinke = $messager->{SLINKE};

    if (!$feedbackArg && $messager->{RECEIVEZONEMASK} != 0x00) {
        $slinke->setIRReceiveZones(ZONEMASK => 0x00,
                                   ERROR    => \my $error,
                                   );
        if ($error) {
            $$errorR = "Unable to turn off receive zones to " .
                "prevent echo.  $error";
        }
    }
}



sub restoreReceiveZones($$) {
#-----------------------------------------------------------------------------
#  Reset the receive zone mask in the Slink-e to what it is supposed
#  to be, because Caller may have temporarily masked the receive zones
#  to prevent echo.
#
#  If our attempt at Set IR Receive Zones fails (probably just because the
#  preceding port message send failed), we send another to ensure that the
#  receive zone mask gets set.
#
#  if $$errorR is undefined and both attempts to restore the mask fail, we
#  set $$errorR to an error message explaining the failure.  If $$errorR is
#  already set, we issue a Perl warning but leave $$errorR alone.
#-----------------------------------------------------------------------------
    my ($messager, $errorR) = @_;
    
    my $slinke = $messager->{SLINKE};

    $slinke->setIRReceiveZones(
                               ZONEMASK => $messager->{RECEIVEZONEMASK},
                               ERROR    => \my $error,
                               );

    if ($error) {
        debug(1, "Set IR Receive Zones command to restore zones after " .
              "Send Port Message failed.  $error");

        $slinke->setIRReceiveZones(
                                   ZONEMASK => $messager->{RECEIVEZONEMASK},
                                   ERROR    => \my $error2,
                                   );
        
        if ($error2) {
            my $msg = "Unable to restore the receive zones after " .
                "successfully sending port message.  " .
                "Slinke::Control::setIRReceiveZones() failed twice, " .
                "most recently with: $error2";
            if ($$errorR) {
                warn($msg);
            } else {
                $$errorR = $msg;
            }
        } else {
            warn("Restore of receive zones after sending a port message " .
                 "failed, but 2nd try succeeded.  $error");
        }
    }
}



=head2 $messager->sendIR(%ARGS)

This function sends an IR port message.  

%ARGS is a hash with these elements:

DATA =>     A reference to an array that describes the pulse set that forms
            the IR message.  Each element in the array is a number
            that describes either an on pulse or an off pulse.  If the
            number is positive, it is an on pulse; negative is off.
            The absolute value of the number is the duration of the
            pulse in microseconds.

            This function does not add any intermessage pause before or
            after.  You must either include that in the pulse set or
            delay between calling sendIR().

ZONES    => A reference to an array that indicates in which IR zones to
            send the message.  If undefined, the method sends in all
            zones.
     
            The array is a list of zone numbers of the zones.  E.g.
     
            ZONES => [ 3, 4 ];
     
            Note that on a Slink-e earlier than Version 2, the only
            valid zone is 0.  This method fails if you specify
            any other zone.

FEEDBACK => This function automatically disables IR receivers in all zones 
            while it is transmitting so that the receiver will not
            capture what the transmitter is sending.  To leave
            receivers enabled (which you might want to do to verify
            that the code got sent properly), set this element to
            true.

ERROR =>    A reference to the variable in which this function returns a
            description of the error if it fails.  If it doesn't fail,
            this function sets it to undef.

=cut

sub sendIR(%) {
    my ($messager, %args) = @_;

    debug(1, "sendIR called");

    my $oldRxZones;
    
    my $slinke = $messager->{SLINKE};
    
    my $errorRet;

    if (!defined($args{DATA})) {
        $errorRet = "You must specify the DATA argument.";
    } elsif (!defined($args{ZONES})) {
        $errorRet = "You must specify the ZONES argument.";
    } elsif ($slinke->{VERSION} < 2.0 && @{$args{ZONES}} != (0)) {
        $errorRet = "For this pre-2.0 Slink-e, the only valid " .
            "zone is 0.  You specified " . join(',', @{$args{ZONES}});
    } else {
        establishTransmitZones($messager, $args{ZONES}, \$errorRet);
        if (!$errorRet) {
            handleFeedback($messager, $args{FEEDBACK}, \$errorRet);
            if (!$errorRet) {
                my @data = pulseTrainToPortMessageData
                    ($args{DATA}, $messager->{IRSAMPLEPERIOD});
                
                $slinke->sendPortMessage(PORT  => "PORT_IR",
                                         DATA  => \@data,
                                         SYNC  => $TRUE,
                                         ERROR => \my $error);
# Simulate a RPT_CMDDATA_UNDERRUN error:
#            $slinke->sendPortMessage(PORT  => "PORT_SER",
#                                     DATA  => [ 0xDF, 0x81 ],
#                                     ERROR => \my $error);
                
                if ($error) {
                    $errorRet = "Failed to send port message " .
                        "(" . scalar(@data) . " bytes).  $error";
                }
                restoreReceiveZones($messager, \$errorRet);
            }
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $messager->refresh(%ARGS)

This function refreshes the Slinke, in case it got itself scrambled somehow.
It resets the various parameters in the Slinke to what they're supposed to
be right now.


%ARGS is a hash with these elements:

ERROR =>    A reference to the variable in which this function returns a
            description of the error if it fails.  If it doesn't fail,
            this function sets it to undef.

=cut

sub refresh(%) {
    my ($messager, %args) = @_;

    debug(1, "refresh called");

    resetSlinke($messager, \my $error);

    if ($args{ERROR}) {
        $ {$args{ERROR}} = $error;
    }
}




sub byteArrayFromHexString($) {
    
    my ($hexString) = @_;
    
    my @byteArray;

    my $byteCount = length($hexString) / 2;

    for (my $i = 0; $i < $byteCount; ++$i) {
        $byteArray[$i] = hex(substr($hexString, 2 * $i, 2));
    }
    
    return \@byteArray;
}



sub processPortMsg($$$$$) {
    my ($slinkeMessage, $samplePeriod, $contentsR, $zoneR, $errorR) = @_;

    $$zoneR = $slinkeMessage->{ZONE};

    if ($slinkeMessage->{PORT} eq "PORT_IR") {
        computePulseTrainFromPortMessageData
            ($slinkeMessage->{CONTENTS}, $samplePeriod,
             $contentsR, \my $error
             );
        if ($error) {
            $$errorR = "port message data from IR port has invalid " .
                "format.  $error.  " .
                "Raw data = '$slinkeMessage->{CONTENTS}'";
        }
    } else {
        $$contentsR = byteArrayFromHexString($slinkeMessage->{CONTENTS});
    }
}



sub handleReceiveZones($$$) {
    my ($messager, $zonesArg, $errorR) = @_;

    my $slinke = $messager->{SLINKE};

    my $zones;
    if (!defined($zonesArg)) {
        if ($slinke->{VERSION} < 2.0) {
            $zones = [ 0 ];
        } else {
            $zones = [ 0, 1, 2, 3, 4, 5, 6, 7 ];
        }
    } else {
        $zones = $zonesArg;
    }
    if ($slinke->{VERSION} < 2.0 && @{$zonesArg} != (0)) {
        $$errorR = "For this pre-2.0 Slink-e, the only valid " .
            "zone is 0.  You specified " . join(',', @{$zonesArg});
    } else {
        establishReceiveZones($messager, $zones, $errorR);
    }
}



=head2 $messager->receiveSlinkeMessage(%ARGS);

This function returns a port message or unsolicted control report from
the Slink-e.

This method first goes through the Slinke::Messager object's queue of
unsolicited reports it received previously from the Slink-e, then
reads from the underlying Slinke::Control object, then from the
Slink-e itself.  If that isn't enough to get a complete port message
or unsolicited control report, the method waits for more reports from
the Slink-e.

Arguments:

    ZONES   => A reference to an array that indicates in which IR zones to
               listen for the message.  If undefined, the method listens in
               all zones.

               The zone mask is set in the Slink-e itself, which
               receives port messages independently of whether this
               method is currently executing.  Hence, this has no
               effect on a message which the Slink-e received before
               this method was called, and it determines the zones in
               which the Slink-e listens for messages after this
               method returns.

               Regardless of this value, the Slink-e always listens on all
               the other (non-IR) ports for port messages as well.

               The array is a list of zone numbers of the zones.  E.g.
     
               ZONES => [ 3, 4 ];
     
               Note that on a Slink-e earlier than Version 2, the only
               valid zone is 0.  This method fails if you specify
               any other zone.

    MESSAGE => a reference to the variable in which
               this function returns a hash reference with these elements:

        TYPE => the type of message.  "CONTROL" means an unsolicited
                control message, and "PORTMSG" means a message that
                tells you that the Slink-e received a port message.

        PORT => the name of the port that generated the message.  The
                port names are: PORT_IR, PORT_PAR, PORT_SLO, PORT_SL1,
                PORT_SL2, PORT_SL3, PORT_SER, PORT_SER.

        For a "PORTMSG" type message:
    
            CONTENTS => a reference to an array of values that describe
                        the port message received.  If it is from the
                        IR port, this array has the same format as
                        what you pass to sendIR().  If it is from any
                        other port, it is an array where each element
                        represents one byte of the port message data.
                        The element is a number whose value is what
                        the port message data byte encodes in pure
                        binary.  E.g. if the port message data is
                        0x102300, the array is (16, 35, 0).

            For an IR message (from port "PORT_IR"):

                ZONE => which IR zone's receiver (0-8) received the
                        data.  Note that you must have a Slink-e of
                        version 2.0 or higher for ZONE to be greater
                        than 0

        For a "CONTROL" type message:

            REPORTTYPE => the Slinke report type as a string, e.g.
                          "RPT_CMDDATA_UNDERRUN".  This always starts with
                          "RPT_".

            DATA       => the value of the databytes the Slinke sent with the 
                          message, in hexadecimal.  E.g. "01FF".

    ERROR => a reference to the variable in which this function
             returns a description of the error if it fails.  If it
             doesn't fail, this function does not set the variable.


=cut

sub receiveSlinkeMessage(%) {

    my ($messager, %args) = @_;

    my $slinke = $messager->{SLINKE};

    my $errorRet;

    handleReceiveZones($messager, $args{ZONES}, \$errorRet);

    $slinke->receiveFromSlinke(MESSAGE => \my $slinkeMessage,
                               ERROR   => \$errorRet);

    if (!$errorRet) {
        if (!defined($slinkeMessage)) {
            $errorRet = "INTERNAL ERROR.  " .
                "Slinke::Receive::receiveFromSlinke() returned undefined " .
                "message.";
        } else {
            if (!defined($slinkeMessage->{PORT})) {
                die("undefined PORT");
            }
            my %message;

            $message{TYPE} = $slinkeMessage->{TYPE};
            $message{PORT} = $slinkeMessage->{PORT};
            
            if ($slinkeMessage->{TYPE} eq "PORTMSG") {
                processPortMsg($slinkeMessage, $messager->{IRSAMPLEPERIOD},
                               \$message{CONTENTS}, \$message{ZONE}, 
                               \$errorRet);
            } else {
                $message{REPORTTYPE} = $slinkeMessage->{REPORTTYPE};
                $message{DATA}       = $slinkeMessage->{DATA};
            }

            if (defined($args{MESSAGE})) {
                $ {$args{MESSAGE}} = \%message;
            }
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub forkedCopy() {
    my ($messager) = @_;

# TODO: We really must figure out the right way to solve this problem
# of objects getting cloned when we fork.

    $messager->{FORKED_COPY} = $TRUE;
    if ($messager->{SLINKE}) {
        $messager->{SLINKE}->{FORKED_COPY} = $TRUE;
    }
}



sub createObject($$) {

    my ($class, %args) = @_;

    my %messager;

    my $errorRet;

    if (!defined($args{SLINKE})) {
        $errorRet = "You must specify the SLINKE argument";
    } else {
        $messager{SLINKE} = $args{SLINKE};
        
        $debugLevel = $args{DEBUGLEVEL} || 0;
        
        if ($args{NOINIT}) {
            loadSlinkeSettings(\%messager, \$errorRet);
        } else {
            $messager{IRSAMPLEPERIOD} = 
                $args{IRSAMPLEPERIOD} || $DEFAULT_SAMPLE_PERIOD;
            $messager{RECEIVEZONEMASK} = 0xFF;
            $messager{TRANSMITZONEMASK} = 0xFF;
            $messager{IRTIMEOUT} = 
                $args{IRTIMEOUT} || $DEFAULT_TIMEOUT_PERIODS;
            
            resetSlinke(\%messager, \$errorRet);
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
    my $messager = \%messager;
    bless($messager, $class);
    if (defined($args{MESSAGER})) {
        $ {$args{MESSAGER}} = $messager;
    }
}

# Return true to 'use':
1;
