package Slinke::Irdb;
=head1 NAME

Slinke::Irdb - module to maintain and use a database of infrared (IR) signals
with the Slink-e device

=head1 SYNOPSIS

   TBD

=head1 DESCRIPTION

This package provides facilities for maintaining and accessing a 
database of descriptions of IR (infrared) messages -- the kind of
messages that come from a TV remote control.

You can use Slinke::Irdb in conjunction with Slinke.pm and a Slink-e
device to easily have a Perl program send commands to your
IR-controlled equipment and use your IR remote control to control a
Perl program.

Slink-e is a device that, among other things, generates and receives
IR (infrared) messages.  Nirvis Systems (http://www.nirvis.com) makes
Slink-e.

The Slinke::Control module gives you a low level interface to a Slink-e
device that you can use to generate and receive IR messages.  At this
level, you work with bits and encodings and timings.

The Slinke::Irdb module works in conjunction with Slinke.pm to provide a
higher level interface in which you say things like, "send the 'play'
command to component 'vcr1'.  It works via a database which contains
the details of the encoding of various messages for various
IR-controlled equipment.  The database lives in a file.

The quintessential example of using Slinke::Irdb is the program 'slinke'.

=head1 METHODS

=cut

use strict;
use Exporter;
use Data::Dumper;
use English;
use Errno ":POSIX";
use POSIX qw(floor);
use Time::HiRes;
use Slinke::Messager;
use Slinke::Ircode;
use Slinke::Foundation qw(max min);

my $TRUE=1; my $FALSE = 0;


# An IR pulse train is a representation of a train of carrier on
# pulses and carrier off pulses.  It can be any part of an IR message
# or an entire IR message or multiple IR messages.  It is represented
# in this program by an array reference.  Each element in the array
# corresponds to one pulse in the train.  It is a floating point
# decimal number.  If it is negative, in represents an off pulse;
# positive means on.  The magnitude of the value is the duration of
# the pulse in microseconds.  The array thus consists of 1 or more
# alternating positive and negative numbers.  Note that these pulses
# represent a continuous time interval -- there's no such thing as a
# time when there is no pulse in progress.
#
# An IR pulse train happens to be what a Slinke::Control object's sendIR()
# method takes as an argument.  Pulse trains also form much of the
# contents of our IR database.


#
# %{$database} is a hash of references to hashes, named as follows:
#
#    COMMANDS:         key = IR code
#                      value = [ component name, command name ]
#    COMMAND_LOOKUP:   key = component name
#                      value = reference to hash:
#                        key = command name
#                        value = IR code
#    COMPONENTS:       key = component name
#                      value = \{ "HEAD"     =>[ headstring ],
#                                 "CONTHEAD  =>[ continue_headstring ],
#                                 "TAIL"     =>[ tailstring ],
#                                 "PAUSETIME"=>pause time in microseconds
#                                 "ZERO"     =>[ zerostring ],
#                                 "ONE"      =>[ onestring ],
#                                 "WEIGHT"   =>how many samples were averaged
#                                              to get head, tail, etc.
#                                 "CODELENGTH" =>
#                                              number of bits in an IR code
#                                 "COMPONENT_CODE" =>
#                                              component code as a binary
#                                              string
#                                 "COMMAND_SEPARATION" => 
#                                              microseconds by which you must
#                                              separate commands.
#                                }
#                         HEAD, CONTHEAD, TAIL, ZERO, and ONE are IR
#                         pulse trains.  See definition above.
#
#                         CONTHEAD is the pulse train for the head sequence
#                         of a "continue" IR message, which is an IR message
#                         that says, "He's still holding down the button."
#                         This is undefined for a component that does not
#                         have a continue IR message.  Most don't.
#
#                         PAUSETIME is the minimum intermessage pause time,
#                         in microseconds.
#
#                         The decoder we use assumes that ZERO and ONE are
#                         one on pulse followed by one off pulse, so that's
#                         all you'll see in database entries that were learned
#                         through this library.  But in theory, these could
#                         be any IR message string.
#
#                         COMMMAND_SEPARATION excludes the minimum
#                         intermessage pause.  I.e. the total off time
#                         between two independent commands must be at
#                         least COMMAND_SEPARATION plus PAUSETIME
#                         microseconds.
#                         
#
#    COMPONENT_LOOKUP: key = component code
#                      value = component name
#
# E.g. $database{COMMANDS}->...
#
# This database format has some problems, that should probably be addressed
# some day:
#
#
#   - COMMANDS and COMMAND_LOOKUP treat an IR code as opaque -- not
#     composed of component code and command code.  Instead, there
#     should be a separate COMMANDS hash for each component and 
#     COMMAND_LOOKUP should contain command codes, not full IR codes.
#
#   - The indexes COMMANDS and COMPONENT_LOOKUP should be built
#     at load time, not stored in the database.
#
#   - The names of COMMANDS and COMMAND_LOOKUP should be swapped.
#
# %{$database} has a few dynamic state elements too:
#
#    FILENAME:         Name of file in which database resides.
#
#    NEWFILE:          Database was started new because file didn't exist.
#  
#    MODIFIED:         Database in %database is newer than in the file.


my %msgHistory;
# %msgHistory is defined as follows:
#
# $msgHistory{$zone} is the history of messages received in Zone
# number $zone via this package.  It is a reference to a hash with
# these keys:
#
#   TIME     => Date/time a message was most recently received.  This is a
#               reference to a list of (seconds, microseconds), as used by with
#               the Time::HiRes package.
#
#   COMPNAME => Component name of component of the message referred to above.
#  
# $msgHistory{$zone} is undefined if no message has ever been received in
# zone $zone.



my $debugLevel;

sub debug {
    my ($level, $message) = @_;
    
    if ($debugLevel >= $level) {
        print("Slinke::Irdb: $message\n");
    }
}



sub showPulses($) {

    my ($pulseTrain) = @_;

    my ($pulsesOnLine, $sum);

    $pulsesOnLine = 0;
    $sum = 0;
    
    print("Pulses:    ");   
    foreach (@{$pulseTrain}) {
        if ($pulsesOnLine >= 10) {
            print("\n");
            print("           ");
            $pulsesOnLine = 0;
        }
        printf("%+5d ", $_);
        $sum += abs($_);
        ++$pulsesOnLine;
    }
    print("\n");
    print("Tot time:  " . $sum/1e6 . "s\n");
}



sub adaptOldDatabase($) {
    my ($database) = @_;
    
    while (my ($componentCode, $compName) = 
           each(%{$database->{COMPONENT_LOOKUP}})) {

        my $componentR = $database->{COMPONENTS}{$compName};

        if (!exists($componentR->{COMPONENT_CODE})) {
            $componentR->{COMPONENT_CODE} = $componentCode;
        }
        if (!exists($componentR->{CODELENGTH})) {
            my $commandHashR = $database->{COMMAND_LOOKUP}->{$compName};
            my ($sampleCmdName, $sampleIrCode) = each(%{$commandHashR});
            if (defined($sampleIrCode)) {
                $componentR->{CODELENGTH} = length($sampleIrCode);
            }
        }
        if (!exists($componentR->{COMMAND_SEPARATION})) {
            $componentR->{COMMAND_SEPARATION} = 150000;
        }
        if (!exists($componentR->{PAUSETIME})) {
            $componentR->{PAUSETIME} = 100000;
        }
    }
}



sub dbStatus($) {
    
    my ($database) = @_;

    my %dbStatus;

    $dbStatus{FILENAME}        = $database->{FILENAME};
    $dbStatus{COMPONENT_COUNT} = scalar(keys(%{$database->{COMPONENTS}}));
    $dbStatus{COMMAND_COUNT}   = scalar(keys(%{$database->{COMMANDS}}));
    $dbStatus{NEWFILE}         = $database->{NEWFILE};
    $dbStatus{MODIFIED}        = $database->{MODIFIED};

    return \%dbStatus;

}



sub readDatabase($$$) {
    my ($database, $statusR, $errorR) = @_;

    my $file = $database->{FILENAME};

    my (%COMMANDS, %COMPONENT_LOOKUP, %COMPONENTS, %COMMAND_LOOKUP);

    local $/ = "";
    my $openSucceeded = open(FILE, "<$file");

    if ($openSucceeded) {
        $database->{NEWFILE} = $FALSE;
        my $string = <FILE>;
        close FILE;
        if (!defined($string)) {
            $$errorR = "Unable to read from file '$file' after opening it.";
        } else {
            eval($string);
            $$errorR = undef;
        } 
    } else {
        if ($ERRNO == ENOENT) {
            # No such file -- start a fresh database.
            $$errorR = undef;
            $database->{NEWFILE} = $TRUE;
            %COMPONENTS = ();
            %COMMANDS = ();
            %COMMAND_LOOKUP = ();
            %COMPONENT_LOOKUP= ();
        } else {
            $$errorR = "Unable to open database file '$file'.  " .
                "Errno is '$ERRNO'\n";
        }
    }
    $database->{COMMANDS} = \%COMMANDS;
    $database->{COMPONENT_LOOKUP} = \%COMPONENT_LOOKUP;
    $database->{COMPONENTS} = \%COMPONENTS;
    $database->{COMMAND_LOOKUP} = \%COMMAND_LOOKUP;

    adaptOldDatabase($database);

    $database->{MODIFIED} = $FALSE;

    $$statusR = dbStatus($database);
}



sub writeDatabase($$) {
    my ($database, $errorR) = @_;

    my $file = $database->{FILENAME};

    my $success = open(FILE, "> $file");
    if (!$success) {
        $$errorR = "Failed to open '$file' for output.  " .
            "Database updates lost.\n";
    } else {
        print FILE Data::Dumper->Dump( 
            [ $database->{COMMANDS}, 
              $database->{COMPONENTS}, 
              $database->{COMPONENT_LOOKUP}, 
              $database->{COMMAND_LOOKUP} ], 
            [ qw( *COMMANDS *COMPONENTS 
                  *COMPONENT_LOOKUP *COMMAND_LOOKUP ) ] );
        close(FILE);
        $database->{MODIFIED} = $FALSE;
    }
}



sub getIrCode($$$) {
    my ($database, $compName, $commandName) = @_;

    return $database->{COMMAND_LOOKUP}{$compName}->{$commandName};
}



sub componentInfo($$) {
    my ($database, $compName) = @_;

    return $database->{COMPONENTS}{$compName};
}



sub componentsInfo($) {

    my ($database) = @_;

    my %componentsInfo;
    
    foreach my $compName (keys(%{$database->{COMPONENTS}})) {
        $componentsInfo{$compName} = ();
    }

    # Now patch in the component codes

    while (my ($componentCode, $compName) = 
           each(%{$database->{COMPONENT_LOOKUP}})) {

        my $componentInfoR = $database->{COMPONENTS}->{$compName};

        $componentsInfo{$compName}{COMPONENT_CODE} = $componentCode;
        $componentsInfo{$compName}{PROTOCOL} = compProtocol($componentInfoR);
        $componentsInfo{$compName}{CODELENGTH} = $componentInfoR->{CODELENGTH};
    }

    foreach (keys(%componentsInfo)) {
        if (!defined($componentsInfo{$_}->{COMPONENT_CODE})) {
            die("No component code in database for component '$_'");
        }
    }
    return %componentsInfo;
}



sub commandsInfo($$) {
    my ($database, $componentName) = @_;
#-----------------------------------------------------------------------------
#  Return information about all the commands for component $componentName.
#
#  Return it as a hash with command name as key.  The value of a hash
#  element is the full IR code for the command.
#-----------------------------------------------------------------------------
    my %commandInfo;

    %commandInfo = ();  # initial value 

    my $commandHashR = $database->{COMMAND_LOOKUP}{$componentName};
    if (defined($commandHashR)) {
        foreach my $commandName (keys(%$commandHashR)) {
            my $irCode = $commandHashR->{$commandName};
            $commandInfo{$commandName} = $irCode;
        }    
    }
    return %commandInfo;
}



sub constructIrMessage($$$) {
    my ($irCode, $componentR, $count) = @_;
#-----------------------------------------------------------------------------
#  Return an IR pulse train that consists of $count IR messages for 
#  Component $$componentR, carrying IR code $irCode.
#
#  Every message is followed by a minimal intermessage pause.  The last
#  one is extra long to provide command separation.
#
#  $irCode is a string of '1' and '0' characters.
#
#  See the top of this file for a description of an IR pulse train.
#
#  The pulse train includes a inter-message pause (OFF pulse).
#-----------------------------------------------------------------------------
    if (!defined($count)) {
        $count = 1;
    }

    my @pulseTrain;

    for (my $i = 0; $i < $count; ++$i) {
        # Put the head segment in the stream.
        push(@pulseTrain, @{$componentR->{"HEAD"}});

        # put the IR code for the command in the stream.
        foreach my $i (split(/ */, $irCode)) {
            push(@pulseTrain, @{$componentR->{$i == 0 ? "ZERO" : "ONE"}});
        }
        
        # Put a tail code in the stream.
        push(@pulseTrain, @{$componentR->{"TAIL"}});

        # Add a minimal intermessage pause time to ensure the next
        # IR message that gets sent through this Slink-e doesn't get
        # confused with this one.

        push(@pulseTrain, - $componentR->{"PAUSETIME"});
    }

    # Add an extra amount to the last intermessage pause, to make
    # sure the next message sent through this Slink-e doesn't look like
    # a continuation of the same command.  (Most remotes send a series of
    # identical messages to indicate that the user is holding the button
    # down).

    push(@pulseTrain, -$componentR->{COMMAND_SEPARATION});

    return @pulseTrain;
}



sub compProtocol($) {
    my ($componentInfo) = @_;

    my $protocol;

    my @zero = @{$componentInfo->{ZERO}};
    my @one  = @{$componentInfo->{ONE}};
    
    if ( $zero[0] == $one[0] ) {
        $protocol = "STANDARD";
    } else {
        $protocol = "SONY";
    }
    return $protocol;
}



sub composeIrCode($$$) {
    my ($componentCode, $commandCode, $protocol) = @_;

    if (defined($protocol)) {
        if ($protocol eq "SONY") {
            return $commandCode . $componentCode;
        } else {
            return $componentCode . $commandCode;
        }
    }
}



sub recordHistory($$) {
    my ($msgHistoryR, $compName) = @_;
    
    my $msgHistory;

    if (defined($$msgHistoryR)) {
        $msgHistory = $$msgHistoryR;
    } else {
        $msgHistory = \my %msgHistory;
        $$msgHistoryR = $msgHistory;
    }
    $msgHistory->{COMPNAME} = $compName;
    $msgHistory->{TIME} = [gettimeofday()];
}



=head2 $database->isContinuationMessage(%ARGS)

This method tell you whether a certain pulse set is a continuation IR
message -- the special IR message some components have that says, 
"User is still holding down the button."

%ARGS is defined as follows:

  PULSESET    => Reference to pulse set array.  E.g. [ 9000, -2200, 700 ]

  ELAPSEDTIME => Amount of time that has passed since the IR message of
                 which this is suspected of being a continuation.  This
                 is a number of seconds, not necessarily an integer.

  COMPONENT   => Name of component for the message mentioned above.
  
=cut

sub isContinuationMessage($%) {
    my ($database, %args) =@_;

    my $pulseSet    = $args{PULSESET};
    my $elapsedTime = $args{ELAPSEDTIME};
    my $compName    = $args{LASTCOMPONENT};

    my $retval;

    if (@{$pulseSet} == 3) {
        # It's the right length for a continuation message.

        if ($elapsedTime < 0.5) {
            # We received a message in this zone less than 500 mS ago.
            if (defined($compName)) {
                my $component = $database->{COMPONENTS}->{$compName};
                if (defined($component)) {
                    if ($component->{CONTHEAD}) {
                        # The component defines continuation messages
                        
                        # We really ought to check the pulse train
                        # against the database to see if it is a
                        # plausible cointinuation message or just some
                        # noise, but we're too lazy.  We'll wait until
                        # it becomes an issue.
                        
                        $retval = $TRUE;
                    }
                }
            }
        }
    }
    return $retval;
}



sub isContinuation($$$) {

    my ($database, $message, $zone) = @_;

    my $zoneHistory = $msgHistory{$zone};

    return isContinuationMessage
        ($database, (
                     PULSESET      => $message->{DATA},
                     ELAPSEDTIME   => tv_interval($zoneHistory->{TIME}),
                     LASTCOMPONENT => $zoneHistory->{COMPNAME}
                     )
         );
}



sub sendIrMessage2($$$$$$$$) {
    my ($database, $messager, $compName, $commandName, $wantFeedback, 
        $count, $zones, $errorR) = @_;
    
    debug(2, "Sending '$commandName' command for component '$compName'");
    my $commandLookupTable = $database->{COMMAND_LOOKUP}{$compName};
    if (!defined($commandLookupTable)) {
        $$errorR = "There is no component named $compName";
    } else {
        my $irCode = $commandLookupTable->{$commandName};
        if (!defined($irCode)) {
            $$errorR = "There is no command '$commandName' " .
                "for component '$compName'"; 
        } else {
            my $component = $database->{COMPONENTS}{$compName};
            
            my @pulseTrain = constructIrMessage($irCode, $component, $count);
            
            if ($debugLevel > 2) {
                showPulses(\@pulseTrain);
            }
            $messager->sendIR(DATA     => \@pulseTrain, 
                              ZONES    => $zones,
                              FEEDBACK => $wantFeedback,
                              ERROR    => \my $error,
                              );
            if ($error) {
                $$errorR = "Failed to send composed IR message.  $error";
            }
        }
    }
}



=head2 $database->sendIrMessage(%ARGS)

This method sends an IR message containing a command from the database.

%ARGS is defined as follows:

  SLINKEMSGR=> A Slinke::Messager object for the Slink-e.

  COMPONENT => The component name for the command.

  COMMAND   => The command name.

  ZONES     => A reference to an array that indicates in which IR zones to
               send the message.  If undefined, the method sends in all
               zones.

               The array is a list of zone numbers of the zones.  E.g.

               ZONES => [ 3, 4 ];

               Note that on a Slink-e earlier than Version 2, the only
               valid zone is 0.  This method fails if you specify
               any other zone.

  COUNT     => Number of messages to be sent, in quick succession so that
               they look like a continuation to a component that recognizes
               repeated messages as a continuation (user holding down button).
               Default is 1.

  WANTFEEDBACK=> True if you want the Slink-e to receive your message so you
               can then read it with receiveIrMessage().  False if you 
               want the Slink-e to mask off all IR receiving while it 
               sends the message, so as to avoid receiving an echo of this
               message.

  ERROR     => Reference to an error string variable.

               If the method is unable to receive a message or unable
               to extract a component code and command code from it,
               it returns a text explanation of the problem as
               this value.  Otherwise, it returns this value undefined.

=cut

sub sendIrMessage(%) {
    my ($database, %args) = @_;

    my $errorRet;

    if (!$args{SLINKEMSGR}->isa("Slinke::Messager")) {
        $errorRet = "SLINKEMSGR argument is not a Slinke::Messager object";
    } else {
        my $count = defined($args{COUNT}) ? $args{COUNT} : 1;
        if ($count !~ m{\d+} || $count == 0) {
            $errorRet = "COUNT argument is not a positive integer";
        } else {
            my $zones = 
                defined($args{ZONES}) ? $args{ZONES} : [0,1,2,3,4,5,6,7];
            
            if (ref($zones) != "ARRAY") {
                $errorRet = "ZONES argument is not an array reference.";
            } else {
                sendIrMessage2($database, $args{SLINKEMSGR}, 
                               $args{COMPONENT}, $args{COMMAND}, 
                               $args{WANTFEEDBACK},
                               $count, $zones, \$errorRet);
            }
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub timeIrMessage2($$$$$$) {

    my ($database, $compName, $commandName, $count, $periodR, $errorR) = @_;

    my $commandLookupTable = $database->{COMMAND_LOOKUP}{$compName};
    if (!defined($commandLookupTable)) {
        $$errorR = "There is no component named $compName";
    } else {
        my $irCode = $commandLookupTable->{$commandName};
        if (!defined($irCode)) {
            $$errorR = "There is no command '$commandName' " .
                "for component '$compName'"; 
        } else {
            my $component = $database->{COMPONENTS}{$compName};
            
            my @pulseTrain = 
                constructIrMessage($irCode, $component, $count);
            
            my $totalPeriod;
            
            $totalPeriod = 0;   # initial value
            foreach my $pulsePeriod (@pulseTrain) {
                $totalPeriod += abs($pulsePeriod);
            }
            $$periodR = $totalPeriod;
        }
    }
}
    


=head2 $database->timeIrMessage(%ARGS)

This method calculates how much time it an IR message (or series of
them) for a command in the database takes.

The total includes one minimal intermessage pause after each message,
and an additional pause after the last one to provide command
separation.

%ARGS is defined as follows:

  COMPONENT => The component name for the command.

  COMMAND   => The command name.

  COUNT     => Number of messages to be sent, in quick succession so that
               they look like a continuation to a component that recognizes
               repeated messages as a continuation (user holding down button).
               Default is 1.

  PERIOD    => Reference to a variable which this method sets to the
               requested period, in seconds.

  ERROR     => Reference to an error string variable.

               If the method is unable to receive a message or unable
               to extract a component code and command code from it,
               it returns a text explanation of the problem as
               this value.  Otherwise, it returns this value undefined.

=cut

sub timeIrMessage(%) {
    my ($database, %args) = @_;

    my $errorRet;
    my $period;

    my $count = defined($args{COUNT}) ? $args{COUNT} : 1;
    if ($count !~ m{\d+} || $count == 0) {
        $errorRet = "COUNT argument is not a positive integer";
    } else {
        timeIrMessage2($database, $args{COMPONENT}, $args{COMMAND},
                       $count, \my $microSeconds, \$errorRet);

        if (defined($args{PERIOD})) {
            $ {$args{PERIOD}} = $microSeconds/1E6;
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub signalHandler {
    my ($signal) = @_;
    die("Signal $signal received");
}



sub compNameFromCode($) {
    my ($database, $compCode) = @_;

    my $retval;

    if (defined($compCode)) {
        $retval = $database->{COMPONENT_LOOKUP}{$compCode};
    }
    return $retval;
}



sub lookupCommandCode($$$) {
    my ($database, $compName, $commandCode, $commandNameR, $errorR) = @_;

    my $retval;

    if (defined($compName) && defined($commandCode)) {
        my $componentInfo = $database->{COMPONENTS}{$compName};

        my $expectedCodelen = int($componentInfo->{CODELENGTH} + 0.5) 
            - length($componentInfo->{COMPONENT_CODE});

        if (length($commandCode) != $expectedCodelen) {
            $$errorR = 
                "Command code is wrong length.  For Component '$compName', " .
                "it should be $expectedCodelen bits, but it " .
                "is " . length($commandCode) . " bits.";
        } else {
            my $irCode = composeIrCode($componentInfo->{COMPONENT_CODE}, 
                                       $commandCode, 
                                       compProtocol($componentInfo));
            $$commandNameR = $database->{COMMANDS}{$irCode}->[1];
        }                
    }
}



sub interpretIrMessage($$$$$) {
    my ($database, $irMessage, $compNameR, $commandNameR, $errorR) = @_;

    if (!defined($irMessage)) {
        die("INTERNAL ERROR: no message passed to interpretIrMessage");
    }
    
    my $compCode = $irMessage->{COMPONENT_CODE};

    if (defined($compCode)) {
        my $compName = $database->{COMPONENT_LOOKUP}{$compCode};
        if (defined($compName)) {
            my $componentR = $database->{COMPONENTS}{$compName};
            my $expectedCodelen = int($componentR->{CODELENGTH} + 0.5);
            if (length($irMessage->{CODE}) != $expectedCodelen) {
                $$errorR = 
                    "IR code is wrong length.  For Component '$compName', " .
                    "it should be $expectedCodelen bits, but it " .
                    "is " . length($irMessage->{CODE}) . " bits.";
            } else {
                if (defined($irMessage->{CODE})) {
                    my $commandName = 
                        $database->{COMMANDS}{$irMessage->{CODE}}->[1];
                    if (defined($commandNameR)) {
                        $$commandNameR = $commandName;
                    }
                }
            }                
            if (defined($compNameR)) {
                $$compNameR = $compName;
            }
        }
    }
}



=head2 $database->receiveIrMessage(%ARGS)

This method waits until the Slink-e receives an IR message and returns
the command and component names that describe the command in the message.

This method discards all port messages the Slink-e reports that are
not IR port messages.

%ARG is defined as follows:

  SLINKEMSGR=> A Slinke::Messager object for the Slink-e.

  COMPONENT => Reference to variable which this method sets to the name 
               of component for IR message received.  If the IR code in
               the message does not indicate a component in the
               database and the IR message is not a special continuation
               message, the method returns success with this value 
               undefined.

  COMMAND   => Reference to variable which this method sets to the name
               of the command received.  If the IR code in the message
               does not indicate a command in the database, the method
               returns success with this value undefined.  If there is
               no IR code, same thing.

  CONTINUE  => Reference to variable which this method sets to a boolean
               value.  True means the IR message received is a special
               continuation message.  See below.

  ZONE      => The number of the IR zone in which the IR message was received.

  ERROR     => Reference to an error string variable.

               If the method is unable to receive a message or unable
               to extract a component code and command code from it,
               it returns a text explanation of the problem as
               this value.  Otherwise, it returns this value undefined.

  Continuation messages: Some components have a special IR message
  that means "User is still holding the button down."  That message
  contains no IR code, so there's no way for us to recognize it
  independently.  Instead, we recognize it as follows: If the
  component from which we received the previous IR message in the same
  zone has a continuation IR message and we received that previous
  message less than 500 milliseconds ago, and the message we just
  received has the form of that component's continuation message, then
  we recognize the message as a continuation message for that
  component.  In this case, we return the component name as COMPONENT,
  undefined COMMAND, and CONTINUE true.
    
  Note that this continuation message magic is the only way the Irdb
  object maintains state of Slink-e accesses.

=cut
sub receiveIrMessage(%) {

    my ($database, %args) = @_;

    my $messager = $args{SLINKEMSGR};
    
    my $errorRet;

    if (!$messager->isa("Slinke::Messager")) {
        $errorRet = "Slinke messager object is not a Slinke::Messager";
    } else {
        my $message;

        eval {
            local $SIG{"INT"} = \&signalHandler;
            
            do {
                debug(2, "Waiting for message from Slink-e");
                $messager->receiveSlinkeMessage(\$message, \my $error);
                debug(2, "Message from Slink-e port $message->{PORT}");
            } while ($message->{PORT} ne "PORT_IR");
            
            pop(@{$message->{DATA}});
        };

        if ($@) {
            # The eval block terminated abnormally.
            if ($@ =~ m{^Signal INT received}) {
                # It was terminated by our signal handler; we'll just
                # return an undefined value.
            } else {
                $errorRet = "Died while trying to receive IR code.  $@";
            }
        } else {
            my $zone = $message->{ZONE};

            my ($continue, $compName, $commandName);
            if (isContinuationMessage($message->{DATA}, 
                                      $msgHistory{$zone})) {
                $continue = $TRUE;
                $compName = msgHistory{$zone}->COMPNAME;
            } else {
                Slinke::Ircode::decodeIrMessage
                    ($message->{DATA}, \my $irMessage, \my $error);
                if (!$error) {
                    $database->interpretIrMessage($irMessage,
                                                  \$compName, 
                                                  \$commandName, 
                                                  \$errorRet);
                }
            }
            if (!$errorRet) {
                recordHistory(\$msgHistory{$zone}, $compName);

                if (defined($args{ZONE})) {
                    $ {$args{ZONE}} = $zone;
                }
                if (defined($args{CONTINUE})) {
                    $ {$args{CONTINUE}} = $continue;
                }
                if (defined($args{COMMAND})) {
                    $ {args{COMMAND}} = $commandName;
                }
                if (defined($args{COMPONENT})) {
                    $ {args{COMPONENT}} = $compName;
                }
            }
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub averageInComponent($$$) {

    my ($database, $componentName, $irMessage) = @_;

    my $component = $database->{COMPONENTS}{$componentName};

    if (!defined($component->{WEIGHT})) {
        # Must be from an old database before WEIGHT existed.
        $component->{WEIGHT} = 1;
    }

    my $weight = $component->{WEIGHT};
        # The weight that the values in the database have; i.e. the number
        # of previous observations of which they are an average.

    if ($irMessage->{PAUSETIME}) {
        if ($component->{PAUSETIME}) {
            my $newMean = ($component->{PAUSETIME} * $weight + 
                           $irMessage->{PAUSETIME} * 1) / ($weight + 1 );

            $component->{PAUSETIME} = floor($newMean + 0.5);
        } else {
            $component->{PAUSETIME} = $irMessage->{PAUSETIME};
        }
    }

    if ($#{$irMessage->{HEAD}} == $#{$component->{HEAD}}) {
        for (my $i=0; $i <= $#{$irMessage->{HEAD}}; ++$i) {
            my $newMean = ($component->{HEAD}->[$i] * $weight + 
                           $irMessage->{HEAD}->[$i] * 1) / ($weight + 1);
            $component->{HEAD}->[$i] = floor($newMean + 0.5);
        }
    }

    if ($#{$irMessage->{TAIL}} == $#{$component->{TAIL}}) {
        for (my $i=0; $i <= $#{$irMessage->{TAIL}}; ++$i) {
            my $newMean = ($component->{TAIL}->[$i] * $weight + 
                           $irMessage->{TAIL}->[$i] * 1) / ($weight + 1);
            $component->{TAIL}->[$i] = floor($newMean + 0.5);
        }
    } elsif ($#{$component->{TAIL}} < 0) {
        $component->{TAIL} = $irMessage->{TAIL};
    }

    if ($#{$irMessage->{ZERO}} == $#{$component->{ZERO}}) {
        for (my $i=0; $i <= $#{$irMessage->{ZERO}}; ++$i) {
            my $newMean = 
                ($component->{ZERO}->[$i] * $weight + 
                 $irMessage->{ZERO}->[$i] * 1) / ($weight + 1);
            $component->{ZERO}->[$i] = floor($newMean + 0.5);
        }
    }

    if ($#{$irMessage->{ONE}} == $#{$component->{ONE}}) {
        for (my $i=0; $i <= $#{$irMessage->{ONE}}; ++$i) {
            my $newMean = 
                ($component->{ONE}->[$i] * $weight + 
                 $irMessage->{ONE}->[$i] * 1) / ($weight + 1);
            $component->{ONE}->[$i] = floor($newMean + 0.5);
        }
    }

    my $codelen = ($component->{CODELENGTH} * $weight +
                   length($irMessage->{CODE})) / ($weight + 1);
         
    # Since we know the code length is really an integer, and any deviation
    # is due to infrequent errors, which should be ignored, we round off a
    # little here.
    
    my $roundedCodelen = int($codelen + 0.5);
    if (abs($roundedCodelen - $codelen) < 0.1) {
        $component->{CODELENGTH} = $roundedCodelen;
    } else {
        $component->{CODELENGTH} = $codelen;
    }
    ++$component->{WEIGHT};
    $database->{MODIFIED} = $TRUE;
}



sub addNewComponent($$$) {
    my ($database, $componentName, $irMessage) = @_;
#-----------------------------------------------------------------------------
#  Add to the database a new component, one which sends messages like
#  $irMessage.  Name it $componentName.
#
#  Results undefined if there is already a component by that name or
#  $irMessage is not a valid message (in a format we understand).
#-----------------------------------------------------------------------------
    if (!defined($irMessage->{COMPONENT_CODE})) {
        die("IR message with undefined component code " .
            "passed to addNewComponent");
    }
    $database->{COMPONENT_LOOKUP}{$irMessage->{COMPONENT_CODE}} = 
        $componentName;
    
    my $componentR;
    
    $componentR->{COMPONENT_CODE} = $irMessage->{COMPONENT_CODE};
    $componentR->{HEAD}           = $irMessage->{HEAD};
    $componentR->{TAIL}           = $irMessage->{TAIL};
    $componentR->{ZERO}           = $irMessage->{ZERO};
    $componentR->{ONE}            = $irMessage->{ONE};
    $componentR->{PAUSETIME}      = $irMessage->{PAUSETIME};
    $componentR->{CODELENGTH}     = length($irMessage->{CODE});
    $componentR->{WEIGHT}         = 1;

    $database->{COMPONENTS}{$componentName} = $componentR;
    $database->{COMMAND_LOOKUP}{$componentName} = { };

    $database->{MODIFIED} = $TRUE;
}



sub deleteComponent($$$) {
    my ($database, $componentName, $errorR) = @_;

    if (!exists($database->{COMPONENTS}{$componentName})) {
        $$errorR = "No such component: '$componentName'\n";
    } else {
        delete($database->{COMPONENTS}{$componentName});
        delete($database->{COMMAND_LOOKUP}{$componentName});
        foreach (keys(%{$database->{COMPONENT_LOOKUP}})) {
            if ($database->{COMPONENT_LOOKUP}{$_} eq $componentName) {
                delete($database->{COMPONENT_LOOKUP}{$_});
            }
        }
        $database->{MODIFIED} = $TRUE;
    }
}



sub addNewCommand($$$$$$) {
#-----------------------------------------------------------------------------
#  Add a command to the database.
#
#  $compName must be a known component.
#
#  $commandCode must be the proper length for command codes for the
#  specified component.
#
#  The specified component must not already have a command with the
#  command code $commandCode.
#-----------------------------------------------------------------------------
    my ($database, $compName, $commandName, $commandCode,
        $errorR) = @_;

    my $componentInfo = $database->{COMPONENTS}{$compName};
    if (!defined($componentInfo)) {
        $$errorR = "No such component: '$compName'";
    } else {
        my $componentCode = $componentInfo->{COMPONENT_CODE};
        my $irCode = composeIrCode($componentCode, $commandCode, 
                                   compProtocol($componentInfo));

        my $expectedCodelen = int($componentInfo->{CODELENGTH} + 0.5);
        if (length($irCode) != $expectedCodelen) {
            $$errorR = "Wrong length IR code.  Component $compName has " .
                "$expectedCodelen bit IR codes.  This one is " .
                length($irCode) . " bits.";
        } else {
            my $existingCommandName = $database->{COMMANDS}{$irCode}->[1];

            if (defined($existingCommandName)) {
                $$errorR = "Command code is for already known command " .
                    "'$existingCommandName'";
            } else {
                my $componentCommandTableR = 
                    $database->{COMMAND_LOOKUP}{$compName};

                if (defined($componentCommandTableR->{$commandName})) {
                    $$errorR = "Command named '$commandName' already " .
                        "exists for component '$compName'";
                } else {
                    $database->{COMMANDS}{$irCode} = 
                        [$compName, $commandName];
                    $componentCommandTableR->{$commandName} = $irCode;
                    $database->{MODIFIED} = $TRUE;
                }
            }
        }
    }
}



sub deleteCommand($$$$) {
    my ($database, $compName, $commandName, $errorR) = @_;

    my $commandHashR = $database->{COMMAND_LOOKUP}{$compName};
    if (!exists($commandHashR->{$commandName})) {
        $$errorR = "The command '$commandName' for component '$compName' " .
            "does not exist.\n";
    } else {
        my $irCode = $commandHashR->{$commandName};
                
        delete($database->{COMMANDS}{$irCode});
        delete($database->{COMMAND_LOOKUP}{$compName}->{$commandName});
        
        $database->{MODIFIED} = $TRUE;
    }
}



sub createObject($%) {

    my ($class, %args) = @_;

    my $errorRet;
        # Description of why we can't create the object.  Undefined if
        # we haven't given up yet.

    my $errorRet;

    $debugLevel = $args{DEBUG} || 0;

    if (!defined($args{FILE})) {
        $errorRet = "You must specify the FILE argument.";
    } else {
        my $database;

        $database->{FILENAME} = $args{FILE};
        readDatabase($database, \my $status, \my $error);
        if ($error) {
            $errorRet = "unable to read database from file '$args{FILE}'.  " .
                "$error";
        } else {
            bless($database, $class);

            debug(1, "Slinke::Irdb object constructed in class '$class'");

            if (defined($args{DATABASE})) {
                $ {$args{DATABASE}} = $database;
            }
            if (defined($args{STATUS})) {
                $ {$args{STATUS}} = $status;
            }
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub DESTROY($) {
    my ($database) = @_;
# This is, by virtue of its name, the destructor for a Slinke::Irdb object.
# The Perl interpreter calls it when the last reference to the object goes
# away.

    if ($database->{FORKED_COPY}) {
        debug(1, "Skipping destroy of forked copy of object");
    } else {
        debug(1, "Slinke::Irdb object being destroyed");
    }
}

# And now return a TRUE value to tell Perl that the 'use' went OK.
1;


=head1 AUTHOR

Bryan Henderson <bryanh@giraffe-data.com> San Jose CA 03.09.27

Contributed to the public domain by its author.

