package Slinke::Ircode;

use strict;
use warnings;
use Data::Dumper;
use English;
use POSIX qw(floor ceil);

my $TRUE=1; my $FALSE = 0;


sub pulseIsImPause($) {
    
    my ($pulse) = @_;

    return ($pulse < 0 && -$pulse > 6000);
}



sub extractFirstMsg($) {

    my @firstMsg;

    my ($pulseTrain) = @_;

    @firstMsg = ();
    
    for (my $i = 0;
         $i <= $#{$pulseTrain} && !pulseIsImPause($pulseTrain->[$i]);
         ++$i) {
        push(@firstMsg, $pulseTrain->[$i]);
    }
    return @firstMsg;
}



sub mean(@) {
    
    if (@_ == 0) {
        return undef;
    } else {
        my $sum;
        $sum = 0;
        
        foreach(@_) {
            $sum += $_;
        }
        return $sum / @_;
    }
}



sub min(@) {

    my $min;

    if (@_ > 0) {
        $min = $_[0];
        foreach(@_) {
            if ($_ < $min) {
                $min = $_;
            }
        }
    }
    return $min;
}



sub max(@) {
    
    my $max;

    if (@_ > 0) {
        $max = $_[0];
        foreach(@_) {
            if ($_ > $max) {
                $max = $_;
            }
        }
    }
    return $max;
}



sub clusterMeans(@) {
#-----------------------------------------------------------------------------
#  Return the mean of the elements of the list that are closest to the
#  mininum and the mean of the elements that are closest to the maximum.
#  Consider an element that is equidistant to be closer to the maximum.
#
#  (This is useful if you know the list is bimodal).
#-----------------------------------------------------------------------------
    my ($min, $max) = (min(@_), max(@_));

    my ($sumlo, $countlo, $sumhi, $counthi);
    $sumlo = 0; $sumhi = 0; $countlo = 0; $counthi = 0;
    foreach(@_) {
        if ($max - $_ > $_ - $min) {
            $sumlo += $_;
            $countlo += 1;
        } else {
            $sumhi += $_;
            $counthi += 1;
        }
    }
    my ($meanlo, $meanhi);
    
    if ($countlo > 0) {
        $meanlo = $sumlo/$countlo;
    }
    if ($counthi > 0) {
        $meanhi = $sumhi/$counthi;
    }

    return($meanlo, $meanhi);
}    



sub measureCodePulses($$$) {
    my ($codePulseTrainR, $pulseProfileR, $errorR) = @_;

    my %pulseProfile;

    my (@onPulses, @offPulses);

    for (my $i = 0; $i <= $#{$codePulseTrainR} && !$$errorR; $i += 2) {
        my $onPulse = $codePulseTrainR->[$i];
        my $offPulse = $codePulseTrainR->[$i+1];

        if (!defined($offPulse)) {
            $$errorR = "IR code segment contains an odd number of pulses.";
        } elsif ($onPulse == 0 || $offPulse == 0) {
            $$errorR = "IR code segment contains a zero duration pulse.";
        } elsif ($onPulse < 0) {
            $$errorR = "IR code segment contains an off pulse where an on " .
                "pulse belongs.";
        } elsif ($offPulse >= 0) {
            $$errorR = "IR code segment contains an on pulse where an off " .
                "pulse belongs.";
        } else {
            push(@onPulses, $onPulse);
            push(@offPulses, -$offPulse);
        }
    }

    $pulseProfile{ON}->{MEAN} = mean(@onPulses);
    $pulseProfile{OFF}->{MEAN} = mean(@offPulses);
    ($pulseProfile{ON}->{MEANLO}, $pulseProfile{ON}->{MEANHI}) = 
        clusterMeans(@onPulses);
    ($pulseProfile{OFF}->{MEANLO}, $pulseProfile{OFF}->{MEANHI}) = 
        clusterMeans(@offPulses);

    $$pulseProfileR = \%pulseProfile;
}



sub varyingPulseType($) {
    my ($pulseProfile) = @_;
#-----------------------------------------------------------------------------
#  Return "ON" or "OFF", indicating which kind of pulse, according to the
#  pulse profile $pulseProfile, varies.
#
# In a valid code segment pulse train, either the on pulses
# differ in duration or the off pulses differ in duration, but
# not both.  Of course, there is variation due to equipment 
# precision and sampling that we have to ignore.  
#
# We just say that
# whichever kind of pulse exhibits the strongest bimodal character,
# that's the one that distinguishes, and variation in the other is
# just noise.
#-----------------------------------------------------------------------------
    my $onSpread;
    if (!defined($pulseProfile->{ON}->{MEANLO}) ||
        !defined($pulseProfile->{ON}->{MEANHI})) {
        $onSpread = 0;
    } else {
        $onSpread = $pulseProfile->{ON}->{MEANHI} - 
            $pulseProfile->{ON}->{MEANLO};
    }
    
    my $offSpread;
    if (!defined($pulseProfile->{OFF}->{MEANLO}) ||
        !defined($pulseProfile->{OFF}->{MEANHI})) {
        $offSpread = 0;
    } else {
        $offSpread = 
            $pulseProfile->{OFF}->{MEANHI} - 
                $pulseProfile->{OFF}->{MEANLO};
    }

    my $retval;
    if ($onSpread > $offSpread) {
        $retval = "ON";
    }
    if ($onSpread < $offSpread) {
        $retval = "OFF";
    }
    return $retval;
}



sub computeZeroOneEncoding($$$$$$) {
    my ($pulseProfile, $zeroPulseTrainR, $onePulseTrainR, 
        $differentiatorR, $thresholdR, $errorR) = @_;

    if (!defined($pulseProfile->{ON}->{MEAN})) {
        $$errorR = "there are no ON pulses in the IR code pulse train!";
    } elsif (!defined($pulseProfile->{OFF}->{MEAN})) {
        $$errorR = "there are no OFF pulses in the IR code pulse train!";
    } else {
        my ($zeroPulseTrain, $onePulseTrain, $differentiator, $threshold);

        my $varyingPulseType = varyingPulseType($pulseProfile);

        if (!defined($varyingPulseType)) {
            $$errorR = "The pulse trains for all bits in this code segment " .
                "are identical.";
        } else {
            if ($varyingPulseType eq "ON") {
                # The off pulse (2nd pulse) is the same for both; the on pulse
                # differentiates zero from one.
                $differentiator = 0;
                $threshold = mean($pulseProfile->{ON}->{MEANHI},
                                  $pulseProfile->{ON}->{MEANLO});
                $zeroPulseTrain->[0] = $pulseProfile->{ON}->{MEANLO};
                $onePulseTrain->[0] = $pulseProfile->{ON}->{MEANHI};
                $zeroPulseTrain->[1] = -$pulseProfile->{OFF}->{MEAN};
                $onePulseTrain->[1] = -$pulseProfile->{OFF}->{MEAN};
            } else {
                # The on pulse (1st pulse) is the same for both; the off pulse
                # differentiates zero from one.
                $differentiator = 1;
                $threshold = mean($pulseProfile->{OFF}->{MEANHI},
                                  $pulseProfile->{OFF}->{MEANLO});
                $zeroPulseTrain->[0] = $pulseProfile->{ON}->{MEAN};
                $onePulseTrain->[0] = $pulseProfile->{ON}->{MEAN};
                $zeroPulseTrain->[1] = -$pulseProfile->{OFF}->{MEANLO};
                $onePulseTrain->[1] = -$pulseProfile->{OFF}->{MEANHI};
            }
            $$zeroPulseTrainR = $zeroPulseTrain;
            $$onePulseTrainR = $onePulseTrain;
            $$differentiatorR = $differentiator;
            $$thresholdR = $threshold;
        }
    }
}



sub computeImPauseLength($) {

    my ($pulseTrain) = @_;
    
    my ($imPauseSum, $imPauseCount);
    $imPauseSum = 0;
    $imPauseCount = 0;
    foreach my $pulse (@{$pulseTrain}) {
        if (pulseIsImPause($pulse)) {
            $imPauseSum += -$pulse;
            $imPauseCount += 1;
        }
    }

    my $imPauseLength;

    if ($imPauseCount > 0) {
        $imPauseLength = $imPauseSum / $imPauseCount;
    }
    return $imPauseLength;
}



sub decodeIrCode($$$) {
    my ($codePulseTrain, $differentiator, $threshold) = @_;
#-----------------------------------------------------------------------------
#  Decode the pulse train $codePulseTrain, which is the pulse train for
#  the IR code segment of an IR message (no head or tail).
#
#  $differentiator is 0 if the first pulse (on pulse) is what differentiates
#  a zero from a one; 1 if the second pulse (off pulse) does.
#
#  $threshold is the pulse duration (in microseconds) that divides a zero
#  from a one.  If the differentiating pulse is longer than $threshold,
#  it means one.  Shorter means zero.  Equal means zero.
#
#  The return value is a string of '0' and '1' characters.
#
#  Assume the code pulse train is properly constructed of on/off pairs of
#  pulses.
#-----------------------------------------------------------------------------
    my $retval;

    $retval = "";  # initial value

    for (my $i = 0; $i <= $#{$codePulseTrain}; $i += 2) {
        my $pulseDuration = abs($codePulseTrain->[$i + $differentiator]);
        $retval .= $pulseDuration > $threshold ? '1' : '0';
    }
    return $retval;
}


sub decomposeIrCode($$$$) {
    my ($irCode, $ircodeFmt, $componentCodeR, $commandCodeR) = @_;
#-----------------------------------------------------------------------------
#  Decompose the IR code $ircode, assuming it is in format $ircodeFmt,
#  into a component code and a command code.  
#
#  If $irCode or $ircodeFmt is undefined or $ircode isn't valid for the
#  format $irFmt, return undef for $$componentCodeR and $$commandCodeR.
#-----------------------------------------------------------------------------
    if (defined($irCode) && defined($ircodeFmt)) {
        if ($ircodeFmt eq "SONY") {
            if (length($irCode) >= 4) {
                # Sony equipment - device is last 4 bits
                # Bryan's experiments show the code being 11 bits from the Sony
                #   VCR, 19 bits from the Sony DVD player.
                $$componentCodeR = substr($irCode, -4);
                $$commandCodeR = substr($irCode, 0, -4);
            }
        } else { 
            # See below for notes on how we came up with this heuristic.
            my $componentCodeLength;
            if (length($irCode) >= 14) {
                if (length($irCode) >= 48) {
                    $componentCodeLength = 32;
                } elsif (length($irCode) >= 32) {
                    $componentCodeLength = 16;
                } elsif (length($irCode) >= 16) {
                    $componentCodeLength = 5;
                } else {
                    $componentCodeLength = 4;
                }
                $$componentCodeR = substr($irCode, 0, $componentCodeLength);
                $$commandCodeR = substr($irCode, $componentCodeLength);
            }
        }
    }
}

# NOTES ON IR CODE DECOMPOSITION:
#
# 32 bits with 16 of component is very common.
# In a quick survey, Bryan also saw 48/39, 16/8, 24/7, 24/5, 17/9,
# 14/4, 24/5, 24/6, 22/5, and 48/32.
# So we just use the following heuristic.  If we guess too long on
# the component code, a single remote will look like
# multiple components.  If we guess too short, two remotes will 
# look like one.
#
# Remotes with codes < 16 bits did not give reliably interpretable
# messages in my test.  Maybe the sample rate was too low and these
# are really supposed to be longer codes.
#
# It looks like it may be possible to decompose a component code.
# On a Quasar 4-component remote, the first half of the 32 bit
# component code was the same for all four.
#
# My ancient 5-function Sharp VCR remote has a 14 bit IR code where
# only Bits 4-6 change.  But the heuristic here says the first 4
# are the component code.
#
# We may have to stop assuming that a component code uniquely
# identifies a component and add the code length and/or encoding
# as a distinguisher.



=head2 decodeIrMessage($pulseTrain, $irMessageR)

This subroutine deduces from a pulse train the code that the component uses,
then decodes the pulse train to extract the IR code, and from that the
component code and command code.

Here are our assumptions about the code:

  - An intermessage pause is more than 6000 microseconds long and no other
    off pulse in the code is that long.

  - The head code, if any, is one on and one off pulse.

  - The tail code is one on pulse.

  - For the encodings of a 1 bit and a 0 bit in the IR code:

    - Each is one on pulse followed by one off pulse.

    - The longer pulse train is the 1 bit.

    - Either the on pulse or the off pulse is the same duration for
      both 0 and 1.

The return value $$irMessageR is a hash reference.  The hash has the
following keys:

    A pulse train is described by a reference to an array.  The
    array has one element per pulse, in order.  The element is a
    number.  The number is positive to indicate an on pulse, negative
    for off.  Its magnitude is the duration of the pulse in
    microseconds.

   HEAD => The "head" pulse train

   TAIL => The "tail" pulse train

   ONE =>  The pulse train that represents a 1 in the IR code

   ZERO => The pulse train that represents a 0 in the IR code

   PAUSETIME =>
           The duration in microseconds of an intermessage pause. 
           Undefined if only one IR message in the pulse set.

   IRCODE_FMT =>
           "SONY" or "STANDARD", indicating which of the two IR code layouts
           is used in this IR message.

   CODE => The IR code, as a string of '0' and '1' characters.

   COMPONENT_CODE =>
           The component code, as a string of '0' and '1' characters

   COMMAND_CODE =>
           The command code, as a string of '0' and '1' characters


=cut

sub decodeIrMessage($$$) {

    my ($pulseTrain, $irMessageR, $errorR) = @_;
    
    my %irMessage;

    my @pulseTrainMsg1 = extractFirstMsg($pulseTrain);
    
    if (@pulseTrainMsg1 < 3) {
        $$errorR = "The first IR message in the pulse set contains " .
            "fewer than 3 pulses.";
    } else {    
        # TODO: analyze pulse train to see if there is really a head
        # code.
        my @headPulseTrain = @pulseTrainMsg1[0..1];
        my @tailPulseTrain = @pulseTrainMsg1[-1..-1];
        my @codePulseTrain = @pulseTrainMsg1[2..$#pulseTrainMsg1-1];

        if ($headPulseTrain[0] < 0 || $headPulseTrain[1] >=0) {
            $$errorR = "The head pulse train is not an on pulse followed " .
                "by an off pulse.";
        } elsif ($tailPulseTrain[0] < 0) {
            $$errorR = "The tail pulse is not an on pulse.";
        } else {
            measureCodePulses(\@codePulseTrain, \my $pulseProfile, $errorR);

            if (!$$errorR) {
                my $error;
                computeZeroOneEncoding($pulseProfile, 
                                       \my $zeroPulseTrain, \my $onePulseTrain,
                                       \my $differentiator, \my $threshold,
                                       \$error);
                if ($error) {
                    $$errorR = "Unable to figure out the zero/one encoding " .
                        "for the IR code segment.  $error";
                } else {
                    $irMessage{ZERO} = $zeroPulseTrain;
                    $irMessage{ONE}  = $onePulseTrain;
                    my $irCodeFmt;
                    if (defined($threshold)) {
                        $irCodeFmt = 
                            $differentiator == 0 ? "SONY" : "STANDARD";
                    }
                    $irMessage{IRCODE_FMT} = $irCodeFmt;

                    my $irCode = decodeIrCode(\@codePulseTrain, 
                                              $differentiator, $threshold);
                    decomposeIrCode($irCode, $irCodeFmt, 
                                    \my $compCode, \my $commandCode);
                    
                    $irMessage{CODE} = $irCode;
                    $irMessage{COMPONENT_CODE} = $compCode;
                    $irMessage{COMMAND_CODE} = $commandCode;
                }
            }
        }
        $irMessage{HEAD} = \@headPulseTrain;
        $irMessage{TAIL} = \@tailPulseTrain;
    }

    $irMessage{PAUSETIME} = computeImPauseLength($pulseTrain);
    
    $$irMessageR = \%irMessage;
}


# And now return a TRUE value to tell Perl that the 'use' went OK.
1;


=head1 AUTHOR

Bryan Henderson <bryanh@giraffe-data.com> San Jose CA 03.11.29

Contributed to the public domain by its author.
