#-----------------------------------------------------------------------------
#  This module contains various definitions useful in multiple other modules
#  in the Slinke package.
#-----------------------------------------------------------------------------

package Slinke::Foundation;

use strict;
use warnings;
use Exporter;
use vars qw( @ISA $VERSION @EXPORT @EXPORT_OK);
use English;

$VERSION = 1.00;
@ISA = qw(Exporter);

my ($TRUE, $FALSE) = (1,0); 

our $SLINKE_CLK      = 20.0e6;
our $PORT_IR_MAXML   = 15;
our $MAXDATABLOCK    = 30;
    # This is the maximum number of bytes that a single Send Port Message
    # command or Port Message Received report can contain.

push(@EXPORT, qw($SLINKE_CLK $PORT_IR_MAXML $MAXDATABLOCK));

our $IPCTYPE_SLINKE = 1; push(@EXPORT, '$IPCTYPE_SLINKE');

our $CONTROL_TYPE = 0x1F; push(@EXPORT, '$CONTROL_TYPE');
    # This is the value of the report type field of a Control report
    # and of the command type field of a Control command.
our $END_OF_PORT_MESSAGE_TYPE = 0x00; 
push(@EXPORT, '$END_OF_PORT_MESSAGE_TYPE');
    # Same as above, but for a Port Message End Received report or
    # Send Port Message End command

our %PORTNUMBER =
    ( 
      PORT_SL0 => 0, 
      PORT_SL1 => 1,
      PORT_SL2 => 2, 
      PORT_SL3 => 3,
      PORT_IR  => 4, 
      PORT_PAR => 5,
      PORT_SER => 6, 
      PORT_NON => 7,
      );
push(@EXPORT, '%PORTNUMBER');

our %PORTNAME = reverse %PORTNUMBER; push(@EXPORT, '%PORTNAME');

our %COMMAND_TYPE = 
    ( 
      CMD_DISABLE        => { ACTIONCODE => 0x02, 
                              DATALEN => 0,
                              EXPECTED_RESPONSE => 'RPT_DISABLE'
                              },  
      CMD_ENABLE         => { ACTIONCODE => 0x03, 
                              DATALEN => 0,
                              EXPECTED_RESPONSE => 'RPT_ENABLE'
                              },  
      CMD_SENDODDBITS    => { ACTIONCODE => 0x04, 
                              DATALEN => 2,
                              EXPECTED_RESPONSE => undef
                              },  
      CMD_SETIRSAMP      => { ACTIONCODE => 0x04, 
                              DATALEN => 2,
                              EXPECTED_RESPONSE => 'RPT_IRSAMPIS'
                              },  
      CMD_GETIRSAMP      => { ACTIONCODE => 0x05, 
                              DATALEN => 2,
                              EXPECTED_RESPONSE => 'RPT_IRSAMPIS'
                              },  
      CMD_SETIRCARRIER   => { ACTIONCODE => 0x06, 
                              DATALEN => 2,
                              EXPECTED_RESPONSE => 'RPT_IRCARRIERIS'
                              },  
      CMD_GETIRCARRIER   => { ACTIONCODE => 0x07, 
                              DATALEN => 2,
                              EXPECTED_RESPONSE => 'RPT_IRCARRIERIS'
                              },  
      CMD_SETIRTIMEOUT   => { ACTIONCODE => 0x0C, 
                              DATALEN => 2,
                              EXPECTED_RESPONSE => 'RPT_IRTIMEOUTIS'
                              },  
      CMD_GETIRTIMEOUT   => { ACTIONCODE => 0x0D, 
                              DATALEN => 2,
                              EXPECTED_RESPONSE => 'RPT_IRTIMEOUTIS'
                              },  
      CMD_SETIRMINLEN    => { ACTIONCODE => 0x0E, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_IRMINLENIS'
                              },  
      CMD_GETIRMINLEN    => { ACTIONCODE => 0x0F, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_IRMINLENIS'
                              },  
      CMD_SETIRTXZONES   => { ACTIONCODE => 0x08, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => undef
                              },  
      CMD_GETIRTXZONES   => { ACTIONCODE => 0x13, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_IRTXZONESIS'
                              },  
      CMD_SETIRRXZONES   => { ACTIONCODE => 0x09, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_IRRXZONESIS'
                              },  
      CMD_GETIRRXZONES   => { ACTIONCODE => 0x12, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_IRRXZONESIS'
                              },  
      CMD_SETIRROUTING   => { ACTIONCODE => 0x0A, 
                              DATALEN => 16,
                              EXPECTED_RESPONSE => 'RPT_IRROUTINGIS'
                              },  
      CMD_GETIRROUTING   => { ACTIONCODE => 0x10, 
                              DATALEN => 16,
                              EXPECTED_RESPONSE => 'RPT_IRROUTINGIS'
                              },  
      CMD_SETIRRXPOL     => { ACTIONCODE => 0x0B, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_IRRXPOLIS'
                              },  
      CMD_GETIRRXPOL     => { ACTIONCODE => 0x11, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_IRRXPOLIS'
                              },  
      CMD_SETBAUD        => { ACTIONCODE => 0x08, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_BAUDIS'
                              },  
      CMD_GETBAUD        => { ACTIONCODE => 0x09, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_BAUDIS'
                              },  
      CMD_SETHSMODE      => { ACTIONCODE => 0x10, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_HSMODEIS'
                              },  
      CMD_GETHSMODE      => { ACTIONCODE => 0x11, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_HSMODEIS'
                              },  
      CMD_SETDIR         => { ACTIONCODE => 0x12, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_DIRIS'
                              },  
      CMD_GETDIR         => { ACTIONCODE => 0x13, 
                              DATALEN => 1,
                              EXPECTED_RESPONSE => 'RPT_DIRIS'
                              },  
      CMD_SAMPLE         => { ACTIONCODE => 0x14, 
                              DATALEN => 0,
                              EXPECTED_RESPONSE => undef
                              },  
      CMD_GETVERSION     => { ACTIONCODE => 0x0B,
                              DATALEN => 0,
                              EXPECTED_RESPONSE => 'RPT_VERSIONIS'
                              },  
      CMD_GETSERIALNO    => { ACTIONCODE => 0x0C, 
                              DATALEN => 0,
                              EXPECTED_RESPONSE => 'RPT_SERIALNOIS'
                              },  
      # SETSERIALNO is not in the Slink-e documentation.
      CMD_SETSERIALNO    => { ACTIONCODE => 0x0D, 
                              DATALEN => 8,
                              EXPECTED_RESPONSE => 'RPT_SERIALNOIS'
                              },  
      CMD_SAVEDEFAULTS   => { ACTIONCODE => 0x0E, 
                              DATALEN => 0,
                              EXPECTED_RESPONSE => 'RPT_DEFAULTSSAXVED'
                              },  
      CMD_LOADDEFAULTS   => { ACTIONCODE => 0x0F, 
                              DATALEN => 0,
                              EXPECTED_RESPONSE => 'RPT_DEFAULTSLOADED'
                              },  
      CMD_RESUME         => { ACTIONCODE => 0xAA, 
                              DATALEN => 0,
                              EXPECTED_RESPONSE => undef
                              },  
      CMD_RESET          => { ACTIONCODE => 0xFF, 
                              DATALEN => 0,
                              EXPECTED_RESPONSE => undef
                              },  
      # The PLAYMACRO commands are not in the Slink-e documenation.
      # I don't know what they are -- apparently they were in Colby's
      # aboriginal code.
      #CMD_PLAYMACRO1     => { ACTIONCODE => 0x10, 
      #                        EXPECTED_RESPONSE => undef
      #                        },  
      #CMD_PLAYMACRO2     => { ACTIONCODE => 0x11, 
      #                        EXPECTED_RESPONSE => undef
      #                        },  
      #CMD_STOREMACRO1    => { ACTIONCODE => 0x12, 
      #                        EXPECTED_RESPONSE => undef
      #                        },  
      #CMD_STOREMACRO2    => { ACTIONCODE => 0x13, 
      #                        EXPECTED_RESPONSE => undef
      #                        },  
      );
push(@EXPORT, '%COMMAND_TYPE');

our %COMMANDTYPE_INDEX = 
(
 0x02 => 'CMD_DISABLE',
 0x03 => 'CMD_ENABLE',
 0x04 => { PORT_IR  => 'CMD_SETIRSAMP',
           PORT_SL0 => 'CMD_SENDODDBITS',
           PORT_SL1 => 'CMD_SENDODDBITS',
           PORT_SL2 => 'CMD_SENDODDBITS',
           PORT_SL3 => 'CMD_SENDODDBITS',
       },
 0x05 => 'CMD_GETIRSAMP',
 0x06 => 'CMD_SETIRCARRIER',
 0x07 => 'CMD_GETIRCARRIER',
 0x08 => { PORT_IR  => 'CMD_SETIRTXZONES',
           PORT_SER => 'CMD_SETBAUD',
       },
 0x09 => { PORT_IR  => 'CMD_SETIRRXZONES',
           PORT_SER => 'CMD_GETBAUD',
       },
 0x0A => 'CMD_SETIRROUTING',
 0x0B => { PORT_IR  => 'CMD_SETIRRXPOL',
           PORT_NON => 'CMD_GETVERSION',
       },
 0x0C => { PORT_IR  => 'CMD_SETIRTIMEOUT',
           PORT_NON => 'CMD_GETSERIALNO',
       },
 0x0D => { PORT_IR  => 'CMD_GETIRTIMEOUT',
           PORT_NON => 'CMD_SETSERIALNO',
       },
 0x0E => { PORT_IR  => 'CMD_SETIRMINLEN',
           PORT_NON => 'CMD_SAVEDEFAULTS',
       },
 0x0F => { PORT_IR  => 'CMD_GETIRMINLEN',
           PORT_NON => 'CMD_LOADDEFAULTS',
       },
 0x10 => { PORT_IR  => 'CMD_GETIRROUTING',
           PORT_PAR => 'CMD_SETHSMODE',
       },
 0x11 => { PORT_IR  => 'CMD_GETIRRXPOL',
           PORT_PAR => 'CMD_GETHSMODE',
       },
 0x12 => { PORT_IR  => 'CMD_GETIRRXZONES',
           PORT_PAR => 'CMD_SETDIR',
       },
 0x13 => { PORT_IR  => 'CMD_GETIRTXZONES',
           PORT_PAR => 'CMD_GETDIR',
       },
 0x14 => 'CMD_SAMPLE',
 0xAA => 'CMD_RESUME',
 0xFF => 'CMD_RESET',
 );
push(@EXPORT, '%COMMANDTYPE_INDEX');


push(@EXPORT, 'commandTypeFromActionCode');
sub commandTypeFromActionCode($$) {
    my ($actionCode, $port) = @_;

    my $commandType;

    my $index1 = $COMMANDTYPE_INDEX{$actionCode};

    if ($index1) {
        if (ref($index1)) {
            $commandType = $index1->{$port};
        } else {
            $commandType = $index1;
        }
    }
    return $commandType;
}



sub validatect($$) {
    my ($commandType, $actionCode) = @_;
    
    if (!exists($COMMAND_TYPE{$commandType})) {
        die("Non-existent command type '$commandType' " .
            "in COMMANDTYPE_INDEX");
    } elsif ($COMMAND_TYPE{$commandType}->{ACTIONCODE} != $actionCode) {
        die("COMMAND_TYPE and COMMANDTYPE_INDEX disagree on command " .
            "type $commandType");
    }
}



# Validate %COMMANDTYPE_INDEX
while (my ($actionCode, $indexValue) = each(%COMMANDTYPE_INDEX)) {
    if (ref($indexValue)) {
        foreach my $commandType (values(%{$indexValue})) {
            validatect($commandType, $actionCode);
        }
    } else {
        validatect($indexValue, $actionCode);
    }
}



our %REPORT_TYPE = ( 
  'RPT_INVALID_BAUD'          => { ACTIONCODE => 0x84, 
                                   RESPONSE => $TRUE , DATALEN => 0 }, 
  'RPT_ILLEGAL_CMD'           => { ACTIONCODE => 0xFF, 
                                   RESPONSE => $TRUE , DATALEN => 0 },
  'RPT_DEFAULTSLOADED'        => { ACTIONCODE => 0x0F, 
                                   RESPONSE => $TRUE , DATALEN => 0 },
  'RPT_DEFAULTSSAVED'         => { ACTIONCODE => 0x0E, 
                                   RESPONSE => $TRUE , DATALEN => 0 },
  'RPT_PORTDISABLED'          => { ACTIONCODE => 0x02, 
                                   RESPONSE => $TRUE , DATALEN => 0 },
  'RPT_PORTENABLED'           => { ACTIONCODE => 0x03, 
                                   RESPONSE => $TRUE , DATALEN => 0 },
  'RPT_BAUDIS'                => { ACTIONCODE => 0x08, 
                                   RESPONSE => $TRUE , DATALEN => 1 },
  'RPT_DIRIS'                 => { ACTIONCODE => 0x12, 
                                   RESPONSE => $TRUE , DATALEN => 1 },
  'RPT_HSMODEIS'              => { ACTIONCODE => 0x10, 
                                   RESPONSE => $TRUE , DATALEN => 1 },
  'RPT_IRCARRIERIS'           => { ACTIONCODE => 0x06, 
                                   RESPONSE => $TRUE , DATALEN => 2 },
  'RPT_IRSAMPIS'              => { ACTIONCODE => 0x04, 
                                   RESPONSE => $TRUE , DATALEN => 2 },
  'RPT_IRMINLENIS'            => { ACTIONCODE => 0x0E, 
                                   RESPONSE => $TRUE , DATALEN => 1 },
  'RPT_IRROUTINGIS'           => { ACTIONCODE => 0x0A, 
                                   RESPONSE => $TRUE , DATALEN => 16 },
  'RPT_IRRXZONESIS'           => { ACTIONCODE => 0x09, 
                                   RESPONSE => $TRUE , DATALEN => 1 },
  'RPT_IRRXPOLIS'             => { ACTIONCODE => 0x0B, 
                                   RESPONSE => $TRUE , DATALEN => 1 },
  'RPT_IRTIMEOUTIS'           => { ACTIONCODE => 0x0C, 
                                   RESPONSE => $TRUE , DATALEN => 2 },
  'RPT_IRTXZONESIS'           => { ACTIONCODE => 0x08, 
                                   RESPONSE => $TRUE , DATALEN => 1 },
  'RPT_LASTRXZONEIS'          => { ACTIONCODE => 0x01, 
                                   RESPONSE => $FALSE, DATALEN => 1 },
  'RPT_SERIALNOIS'            => { ACTIONCODE => 0x0C, 
                                   RESPONSE => $TRUE , DATALEN => 8 },
  'RPT_VERSIONIS'             => { ACTIONCODE => 0x0B, 
                                   RESPONSE => $TRUE , DATALEN => 1 },
  'RPT_INVALID_SAMP'          => { ACTIONCODE => 0x82, 
                                   RESPONSE => $TRUE , DATALEN => 0 },
  'RPT_CMDDATA_UNDERRUN'      => { ACTIONCODE => 0x81, 
                                   RESPONSE => $FALSE, DATALEN => 0 }, 
      # This is known as "Transmit Timeout" in Nirvis documentation.
  'RPT_ODDBITS'               => { ACTIONCODE => 0x04, 
                                   RESPONSE => $FALSE, DATALEN => 0 },
  'RPT_RX_ERROR'              => { ACTIONCODE => 0x80, 
                                   RESPONSE => $FALSE, DATALEN => 0 },
  'RPT_SEEPROMWRERR'          => { ACTIONCODE => 0x8F, 
                                   RESPONSE => $TRUE , DATALEN => 0 },  
  'RPT_RS232_FRAME_ERROR'     => { ACTIONCODE => 0x85, 
                                   RESPONSE => $FALSE, DATALEN => 0 },
  'RPT_CMDDATA_OVERRUN'       => { ACTIONCODE => 0x83, 
                                   RESPONSE => $FALSE, DATALEN => 0 },
  'RPT_CMD_BYTE_MISSED'       => { ACTIONCODE => 0x86, 
                                   RESPONSE => $FALSE, DATALEN => 0 },
                             );

push(@EXPORT, '%REPORT_TYPE');

our %REPORTTYPE_INDEX = 
(
 0x01 => 'RPT_LASTRXZONEIS',
 0x02 => 'RPT_PORTDISABLED',
 0x03 => 'RPT_PORTENABLED',
 0x04 => { PORT_IR  => 'RPT_IRSAMPIS',
           PORT_SL0 => 'RPT_ODDBITS',
           PORT_SL1 => 'RPT_ODDBITS',
           PORT_SL2 => 'RPT_ODDBITS',
           PORT_SL3 => 'RPT_ODDBITS',
       },
 0x06 => 'RPT_IRCARRIERIS',
 0x08 => { PORT_SER => 'RPT_BAUDIS',
           PORT_IR  => 'RPT_IRTXZONESIS',
       },
 0x09 => 'RPT_IRRXZONESIS',
 0x0A => 'RPT_IRROUTINGIS',
 0x0B => { PORT_IR  => 'RPT_IRRXPOLIS',
           PORT_NON => 'RPT_VERSIONIS',
       },
 0x0C => { PORT_IR  => 'RPT_IRTIMEOUTIS',
           PORT_NON => 'RPT_SERIALNOIS',
       },
 0x0E => { PORT_IR  => 'RPT_IRMINLENIS',
           PORT_NON => 'RPT_DEFAULTSSAVED',
       },
 0x0F => 'RPT_DEFAULTSLOADED',
 0x10 => 'RPT_HSMODEIS',
 0x12 => 'RPT_DIRIS',
 0x80 => 'RPT_RX_ERROR',
 0x81 => 'RPT_CMDDATA_UNDERRUN',
 0x82 => 'RPT_INVALID_SAMP',
 0x83 => 'RPT_CMDDATA_OVERRUN',
 0x84 => 'RPT_INVALID_BAUD',
 0x85 => 'RPT_RS232_FRAME_ERROR',
 0x86 => 'RPT_CMD_BYTE_MISSED',
 0x8F => 'RPT_SEEPROMWRERR',
 0xFF => 'RPT_ILLEGAL_CMD',
 );
push(@EXPORT, '%REPORTTYPE_INDEX');


push(@EXPORT, 'reportTypeFromActionCode');
sub reportTypeFromActionCode($$) {
    my ($actionCode, $port) = @_;

    my $reportType;

    my $index1 = $REPORTTYPE_INDEX{$actionCode};

    if ($index1) {
        if (ref($index1)) {
            $reportType = $index1->{$port};
        } else {
            $reportType = $index1;
        }
    }
    return $reportType;
}



sub validatert($$) {
    my ($reportType, $actionCode) = @_;
    
    if (!exists($REPORT_TYPE{$reportType})) {
        die("Non-existent report type '$reportType' " .
            "in REPORTTYPE_INDEX");
    } elsif ($REPORT_TYPE{$reportType}->{ACTIONCODE} != $actionCode) {
        die("REPORT_TYPE and REPORTTYPE_INDEX disagree on report " .
            "type $reportType");
    }
}



# Validate %REPORTTYPE_INDEX
while (my ($actionCode, $indexValue) = each(%REPORTTYPE_INDEX)) {
    if (ref($indexValue)) {
        foreach my $reportType (values(%{$indexValue})) {
            validatert($reportType, $actionCode);
        }
    } else {
        validatert($indexValue, $actionCode);
    }
}



our %ERRORS = 
# Error responses.
# The keys of this hash are action codes.  Every report action code that
# an error report can have is represented.  A report that has one of these
# action codes is necessarily an error report.
#
# Each value is a hash reference with the following keys:
#
#   DESCRIPTION => text description of the meaning of a report with this
#                  action code.
#
#   CRITICAL    => logical: the report describes a Slink-e critical error.
#                  A Slink-e critical error is one that puts the Slink-e into
#                  the need-resume state.
#   ASYNC       => logical: the action code is one that _could_ describe the
#                  failure of a commmand that does not have a response.
#                  For example, a Send Port Message command does not have a
#                  response, but can fail due to a data underrun, which would
#                  generate a RPT_MSGDATA_UNDERRUN message.  So
#                  RPT_MSGDATA_UNDERRUN has ASYNC=TRUE.  But note that not
#                  all RPT_MSGDATA_UNDERRUN reports report failures of commands
#                  that have no response.  One could see such a report as the
#                  response to a Set IR Sample Period command, for example.
    ( 
      RPT_INVALID_BAUD        => {
          DESCRIPTION => 'Invalid Baud Rate',
          CRITICAL    => $TRUE,
          ASYNC       => $FALSE,
      },
      RPT_ILLEGAL_CMD         => {
          DESCRIPTION => 'Illegal Command', 
          CRITICAL    => $TRUE,
          ASYNC       => $FALSE,
      },
      RPT_INVALID_SAMP        => {
          DESCRIPTION => 'Invalid Sample Period', 
          CRITICAL    => $FALSE,
          ASYNC       => $FALSE,
      },
      RPT_CMDDATA_UNDERRUN    => {
          DESCRIPTION => 'Port message data underrun (DTE did not send ' .
              'port message data to the Slink-e as fast as the port needed ' .
              'it to maintain sending the port message)',
          CRITICAL    => $TRUE, 
          ASYNC       => $TRUE,
      },
      RPT_SEEPROMWRERR        => {
          DESCRIPTION => 'SEEPROM Write Error', 
          CRITICAL    => $FALSE,
          ASYNC       => $FALSE,
      },
      RPT_RS232_FRAME_ERROR => {
          DESCRIPTION => 'Receive framing error', 
          CRITICAL    => $TRUE,
          ASYNC       => $TRUE,
      },
      RPT_CMDDATA_OVERRUN   => {
          DESCRIPTION => 'Receive overrun', 
          CRITICAL    => $TRUE,
          ASYNC       => $TRUE,
      },
      RPT_CMD_BYTE_MISSED   => {
          DESCRIPTION => 'Command byte missed; Slink-e busy', 
          CRITICAL    => $TRUE,
          ASYNC       => $TRUE,
      },
      # You might thing RPT_RX_ERROR would be an error, but it just an
      # unsolicited message telling you about a problem on the S-link bus,
      # not anything wrong with a Slink-e operation.
      );
push(@EXPORT, '%ERRORS');



# Validate %ERRORS

foreach (keys(%ERRORS)) {
    if (!$REPORT_TYPE{$_}) {
        die("Report type '$_' in \%ERRORS is not in \%REPORT_TYPE");
    }
}

our %BAUDRATEARG = (
                     2400 => 0,
                     4800 => 1,
                     9600 => 2,
                    19200 => 3,
                    38400 => 4,
                    );

push(@EXPORT, '%BAUDRATEARG');


sub max($$) {
    return $_[0] > $_[1] ? $_[0] : $_[1];
}

sub min($$) {
    return $_[0] < $_[1] ? $_[0] : $_[1];
}

push(@EXPORT_OK, 'max', 'min');

1;
