package Slinke::Control;

=head1 NAME

Slinke - module to control the Slink-e product from by Nirvis.
See <http://giraffe-data.com/slinke> or <http://www.nirvis.com> for 
information on Slink-e.

=head1 SYNOPSIS

=head1 DESCRIPTION

Slink-e is a product that can speak to many different Sony products over 
the S-Link port.  Also, it can receive and transmit infrared signals over
transmitters and receivers in 8 separate zones via the IR port and send
and receive via an ISA parallel port too.

The Slink-e has an RS-232 (serial) interface and a bit-level protocol you
use on that interface to send and receive messages on the ports and control
the Slink-e in general.

This module turns that bit-level RS-232 interface into a perl-friendly
interface you can use easily to control the Slink-e with a Perl program.


Bryan Henderson adapted most of this code from a similar module named
Slinke.pm (available via CPAN) by Brian Paulsen.  Brian's code uses
techniques he learned from C++ code from Colby Boles.  The Perl code
you are reading does not contain anything copied Colby's work.  The
comments are all original.

=head1 METHODS

=cut

use strict;
use warnings;
use Exporter;
use Carp;
use vars qw(@ISA $VERSION @EXPORT);
use Errno qw(:POSIX);
use English;
use POSIX qw(SIGTERM floor setsid);
use IPC::Msg;
use IPC::SysV qw(IPC_PRIVATE IPC_NOWAIT MSG_NOERROR S_IRWXU);
use Time::HiRes qw(setitimer ITIMER_REAL);
use Slinke::Receive;
use Slinke::Foundation qw(:DEFAULT max min);
use Slinke::Serial;


$VERSION = 1.00;
@ISA = qw(Exporter);

my ($TRUE, $FALSE) = (1,0); 


my $debugLevel;

sub debug {
    my ($level, $message) = @_;
    
    if ($debugLevel >= $level) {
        print("Slinke::Control: $message\n");
    }
}



sub receiver($) {
    my ($slinke) = @_;

    if (!defined($slinke->{RECEIVEMSGQ})) {
        carp("RECEIVEMSGQ not defined for Slinke object");
    }
    if (!defined($slinke->{RESPONSEMSGQ})) {
        carp("RESPONSEMSGQ not defined for Slinke object");
    }

    # We become our own session, so that we don't have a controlling
    # terminal, so ctl-C doesn't affect us.
    setsid();
    
    require Slinke::Receiver;
    
    Slinke::Receiver::receiver
        (
         SERIALPORTFH => $slinke->{SERIALPORTFH},
         RECEIVEMSGQ  => $slinke->{RECEIVEMSGQ},
         RESPONSEMSGQ => $slinke->{RESPONSEMSGQ},
         MULTIZONE    => ($slinke->{VERSION} >= 2.0),
         DEBUG        => $debugLevel,
         );
}



sub createReceiverMsgQ($$$) {
    my ($slinke, $receiveQKey, $errorR) = @_;

    # We set up two SysV IPC message queues for communication between the
    # receiver process and the outside world:
    #
    # - The receive message queue is where the receiver delivers unsolicited
    #   reports about the outside world such as Port Message Received.
    #
    # - The response message queue is where the receiver delivers command
    #   responses such as the IR Sample Period Is report that the Slink-e
    #   sends in response to a Set IR Sample Period command.

    my $keyarg = defined($receiveQKey) ? $receiveQKey : IPC_PRIVATE;

    my $receiveMsgQ = IPC::Msg->new($keyarg, S_IRWXU);
    if (!defined($receiveMsgQ)) {
        $$errorR = "Unable to access receive IPC message queue";
    } else {
        my $responseMsgQ = IPC::Msg->new(IPC_PRIVATE, S_IRWXU);
        if (!defined($responseMsgQ)) {
            $$errorR = "Unable to create response IPC message queue.";
        } else {
            $slinke->{RESPONSEMSGQ} = $responseMsgQ;
            $slinke->{RECEIVEMSGQ} = $receiveMsgQ;
            if ($keyarg == IPC_PRIVATE) {
                $slinke->{PRIVATE_RECEIVEMSGQ} = $TRUE;
            }
        }
        if ($$errorR) {
            if ($keyarg == IPC_PRIVATE) {
                $receiveMsgQ->remove();
            }
        }
    }
}



sub createReceiver($$) {
    my ($slinke, $errorR) = @_;

    my $forkRc = fork();
    if (defined($forkRc)) {
        if ($forkRc == 0) {
            # We are the child -- the receiver process

            # A hack until we figure out this forking thing:
            $slinke->{FORKED_COPY} = $TRUE;

            receiver($slinke);

            exit(0);
        } else {
            # We are the parent
            $slinke->{PARENT} = $TRUE;
            $slinke->{RECEIVER_PID} = $forkRc;
            debug(2, "Receiver process created.  " .
                  "Pid = $slinke->{RECEIVER_PID}");
            
        }
    } else {
        $$errorR = "Unable to create the receiver process.  " .
            "fork() failed.  errno=$ERRNO";
    }
}



sub destroyReceiver($$) {

    my ($slinke, $errorR) = @_;

    if (defined($slinke->{RECEIVER_PID})) {
        my $pid = $slinke->{RECEIVER_PID};
        my $success = kill(SIGTERM, $pid);

        if (!$success) {
            if ($ERRNO eq ESRCH) {
                warn("Receiver process already dead when we went to kill it.");
            } else {
                $$errorR = "Unable to kill receiver process.  Errno=$ERRNO";
            }
        } else {
            my $rc = waitpid($pid, 0);
            if ($rc == -1) {
                warn("waitpid() after killing receiver process failed.");
            }
        }
    }
}



sub doResume($$);



sub drainResponse($) {
#-----------------------------------------------------------------------------
#  Receive any bytes the Slink-e wants to send us, or are already in 
#  serial port buffers, so we know the next bytes received through this 
#  object are a response to the next command sent through it.
#
#  Of course, if the Slink-e receives a port message after we return,
#  that will break that assumption.
#-----------------------------------------------------------------------------
    my ($slinke) = @_;

    debug(4, "Draining previously sent response data");

    my $empty;

    $empty = $FALSE;
    
    while (!$empty) {
        my $receivedChar = 
            Slinke::Serial::timedRead($slinke->{SERIALPORTFH}, 1, 0.060);
        if ($receivedChar eq '') {
            $empty = $TRUE;
        } else {
            my $receivedValue = ord($receivedChar);
            debug(5, "Received character " . 
                  sprintf("0x%02x", $receivedValue));
        }    
    }
}



sub writeToPort {
    my ($slinke, @bytes) = @_;
#-----------------------------------------------------------------------------
#  Write the bytes @bytes to the serial port attached to the Slink-e.
#
#  Each element of @bytes is a number representing one byte.  The byte is
#  the pure binary representation of that number.  E.g. an ASCII "A" is 65.
#-----------------------------------------------------------------------------
    my $chars;
    foreach my $byte (@bytes) {
        if ($byte =~ /^\d*$/) {
            $chars .= chr($byte);
        }
    }
    debug(5, "Writing " . scalar(@bytes) . " bytes to serial port file: " .
          "0x" . unpack("H*", $chars));

    my $bytesWritten = syswrite($slinke->{SERIALPORTFH}, $chars);

    return $bytesWritten;
}



sub sendGetVersion($$) {
#-----------------------------------------------------------------------------
#  Send the two-byte Get Version command to the Slink-e.  Don't wait
#  for the response.
#
#  In case you're wondering why we have this special subroutine just for
#  this command that talks directly the serial port instead of using the
#  generic Slink-e command subroutines:  It's so you can use this to
#  get communications established in the first place.  It works before
#  everything is initialized, and gives very specific error information
#  so the user can figure out what's wrong.
#-----------------------------------------------------------------------------
    my ($slinke, $errorR) = @_;

    debug(3,"Sending special Get Version command");
    # Send a Command Byte 0 indicating a control message for no port.
    my $portAndType = ($PORTNUMBER{PORT_NON} << 5 ) + $CONTROL_TYPE;
        
    my $bytesWritten = syswrite($slinke->{SERIALPORTFH}, chr($portAndType));

    if (!$bytesWritten) {
        $$errorR = "Unable to transmit Get Version command byte 0 " .
                   "to device.  syswrite() failed.";
    } else {
        debug(5, "Sent byte " . sprintf("0x%02x", $portAndType) . 
              " to device");

        # Send Command Byte 1 indicating Get Version 
        my $actionCode = $COMMAND_TYPE{CMD_GETVERSION}->{ACTIONCODE};
        
        my $bytesWritten = syswrite($slinke->{SERIALPORTFH}, chr($actionCode));
        if (!$bytesWritten) {
            $$errorR = "Unable to transmit Get Version command byte 1 " .
                       "to device.  syswrite() failed.";
        } else {
            debug(5, "Sent byte " . sprintf("0x%02x", $actionCode) . 
                  " to device");
        }
    }
}



sub receiveGetVersionResponse($$$) {
#-----------------------------------------------------------------------------
#  Receive what we expect to be a Get Version command response from the
#  Slink-e.  Receive directly from the Slink-e, not via the receiver 
#  process.  Use this with sendGetVersion() above.
#-----------------------------------------------------------------------------
    my ($slinke, $versionR, $errorR) = @_;

    debug(4, "Attempting to receive Get Version response");

    my $getVersResp = $REPORT_TYPE{RPT_VERSIONIS}->{ACTIONCODE};
    
    my $receivedChars = 
        Slinke::Serial::timedRead($slinke->{SERIALPORTFH}, 3, 0.060);

    debug(5, "Received: 0x" . unpack("H*", $receivedChars));

    if (length($receivedChars) < 3) {
        $$errorR = 
            "Not enough bytes received in response to Get Version " .
            "command.  Expected 3 byte Get Version response, but in " .
            "60 milliseconds, got " .
            (length($receivedChars) == 0 ? 
             "none" : 
             "only these: 0x" . unpack("H*", $receivedChars)
             );
    } elsif (substr($receivedChars, 0, 1) ne chr(0xff)) {
        $$errorR = 
            "First character of Get Version response is " .
            "0x" . unpack("H*", substr($receivedChars, 0, 1)) .
            " instead of 0xff";
    } elsif (substr($receivedChars, 1, 1) ne chr($getVersResp)) {
        $$errorR = 
            "Second character of Get Version response is " . 
            "0x" . unpack("H*", substr($receivedChars, 1, 1)) .
            " instead of 0x" . unpack("H*", chr($getVersResp));
    } else {
        # We got a perfect Get Version response.
        my $versionByte = ord(substr($receivedChars, 2,1));
        if (defined($versionR)) {
            $$versionR = ($versionByte >> 4) . "." . ($versionByte & 0xF);
            debug(3, "Got version response '$$versionR'");
        }
    }
}



sub executeGetVersion($$$) {
    my ($slinke, $versionR, $errorR) = @_;
#-----------------------------------------------------------------------------
#  Execute a Get Version command at the Slink-e (send command, wait for 
#  response).
#
#  We do this without higher level functions of this module and with
#  detailed error reporting so Caller can use this 1) as part of
#  object initialization (Some initialization depends on what the
#  Slink-e version is) and 2) to verify that Slink-e commmunication is
#  working before moving on to use higher level subroutines such as
#  txrx() to talk to the Slink-e.
#
#  Note that this subroutine will not work if there is a receiver process
#  running, because it will suck up the response to our command.
#
#  Get Version tells the Slink-e to respond with its product version 
#  number.  It is a very simple command for the Slink-e to execute.
#-----------------------------------------------------------------------------
    sendGetVersion($slinke, \my $error);
    if ($error) {
        $$errorR = "Unable to send Get Version command to Slink-e.  $error";
    } else {
        receiveGetVersionResponse($slinke, $versionR, \my $error);
        if ($error) {
            $$errorR = "Sent Get Version command to Slink-e, but " .
                "did not receive a proper response.  $error";
        }
    }
}


sub tryToMakeContact($$) {
#-----------------------------------------------------------------------------
#  Try to conduct a trivial transaction with the Slink-e using current
#  DTE serial port settings.
#
#  If it works, return $$errorR undefined.  Otherwise, return a description
#  of how it failed as $$errorR.
#-----------------------------------------------------------------------------
    my ($slinke, $errorR) = @_;

    resetComm($slinke, $errorR);
    drainResponse($slinke);
    if (!$$errorR) {
        executeGetVersion($slinke, undef, \my $error);
        if ($error) {
            $$errorR = "Unable to complete a Get Version command.  $error";
        } else {
            # It worked; we're all synced up!
        }
    }
}



sub establishCommunication($$) {

    my ($slinke, $errorR) = @_;

    my $error;

    my $origBaudrate = Slinke::Serial::getBaud($slinke->{SERIALPORTFH});

    my $communicationEstablished;

    if (defined($origBaudrate)) {
        tryToMakeContact($slinke, \$error);

        if ($error) {
            debug(2, "Unable to make contact at original baud rate.  $error");
            $communicationEstablished = $FALSE;
        } else {
            $communicationEstablished = $TRUE;
        }
    }

    for (my $i=4; $i>=0 && !$communicationEstablished; --$i) {
        my $trialBaudRate = 2400 * (1 << $i);
        
        debug(4, "Trying baud rate $trialBaudRate");
        Slinke::Serial::setBaud($slinke->{SERIALPORTFH}, $trialBaudRate);
        
        my $trialError;
        tryToMakeContact($slinke, \$trialError);
        if ($trialError) {
            debug(2, "Unable to make contact at trial baud rate.  " .
                  "$trialError");
        }
        $communicationEstablished = !$trialError;
    } 
    if (!$communicationEstablished) {
        $$errorR = "Could not communicate with Slink-e with current DTE " .
                   "settings.  " .
                   "Attempts at various other baud rates also failed.";

        if (defined($origBaudrate)) {
            setDteBaud($slinke->{SERIALPORTFH}, $origBaudrate);
        }
    }
}



sub setupSlinke($$) {
    my ($slinke, $errorR) = @_;

    Slinke::Serial::initSerialPort($slinke->{SERIALPORTFH});
    debug(2, "Serial port initialized.");
    establishCommunication($slinke, \my $error);
    if ($error) {
        $$errorR = "Unable to establish communication with Slink-e " .
            "in spite of successful access to the DTE (near-end) " .
            "serial port.  $error";
    } else {
        executeGetVersion($slinke, \$slinke->{VERSION}, \my $error);
        if ($error) {
            $$errorR = "Unable to get version info from Slink-e.  $error";
        } else {
            createReceiver($slinke, \my $error);
            if ($error) {
                $$errorR = "Failed to create receiver.  $error";
            }
        }
    }
}



sub drainMsgQueue($$) {

    my ($msgQ, $numberDrainedR) = @_;

    my $queueEmpty;
    my $numberDrained;
    $numberDrained = 0;
    $queueEmpty = $FALSE;
    while (!$queueEmpty) {
        my $msgType = $msgQ->rcv(my $msg, 30, 0, IPC_NOWAIT | MSG_NOERROR);
        if (!defined($msgType)) {
            $queueEmpty = $TRUE;
        } else {
            ++$numberDrained;
            if ($numberDrained % 100 == 0) {
                warn("$numberDrained messages drained from receive queue " .
                     "so far.  Probably something wrong.");
            }
            debug(1, "drained msg '$msg'");
        }
    }
    $$numberDrainedR = $numberDrained;
}



sub signalHandler {
    my ($signal) = @_;
    debug(1, "Signal $signal received by Slinke::Control signal handler.");
    die("Signal $signal received");
}



sub rcvMsgQTimeout($$$$$$) {
    my ($msgQ, $maxSize, $timeout, $msgR, $msgTypeR, $errorR) = @_;
#-----------------------------------------------------------------------------
#  This is the IPC:Msg::rcv() function, except with a timeout and a more
#  modern interface.
#-----------------------------------------------------------------------------

    my $errorRet;

    eval {
        local $SIG{"ALRM"} = \&signalHandler;

        # The following causes a SIGALRM signal after $timeout seconds.
        # Our SIGALRM handler does a die(), which means this eval block
        # exits immediately with $@ set tellingly.
        setitimer(ITIMER_REAL, $timeout);

        my $msgType = $msgQ->rcv(my $msg, $maxSize);

        # Cancel the SIGALRM
        setitimer(ITIMER_REAL, 0);
    
        if (!defined($msgType)) {
            $errorRet = "IPC::Msg::rcv failed.  Errno=$ERRNO";
        } else {
            debug(4, "Received '$msgType' type msg " .
                  "from response msg queue: '$msg'");
            $$msgR     = $msg;
            $$msgTypeR = $msgType;
        }
    };
    if ($@) {
        # The eval block terminated abnormally.
        if ($@ =~ m{^Signal ALRM received}) {
            # It was terminated by our signal handler; That means the receive
            # timed out.
            $errorRet = "Receive timed out on command response message queue";
        } else {
            # We were supposed to die; our eval accidentally saved us.
            die("Slinke::Control::rcvMsgQTimeout died.  $@");
        }
    }
    $$errorR = $errorRet;
}

        

sub receiveResponse($%) {
    my ($slinke, %args) = @_;
#-----------------------------------------------------------------------------
#  Receive one report from the response message queue.
#
#  The response message queue is where the receiver process sends a
#  report that it receives from the Slink-e when it determines that it
#  is a response to a command.
#
#  If no response is presently in the queue, wait for one.  But if one 
#  doesn't come within the timeout period, return failure.  The timeout 
#  period is $args{TIMEOUT} seconds, or 1 second if unspecified.
#
#  A note about the timeout: When you send a command to the Slink-e,
#  it takes longer to receive a response than just the time to execute
#  that command and transmission time.  Previous Slink-e commands
#  which don't have responses, particularly Send Port Message
#  commands, may be queued up between here and the Slink-e, and you
#  have to wait for them too.  Not only that, you have to wait for
#  the receiver process to wake up and receive the response and put
#  it into the response SYSV message queue, and then for this process
#  to wake up and read the message out.
#
#  If there are additional responses in the pipeline, fail.
#
#  Return the response as ${$args{RESPONSE}}.
#-----------------------------------------------------------------------------
    my $msgQ = $slinke->{RESPONSEMSGQ};
    my $timeout = $args{TIMEOUT} || 1.0;  # timeout in seconds

    my $errorRet;
    
    debug(3, "Getting next response from response SYSV IPC queue...");

    rcvMsgQTimeout($msgQ, 256, $timeout, \my $msg, \my $msgType, \$errorRet);

    if ($args{RESPONSE}) {
        $ {$args{RESPONSE}} = $msg;
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub parseReportMsg($$$) {
    my ($reportString, $reportR, $errorR) = @_;
#-----------------------------------------------------------------------------
#  Parse a report from the message queue, which is a string like
#  "PORT_IR RPT_BAUDIS 04", and return the individual pieces of information
#  it gives as a hash at $$reportR:
#
#    PORT       => e.g. "PORT_IR"
#    REPORTTYPE => e.g. "RPT_BAUDIS"
#    DATA       => 04  (may be null string, but never undefined)
#-----------------------------------------------------------------------------
    if ($reportString =~ m{ ^ (PORT_\S+) \s (RPT_\S+) (\s (.+))? \s* $ }x) {
        # It's a valid report string

        my ($port, $reportType, $data) = ($1, $2, $4);

        $reportR->{PORT} = $port;
        $reportR->{REPORTTYPE} = $reportType;
        $reportR->{DATA} = $data || "";

        # TODO: $reportR->{DATA} should be an array reference with one
        # array element per byte of report data, equal to the
        # number that that byte encodes in pure binary.
    } else {
        $$errorR = "Does not have the format PORT_xxx RPT_xxx HHHHHH...";
    }
}



sub interpretResponse($$$$$) {
    my ($responseR, $expectedResponse, $sentPort, 
        $responseDataR, $errorR) = @_;

    if (exists($ERRORS{$responseR->{REPORTTYPE}})) {
        if ($responseR->{PORT} ne $sentPort) {
            if ($ERRORS{$responseR->{REPORTTYPE}}->{ASYNC}) {
                $$errorR = "$responseR->{REPORTTYPE} error report received " .
                    "for prior command to port $responseR->{PORT}";
            } else {
                $$errorR = "$responseR->{REPORTTYPE} report received for " .
                "wrong port ($responseR->{PORT}).  Command was sent to " .
                "Port $sentPort";
            }
        } else {
            $$errorR = "Slink-e reports that command failed.  " .
                $ERRORS{$responseR->{REPORTTYPE}}->{DESCRIPTION};
            if ($ERRORS{$responseR->{REPORTTYPE}}->{ASYNC}) {
                $errorR .= "Could be from a prior command.  ";
            }
        }
    } else {
        # This is a normal command response -- not an error report.

        if ($responseR->{PORT} ne $sentPort) {
            $$errorR = "$responseR->{REPORTTYPE} report received for " .
                "wrong port ($responseR->{PORT}).  " .
                "Command was sent to Port $sentPort";
        } elsif ($expectedResponse ne $responseR->{REPORTTYPE}){
            $$errorR = "Received $responseR->{REPORTTYPE} report " .
                "instead of the expected $expectedResponse report";
        } else {
            $$responseDataR = $responseR->{DATA};
        }
    }
}



sub getResponse($$$$$$) {
    my ($slinke, $cmd, $sentPort, $timeout, $responseDataR, $errorR) = @_;
#-----------------------------------------------------------------------------
#  Wait for and receive from the response SYSV IPC message queue the
#  response for a $cmd command (which Caller presumably sent before calling
#  us) to port $sentPort.
#
#  Return the response data from that response as $$responseDataR, in
#  hexadecimal.
#
#  If the command type $cmd does not have a response, return success with
#  $$responseDataR undefined.
#
#  If the command type does have a response, but we don't receive one within
#  a reasonable amount of time, fail.  If the response we get is of a type
#  that is not possible for this command, fail.
#-----------------------------------------------------------------------------
    my $expectedResponse = $cmd->{EXPECTED_RESPONSE};
    
    if (defined $expectedResponse) {
        receiveResponse($slinke, 
                        TIMEOUT  => $timeout, 
                        RESPONSE => \my $responseString,
                        ERROR    => $errorR,
                        );

        if (!defined($$errorR)) {
            debug(2, "Got response '$responseString'");
            
            parseReportMsg($responseString, \my %response, \my $error);
            if ($error) {
                $$errorR = "INTERNAL ERROR.  Syntactically invalid " .
                    "response string '$responseString'.  $error";
            } else {
                interpretResponse(\%response, $expectedResponse, $sentPort, 
                                  $responseDataR, $errorR);
            }
        }
    }
}



sub txrxit($$$$$$) {
    my ($slinke, $cmd, $port, $args, $responseDataR, $errorR) = @_;
#-----------------------------------------------------------------------------
#  Caller must ensure that the the response message queue is empty, so that
#  we know that the next response is the response to the command we send
#  (if the command in fact has a response).
#-----------------------------------------------------------------------------
    my $portAndType = ($PORTNUMBER{$port} << 5) | $CONTROL_TYPE;
    my $action = $cmd->{ACTIONCODE};
    
    my $success = writeToPort($slinke, 
                              $portAndType, $action, 
                              @{$args});
    if (!$success) {
        $$errorR = "write to serial port failed.";
    } else {
        # Note that the timeout here should be based on how big the command
        # is (because transmission time is the primary factor in how long
        # a command takes) and how many other commands (that don't have
        # responses) might be before it in the queue.
        
        my $timeout = 1;  # 1 second
    
        getResponse($slinke, $cmd, $port, $timeout, $responseDataR, 
                    \my $error);
        
        if ($error) {
            my $actionHex = unpack("H*", chr($action));
            $$errorR = "Sent 0x$actionHex command OK, but didn't get valid " .
                "response.  $error";
        }
    }
}
            


sub doResume($$) {
    my ($slinke, $errorR) = @_;

    debug(1, "Issuing a Resume command because " .
          "Slink-e is in need-resume state.");
    
    txrxit($slinke, $COMMAND_TYPE{'CMD_RESUME'}, "PORT_NON", [], 
           \my $response, \my $error);
    if ($error) {
        $$errorR = "Failed to execute Resume command.  $error";
        warn($$errorR);
    } else {
        $$errorR = undef;
    }
}



sub checkForErrors($$$) {
    my ($slinke, $continue, $errorR) = @_;
#-----------------------------------------------------------------------------
#  Drain any messages from the receive queue in preparation for sending a
#  new command.
#
#  $continue means the new command is a continuation of a sequence, so
#  errors from previous activity should be recognized as errors on the new
#  command.  If $continue is true and there is anything in the receive queue,
#  we return an error description as $$errorR.  Otherwise, we return $$errorR
#  undef.
#-----------------------------------------------------------------------------
    my $msgQ = $slinke->{RESPONSEMSGQ};

    my $responseQueueEmpty;

    $responseQueueEmpty = $FALSE;  # initial value

    debug(4, "Checking response message queue for prior errors");

    my $errorRet;

    # There could be multiple errors.  The last message in the queue
    # takes precedence, except that if a Resume fails, that takes precedence
    # over everything.

    while (!$responseQueueEmpty) {
        my $msgType = $msgQ->rcv(my $msg, 256, 0, IPC_NOWAIT | MSG_NOERROR);

        if (defined($msgType)) {
            debug(4, "Got prior report report '$msg'");
            parseReportMsg($msg, \my %report, \my $error);

            if (!exists($ERRORS{$report{REPORTTYPE}})) {
                $errorRet = "Non-error report $report{REPORTTYPE} received " .
                    "from response SYSV IPC message queue when no command " .
                    "outstanding.";
            } else {
                if (!$ERRORS{$report{REPORTTYPE}}->{ASYNC}) {
                    $errorRet = "Non-async error report $report{REPORTTYPE} " .
                        "received from response SYSV IPC message queue when " .
                        "no command outstanding.";
                } else {
                    if ($continue) {
                        $errorRet = "Previous related command failed.  " .
                            $ERRORS{$report{REPORTTYPE}}->{DESCRIPTION};
                    }
                }
            }
        } else {
            $responseQueueEmpty = $TRUE;
        }
    }
    $$errorR = $errorRet;
}



sub txrx($%) {
    my ($slinke, %args) = @_;
#-----------------------------------------------------------------------------
#  Send one command to the Slink-e.  If the command is supposed to have a
#  response, wait for, receive, and return that.
#
#  This is ONE command -- not multiple commands as required to send a 
#  single port message.
#
#  %args:
#     COMMAND  => Slink-e command (e.g. "CMD_SETIRSAMP")
#     PORT     => Slink-e port (e.g. "PORT_IR")
#     ARGS     => reference to array of whole numbers defining the Slink-e 
#                 command arguments.  Each byte of the argument is one
#                 element of the array, in pure binary.
#     CHAINED  => Logical: true means this command is related to a previous
#                 one, and therefore if there is delayed error status from
#                 a previous command, this method should fail without
#                 even trying the command.  False means this is a new 
#                 operation, so if previous error status needs clearing,
#                 do it before executing the command.
#     RESPONSE => reference to a variable that this method sets to the
#                 value of the Slink-e's response to the command.  If 
#                 the command has no response, it is undef.  Otherwise, it
#                 is a hexadecimal string, e.g. "12FF".
#     ERROR    => reference to error description variable (e.g. \my $error)
#-----------------------------------------------------------------------------
    my $errorRet;

    my $command = $args{COMMAND};
    if (!defined($command)) {
        $errorRet = "You must specify the COMMAND argument";
    } else {
        my $cmd = $COMMAND_TYPE{$command};

        if (!defined($cmd)) {
            $errorRet = "Unknown command '$command'";
        } else {
            if (!defined($args{PORT})) {
                $errorRet = "You must specify the PORT argument";
            } else {
                checkForErrors($slinke, $args{CHAINED}, \$errorRet);
                if (!$errorRet) {
                    if (!$args{CHAINED}) {
                        # Send a resume that is probably not necessary, just
                        # for robustness.
                        txrxit($slinke, $COMMAND_TYPE{'CMD_RESUME'}, 
                               "PORT_NON", [], 
                               \my $resumeResp, \my $resumeError);
                    }
                    debug(2, "Sending $command command to Slink-e");
                    txrxit($slinke, $cmd, $args{PORT}, $args{ARGS},
                           \my $responseData, \$errorRet);
                    
                    debug(3, "$command command completed. " .
                          "Response = " . ($responseData || "undef"));
                    if ($args{RESPONSE}) {
                        $ {$args{RESPONSE}} = $responseData;
                    }
                }
            }
        }
    }
    my $errorR = $args{ERROR};
    if (defined($errorR)) {
        $$errorR = $errorRet;
    }
    if ($errorRet) {
        debug(1, "txrx() failed to execute command.  $errorRet");
    }
}



=head2 $slinke->receiveFromSlinke(%ARGS);

This function receives and returns a port message or unsolicited control
report from the Slink-e.  When it receives and returns a port message,
it is an entire port message, not just a single report from the Slink-e.

This method first goes through the Slinke::Control object's queue of 
unsolicited reports it received previously from the Slink-e, then reads
from the Slink-e itself.  If that isn't enough to get a complete port
message or unsolicited control report, the method waits for more reports
from the Slink-e.

Arguments:

    C<$ARGS{MESSAGE}> is a reference to the variable in which
    this function returns a hash reference with these elements:

        C<TYPE> identifies the type of message.  "CONTROL" means an
        unsolicited control message, and "PORTMSG" means a port message.

        C<PORT> contains the name of the port that generated the
        message.  The port names are: PORT_IR, PORT_PAR, PORT_SLO,
        PORT_SL1, PORT_SL2, PORT_SL3, PORT_SER, PORT_NON.

        For a "PORTMSG" type message:

            C<CONTENTS> is a hexadecimal representation of the port
            message data (e.g. "08F240AA").

            For an IR message (from port "PORT_IR"):

                C<ZONE> tells which IR zone's receiver (0-8) received the
                data.  Note that you must have a Slink-e of version 2.0 or
                higher for ZONE to be greater than 0

        For a "CONTROL" type message:

            C<REPORTTYPE> is the Slink-e report type as a string, e.g.
            "RPT_CMDDATA_UNDERRUN".  This always starts with "RPT_".

            C<DATA> is the value of the databytes the Slink-e sent with the 
            message, in hexadecimal.  E.g. "01FF".

    C<$ARGS{ERROR}> is a reference to the variable in which this function
    returns a description of the error if it fails.  If it doesn't fail,
    this function does not set the variable.


=cut

sub receiveFromSlinke(%) {

    Slinke::Receive::receiveFromSlinke(@_);
}



sub irZoneList() {
=head2 $slinke->irZoneList()

This returns a list of the IR zones that exist in this Slink-e device.
Each element in the list is a zone number.

Older Slink-es have one IR zone: Zone 0.  Newer ones have 8: 0-7.

=cut


    my ($slinke) = @_;

    my @zoneList;

    if (defined($slinke->{VERSION})) {
        if ($slinke->{VERSION} < 2.0) {
            @zoneList = (0);
        } else {
            @zoneList = (0, 1, 2, 3, 4, 5, 6, 7);
        }
    }
    return @zoneList;
}



sub resetComm($$) {
#-----------------------------------------------------------------------------
#  This subroutine sends 3 bytes to the Slink-e, in very primitive fashion,
#  in order to get it into a known state while assuming as little as possible
#  about the current state, and without concern about wiping out whatever
#  the device might be in the middle of.
#
#  This should not cause Slink-e to generate any response.
#-----------------------------------------------------------------------------
    my ($slinke, $errorR) = @_;

    # Send an End Of Data command to PORT_NON just in case the Slink-e
    # is in the middle of some message right now.  Note that a zero in
    # what is otherwise the length field means End Of Data.
    my $cmd = ($PORTNUMBER{PORT_NON} << 5) + 0;
    
    debug(3,"Sending End Of Data");
    my $bytesWritten = syswrite($slinke->{SERIALPORTFH}, chr($cmd));
    if (!$bytesWritten) {
        $$errorR = "Unable to transmit End Of Data byte.  " .
                   "syswrite() failed.";
    } else {
        debug(5, "Sent byte " . sprintf("0x%02x", $cmd) . " to device");

        debug(3,"Sending Resume command");
        # Send Command Byte 0 indicating a control command for no port.
        my $cmd0 = ($PORTNUMBER{PORT_NON} << 5) + $CONTROL_TYPE;
        
        my $bytesWritten = syswrite($slinke->{SERIALPORTFH}, chr($cmd0));
        if (!$bytesWritten) {
            $$errorR = "Unable to transmit Resume command byte 0 " .
                       "to device.  syswrite() failed.";
        } else {
            debug(5, "Sent byte " . sprintf("0x%02x", $cmd0) . 
                  " to device");

            # Send Command Byte 1 indicating Resume (throw away prior error
            # state and resume normal communication).
            my $cmd1 = $COMMAND_TYPE{CMD_RESUME}->{ACTIONCODE};
            
            my $bytesWritten = syswrite($slinke->{SERIALPORTFH}, chr($cmd1));
            if (!$bytesWritten) {
                $$errorR = "Unable to transmit Resume command byte 1 " .
                           "to device.  syswrite() failed.";
            } else {
                debug(5, "Sent byte " . sprintf("0x%02x", $cmd1) . 
                      " to device");
            }
        }
    }
}



=head2 $slinke->getReceiveQ(%)

This method returns an IPC::SysV::Msg object to access the SYSV IPC
message queue to which the Slink-e object sends all port messages and
unsolicited control reports that the Slink-e receives.

Here is an example of receiving a port message:

    $slinke->getReceiveQ{\my $msgQ};
    $msgQ->rcv(my $message, 256);
    print($message);

    This would print something like this, after you press a button on a 
    remote control (assuming there are no other previously queued Slink-e
    reports):

    PORTMSG PORT_IR FF40F0 3
 
    Here, "FF40F0" is the (unrealistically short) port message data and 3
    is the IR zone in which the Slink-e received the port message.


=cut

sub getReceiveQ(%) {
    my ($slinke, %args) = @_;

    if ($args{MSGQ}) {
        $ {$args{MSGQ}} = $slinke->{RECEIVEMSGQ};
    }
}




=head2 $slinke->syncComm($errorR)

This method sets the DTE serial line parameters to match the
Slink-e's and confirms that communication with the Slink-e is working.
When you first approach a Slink-e, you must do this because if the DTE
(the serial port in the computer) and the DCE (the serial port in the
Slink-e) are not speaking the same language (baud rate, etc.) or
there's something wrong with the line, nothing else will work and it
will fail in strange ways.

After you have established communication, you can use the setBaud()
method to change the baud rates on both ends simultaneously.

This method works by trial and error.  It sends a simple command and
receives the response.  It does this at various line settings until it
gets a sensible response, in which case it knows it has the right
settings.

Argument is a hash.  The elements are:

  C<ERROR> => reference to a variable which this method sets to a
              text description of why it failed, or undef if it didn't.


=cut

sub syncComm {
    my ($slinke, %args) = @_;

    debug(2, "syncComm() called.");

    my $errorRet;
    
    destroyReceiver($slinke, \my $error);
    if ($error) {
        $errorRet = "Unable to destroy the receiver.  We cannot attempt " .
            "to synchronize communication while the receiver process " .
            "is sucking up Slink-e's responses.";
    } else {
        establishCommunication($slinke, \$errorRet);
        
        createReceiver($slinke, \my $error);
        if ($error && !$errorRet) {
            $errorRet = "Unable to create a receiver after successfully " .
                "establishing communication with Slink-e.  $error";
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->getVersion(%ARGS)

This returns the version of the Slink-e.

Argument is a hash.  Elements are:

    VERSION => A reference to the variable in which this function 
               returns the version number.  It is in decimal, e.g "3.0".

    ERROR   => A reference to an error string variable.

=cut

sub getVersion(%) {
    my ($slinke, %args) = @_;

    my $errorRet;

    debug(1, "getVersion() called");

    $slinke->txrx(COMMAND  => "CMD_GETVERSION",
                  PORT     => "PORT_NON",
                  RESPONSE => \my $response,
                  ERROR    => \my $error,
                  );

    if ($error) {
        $errorRet = 
            "Unable to execute Slink-e CMD_GETVERSION command.  $error";
    } else {
        my $version;

        if (defined $response) {
            my $numericalResponse = hex($response);
            my $majorVersion = $numericalResponse >> 4;
            my $minorVersion = $numericalResponse & 0xF;
            $version = "$majorVersion.$minorVersion";
            debug(1, "getVersion() returning '$version'");
            if ($args{VERSION}) {
                $ {$args{VERSION}} = $version;
            }        
        } else {
            $errorRet = "Slink-e CMD_GETVERSION command completed with " .
                "no response.";
        }
    }
}



=head2 $slinke->getSerialNumber()

This gets the 8 byte serial number of the Slink-e.

Argument is a hash.  Elements are:

    SERIALNO => A reference to the variable in which this function 
                returns the serial number.  It returns it in hexadecimal
                (16 characters).

    ERROR    => A reference to an error string variable.

=cut

sub getSerialNumber(%) {
    my ($slinke, %args) = @_;

    my $errorRet;

    debug(1, "getSerialNumber() called");

    $slinke->txrx(COMMAND  => "CMD_GETSERIALNO",
                  PORT     => "PORT_NON",
                  RESPONSE => \my $data,
                  ERROR    => \my $error,
                  );

    if ($error) {
        $errorRet = 
            "Unable to execute Slink-e CMD_GETSERIALNO command.  $error";
    } else {
        debug(1, "getSerialNumber() returning '$data'");

        if ($args{SERIALNO}) {
            $ {$args{SERIALNO}} = $data;
        }        
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->enablePort($port)

Enables reception on specified port.  

If port == C<PORT_NON>, enables ports globally.

=cut

sub enablePort {
    my ($slinke, $port) = @_;

    if ($port eq "PORT_SER") {
        warn("enable/disable is not valid on serial port");
    } else {
        $slinke->txrx(COMMAND => "CMD_ENABLE",
                      PORT    => $port,
                      ERROR   => \my $error,
                      );
        
        if ($error) {
            warn("Error trying to enable $port.  $error"); 
        }
    }
}



=head2 $slinke->disablePort($port)

Disables reception on specified port.

If port == C<PORT_NON>, disables ports globally.

=cut

sub disablePort {
    my ($slinke, $port) = @_;

    if ($port eq "PORT_SER") {
        warn("enable/disable is not valid on serial port");
    } else {
        $slinke->txrx(COMMAND => "CMD_DISABLE",
                      PORT    => $port,
                      ERROR   => \my $error,
                      );
        
        if ($error) {
            warn("Error trying to disable $port.  $error");
        }
    }
}



sub getBaudFromSlinke($$$) {
    my ($slinke, $baudrateR, $errorR) = @_;

    debug(3, "current DTE baud rate is " . 
          Slinke::Serial::getBaud($slinke->{SERIALPORTFH}));

    $slinke->txrx(COMMAND  => "CMD_GETBAUD",
                  PORT     => "PORT_SER",
                  RESPONSE => \my $response,
                  ERROR    => \my $error,
                  );
    if ($error) {
        $$errorR = "Unable to execute Slink-e CMD_GETBAUD command.  $error";
    } else {
        $$errorR = undef;
        my $baudrateNumber = hex($response);
        $$baudrateR = 2400 * (1 << $baudrateNumber);
    }
}



=head2 $slinke->getBaud(%)

This returns the baud rate in bps of the DCE serial port in the Slink-e and
the DTE serial port through which we talk to the Slink-e.

This isn't terribly useful, because you'd have already to know the baud
rate of the Slink-e in order to send the command and get the response.
But it exists for mathematical completeness.

argument is a hash.  Elements are:

  BAUD =>   A reference to a variable that this method sets to the baud
            rate, in bits per second.
  ERROR =>  A reference to an error string variable.

Use syncComm() to sync up baud rates in the first place.

=cut

sub getBaud(%) {
    my ($slinke, %args) = @_;

    debug(2, "getBaud() called.");
    
    my $errorRet;

    getBaudFromSlinke($slinke, \my $baudrate, \$errorRet);

    if ($args{BAUD}) {
        $ {$args{BAUD}} = $baudrate;
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->setBaud(%)

This sets the baud rate in bps of the DCE serial port in the Slink-e and
the DTE serial port through which we talk to the Slink-e.

In order for this function to succeed, the DTE and DCE baud rates must
match upon entry.  We conduct the transaction to set the new baud rate
at the current baud rate.  The Slink-e changes its baud rate after the
transaction is done and we set the DTE baud rate then too.  Thus, future
communications will happen at the new baud rate.

argument is a hash.  Elements are:

  BAUD =>   The baud rate, in bits per second.
            The only valid ones (per Slink-e) are 2400, 4800, 9600, 19200,
            and 38400.
  ERROR =>  A reference to an error string variable.

=cut

sub setBaud {
    my ($slinke, %args) = @_;

    debug(2, "setBaud() called.");
    
    my $errorRet;

    if (!defined($args{BAUD})) {
        $errorRet = "You must specify the BAUD argument";
    } else {
        my $baudRateArg = $BAUDRATEARG{$args{BAUD}};

        if (!defined($baudRateArg)) {
            $errorRet = "$args{BAUD} is not a baud rate " .
                "of which the Slink-e is capable";
        } else {
            $slinke->txrx(COMMAND  => "CMD_SETBAUD",
                          PORT     => "PORT_SER",
                          ARGS     => [$baudRateArg],
                          RESPONSE => \my $response,
                          ERROR    => \my $error,
                          );
            if ($error) {
                $errorRet = 
                    "Unable to execute Slink-e CMD_SETBAUD command.  $error";
            } else {
                my $slinkeBaudNumber = hex($response);
                my $baud = 2400 * (1 << $slinkeBaudNumber);
                setDteBaud($slinke->{SERIALPORTFH}, $baud);
                if ($baud != $args{BAUD}) {
                    $errorRet = "Set Baud command executed fine, but its " .
                        "return value says the baud rate was set to $baud " .
                        "instead of $args{BAUD}!";
                }
            }
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->baudrate()

This returns the baud rate in bps of the DTE serial port.

This method does not involve the Slink-e itself.

Use syncComm() to sync up baud rates in the first place.

=cut

sub baudrate() {
    my ($slinke) = @_;

    debug(2, "baudrate() called.");
    
    return Slinke::Serial::getBaud($slinke->{SERIALPORTFH});
}



=head2 $slinke = Slinke::Control->createObject(...);

Slinke::Control->new(SLINKE => \my $slinke,
                     DEVICE => $device)

Creates a C<Slinke::Control> object.

Argument is a hash.  The elements are:

  C<SLINKE> => reference to a variable which this method sets to the newly
               created Control::Slinke object.  Optional (but it would be
               strange not to specify it).

  C<DEVICE> => name of the serial device that the Slink-e is connected to, e.g.
               /dev/ttyS0

  C<SERIALPORTFH> => 
               a file handle representing the serial port to
               which the Slink-e is connected.

  You must specify either DEVICE or SERIALPORTFH, but not both.

  C<ERROR> =>  a reference to a variable that this method sets to a text
               description of why it cannot create an object or undef if
               it does create it.  Optional.

  C<DEBUG> =>  debug level for debug messages.  The higher the number, the
               more detail debug messages have.  0 means no debug messages.
               Default is 0.

=cut



sub createObject {
    my ($class, %args) = @_;

    my $errorRet;
        # Description of why we can't create the object.  Undefined if
        # we haven't given up yet.

    my $slinke;

    $slinke = {};

    $debugLevel = $args{DEBUG} || 0;
    bless ($slinke, $class);

    if (defined($args{SERIALPORTFH})) {
        $slinke->{SERIALPORTFH} = $args{SERIALPORTFH};
    } else {
        if (!defined($args{DEVICE})) {
            $errorRet = "You must specify either SERIALPORTFH or DEVICE";
        } else {
            my $success = open($slinke->{SERIALPORTFH}, '+<', $args{DEVICE});
            if (!$success) {
                $errorRet = "Unable to open the serial device file " .
                            "'$args{DEVICE}'.  errno = $ERRNO";
            }
        }
    }
    if (!$errorRet) {
        createReceiverMsgQ($slinke, $args{RECEIVEMSGQKEY}, \my $error);
        if ($error) {
            $errorRet = "Unable to create SYSV IPC message queues.  $error";
        } else {
            setupSlinke($slinke, \$errorRet);
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
    if ($args{SLINKE}) {
        $ {$args{SLINKE}} = $slinke;
    }
}



sub DESTROY {
# This is, by virtue of its name, the destructor for a Slinke object.
# The Perl interpreter calls it when the last reference to the object goes
# away.

    my ($slinke) = @_;

    if ($slinke->{FORKED_COPY}) {
        debug(4, "Destructor for forked copy of Slinke object " .
              "doing nothing.");
    } else {
        debug(1, "Slinke object being destroyed.");
        
        if (defined($slinke->{RECEIVEMSGQ})) {
            if ($slinke->{PRIVATE_RECEIVEMSGQ}) {
                $slinke->{RECEIVEMSGQ}->remove;
            }
            $slinke->{RECEIVEMSGQ} = undef;
        }
        if (defined($slinke->{RESPONSEMSGQ})) {
            $slinke->{RESPONSEMSGQ}->remove;
            $slinke->{RESPONSEMSGQ} = undef;
        }
        if (defined($slinke->{UNSOLERRMSGQ})) {
            $slinke->{UNSOLERRMSGQ}->remove;
            $slinke->{UNSOLERRMSGQ} = undef;
        }

        destroyReceiver($slinke, \my $error);

        if (defined($slinke->{SERIALPORTFH})) {
            debug(5, "Closing serial port.");
            close($slinke->{SERIALPORTFH});
            $slinke->{SERIALPORTFH} = undef;
        }
    }
}



sub getIRSamplingPeriodFromSlinke($$$) {
    my ($slinke, $samplePeriodR, $errorR) = @_;

    $slinke->txrx(COMMAND  => "CMD_GETIRSAMP",
                  PORT     => "PORT_IR",
                  RESPONSE => \my $response,
                  ERROR    => \my $error,
                  );

    if (defined($error)) {
        $$errorR = "Unable to execute Slink-e CMD_GETIRSAMP command.  $error";
    } else {
        if (length($response) != 4) {
            $$errorR = "Slink-e CMD_GETIRSAMP command returned " .
                "garbage: '$response'";
        } else {
            my $count = hex($response);
            $$samplePeriodR = $count/($SLINKE_CLK / 4.0);
        }
    }
}



=head2 $slinke->getIRSamplingPeriod()

This returns the infrared sampling period of the Slink-e.  Values can
range from 50 microseconds to 1 millisecond.

The argument is a hash.  Elements are:

  PERIOD => A reference to a variable that this method sets to the 
            sampling period, in seconds.
  ERROR  => A reference to an error string variable.

The IR sampling period determines the maximum timing resolution which
can be achieved when decoding IR signals.  In general, the sampling
period should be at least 3 times shorter than the shortest pulse you
wish to detect.  Short sampling periods are necessary when acquiring
timing information about previously unknown IR message formats, but
are not necessarily needed to output a known IR message format since
the sampling period need only be the least common multiple of the
pulse widths in the message.

The IR sampling period is also used as a timebase for parallel port
output signals.

=cut

sub getIRSamplingPeriod(%) {
    my ($slinke, %args) = @_;

    my $errorRet;

    getIRSamplingPeriodFromSlinke($slinke, \my $samplePeriod, \$errorRet);

    if ($args{PERIOD}) {
        $ {$args{PERIOD}} = $samplePeriod;
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub setIrfs($$$) {
    my ($slinke, $samplePeriod, $errorR) = @_;

    my $count = floor($SLINKE_CLK/4*$samplePeriod + 0.5);
        
    my $d1 = $count >> 8;
    my $d2 = $count & 0xff;
    
    $slinke->txrx(COMMAND  => "CMD_SETIRSAMP",
                  PORT     => "PORT_IR",
                  ARGS     => [$d1, $d2],
                  RESPONSE => \my $response,
                  ERROR    => \my $error,
                  );
    
    if ($error) {
        $$errorR = "Failed to execute Slink-e command.  $error";
    } else {
        if (length($response) != 4) {
            $errorR = "Slink-e returned garbage response to " .
                "CMD_SETIRSAMP command: '$response'";
        } else {
            my $attemptedPeriod = $count / ($SLINKE_CLK / 4.0);
            my $retPeriod = hex($response)/ ($SLINKE_CLK / 4.0);
            
            if ($retPeriod != $attemptedPeriod) {
                $$errorR = "Tried setting samplerate " .
                    "of $attemptedPeriod, but Slink-e says it set " .
                    "$retPeriod";
            }
        }
    }
}



=head2 $slinke->setIRSamplingPeriod(%ARGS)

This sets the infrared sampling period of the Slink-e.  Values can
range from 50 microseconds to 1 millisecond in 1/5 microsecond
steps.

The argument is a hash.  The elements are:

  C<PERIOD> =>   The sampling period to set, in seconds.  The sampling
                 period gets set to the nearest valid sampling period
                 (only certain discrete values are possible)
                 except that if this argument is too small or too large,
                 the method just fails.

The IR sampling period determines the maximum timing resolution which
can be achieved when decoding IR signals.  In general, the sampling
period should be at least 3 times shorter than the shortest pulse you
wish to detect.  Short sampling periods are necessary when acquiring
timing information about previously unknown IR message formats, but
are not necessarily needed to output a known IR message format since
the sampling period need only be the least common multiple of the
pulse widths in the message.

The IR sampling period is also used as a timebase for parallel port
output signals.

=cut

sub setIRSamplingPeriod(%) {
    my ($slinke, %args) = @_;

    debug(1, "setIRSamplingPeriod() called.");

    my $samplePeriod = $args{PERIOD};

    my $errorRet;

    if (!defined($slinke)) {
        $errorRet = "method not called on an object";
    } elsif (!defined($samplePeriod)) {
        $errorRet = "you must specify the PERIOD argument";
    } else {
        my $maxper = 1e-3;

        if ($samplePeriod > $maxper) {
            $errorRet = "$samplePeriod is too long a sampling period " .
                "($maxper is the longest)";
        } else {
            setIrfs($slinke, $samplePeriod, \$errorRet);
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->getIRCarrier(%ARGS)

This gets the current IR carrier frequency of the Slink-e.

The argument is a hash.  The elements are:

  CARRIERFREQ => A reference to a variable that this method sets to the
                 carrier frequency, in hertz.
  ERROR       =>  A reference to an error string variable.


=cut

sub getIRCarrier {
    my ($slinke, %args) = @_;

    my $errorRet;

    $slinke->txrx(COMMAND  => "CMD_GETIRCARRIER",
                  PORT     => "PORT_IR",
                  RESPONSE => \my $response,
                  ERROR    => \my $error,
                  );

    if ($error) {
        $errorRet = "Unable to execute Slink-e " .
            "CMD_GETIRCARRIER command.  $error";
    } else {
        my ($hexd1, $hexd2) = $response =~ /(..)(..)/;

        my $d1 = hex($hexd1);
        my $d2 = hex($hexd2);

        my $carrierFreq;
        if ($d1 == 0 && $d2 == 0) {
            $carrierFreq = 0;
        } else {
            $carrierFreq = ($SLINKE_CLK / 4)/((1 << $d1)*($d2+1));
        }
        if ($args{CARRIERFREQ}) {
            $ {$args{CARRIERFREQ}} = $carrierFreq;
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->setIRCarrier(%ARGS)

This sets the IR carrier frequency of the Slink-e.  Note that because
of the way that the frequency gets set, it will be very unlikely that
you will be able to set the exact frequency that you want.  However,
the Slink-e should be able to handle your frequency within several
hundred hertz.

The argument is a hash.  Elements are:

  REQCARRIERFREQ => The frequency you want to set.

  CARRIERFREQ    => A reference to a variable that this method sets to the 
                    actual carrier frequency that got set.

  ERROR          => A reference to an error string variable.

=cut

sub setIRCarrier {
    my ($slinke, %args) = @_;

    my $errorRet;

    debug(1, "setIRCarrier() called.");

    if (!defined($args{REQCARRIERFREQ})) {
        carp("You must supply a REQCARRIERFREQ argument");
        return;
    }
    my $frequency = $args{REQCARRIERFREQ};

    if ($frequency != 0.0) {
        my $count = $SLINKE_CLK/4.0/$frequency;
        if ($count == 0) {
            my $max = $SLINKE_CLK/4.0;
            $errorRet = "$frequency is too high of a carrier frequency " .
                "($max is the max)";
        } elsif ($count > 8*256) {
            my $min = $SLINKE_CLK/4.0/8.0/256.0;
            $errorRet = "$frequency is too low of a carrier frequency " .
                "($min is the minimum)";
        } else {
            my $d1;
            if ($count < 256) {
                $d1 = 0;
            } else {
                $d1 = POSIX::ceil(log(($count >> 8))/log(2.0));
            }
        
            my $d2 = int($count / (1 << $d1)) - 1;

            $slinke->txrx(COMMAND  => "CMD_SETIRCARRIER",
                          PORT     => "PORT_IR",
                          ARGS     => [$d1, $d2],
                          RESPONSE => \my $response,
                          ERROR    => \my $error,
                  );

            if ($error) {
                $errorRet = "Unable to execute Slink-e CMD_SETIRCARRIER " .
                    "command.  $error";
            } else {
                my ($hext1, $hext2) = $response =~ /(..)(..)/;
                my $t1 = hex($hext1);
                my $t2 = hex($hext2);
                
                my $frequencySet;
                if ($t1 == 0 && $t2 == 0) {
                    $frequencySet = 0;
                } else {
                    $frequencySet = ($SLINKE_CLK / 4)/((1 << $t1)*($t2+1));
                }
                
                if ($d1 != $t1 || $d2 != $t2) {
                    $errorRet = 
                        "Tried setting frequency of $frequency, but " .
                        "$frequencySet got set";
                } else {
                    if ($args{CARRIERFREQ}) {
                        $ {$args{CARRIERFREQ}} = $frequencySet;
                    }
                }
            }
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->getIRTimeoutPeriod(%ARGS)

This returns the IR timeout period of the Slink-e as measured in sample
periods.  The timeout period defines how long the IR receiver module must
be inactive for the Slink-e to consider a message to be completed.

%ARG is defined as follows:

  PERIOD => Reference to a scalar variable to which this method sets
            the timeout period, in seconds.

  ERROR  => A reference to an error string variable.

Control the timeout period with $slinke->setIRTimeoutPeriod().

=cut

sub getIRTimeoutPeriod {
    my ($slinke, %args) = @_;

    my $errorRet;

    $slinke->txrx(COMMAND  => "CMD_GETIRTIMEOUT",
                  PORT     => "PORT_IR",
                  RESPONSE => \my $response,
                  ERROR    => \my $error,
                  );

    if ($error) {
        $errorRet = "Unable to execute slinke GETIRTIMEOUT command.  $error";
    } else {
        my ($hexd1, $hexd2) = $response =~ m{(..)(..)};

        my $periods = hex($hexd1)*256 + hex($hexd2);
        
        if (defined($args{PERIOD})) {
            $ {$args{PERIOD}} = $periods;
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->setIRTimeoutPeriod(%)

This sets the IR timeout period of the Slink-e as measured in sample
periods.  The timeout period defines how long an off pulse the Slink-e
considers to delimit a pulse set.  I.e. while the Slink-e is receiving 
a pulse set, after it has seen an off pulse this long, it calls the 
pulse set complete and reports it on the programming interface.  The
next on pulse the Slink-e sees is part of the next pulse set.

Typically, you would set this longer than any off pulse that would be
part of an IR message, but shorter than the shortest intermessage
pause time.  That way, each pulse set the Slink-e reports is a single
IR message.  

But note that most remote controls send a stream of IR messages while
you hold down a button.  If you set the IR timeout period as described
above, you will see a stream of reports.  If you set it longer, you
can see the whole stream in one report.  Setting the timeout extra
long is also the only way to find out what the intermessage pauses in
a stream like this look like.

Argument is a hash.  Elements are:

  PERIOD => The IR timeout period, as a number of sample periods.

  ERROR =>  Reference to a variable in which this method returns a 
            description of the error it encountered, or undef if it did
            not encounter an error

There is no return value.

=cut

sub setIRTimeoutPeriod {
    my ($slinke, %args) = @_;
    
    debug(1, "setIRTimeoutPeriod() called.");

    my $errorRet;

    if (!defined($args{PERIOD})) {
        $errorRet = "You must specify the PERIOD argument";
    } else {
        if ($args{PERIOD} !~ m{\d+}) {
            $errorRet = "period must be a positive integer";
        } elsif ($args{PERIOD} == 0) {
            $errorRet = "period must be at least one sample period";
        } elsif ($args{PERIOD} > 65536) {
            $errorRet = "65536 sample periods is the maximum allowable " .
                "timeout period.  You specified $args{PERIOD}";
        } else {
            my $d1 = $args{PERIOD} >> 8;
            my $d2 = $args{PERIOD} & 0xff;
    
            $slinke->txrx(COMMAND  => "CMD_SETIRTIMEOUT",
                          PORT     => "PORT_IR",
                          ARGS     => [$d1, $d2],
                          RESPONSE => \my $response,
                          ERROR    => \my $error,
                          );
            
            if ($error) {
                $errorRet = "Failed to execute Slink-e CMD_SETIRTIMEOUT " .
                    "command.  $error";
            } else {
                if (!defined($response)) {
                    $errorRet = "Slink-e command failed to return a " .
                        "response, as it's required to do";
                } elsif (length($response) != 2*2) {
                    $errorRet = "Slink-e command return the wrong number of " .
                        "bytes of response.";
                } else {
                    my $responsePeriods = hex($response);
                    if ($responsePeriods != $args{PERIOD}) {
                        $errorRet = "Tried setting timeout period of " .
                            "$args{PERIOD} sample periods, " .
                            "but Slink-e Set IR Timeout command's " .
                            "return value says it set it to $responsePeriods";
                    }
                }
            }
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->getIRMinimumLength(%ARGS)

This gets the length of the shortest IR message in port message data
bytes which the Slink-e will consider a valid message.

Arguments:

  MINLENGTH => The minimum message length in bytes of port message data.

  ERROR     =>  Reference to a variable in which this method returns a 
                description of the error it encountered, or undef if it did
                not encounter an error

=cut

sub getIRMinimumLength(%) {
    my ($slinke, %args) = @_;

    my $errorRet;

    $slinke->txrx(COMMAND  => "CMD_GETIRMINLEN",
                  PORT     => "PORT_IR",
                  RESPONSE => \my $response,
                  ERROR    => \my $error,
                  );

    if ($args{MINLENGTH}) {
        $ {$args{MINLENGTH}} = hex($response);
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->setIRMinimumLength(%ARGS)

This sets the length of the shortest IR receive message in bytes which
will be considered a valid message.  IR receiver modules such as the one
on the Slink-e tend to be very sensitive to both optical and electrical
noise, causing them to occasionally generate false pulses when there is
no actual IR signal.  The false pulses are generally of short duration 
and do not contain the large number of on/off alternations present in a
true IR remote signal.  By setting a minimum message length, false pulses
will be ignored and not reported to the host.  The minimum length can
range from 0 to 15 bytes.

%ARG is defined as follows:

  MINLENGTH => Minimum message length, in bytes.

  ERROR     => A reference to an error string variable.


=cut

sub setIRMinimumLength(%) {
    my ($slinke, %args) = @_;

    debug(1, "setIRMinimumLength() called.");

    my $errorRet;

    if (!defined($args{MINLENGTH})) {
        $errorRet = "You must specify the MINLENGTH argument";
    } elsif ($args{MINLENGTH} !~ m{\d+}) {
        $errorRet = "MINLENGTH must be a positive integer.";
    } elsif ($args{MINLENGTH} > $Slinke::Foundation::PORT_IR_MAXML) {
        $errorRet = "$args{MINLENGTH} is too long for the minimum message " .
            "length.  The maximum minimum allowed is " .
            "$Slinke::Foundation::PORT_IR_MAXML bytes.";
    } else {
        $slinke->txrx(COMMAND  => "CMD_SETIRMINLEN",
                      PORT     => "PORT_IR",
                      ARGS     => [$args{MINLENGTH}],
                      RESPONSE => \my $newMinLength,
                      ERROR    => \my $error,
                      );
        if ($error) {
            $errorRet = "Unable to execute CMD_SETIRMINLEN Slink-e " .
                "command.  $error";
        } else {
            my $numericalNewMinLength = hex($newMinLength);

            if ($numericalNewMinLength != $args{MINLENGTH}) {
                $errorRet = "Tried setting IR minimum message length of " .
                    "$args{MINLENGTH} - $numericalNewMinLength set";
            }
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub getIRZones($$%) {
    my ($direction, $slinke, %args) = @_;

    my $errorRet;

    if ($slinke->{VERSION} < 2.0) {
        $errorRet = "This method works only on Slink-e version 2.0 or " .
            "better.  This Slink-e is version $slinke->{VERSION}";
    } else {
        my $command = $direction eq "TRANSMIT" ? 
            "CMD_GETIRTXZONES" : "CMD_GETIRRXZONES";
        
        $slinke->txrx(COMMAND  => $command,
                      PORT     => "PORT_IR",
                      RESPONSE => \my $response,
                      ERROR    => \my $error,
                      );
        
        if ($error) {
            $errorRet = "Unable to execute Slink-e $command command.  $error";
        } else {
            if (defined($args{ZONEMASK})) {
                $ {$args{ZONEMASK}} = hex($response);
            }
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



sub setIRZones($$%) {

    my ($direction, $slinke, %args) = @_;

    my $errorRet;

    if ($slinke->{VERSION} < 2.0 ) {
        $errorRet = "This method requires a Slink-e version 2.0 or better.  " .
            "This Slink-e version is $slinke->{VERSION} ";
    } elsif (!defined($args{ZONEMASK})) {
        $errorRet = "You must specify the ZONEMASK argument.";
    } elsif ($args{ZONEMASK} !~ m{\d+}) {
        $errorRet = "ZONEMASK number must be a positive integer.  You " .
            "specified '$args{ZONEMASK}'";
    } elsif ($args{ZONEMASK} < 0 || $args{ZONEMASK} > 255) {
        $errorRet = "ZONEMASK argument must be in the range 0-0xFF.  " .
            "You specified $args{ZONEMASK}";
    } else {
        my $command = $direction eq "TRANSMIT" ? 
            "CMD_SETIRTXZONES" : "CMD_SETIRRXZONES";
        
        $slinke->txrx(COMMAND  => $command,
                      PORT     => "PORT_IR",
                      ARGS     => [$args{ZONEMASK}],
                      RESPONSE => \my $newzonemask,
                      ERROR    => \my $error,
                      );
        if ($error) {
            $errorRet = "Unable to execute $command Slink-e " .
                "command.  $error";
        } else {
            if ($direction eq "RECEIVE") {
                my $numericalNewzonemask = hex($newzonemask);
                
                if ($numericalNewzonemask != $args{ZONEMASK}) {
                    my $p = "0x" . uc(sprintf("%02x", $args{ZONEMASK}));
                    my $d = "0x" . uc(sprintf("%02x", $numericalNewzonemask));
                    $errorRet = "Tried setting IR receive zone mask " .
                        "of $p - $d set";
                }
            } else {
                # Slink-e makes a special case of Set Transmit Zones because
                # it knows people send it frequently (often before each port
                # message send), so it doesn't provide the verification 
                # response, as it does for Set Receive Zones.
            }
        }
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->getIRTransmitZones(%ARGS)

This returns the list of zones in which the the Slink-e transmits IR
signals.  

Argument is a hash with the following elements:

  ZONEMASK => Reference to a scalar variable to which this method sets
              the zone mask.  The value is a Perl number.  To interpret
              it, look at the binary representation of it.  The bits
              represent the 8 IR zones, IR0 being the LSB, IR7 the MSB.
              A "1" indicates the Slink-e is set to transmit in that
              zone.

  ERROR =>    A reference to an error string variable.

Control the set of transmit zones with $slinke->setIRTransmitZones().

This method requires a Slink-e version of 2.0 or above.  It fails
if the Slink-e version is earlier, in which case there is only one IR
zone.

=cut

sub getIRTransmitZones(%) {
    my ($slinke, %args) = @_;

    debug(1, "getIRTransmitZones() called.");

    getIRZones("TRANSMIT", $slinke, %args);
}



=head2 $slinke->setIRTransmitZones(%ARGS)

This sets the set of zones in which the Slink-e transmits IR signals.

%ARG is defined as follows:

  ZONEMASK => This gives the zone mask.  It is a Perl number.  
              To interpret it, look at the binary representation of
              it.  The bits represent the 8 IR zones, IR0 being the
              LSB, IR7 the MSB.  A "1" indicates the Slink-e transmits
              in that zone.

  ERROR =>    A reference to an error string variable.

Query the current set of transmit zones with $slinke->getIRTransmitZones().

This method requires a Slink-e version of 2.0 or above.  It fails
if the Slink-e version is earlier, in which case there is only one IR zone.

=cut

sub setIRTransmitZones {
    my ($slinke, %args) = @_;

    debug(1, "setIRTransmitZones() called.");

    setIRZones("TRANSMIT", $slinke, %args);
}



=head2 $slinke->getIRReceiveZones(%ARGS)

This returns the list of zones in which the the Slink-e receives IR
signals.  

%ARG is defined as follows:

  ZONEMASK => Reference to a scalar variable to which this method sets
              the zone mask.  The value is a Perl number.  To interpret
              it, look at the binary representation of it.  The bits
              represent the 8 IR zones, IR0 being the LSB, IR7 the MSB.
              A "1" indicates the Slink-e is set to transmit in that
              zone.

  ERROR =>    A reference to an error string variable.

Control the set of receive zones with $slinke->setIRReceiveZones().

This method requires a Slink-e version of 2.0 or above.  It fails
if the Slink-e version is earlier, in which case there is only one IR
zone.

=cut

sub getIRReceiveZones(%) {
    my ($slinke, %args) = @_;

    debug(1, "getIRReceiveZones() called.");

    getIRZones("RECEIVE", $slinke, %args);
}



=head2 $slinke->setIRReceiveZones(%ARGS)

This sets the set of zones in which the Slink-e receives IR signals.

%ARG is defined as follows:

  ZONEMASK => This gives the zone mask.  It is a Perl number.  
              To interpret it, look at the binary representation of
              it.  The bits represent the 8 IR zones, IR0 being the
              LSB, IR7 the MSB.  A "1" indicates the Slink-e transmits
              in that zone.

  ERROR =>    A reference to an error string variable.

Query the current set of receive zones with $slinke->getIRReceiveZones().

This method requires a Slink-e version of 2.0 or above.  It fails
if the Slink-e version is earlier, in which case there is only one IR zone.

=cut


sub setIRReceiveZones(%) {
    my ($slinke, %args) = @_;
    
    debug(1, "setIRReceiveZones() called.");

    setIRZones("RECEIVE", $slinke, %args);
}



=head2 $slinke->getIRPolarity(%ARGS)

Gets the polarity sense for each of the IR zones.  These settings also
affect the IR routing system.

Argument is a hash.  Elements are:

  POLARITY => Reference to a variable this method sets to the current
              polarity, as follows:

              The value is a Perl number.  To interpret it, look at
              the binary representation of the number.  The bits
              represent the 8 IR zones, IR0 being the LSB, IR7 the
              MSB. A "1" bit indicates that input in that zone is
              active-low (0 Volts means "1"), a "0" bit indicates that
              input in that zone is active-high (5 Volts means "1").

  ERROR =>    Reference to a variable in which this method returns a 
              description of the error it encountered, or undef if it did
              not encounter an error


To set the zone polarities, use $slinke->setIRPolarity().

I<This command requires a Slink-e version of 2.0 and above>

=cut

sub getIRPolarity(%) {
    my ($slinke, %args) = @_;
    
    my $errorRet;

    if ($slinke->{VERSION} < 2.0 ) {
        $errorRet = "Current Slink-e version is $slinke->{VERSION} " .
            "(need 2.0 or greater)";
    } else {
        $slinke->txrx(COMMAND  => "CMD_GETIRRXPOL",
                      PORT     => "PORT_IR",
                      RESPONSE => \my $response,
                      ERROR    => \my $error,
                      );

        if ($error) {
            $errorRet = "Failed to execute Slink-e CMD_GETIRRXPOL " .
                "command.  $error";
        } else {
            if ($args{POLARITY}) {
                $ {$args{POLARITY}} = hex($response);
            }
        }
    }

    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->setIRPolarity(%ARGS)

Sets the polarity sense for each of the IR zones.  These settings
also affect the IR routing system.  

Argument is a hash.  Elements are:

  POLARITY => The polarity, as follows:

              The value is a Perl number.  To interpret it, look at
              the binary representation of the number.  The bits
              represent the 8 IR zones, IR0 being the LSB, IR7 the
              MSB. A "1" bit indicates that input in that zone is
              active-low (0 Volts means "1"), a "0" bit indicates that
              input in that zone is active-high (5 Volts means "1").

  ERROR =>    Reference to a variable in which this method returns a 
              description of the error it encountered, or undef if it did
              not encounter an error

To query the current zone polarities, use $slinke->getIRPolarity().

I<This command requires a Slink-e version of 2.0 and above>

=cut

sub setIRPolarity {
    my ($slinke, %args) = @_;

    my $errorRet;

    if ($slinke->{VERSION} < 2.0) {
        $errorRet = "Current Slink-e version is $slinke->{VERSION} " .
            "(need 2.0 or greater)";
    } else {
        my $polarity = $args{POLARITY};
        if ($polarity < 0 || $polarity > 255) {
            $errorRet = "$polarity is not a valid polarity code " .
                "(0-0xFF is the acceptable range)";
        } else {
            $slinke->txrx(COMMAND  => "CMD_SETIRRXPOL",
                          PORT     => "PORT_IR",
                          ARGS     => [ $polarity ],
                          RESPONSE => \my $response,
                          ERROR    => \my $error,
                          );
            
            my $polaritySet = hex($response);
            
            if ($polaritySet != $polarity) {
                my $req = "0x" . uc(sprintf("%02x", $polarity));
                my $set = "0x" . uc(sprintf("%02x", $polaritySet));
                $errorRet = "Tried setting IR receive polarity of $req " .
                    "but $set got set";
            }
        }
    }
    if ($args{ERROR}) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->getIRRoutingTable()

This response describes the IR routing table. The routelist byte for
each receive zone specifies to which zones the Slink-e echos the
signals received in that receive zone.

The format for this byte is the same as the Set IR transmit zones command.
The carrier byte specifies the carrier frequency to be used in the routed 
signals from a given receive zone.  This byte is equivalent to the CC byte in 
the Set IR carrier command. To reduce data storage requirements, no
prescaler value can be specified and the prescaler is defaulted to 0 
instead. This means that 15.7khz is the lowest available carrier frequency 
for IR routing.

I<This command requires a Slink-e version of 2.0 and above>

=cut

sub getIRRoutingTable {
    my $slinke = shift;
    
    my @retval;

    if ($slinke->{VERSION} < 2.0) {
        warn("Current Slink-e version is $slinke->{VERSION} " .
             "(need 2.0 or greater)");
    } else {
        $slinke->txrx(COMMAND  => "CMD_GETIRROUTING",
                      PORT     => "PORT_IR",
                      RESPONSE => \my $data);

        my @data;
        
        while ($data) {
            my $i;
            ($i, $data) = $data =~ /(..)(.*)/;

            push @data, hex($i);

            ($i, $data) = $data =~ /(..)(.*)/;
            push @data, ($SLINKE_CLK / 4) / (hex( $i ) + 1);
        }
        @retval = @data;
    }
    
    return @retval;
}



=head2 $slinke->setIRRoutingTable(@data)

This command sets up the IR routing table. The routelist byte for each 
receive zone specifies to which zones the Slink-e will echo the 
signals received in that receive zone.
The format for this byte is the same as the Set IR transmit zones command.
The carrier byte specifes the carrier frequency to be used in the routed 
signals from a given receive zone. This byte is equivalent to the CC byte 
in the Set IR carrier command. To reduce data storage requirements, no
prescaler value can be specified and the prescaler is defaulted to 0 
instead. This means that 15.7khz is the lowest available carrier frequency 
for IR routing.

I<This command requires a Slink-e version of 2.0 and above>

=cut

sub setIRRoutingTable {
    my $slinke = shift;
    my @data = @_;
    
    if ($slinke->{VERSION} < 2.0 ) {
        warn("Current Slink-e version is $slinke->{VERSION} " .
             "(need 2.0 or greater)");
        return undef;
    }

    for (my $i=1; $i <= $#data; $i+=2) {
        my $freq = $data[$i];

        if ($freq) {
            my $count = int($SLINKE_CLK/4.0/$freq);
            if (!$count) {
                my $max = $SLINKE_CLK/4.0;
                warn("$freq is too high of a carrier frequency " .
                     "($max is the max)");
                return undef;
            }
            elsif ($count >= 256) {
                my $min = $SLINKE_CLK/4.0/256.0;
                warn("$freq is too low of a carrier frequency " .
                     "($min is the minimum)");
                return undef;
            }

            $data[$i] = $count - 1;
        }
        else {
            $data[$i] = 0; # indicate no carrier
        }
    }
    
    $slinke->txrx(COMMAND  => "CMD_SETIRROUTING",
                  PORT     => "PORT_IR",
                  ARGS     => [ @data ],
                  RESPONSE => \my $data);

    my @tmpdata;
    my @newdata;
    while ($data) {
        my $i;
        ($i, $data) = $data =~ /(..)(.*)/;

        push @newdata, hex($i);
        push @tmpdata, hex($i);

        ($i, $data) = $data =~ /(..)(.*)/;
        push @tmpdata, hex($i);
        push @newdata, ($SLINKE_CLK / 4) / (hex($i) + 1);
    }

    for (my $i=0; $i<=$#data; ++$i) {
        if ($data[$i] != $tmpdata[$i]) {
            warn("Did not get proper return value");
            return undef;
        }
    }
    
    return @newdata;
}



=head2 $slinke->getHandshaking()

Reports the handshaking mode for the Parallel Port. 

The output looks like this: C<[0 0 0 0 0 0 in out]>
where 'in' and 'out' are 0 or 1.

The data has the same interpretation as the response to a Get Handshaking
Mode Slink-e command (for which this is the binary representation).

=cut

sub getHandshaking {
    my $slinke = shift;

    $slinke->txrx(COMMAND  => "CMD_GETHSMODE",
                  PORT     => "PORT_PAR",
                  RESPONSE => \my $response,
                  );

    return hex($response);
}



=head2 $slinke->setHandshaking( $handshaking )

Sets the input and output handshaking mode for the Parallel Port. 

C<$handshaking> looks like this: C<[0 0 0 0 0 0 in out]>
where 'in' and 'out' are 0 or 1.

The argument has the same interpretation as the Set Handshaking Mode
Slink-e command (for which this is the binary representation).

The return value is the new handshaking mode, in the same format as the
argument.

If the method fails, it returns C<undef>.

=cut

sub setHandshaking {
    my ($slinke, $handshaking) = @_;

    my $retval;
    
    if ($handshaking < 0 || $handshaking > 3) {
        warn("$handshaking is not a valid port " .
             "(0-0x03 is the acceptable range)");
    } else {
        $slinke->txrx(COMMAND  => "CMD_SETHSMODE",
                      PORT     => "PORT_PAR",
                      ARGS     => [ $handshaking ],
                      RESPONSE => \my $response);
    
        my $data = hex($response);
        
        if ($data != $handshaking) {
            my $p = "0x" . uc(sprintf("%02x", $handshaking));
            my $d = "0x" . uc(sprintf("%02x", $data));
            warn("Tried setting handshaking mode of $p, but $d got set");
        } else {
            $retval = $data;
        }
    }
    return $retval;
}



=head2 $slinke->getDirection()

Reports which parallel port lines are inputs or outputs. The bits d7:d0 in 
the output correspond 1 to 1 with the Parallel Port I/O lines DIO7:DIO0. 
Setting a direction bit to 1 assigns the corresponding DIO line as an
input, while setting it to 0 make it an output. At startup, all DIO 
lines are configured as inputs. The use of handshaking on lines DISTB/DIO7 
and DOSTB/DIO6 overrides the direction configuration for these lines while
enabled.

=cut

sub getDirection {
    my $slinke = shift;

    $slinke->txrx(COMMAND  => "CMD_GETDIR",
                  PORT     => "PORT_PAR",
                  RESPONSE => \my $response);

    return hex($response);
}



=head2 $slinke->setDirection($direction)

Configures the parallel port lines as inputs or outputs. The bits d7:d0 
in the direction byte correspond 1 to 1 with the Parallel Port I/O lines 
DIO7:DIO0. Setting a direction bit to 1 assigns the corresponding DIO line 
as an input, while setting it to 0 make it an output. At startup, all DIO
lines are configured as inputs. The use of handshaking on lines DISTB/DIO7 
and DOSTB/DIO6 overrides the direction configuration for these lines while
enabled. Slink-e will return a configuration direction equals response to 
verify your command.

=cut

sub setDirection {
    my ($slinke, $direction) = @_;

    my $retval;
    
    if ($direction < 0 || $direction > 255) {
        warn("$direction is not a valid direction setting " .
             "(0-0xFF is the acceptable range)");
    } else {
        $slinke->txrx(COMMAND  => "CMD_SETDIR",
                      PORT     => "PORT_PAR",
                      ARGS     => [ $direction ],
                      RESPONSE => \my $response);

        my $data = hex($response);

        if ($data != $direction) {
            my $p = "0x" . uc(sprintf("%02x", $direction));
            my $d = "0x" . uc(sprintf("%02x", $data));
            warn("Tried setting parallel direction configuration of $p " .
                 "but $d got set");
        } else {
            $retval = $data;
        }
    }
    return $retval;
}



=head2 $slinke->sampleParPort()

Causes the Slink-e to sample the Parallel Port inputs just as if it had 
seen a rising edge on DISTB when input handshaking is enabled. This command 
works whether input handshaking is enabled or not. The Slink-e will
respond with a port receive message containing the Parallel Port data.

Note that this function does I<not> actually return the parallel port data.
To get that, you must call C<Slinke::Receive::receiveFromSlinke()>.

=cut

sub sampleParPort {
    my $slinke = shift;
    
    $slinke->txrx(COMMAND => "CMD_SAMPLE",
                  PORT    => "PORT_PAR",
                  );
}



=head2 $slinke->sendPortMessage(%)

This method causes the Slink-e to send a port message.

The argument is a hash with these elements:

PORT  => The port from which to send the port message.  This must be one
         of C<PORT_IR>, C<PORT_SL0>, C<PORT_SL1>, C<PORT_SL2>, C<PORT_PAR>, or
         C<PORT_SER>.  E.g. PORT => "PORT_IR" .

DATA  => A reference to an array of numbers.  Each number represents a
         byte of port message data, in order.  The byte is the pure binary
         encoding of that number.  E.g. for the port message data 0x10A000,
         DATA => [16,160,0] .

SYNC  => Logical:  Caller wants to wait until the port message has been
         sent and have the call fail if it could not be sent.  If you don't
         specify this, this method may simply buffer the message in the 
         device driver or the Slink-e and return success, and some time later
         an error could occur resulting in the message not actually getting
         sent.  A typical failure is a data underrun, where the device driver
         is unable to send the data to the Slink-e as fast as required to
         generate an uninterrupted port message.

         With SYNC, we make the sending synchronous by sending an
         arbitrary command (the "synchronizing command") that has a
         response, after sending all the port message data, and
         waiting for the response to the synchronizing command.  If
         there was an error generating the port message, the Slink-e
         generated an unsolicited error report and then went into a
         mode in which it will ignore the synchronizing command, so we
         will get the error report in what appears to be response to
         the synchronizing command.  When that happens, we return
         failure from this method.

ERROR => reference to error description variable.  E.g. ERROR => \my $error .

Note that port message data for the IR port has a complex relationship
with the actual ons and offs of the port message, whereas for all the other
ports, the relationship is straightforward bit for bit.

=cut

sub sendPortMessage {

    my $slinke = shift;
    my %args = @_;
    
    my $errorRet;

    debug(1, "sendPortMessage() called.");

    if (!defined($args{PORT})) {
        $errorRet = "You must specify the PORT argument";
    } elsif (!defined($PORTNUMBER{$args{PORT}})) {
        $errorRet = "Invalid PORT value: $args{PORT}";
    } elsif (!defined($args{DATA})) {
        $errorRet = "You must specify the DATA argument.";
    } else {
        my @data = @{$args{DATA}};
        my $port = $args{PORT};

        debug(2, "Port=$port, " . scalar(@data) . " bytes of port msg data");

        # Note that there is no response to a Send Port Message Data
        # or Send Port Message End command, but if we don't send these
        # messages fast enough to keep the port message going, Slink-e
        # sends an unsolicited Command Data Underrun message and
        # ignores further commands until it gets a Resume command.

        my $maxDataEachMessage = $Slinke::Foundation::MAXDATABLOCK;

        my @stream;

        @stream = ();  # initial value

        for (my $i=0; $i < @data; $i += $maxDataEachMessage) {
            my $count = min($maxDataEachMessage, @data - $i);
            my $last = $i + $count - 1;

            debug(2, "Adding Send Port Message command for $count bytes " .
                  "to stream");

            # Add the port/type byte (Byte 0) of the command:
            push(@stream, ($PORTNUMBER{$port} << 5) + $count);
            
            # Add the port message data bytes:
            push(@stream, @data[$i..$last]);
        }

        debug(2, "Adding Send Port Message End command to stream");

        push(@stream, ($PORTNUMBER{$port} << 5) + 0);

        my $status = writeToPort($slinke, @stream);
        if (!defined($status)) {
            warn("Error in sending stream of Send Port Message / " .
                 "Send Port Message End commands to serial port.  (" . 
                 scalar(@stream) . " bytes)");
        }
        debug(2, scalar(@stream) . "-byte stream sent");
        if ($args{SYNC}) {
            debug(2, "Executing synchronizing command (Get Version)");

            txrxit($slinke, $COMMAND_TYPE{'CMD_GETVERSION'}, 'PORT_NON', 
                   [], undef, \my $error);
            if ($error) {
                $errorRet = "Port message data successfully written to " .
                    "serial port, but syncing command failed:  $error";
            }                        
        }
    }

    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->reset(%ARGS)

Warm-boots the Slink-e, resetting all settings to their startup
defaults, including the baud rate.

Be wary of the fact that the startup default baud rate could be
different from the current baud rate.  In that case, you will lose the
ability to do anything with the Slinke::Control object until you do a
->syncComm() to resynchronize communcations.


Argument is a hash.  Elements are:

  ERROR =>    Reference to a variable in which this method returns a 
              description of the error it encountered, or undef if it did
              not encounter an error

=cut

sub reset() {
    my ($slinke, %args) = @_;

    my $errorRet;

    $slinke->txrx(COMMAND => "CMD_RESET",
                  PORT    => "PORT_NON",
                  ERROR   => \my $error,
                  );

    if ($error) {
        $errorRet = "Slink-e CMD_RESET command failed.  $error";
    } else {
        # That Slink-e command may have reset the baud rate, leaving us
        # unable to communicate.  In that case, the following load of
        # Slink-e settings will fail.  We leave it to Caller to do 
        # a syncComm() if necessary to reestablish communication and
        # reload the Slink-e settings.
        
        loadSlinkeSettings($slinke);
        
        # TODO: Should establish a "settings unknown" state in case the 
        # above failed.
    }
    if (defined($args{ERROR})) {
        $ {$args{ERROR}} = $errorRet;
    }
}



=head2 $slinke->loadDefaults()

Causes the Slink-e to set all settings to their startup defaults.  Be
wary of the fact that the startup default baud rate could be different
from the current baud rate.  In that case, you will lose the ability
to do anything with the Slinke::Control object until you do a
->syncComm() to resynchronize communcations.

I<This method requires a Slink-e version of 2.0 and above>

=cut

sub loadDefaults {
    my ($slinke) = @_;

    if ($slinke->{VERSION} < 2.0) {
        warn("Current Slink-e version is $slinke->{VERSION} " .
             "(need 2.0 or greater)");
    } else {
        $slinke->txrx(COMMAND => "CMD_LOADDEFAULTS",
                      PORT    => "PORT_NON",
                      );
        # That Slink-e command may have reset the baud rate, leaving us
        # unable to communicate.  In that case, the following load of
        # Slink-e settings will fail.  We leave it to Caller to do 
        # a syncComm() if necessary to reestablish communication and
        # reload the Slink-e settings.

        loadSlinkeSettings($slinke);

        # TODO: Should establish a "settings unknown" state in case the 
        # above failed.
    }
}



=head2 $slinke->saveDefaults()

Causes the Slink-e to save all of the current user settings as the
startup defaults.

I<This command requires a Slink-e version of 2.0 and above>

=cut

sub saveDefaults {
    my $slinke = shift;

    if ($slinke->{VERSION} < 2.0) {
        warn("Current Slink-e version is $slinke->{VERSION} " .
             "(need 2.0 or greater)");
    } else {
        $slinke->txrx(COMMAND => "CMD_SAVEDEFAULTS",
                      PORT    => "PORT_NON",
                      );
    }
}

1;



=head1 AUTHOR

Bryan Henderson <bryanh@giraffe-data.com>

Derived from Slinke.pm which is by Brian Paulsen <Brian@ThePaulsens.com>

  Copyright 2000, Brian Paulsen.  All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

For further information about the Slink-e, see 
<http://giraffe-data.com/slinke>.

=cut
