/*file contains code for the userdev device driver module*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include "userdev.h"

MODULE_AUTHOR("HariKrishna Vemuri");
MODULE_DESCRIPTION("UserLevel Device Driver Framework");

static int userdev_major = 0;		/*userdev major number*/
MODULE_PARM(userdev_major,"i");
MODULE_PARM_DESC(userdev_major,"The major number for userdev devices (default value assigned by system)");

EXPORT_NO_SYMBOLS;

/*definitions for using block device macros in blk.h*/
# define MAJOR_NR		userdev_major
# define DEVICE_NR(device)	MINOR(device)
# define DEVICE_NAME		USERDEV_BLK_NAME
# define DEVICE_REQUEST		userdev_request
# define DEVICE_NO_RANDOM
# define DEVICE_ON		/*nothing*/
# define DEVICE_OFF		/*nothing*/
# include <linux/blk.h>

# undef SHOW_LOG		/*define it to obtain log*/
# ifdef SHOW_LOG
# define PRINT(format, args...)	printk(format, ##args)
# else
# define PRINT(format, args...)	/*ignore*/
# endif

static struct device_record *attached_process[MAX_USERDEV_PROCESSES];	/*array of pointer to device record structures*/
static int chr_map[MAX_USERDEV_MINORS];	/*map between minor number and userdev id(index into attached_process array) for char device*/
static int blk_map[MAX_USERDEV_MINORS];	/*map between minor number and userdev id(index into attached_process array) for block device*/
static rwlock_t userdev_rwlock = RW_LOCK_UNLOCKED;	/*read-write lock for above 3 data structures*/

static struct proc_dir_entry *userdev_chr_base = NULL;	/*pointer to root of proc entry for userdev char devices*/
static struct proc_dir_entry *userdev_blk_base = NULL;	/*pointer to root of proc entry for userdev char devices*/

static devfs_handle_t userdev_devfs_handle = NULL;	/*device file system handle*/

/*block device arrays*/
static int *userdev_sizes;		/*block device sizes*/
static int *userdev_blksizes;		/*block size of block devices*/
static int *userdev_hardsects;		/*sector size of block devices*/
static int *userdev_maxreadaheads;	/*max read ahead for block devices*/
static int *userdev_maxsectors;		/*max request size for block devices*/

/*file operations structure for userdev device driver */
static struct file_operations userdev_fops = {
	owner:	THIS_MODULE,
	read:	userdev_read,
	write:	userdev_write,
	poll:	userdev_poll,
	ioctl:	userdev_ioctl_chr,
	open:	userdev_open_chr,
	flush:	userdev_flush,
	release:userdev_close_chr,
	fsync:	userdev_fsync,
	fasync:	userdev_fasync,
	readv: 	userdev_readv,
	writev:	userdev_writev
};

/*Block device operations structure for userdev device driver*/
static struct block_device_operations userdev_bdops = {
	open:			userdev_open_blk,
	release:		userdev_close_blk,
	ioctl:			userdev_ioctl_blk,
	check_media_change:	userdev_check_media_change,
	revalidate:		userdev_revalidate,
	mediactl:		userdev_mediactl
};


/*function to cleanup block device global arrays*/
static void userdev_blkdev_cleanup()
{
	if (userdev_sizes) kfree(userdev_sizes);
	if (userdev_blksizes) kfree(userdev_blksizes);
	if (userdev_hardsects) kfree(userdev_hardsects);
	if (userdev_maxreadaheads) kfree(userdev_maxreadaheads);
	if (userdev_maxsectors) kfree(userdev_maxsectors);

	blk_size[userdev_major] = NULL;
	blksize_size[userdev_major] = NULL;
	hardsect_size[userdev_major] = NULL;
	max_readahead[userdev_major] = NULL;
	max_sectors[userdev_major] = NULL;
}


/*function to initialize block device global arrays*/
static int userdev_blkdev_init()
{
	int i;

	/*alocate memory for each of the arrays*/
	userdev_sizes = (int*)kmalloc(MAX_USERDEV_MINORS*sizeof(int), GFP_KERNEL);
	if(!userdev_sizes) goto fail;

	userdev_blksizes = (int*)kmalloc(MAX_USERDEV_MINORS*sizeof(int), GFP_KERNEL);
	if(!userdev_blksizes) goto fail;

	userdev_hardsects = (int*)kmalloc(MAX_USERDEV_MINORS*sizeof(int), GFP_KERNEL);
	if(!userdev_hardsects) goto fail;

	userdev_maxreadaheads = (int*)kmalloc(MAX_USERDEV_MINORS*sizeof(int), GFP_KERNEL);
	if(!userdev_maxreadaheads) goto fail;

	userdev_maxsectors = (int*)kmalloc(MAX_USERDEV_MINORS*sizeof(int), GFP_KERNEL);
	if(!userdev_maxsectors) goto fail;

	/*assign default values to each of the arrays' elements*/
	for(i=0;i<MAX_USERDEV_MINORS;i++)
	{
		userdev_sizes[i] = USERDEV_BLKDEV_SIZE;
		userdev_blksizes[i] = USERDEV_BLKSIZE_SIZE;
		userdev_hardsects[i] = USERDEV_HARDSECT_SIZE;
		userdev_maxreadaheads[i] = USERDEV_MAX_READ_AHEAD;
		userdev_maxsectors[i] = USERDEV_MAX_SECTORS;
	}

	/*assign arrays to global block device arrays' pointers*/
	blk_size[userdev_major] = userdev_sizes;
	blksize_size[userdev_major] = userdev_blksizes;
	hardsect_size[userdev_major] = userdev_hardsects;
	max_readahead[userdev_major] = userdev_maxreadaheads;
	max_sectors[userdev_major] = userdev_maxsectors;

	read_ahead[userdev_major] = USERDEV_READ_AHEAD;
	return 0;
fail:
	userdev_blkdev_cleanup();
	return -1;
}


/*module initialization routine, called when insmod command executed*/
int init_module(void)		/*!!!! KERNEL DEPENDANT !!!! */
{
	int res,i;
	printk("Installing USERDEV driver\n");
	
	for(i=0;i<MAX_USERDEV_PROCESSES;i++)	/*intialize data structures*/
		attached_process[i] = NULL;
	for(i=0;i<MAX_USERDEV_MINORS;i++)
	{
		chr_map[i] = -1;
		blk_map[i] = -1;
	}
	
	res = devfs_register_chrdev(userdev_major,USERDEV_CHR_NAME,&userdev_fops);
	if (res == 0) 
	{
    		if (userdev_major > 0)		/*major number specified as module parameter*/
			printk("Character driver registered at major = %d\n",userdev_major);
		else
		{
			printk("ERROR(%d) occured during character driver registration\n",res);
			return res;
		}
	}
	else if(res < 0)
	{
		printk("ERROR(%d) occured during character driver registration\n",res);
		return res;
	}
	else
	{
    		printk("Character driver registered at major = %d\n",res);
		userdev_major = res;		/*record system assigned major number*/
	}

	/*register with device filesystem*/
	userdev_devfs_handle = devfs_register(NULL, USERDEV_CHR_NAME, DEVFS_FL_DEFAULT, 
					userdev_major, 0, S_IFCHR|S_IRUGO|S_IWUGO, &userdev_fops, NULL);

	/*register a block device with the same major number as the character device*/
	res = register_blkdev(userdev_major, USERDEV_BLK_NAME, &userdev_bdops);
	if(res == 0)
	{
		printk("Block driver registered at major = %d\n",userdev_major);
		/*initialize block device request queue*/
		blk_init_queue(BLK_DEFAULT_QUEUE(userdev_major), userdev_request);
		res = userdev_blkdev_init();
		if(res == 0)
			printk("Block driver arrays initialized\n");
		else
		{
			printk("ERROR(%d): Block driver arrays could not be initialized\n",res);
			return res;
		}
	}
	else
	{
		printk("ERROR(%d) occured during block driver registration\n",res);
		return res;
	}

	userdev_proc_init();	/*construct the userdev root proc entries*/
	return 0;
}


/*module cleanup routine called by rmmod command*/
void cleanup_module(void)		/*!!!! KERNEL DEPENDANT !!!! */
{
	int res,i;
	printk("Uninstalling USERDEV driver\n");
	
	for(i=0;i<MAX_USERDEV_PROCESSES;i++)
		if(attached_process[i] != NULL)
		{
			kfree(attached_process[i]);
			attached_process[i] = NULL;
		}

	devfs_unregister(userdev_devfs_handle);		/*unregister from device file system*/
	res = unregister_chrdev(userdev_major,USERDEV_CHR_NAME); 
	printk("Character Driver unregistration returned %d\n",res);

	res = unregister_blkdev(userdev_major,USERDEV_BLK_NAME); 
	printk("Block Driver unregistration returned %d\n",res);

	/*uninitialize block device request queue*/
	blk_cleanup_queue(BLK_DEFAULT_QUEUE(userdev_major));
	userdev_blkdev_cleanup();

	userdev_proc_cleanup();				/*remove proc filesystem entries*/
}


/*function to obtain userdev id from minor and type of device*/
static inline int process_index(int type, int minor)
{
	return (type == USERDEV_CHR) ? chr_map[minor] : blk_map[minor];
}


/*function to increment reference count of a device record */
static inline void addref(struct device_record *devrec)
{
	if(devrec) atomic_inc(&devrec->refcount);
	PRINT("addref = %d\n",atomic_read(&devrec->refcount));
}


/*function to free a given asynchronous I/O queue*/
static void empty_fasync_queue(struct fasync_struct *fas)
{
	struct fasync_struct *tmp;
	PRINT("Emptying fasync queue\n");
	while(fas!=NULL)
	{
		tmp = fas;
		fas = fas->fa_next;
		kfree(tmp);
	}
}


/*function to free resources held by the attached device*/
static void free_device(int id)
{
	struct device_record *devrec;
	int minor,type,channel;

	PRINT("freeing device\n");
	read_lock(&userdev_rwlock);
	devrec = attached_process[id];
	read_unlock(&userdev_rwlock);

	minor = devrec->minor;
	type = devrec->type;

	if(devrec->proc_entry)		/*remove proc file system entry*/
	{
		char *name;
		name = (char*)kmalloc(15,GFP_KERNEL);
		sprintf(name,"dev%d",minor);
		if(type == USERDEV_CHR)
			del_proc_entry(name, userdev_chr_base);
		else
			del_proc_entry(name, userdev_blk_base);
		kfree(name);
	}

	spin_lock(&devrec->fasync_lock);	/*empty fasync queue*/
	empty_fasync_queue(devrec->fasync_queue);
	spin_unlock(&devrec->fasync_lock);

	if((channel = atomic_read(&devrec->dma_no)) != -1)	/*free dma channel if being used*/
	{
		int flags;
		PRINT("freeing dma channel %d\n",atomic_read(&devrec->dma_no));
		flags = claim_dma_lock();	/*claim lock on dma controller*/
		disable_dma(channel);
		release_dma_lock(flags);	/*release lock on dma contoller*/
		free_dma(atomic_read(&devrec->dma_no));
	}
	if(atomic_read(&devrec->irq_no) != -1)	/*free irq if interrupts being used*/
	{
		PRINT("freeing irq %d\n",atomic_read(&devrec->irq_no));
		free_irq(atomic_read(&devrec->irq_no), (void*)devrec);
	}
	if(devrec->dma_buffer != NULL) 		/*free dma buffer if it still exists*/
		free_pages((unsigned long)devrec->dma_buffer, get_order(devrec->dma_len));
	devrec->dma_len = 0;

	/*close the additional open made during attach, to release inode private data in case of pipes*/
	devrec->writeops->release(devrec->write_dentry->d_inode, devrec->writefp);
	/*decrement dentry reference count and deallocate both dentry and the inode*/
	dput(devrec->write_dentry);

	devrec->readops->release(devrec->read_dentry->d_inode, devrec->readfp);
	dput(devrec->read_dentry);	/*same as above for read descriptor*/
	
	kfree(devrec->devicename);
	kfree(devrec->ioctls);

	write_lock(&userdev_rwlock);
	kfree(attached_process[id]);	/* = devrec*/
	attached_process[id] = NULL;
	if(type == USERDEV_CHR)
		chr_map[minor] = -1;
	else
	{
		blk_map[minor] = -1;
		userdev_sizes[minor] = USERDEV_BLKDEV_SIZE;	/*restore defaults into arrays*/
		userdev_blksizes[minor] = USERDEV_BLKSIZE_SIZE;
		userdev_hardsects[minor] = USERDEV_HARDSECT_SIZE;
		userdev_maxreadaheads[minor] = USERDEV_MAX_READ_AHEAD;
		userdev_maxsectors[minor] = USERDEV_MAX_SECTORS;
	}
	write_unlock(&userdev_rwlock);

	MOD_DEC_USE_COUNT;		/*decrement userdev module use count*/
}


/*
//function to decrement reference count of a device record and free it 
//along with its components if the reference count falls to zero and a request
//for detaching the process has been made. 
//Any waiting processes are woken up and error number is appropriately set
*/
static void remref(struct device_record *devrec)	/*!!!! KERNEL DEPENDANT !!!! */
{
	if(devrec == NULL) return;
	atomic_dec(&devrec->refcount);
	PRINT("remref = %d\n",atomic_read(&devrec->refcount));

	if(atomic_read(&devrec->detach_called))		/*if detach request has been made, wake up all*/
		userdev_wake_up_all(devrec);	/*process waiting for response packet*/
	if((atomic_read(&devrec->refcount) == 0) && (atomic_read(&devrec->detach_called)))
		free_device(process_index(devrec->type, devrec->minor));
}


/*
//interrupt handler for userdev driver
//When an interrupt occurs on an irq specified by the driver process,
//a specific signal is sent to it signifying the occurance of an interrupt 
*/
static void userdev_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	int ret;
    	struct siginfo info;
	struct device_record *dev = (struct device_record*)dev_id;	/*pointer to device record passed by kernel*/
	 
	PRINT("Interrupt occured for %s on irq %d\n",dev->devicename,irq);
	info.si_signo = atomic_read(&dev->sig_no);
    	info.si_errno = 0;
    	info.si_code = SI_INTERRUPT;	/*defined in lpproc.h*/
    	info.si_pid = 0;
    	info.si_uid = 0;  
	info.si_int = dev->minor;
	
	ret = kill_proc_info(info.si_signo, &info, dev->pid);	/*send signal to the process*/
	PRINT("Signal delivery returned %d\n",ret);
}


/*function to send a message to the driver process*/
int userdev_message(struct device_record *p, int type, int reqid, int size, void *data)
{
	struct header head;
	struct message_send ms;
	char *packet;
	int ret,len;
	
	ms.type = type;			/*fill in message packet and convert to string*/
	ms.reqid = reqid;
	ms.size = size;
	ms.data = data;

	head.opcode = OP_MESSAGE;
	head.request_id = genID();
	len = message_send_to_char(&head,&ms,&packet);	

	ret = send(packet,len,p);	/*send packet(string)*/
	kfree(packet);
	PRINT("Sent message packet\n");
	return(ret);
}


/*
//read function of userdev driver
//size, offset to read from are sent across to the driver process and the received 
//data is placed in the buffer
*/
ssize_t userdev_read (struct file *filp, char *buffer, size_t size, loff_t *lpos)
{
	int minor,id,ret;
	struct header head;
	struct read_send rs;
	struct read_recv *rr;
	char *packet;
	int len,i;
	struct device_record *p;

	PRINT("Reading USERDEV\n");
	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	if (minor == 0) return 0;		/*nothing to do if minor = 0 */
	
	read_lock(&userdev_rwlock);
	if ((i = chr_map[minor]) == -1) 	/*as read is only for character devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);				/*check if the process has attached to userdev and increase reference count */
	if (!IS_READ_SUPPORTED(p->func_mask)) { remref(p);  return ERR_FUNC_NOT_SUPPORTED; }

	rs.size = size;				/*fill in data to send*/
	rs.offset = *lpos;
	rs.flags = filp->f_flags;

	id = genID();				/*construct packet header*/
	head.opcode = OP_READ;
	head.request_id = id;
	len = read_send_to_char(&head,&rs,&packet);	/*convert to string*/

	ret = send(packet,len,p);		/*send packet(string)*/
	kfree(packet);
	PRINT("Sent read packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);	/*wait and receive response string*/
	PRINT("Received read packet\n");
	if(ret<0) { remref(p); return ret; }
	rr = read_recv_from_char(packet);	/*retreive response data from string*/

	if(rr->data.len >= 0)			/*success*/
	{
		if(copy_to_user(buffer,rr->data.elements,rr->data.len)) ret = ERR_MEMORY_FAULT;
		else ret = rr->data.len;	/*fill data buffer from response*/
	}
	else ret = rr->data.len;
	*lpos = rr->offset;

	kfree(packet);
	kfree(rr);
	remref(p);
	return ret;
}


/*
//write function for userdev driver
//the data to be written and the offset are sent across to the driver process
//and the response from it is returned 
*/
ssize_t userdev_write (struct file *filp, const char *buffer, size_t size, loff_t *lpos)
{
	int minor,id,ret,len;
	struct header head;
	struct write_send ws;
	struct write_recv *wr;
	char *packet,*buf;
	int i;
	struct device_record *p;

	PRINT("Writing USERDEV\n");
	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	if (minor == 0) return 0;		/*nothing to do if minor = 0 */
	
	read_lock(&userdev_rwlock);
	if ((i = chr_map[minor]) == -1) 	/*as write function is only for character devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);				/*check if the process has attached to userdev and increase reference count*/
	if (!IS_WRITE_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	ws.offset = *lpos;			/*fill in data to send*/
	ws.flags = filp->f_flags;
	ws.data.len = size;
	buf = (char*)kmalloc(size,GFP_KERNEL);
	if (buf == NULL) { remref(p); return ERR_MEMORY_FAULT; }
	if(copy_from_user(buf,buffer,size))	/*copy data from user space and check for memory errors*/
	{
		kfree(buf);
		remref(p);
		return ERR_MEMORY_FAULT;
	}
	ws.data.elements=buf;

	id = genID();				/*construct packet header*/
	head.opcode = OP_WRITE;
	head.request_id = id;
	len = write_send_to_char(&head,&ws,&packet);	/*transform to string*/
	kfree(buf);

	ret = send(packet,len,p);		/*send header and data packet*/
	kfree(packet);
	PRINT("Sent write packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);	/*wait and receive response*/
	PRINT("Received write packet\n");
	if(ret<0) { remref(p); return ret; }
	wr = write_recv_from_char(packet);	/*retreive result from response*/
	ret = wr->num_written;
	*lpos = wr->offset;

	kfree(packet);
	kfree(wr);
	remref(p);
	return ret;
}

/*
//poll function for userdev driver
//make an entry in the input and output wait queues, so that driver process can
//intimate through ioctls about the input/output poll events
//call the userlevel driver function, wait for response and return the result
*/
unsigned int userdev_poll (struct file *filp, struct poll_table_struct *polltab)	/*!!!! KERNEL DEPENDANT !!!! */
{
    	int minor,i,len,id,ret;
	struct header head;
	struct poll_send ps;
	struct poll_recv *pr;
	char *packet;
	struct device_record *p;

	PRINT("Polling USERDEV\n");
	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	if(minor==0) return 0;			/*nothing to do if minor = 0*/

	read_lock(&userdev_rwlock);
	if ((i = chr_map[minor]) == -1) 	/*as poll is only for character devices*/
	{
		read_unlock(&userdev_rwlock);
		return POLLERR;	/*poll function error code*/
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return POLLERR;	/*poll error code is interpreted by the calling generic poll function*/
	if (atomic_read(&p->detach_called)) return POLLERR;			
	addref(p);				/*check if the process has attached to userdev and increase reference count */
	if (!IS_POLL_SUPPORTED(p->func_mask)) { remref(p); return POLLERR; }

    	if (polltab)				/*add to input & output poll wait queue*/
	{
		PRINT("Adding to inq\n");
		spin_lock(&p->poll_in_lock);
		__pollwait(filp, &p->poll_inq, polltab);
		spin_unlock(&p->poll_in_lock);

		PRINT("Adding to outq\n");
		spin_lock(&p->poll_out_lock);
		__pollwait(filp, &p->poll_outq, polltab);
		spin_unlock(&p->poll_out_lock);
	}
	PRINT("current pid = %d\n",current->pid);

	id = genID();				/*construct packet header*/
	head.opcode = OP_POLL;
	head.request_id = id;
	len = poll_send_to_char(&head,&ps,&packet);	/*send packet to call driver process poll*/

	ret = send(packet,len,p);				
	kfree(packet);
	PRINT("Sent poll packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);	/*wait for response and return the result*/
	PRINT("Received poll packet\n");
	if(ret<0) { remref(p); return ret; }
	pr = poll_recv_from_char(packet);

	ret = pr->poll_result;
	kfree(packet);
	kfree(pr);

	current->state = TASK_INTERRUPTIBLE;	/*bcoz intermediate call to pipe read, resets it to running*/
	remref(p);
	return ret;
}


/*ioctl function of userdev for character devices*/
int userdev_ioctl_chr (struct inode *inode, struct file *filp, unsigned int cmd, unsigned long addr)
{
	PRINT("Ioctl USERDEV_CHR\n");
	return userdev_ioctl(inode, filp, cmd, addr, USERDEV_CHR);
}


/*ioctl function of userdev for block devices*/
int userdev_ioctl_blk (struct inode *inode, struct file *filp, unsigned int cmd, unsigned long addr)
{
	PRINT("Ioctl USERDEV_BLK\n");
	return userdev_ioctl(inode, filp, cmd, addr, USERDEV_BLK);
}


/*generic ioctl function called by above 2 functions*/
int userdev_ioctl (struct inode *inode, struct file *filp, unsigned int cmd, unsigned long addr, int type)
{
	int minor,ret,i;

	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	if (minor == 0)				/*minor(0) is a special case used for talking to userdev driver*/
		switch(cmd)
		{
			case USERDEV_ATTACH:	/*attach a driver process to userdev*/
			{
				struct attach_record a_rec;				/*attach record*/
				char *temp;
				int ix;

				if(copy_from_user(&a_rec,(void*)addr,sizeof(a_rec))) return ERR_MEMORY_FAULT;

					/*retreive minor number from device filename*/
				ret = get_minor(a_rec.devicename);		
				if (ret <= 0) return ERR_INVALID_DEVICE;
				minor = MINOR(ret);

				/*permission check: need read-write permission on userdev*/
				if(!(filp->f_flags & O_RDWR)) return ERR_ACCESS_DENIED;

				read_lock(&userdev_rwlock);
				ix = process_index(type, minor);
				read_unlock(&userdev_rwlock);
				if(ix != -1) return ERR_MINOR_NOT_FREE;

				return attach_device(&a_rec, minor, type);	/*do process attaching*/
				break;
			}

			case USERDEV_DETACH:	/*detach driver process from userdev*/
			{
				struct detach_record d_rec;	/*detach record*/
				struct device_record *p;

				if(copy_from_user(&d_rec,(void*)addr,sizeof(d_rec))) return ERR_MEMORY_FAULT;

					/*check validity of the passed userdev id*/
				if (d_rec.id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type,i) == d_rec.id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[d_rec.id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;

					/*check if the current process pid matches*/
					/*with that of the process that called*/
					/*attach to userdev*/
				if(current->pid != p->pid) return ERR_WRONG_PID;
				return detach_device(&d_rec);		/*do process detaching*/
				break;
			}

			case USERDEV_POLL_IN:	/*inform arrival of input data for polling*/
			{
				int id;	/*userdev id from user space*/
				struct device_record *p;

				if(copy_from_user(&id,(void*)addr,sizeof(id))) return ERR_MEMORY_FAULT;

					/*check validity of the passed userdev id*/
				if (id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_DEVICE_NOT_ATTACHED;

					/*check if the current process pid matches*/
					/*with that of the process that called*/
					/*attach to userdev*/
				if(current->pid != p->pid) return ERR_WRONG_PID;
					/*inform corresponding process*/
				PRINT("Input data available\n");
				spin_lock(&p->poll_in_lock);
				wake_up_interruptible(&p->poll_inq);
				spin_unlock(&p->poll_in_lock);
				return 0;
				break;
			}

			case USERDEV_POLL_OUT:	/*inform arrival of output data for polling*/
			{
				int id;	/*userdev id from user space*/
				struct device_record *p;

				if(copy_from_user(&id,(void*)addr,sizeof(id))) return ERR_MEMORY_FAULT;

					/*check validity of the passed userdev id*/
				if (id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[id];
				read_unlock(&userdev_rwlock);

				if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;

					/*check if the current process pid matches*/
					/*with that of the process that called*/
					/*attach to userdev*/
				if(current->pid != p->pid) return ERR_WRONG_PID;
					/*inform corresponding process*/
				PRINT("Output data available\n");
				spin_lock(&p->poll_out_lock);
				wake_up_interruptible(&p->poll_outq);
				spin_unlock(&p->poll_out_lock);
				return 0;
				break;
			}

			case USERDEV_FASYNC_SIGNAL:	/*inform about asynchronous I/O*/
			{
				int id;	/*userdev id from user space*/
				struct device_record *p;

				if(copy_from_user(&id,(void*)addr,sizeof(id))) return ERR_MEMORY_FAULT;

					/*check validity of the passed userdev id*/
				if (id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[id];
				read_unlock(&userdev_rwlock);

				if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;

					/*check if the current process pid matches*/
					/*with that of the process that called*/
					/*attach to userdev*/
				if(current->pid != p->pid) return ERR_WRONG_PID;

				spin_lock(&p->fasync_lock);
				if(p->fasync_queue != NULL)	/*send SIGIO to the waiting process*/
				{
					PRINT("Sending SIGIO(fasync) to %d\n",p->fasync_queue->fa_file->f_owner.pid);
					kill_fasync(&p->fasync_queue,SIGIO,POLL_IN);
				}
				spin_unlock(&p->fasync_lock);
				return 0;
				break;
			}

			case USERDEV_REQUEST_IRQ:/*request for handling interrupts on a given irq and send a given signal to intimate*/
			{
				struct irq_request_record rec;		/*irq request record*/
				struct device_record *p;

				if(copy_from_user(&rec,(void*)addr,sizeof(rec))) return ERR_MEMORY_FAULT;

					/*check validity of the passed userdev id*/
				if (rec.id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == rec.id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[rec.id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
									
					/*request irq from kernel so that the*/
					/*userdev interrupt handler can handle the*/
					/*interrupt by sending a signal to the*/
					/*driver process whenever an interrupt*/
					/*occurs*/
				PRINT("Requesting irq %d for device %s to signal %d\n",rec.irqno,p->devicename,rec.signo);
				ret = request_irq(rec.irqno, userdev_interrupt, SA_INTERRUPT|SA_SHIRQ, p->devicename, (void*)p);
				if (ret == 0)	/*record irq,signal numbers*/
				{
					PRINT("Request irq succeeded\n");
					atomic_set(&p->irq_no, rec.irqno);
					atomic_set(&p->sig_no, rec.signo);
				}
				else PRINT("Request irq failed\n");
				return ret;
				break;
			}

			case USERDEV_FREE_IRQ:	/*free interrupt handling */
			{
				struct irq_free_record rec;		/*free irq record*/
				struct device_record *p;

				if(copy_from_user(&rec,(void*)addr,sizeof(rec))) return ERR_MEMORY_FAULT;
	
					/*check validity of the passed userdev id*/
				if (rec.id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == rec.id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[rec.id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
									
					/*remove handling of interrupts by the*/
					/*userdev interrupt handler*/
				if(atomic_read(&p->irq_no) != -1)	/*if irq had been requested*/
				{
					PRINT("Freeing irq %d for device %s\n",atomic_read(&p->irq_no), p->devicename);
					free_irq(atomic_read(&p->irq_no), (void*)p);
					atomic_set(&p->irq_no, -1);
					atomic_set(&p->sig_no, -1);
				}
				return 0;
				break;
			}

			case USERDEV_REQUEST_DMA:/*request for dma channel*/
			{
				struct dma_request_record rec;		/*dma request record*/
				struct device_record *p;

				if(copy_from_user(&rec,(void*)addr,sizeof(rec))) return ERR_MEMORY_FAULT;

					/*check validity of the passed userdev id*/
				if (rec.id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == rec.id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[rec.id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
									
				PRINT("Requesting dma channel %d for device %s \n",rec.channel,p->devicename);
				ret = request_dma(rec.channel, p->devicename);	/*request dma channel*/
				if (ret == 0)	
				{
					PRINT("Request dma succeeded\n");
					atomic_set(&p->dma_no, rec.channel);
				}
				else PRINT("Request dma failed\n");
				return ret;
				break;
			}

			case USERDEV_FREE_DMA:	/*free dma channel */
			{
				struct dma_free_record rec;		/*free dma channel record*/
				struct device_record *p;
				int channel;

				if(copy_from_user(&rec,(void*)addr,sizeof(rec))) return ERR_MEMORY_FAULT;
	
					/*check validity of the passed userdev id*/
				if (rec.id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == rec.id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[rec.id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
					
				if((channel = atomic_read(&p->dma_no)) != -1)	/*if dma channel had been requested free it*/
				{
					int flags;

					PRINT("Freeing dma channel %d for device %s\n",channel, p->devicename);
					flags = claim_dma_lock();	/*claim lock on dma controller*/
					disable_dma(channel);
					release_dma_lock(flags);	/*release lock on dma contoller*/

					free_dma(atomic_read(&p->dma_no));	/*reset dma channel number*/
					atomic_set(&p->dma_no, -1);

					if (p->dma_buffer != NULL)	/*free dma buffer*/
						free_pages((unsigned long)p->dma_buffer, get_order(p->dma_len));
					p->dma_buffer = NULL;
					p->dma_len = 0;
				}
				return 0;
				break;
			}

			case USERDEV_START_DMA:	/*start dma operation */
			{
				struct dma_start_record rec;		/*start dma operation record*/
				struct device_record *p;
				int channel,flags;

				if(copy_from_user(&rec,(void*)addr,sizeof(rec))) return ERR_MEMORY_FAULT;
	
					/*check validity of the passed userdev id*/
				if (rec.id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == rec.id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[rec.id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
					
				channel = atomic_read(&p->dma_no);
				if(channel == -1) return ERR_INVALID_DMA;	/*if channel not requested return error*/

				/*we allocate a dma buffer(in dma memory zone) from which the dma operation will take place*/
				/*incase of write operation data is copied from user space buffer*/
				/*incase of read operation data is copied from dma_buffer using the USERDEV_COPY_DMA*/
				/*ioctl on receiving an interrupt signalling the operation has completed*/

				if((p->dma_buffer != NULL) &&
						(p->dma_len != rec.count)) 	/*free dma buffer if it exists and is of different length*/
					free_pages((unsigned long)p->dma_buffer, get_order(p->dma_len));
				p->dma_len = rec.count;		/*allocate new dma buffer*/
				p->dma_buffer = (char*)__get_dma_pages(GFP_KERNEL, get_order(p->dma_len));
				if (p->dma_buffer == NULL) return ERR_MEMORY_FAULT;

				if(rec.mode == DMA_MODE_WRITE)	/*in case of write operation copy data from user space buffer*/
					if(copy_from_user(p->dma_buffer,rec.buf,rec.count)) return ERR_MEMORY_FAULT;

				flags = claim_dma_lock();	/*claim lock on dma controller*/
				disable_dma(channel);		/*disable dma operation on the channel*/
				clear_dma_ff(channel);		/*clear the dma flipflop to accept lower byte*/
				set_dma_mode(channel, rec.mode);/*set mode of operation for dam*/
				set_dma_addr(channel, virt_to_bus(p->dma_buffer));/*set memory address for dma operation*/
				set_dma_count(channel, p->dma_len);		  /*set number of bytes to transfer*/
				enable_dma(channel);		/*enable dma on channel*/
				release_dma_lock(flags);	/*release lock on dma contoller*/
				return 0;
				break;
			}

			case USERDEV_CHECK_DMA:	/*check status of an ongoing dma operation */
			{
				struct dma_check_record rec;		/*check dma operation record*/
				struct device_record *p;
				int channel,flags;

				if(copy_from_user(&rec,(void*)addr,sizeof(rec))) return ERR_MEMORY_FAULT;
	
					/*check validity of the passed userdev id*/
				if (rec.id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == rec.id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[rec.id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
					
				channel = atomic_read(&p->dma_no);
				if(channel == -1) return ERR_INVALID_DMA;	/*if channel not requested return error*/

				flags = claim_dma_lock();
				ret = get_dma_residue(channel);
				release_dma_lock(flags);
				return ret;
				break;
			}

			case USERDEV_COPY_DMA:	/*copy data from dma buffer */
			{
				struct dma_copy_record rec;		/*copy dma operation record*/
				struct device_record *p;
				int channel,flags;

				if(copy_from_user(&rec,(void*)addr,sizeof(rec))) return ERR_MEMORY_FAULT;
	
					/*check validity of the passed userdev id*/
				if (rec.id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == rec.id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[rec.id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
					
				channel = atomic_read(&p->dma_no);
				if(channel == -1) return ERR_INVALID_DMA;	/*if channel not requested return error*/

				while(1)	/*check if dma operation has completed, if not wait for 10usec and recheck*/
				{
					flags = claim_dma_lock();
					ret = get_dma_residue(channel);
					release_dma_lock(flags);
					if(!ret) break;		/*once done break out of loop*/
					udelay(10);		/*approximation*/
				}

				if (copy_to_user(rec.buf, p->dma_buffer, p->dma_len)) return ERR_MEMORY_FAULT;
				else return 0;		/*copy data from dma_buffer to user space*/
				break;
			}

			case USERDEV_ENABLE_DMA:	/*enable dma operation */
			{
				int id;
				struct device_record *p;
				int channel,flags;

				if(copy_from_user(&id,(void*)addr,sizeof(int))) return ERR_MEMORY_FAULT;
	
					/*check validity of the passed userdev id*/
				if (id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
					
				channel = atomic_read(&p->dma_no);
				if(channel == -1) return ERR_INVALID_DMA;	/*if channel not requested return error*/

				flags = claim_dma_lock();
				enable_dma(channel);
				release_dma_lock(flags);
				PRINT("dma enabled on channel %d\n",channel);
				return 0;
				break;
			}

			case USERDEV_DISABLE_DMA:	/*disable dma operation */
			{
				int id;
				struct device_record *p;
				int channel,flags;

				if(copy_from_user(&id,(void*)addr,sizeof(int))) return ERR_MEMORY_FAULT;
	
					/*check validity of the passed userdev id*/
				if (id < 0) return ERR_INVALID_ID;

				read_lock(&userdev_rwlock);
				for(i=0;i<MAX_USERDEV_MINORS;i++)
					if (process_index(type, i) == id) break;
				if(i >= MAX_USERDEV_MINORS) 
				{
					read_unlock(&userdev_rwlock);
					return ERR_INVALID_ID;
				}
				p = attached_process[id];
				read_unlock(&userdev_rwlock);

				if(p == NULL) return ERR_INVALID_ID;
					
				channel = atomic_read(&p->dma_no);
				if(channel == -1) return ERR_INVALID_DMA;	/*if channel not requested return error*/

				flags = claim_dma_lock();
				disable_dma(channel);
				release_dma_lock(flags);
				PRINT("dma disabled on channel %d\n",channel);
				return 0;
				break;
			}

			default:	/*by default ignore*/
				return 0;
		}
	else 	/*handle ioctls for other minors*/
	{
		struct header head;
		struct ioctl_send is;
		struct ioctl_recv *ir;
		struct device_record *p;
		char *buf,*packet;
		int len,id;

		read_lock(&userdev_rwlock);
		if ((i = process_index(type,minor)) == -1) 
		{
			read_unlock(&userdev_rwlock);
			return ERR_DEVICE_NOT_ATTACHED;
		}
		p = attached_process[i];
		read_unlock(&userdev_rwlock);

		if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
		if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
		addref(p);			/*check if the process has attached to userdev and increase reference count */
		if (!IS_IOCTL_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

		is.command = cmd;		/*construct the parameter to be sent to process*/

		for(i=0;i<p->num_ioctls;i++)	/*obtain information of the ioctl from device record*/
			if(p->ioctls[i].num==cmd) break;

		if(i == p->num_ioctls) 		/*if not present*/
		{
			if(type == USERDEV_BLK)	/*check if device is a block device*/
			{
				int val;
				switch(cmd)	/*if so check the generic ioctl commands + userdev supported ones*/
				{
					case BLKGETSIZE:	/*get device size in sectors*/
						/*if implemented in kernel*/
							/*Its better to have the following implemented at driver process*/
							/*as it will have the latest values of blocksize and devicesize*/
							/*which are set into kernel arrays only at attach but are not updated*/
							/*incase they get changed, such as when device is auto configured on*/
							/*run as is the case with floppy*/

						/*{
							int size = 0;
							remref(p);
							if (blk_size[userdev_major] != NULL)
								size = blk_size[userdev_major][minor];
							size = size * blksize_size[userdev_major][minor]/hardsect_size[userdev_major][minor];
							if(copy_to_user((void*)addr, &size, sizeof(int)))
								return ERR_MEMORY_FAULT;
							return 0;
						}*/

						/*else redirect call to driver process*/
							is.data.len = sizeof(int);
						break;
						
					case BLKRRPART:		/* Reread partition table*/
						remref(p);
						return -ENOTTY;
						
					case HDIO_GETGEO:	/*get disk geometry - need to be implemented by driver*/
						is.data.len = sizeof(struct hd_geometry);
						break;
						
					default:		/*others handled by generic ioctl function for block devices*/
						remref(p);
						return blk_ioctl(inode->i_rdev, cmd, addr);
				}
			}
			else 	/*report error incase of character devices*/
			{
				remref(p);
				return ERR_UNKNOWN_IOCTL_COMMAND; 
			}
		}
		else 	/*found the ioctl, then assign the size of ioctl data*/
			is.data.len = p->ioctls[i].max_length;
		
		buf = (char*)kmalloc(is.data.len,GFP_KERNEL);
		if (buf == NULL) { remref(p); return ERR_MEMORY_FAULT; }
		if(copy_from_user(buf,(void*)addr,is.data.len))		/*copy ioctl(SET) data if given*/
		{
			kfree(buf);
			remref(p);
			return ERR_MEMORY_FAULT;
		}
		is.data.elements=buf;

		id = genID();				/*construct packet header*/
		head.opcode = OP_IOCTL;
		head.request_id = id;
		len = ioctl_send_to_char(&head,&is,&packet);	/*serialize the request packet*/
		kfree(buf);

		ret = send(packet,len,p);
		kfree(packet);
		PRINT("Sent ioctl packet\n");
		if(ret<0) { remref(p); return ret; }

		ret = receive(&head,&packet,p,id);	/*wait for response and retreive packet*/
		PRINT("Received ioctl packet\n");
		if(ret<0) { remref(p); return ret; }
		ir = ioctl_recv_from_char(packet);	/*construct response*/

		ret = ir->result;		/*copy result and ioctl(GET) data to user space*/
		if(ir->data.len >= 0)
			if(copy_to_user((void*)addr,ir->data.elements,ir->data.len)) ret = ERR_MEMORY_FAULT;

		kfree(packet);
		kfree(ir);
		remref(p);
		return ret;
	}
}


/*open function for character devices*/
int userdev_open_chr(struct inode *inode, struct file *filp)
{
	PRINT("Opening USERDEV_CHR\n");
	return userdev_open(inode, filp, USERDEV_CHR);
}


/*open function for block devices*/
int userdev_open_blk(struct inode *inode, struct file *filp)
{
	PRINT("Opening USERDEV_BLK\n");
	return userdev_open(inode, filp, USERDEV_BLK);
}


/*
//generic open function for userdev
//package in the flags and mode with which the device is to be opened and return the
//result of the device open operation
*/
int userdev_open(struct inode *inode, struct file *filp, int type)
{
    	int minor,i,len,id,ret;
	struct header head;
	struct open_send os;
	struct open_recv *or;
	char *packet;
	struct device_record *p;

	MOD_INC_USE_COUNT;			/*increment userdev module use count*/
    	minor = MINOR(inode->i_rdev);		/*using inode instead of filp, to takecare of block devices*/
    	if (minor==0) return 0;			/*nothing to do if minor = 0 */

	read_lock(&userdev_rwlock);
	if ((i = process_index(type, minor)) == -1) 
	{
		read_unlock(&userdev_rwlock);
		MOD_DEC_USE_COUNT;
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) { MOD_DEC_USE_COUNT; return ERR_DEVICE_NOT_ATTACHED; }
	if (atomic_read(&p->detach_called)) { MOD_DEC_USE_COUNT; return ERR_DEVICE_NOT_ATTACHED; }
	addref(p);				/*check if the process has attached to userdev and increase reference count */
	if (!IS_OPEN_SUPPORTED(p->func_mask)) { remref(p); MOD_DEC_USE_COUNT; return ERR_FUNC_NOT_SUPPORTED; }

	os.flags = filp->f_flags;		/*copy the flags from file structure*/
	os.mode = filp->f_mode;			/*copy open mode from file structure*/

	id = genID();				/*construct packet header*/
	head.opcode = OP_OPEN;
	head.request_id = id;
	len = open_send_to_char(&head,&os,&packet);

	ret = send(packet,len,p);		/*send header followed by packet*/
	kfree(packet);
	PRINT("Sent open packet\n");
	if(ret<0) { remref(p); MOD_DEC_USE_COUNT; return ret; }

	ret = receive(&head,&packet,p,id);	/*wait and receive response*/
	PRINT("Received open packet\n");
	if(ret<0) { remref(p); MOD_DEC_USE_COUNT; return ret; }
	or = open_recv_from_char(packet);

	ret = or->result;			/*return the result from the response packet*/
	kfree(packet);
	kfree(or);
	remref(p);
	return ret;
}


/*
//flush function of userdev
//the user driver flush function is called and the result is returned
*/
int userdev_flush (struct file *filp)
{
    	int minor,i,len,id,ret;
	struct header head;
	struct flush_send fs;
	struct flush_recv *fr;
	char *packet;
	struct device_record *p;

	PRINT("Flushing USERDEV\n");
    	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
    	if (minor==0) return 0;			/*nothing to do if minor = 0 */

	read_lock(&userdev_rwlock);
	if ((i = chr_map[minor]) == -1) 	/*defined only for char devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);				/*check if the process has attached to userdev and increase reference count */
	if (!IS_FLUSH_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	id = genID();				/*construct packet header*/
	head.opcode = OP_FLUSH;
	head.request_id = id;
	len = flush_send_to_char(&head,&fs,&packet);	/*send in the flush request packet*/

	ret = send(packet,len,p);
	kfree(packet);
	PRINT("Sent flush packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);		/*wait and receive response packet*/
	PRINT("Received flush packet\n");
	if(ret<0) { remref(p); return ret; }
	fr = flush_recv_from_char(packet);

	ret = fr->result;				/*return the result from the response packet*/
	kfree(packet);
	kfree(fr);
	remref(p);
	return ret;
}


/*close function for character devices*/
int userdev_close_chr(struct inode *inode, struct file *filp)
{
	PRINT("Closing USERDEV_CHR\n");
	return userdev_close(inode, filp, USERDEV_CHR);
}


/*close function for block devices*/
int userdev_close_blk(struct inode *inode, struct file *filp)
{
	PRINT("Closing USERDEV_BLK\n");
	return userdev_close(inode, filp, USERDEV_BLK);
}


/*
//generic close function for userdev
//send in a close request to the driver process and return back the result
//obtained in the response
*/
int userdev_close(struct inode *inode, struct file *filp, int type)
{
    	int minor,i,len,id,ret;
	struct header head;
	struct close_send cs;
	struct close_recv *cr;
	char *packet;
	struct device_record *p;

    	minor = MINOR(inode->i_rdev);	/*using inode as for block devices filp will be NULL for umount*/
    	if (minor==0) 			/*nothing to do is minor is 0*/
    	{ 
        	MOD_DEC_USE_COUNT;
        	return 0;
    	}

	read_lock(&userdev_rwlock);
	if ((i = process_index(type,minor)) == -1) 
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);			/*check if the process has attached to userdev and increase reference count */
	if (!IS_CLOSE_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	id = genID();			/*construct packet header*/
	head.opcode = OP_CLOSE;
	head.request_id = id;
	len = close_send_to_char(&head,&cs,&packet);	/*send the close request packet*/

	ret = send(packet,len,p);
	kfree(packet);
	PRINT("Sent close packet\n");
	if(ret<0) { remref(p); MOD_DEC_USE_COUNT; return ret; }

	ret = receive(&head,&packet,p,id);		/*wait and receive response packet*/
	PRINT("Received close packet\n");
	if(ret<0) { remref(p); MOD_DEC_USE_COUNT; return ret; }
	cr = close_recv_from_char(packet);

	ret = cr->result;				/*return result of the operation*/
	kfree(packet);
	kfree(cr);

	if(type == USERDEV_CHR) userdev_fasync(-1,filp,0);	/*call only if device is a character device*/
	remref(p);
	MOD_DEC_USE_COUNT;				/*decrement userdev module use count*/
	return ret;
}


/*
//fsync function of userdev
//send in a request for fsync to the driver process and return back the
//response obtained
*/
int userdev_fsync (struct file *filp, struct dentry *dentry, int datasync)
{
    	int minor,i,len,id,ret;
	struct header head;
	struct fsync_send fs;
	struct fsync_recv *fr;
	char *packet;
	struct device_record *p;

	PRINT("Fsync USERDEV\n");
    	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
    	if (minor==0) return 0;		/*do nothing if minor is 0*/

	read_lock(&userdev_rwlock);
	if ((i = chr_map[minor]) == -1) 	/*defined only for char devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);			/*check if the process has attached to userdev and increase reference count */
	if (!IS_FSYNC_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	id = genID();			/*construct packet header*/
	head.opcode = OP_FSYNC;
	head.request_id = id;
	fs.datasync = datasync;		/*fill fields*/
	len = fsync_send_to_char(&head,&fs,&packet);	/*send in fsync request*/

	ret = send(packet,len,p);
	kfree(packet);
	PRINT("Sent fsync packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);	/*wait and receive response*/
	PRINT("Received fsync packet\n");
	if(ret<0) { remref(p); return ret; }
	fr = fsync_recv_from_char(packet);

	ret = fr->result;			/*return the result of the operation*/
	kfree(packet);
	kfree(fr);
	remref(p);
	return ret;
}


/*
//fasync function of userdev, for asynchronous I/O
//add the file structure to the asynchronous I/O wait queue,
//call the driver fasync function and return the result obtained.
//Whenever I/O happens the driver process intimates by sending SIGIO signal 
//using the USERDEV_FASYNC_SIGNAL ioctl call on userdev(0)
*/
int userdev_fasync (int fd, struct file *filp, int on)		/*!!!! KERNEL DEPENDANT !!!! */
{
    	int minor,i,len,id,ret;
	struct header head;
	struct fasync_send fs;
	struct fasync_recv *fr;
	char *packet;
	struct device_record *p;

	PRINT("Fasync USERDEV\n");
    	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
    	if (minor==0) return 0;		/*if minor = 0 do nothing*/

	read_lock(&userdev_rwlock);
	if ((i = chr_map[minor]) == -1) 	/*defined only for char devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);			/*check if the process has attached to userdev and increase reference count*/
	if (!IS_FASYNC_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	spin_lock(&p->fasync_lock);
    	ret = fasync_helper (fd, filp, on, &p->fasync_queue);	/*add filp to asynchronus I/O queue of device record*/
	spin_unlock(&p->fasync_lock);

	if (on)
		PRINT("Will send SIGIO(fasync) to %d\n",p->fasync_queue->fa_file->f_owner.pid);
	PRINT("fasync_helper returned %d\n",ret);
	if (ret < 0) { remref(p); return ret; }

	id = genID();			/*construct packet header*/
	head.opcode = OP_FASYNC;
	head.request_id = id;
	len = fasync_send_to_char(&head,&fs,&packet);	/*send fasync request packet*/

	ret = send(packet,len,p);
	kfree(packet);
	PRINT("Sent fasync packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);		/*wait and receive response packet*/
	PRINT("Received fasync packet\n");
	if(ret<0) { remref(p); return ret; }
	fr = fasync_recv_from_char(packet);

	ret = fr->result;				/*return the result value obtained*/
	kfree(packet);
	kfree(fr);
	remref(p);
	return ret;
}


/*
//Readv function of userdev, to perform scatter read operation,
//Basically data of length equal to that of the cumulative request is read
//and is written in parts to at various addresses in the vector request.
*/
ssize_t userdev_readv (struct file *filp, const struct iovec *iov, unsigned long num, loff_t *lpos)
{
	int minor,id,ret;
	struct header head;
	struct read_send rs;
	struct read_recv *rr;
	char *packet;
	int len,i,j,size,offset,fault,total;
	struct device_record *p;

	PRINT("Vectored Reading USERDEV\n");
	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	if (minor == 0) return 0;		/*nothing to do if minor = 0 */
	
	read_lock(&userdev_rwlock);
	if ((i = chr_map[minor]) == -1) 	/*defined only for char devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);				/*check if the process has attached to userdev and increase reference count */
	if (!IS_READ_SUPPORTED(p->func_mask)) { remref(p);  return ERR_FUNC_NOT_SUPPORTED; }

	size = 0;
	for(j=0;j<num;j++)			/*find total size of data*/
		size = size +iov[j].iov_len;
	rs.size = size;	

	rs.offset = *lpos;			/*fill other fields*/
	rs.flags = filp->f_flags;

	id = genID();				/*construct packet header*/
	head.opcode = OP_READ;
	head.request_id = id;
	len = read_send_to_char(&head,&rs,&packet);	/*convert to string*/

	ret = send(packet,len,p);		/*send packet(string)*/
	kfree(packet);
	PRINT("Sent read packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);	/*wait and receive response string*/
	PRINT("Received read packet\n");
	if(ret<0) { remref(p); return ret; }
	rr = read_recv_from_char(packet);	/*retreive response data from string*/

	if(rr->data.len >= 0)			/*success*/
	{
		offset = total = 0;		/*offset to write at, total written*/
		fault = 0;			/*flag to denote occurance of memory fault*/
		for(j=0;j<num;j++)
		{
			if ((offset + iov[j].iov_len) > rr->data.len)
				len = rr->data.len - offset;
			else			/*copy only upto the requested length*/
				len = iov[j].iov_len;
			if(copy_to_user(iov[j].iov_base, rr->data.elements+offset, len))
			{
				fault = 1;	/*copy fault, so stop*/
				break;
			}
			total = total + len;	/*compute total written and check if exceeds request*/
			offset = offset + iov[j].iov_len;
			if(offset >= rr->data.len) break;
		}
		if(fault) ret = ERR_MEMORY_FAULT;
		else ret = total;		/*return total bytes copied*/
	}
	else ret = rr->data.len;
	*lpos = rr->offset;

	kfree(packet);
	kfree(rr);
	remref(p);
	return ret;
}


/*
//Writev function of userdev, to perform scatter write operation
//Basically the data in the iovec vector is combined into a single string before
//performing the write operation
*/
ssize_t userdev_writev (struct file *filp, const struct iovec *iov, unsigned long num, loff_t *lpos)
{
	int minor,id,ret,len;
	struct header head;
	struct write_send ws;
	struct write_recv *wr;
	char *packet,*buf;
	int i,j,size,offset;
	struct device_record *p;

	PRINT("Vectored Writing USERDEV\n");
	minor = MINOR(filp->f_dentry->d_inode->i_rdev);
	if (minor == 0) return 0;		/*nothing to do if minor = 0 */
	
	read_lock(&userdev_rwlock);
	if ((i = chr_map[minor]) == -1) 	/*defined only for char devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);				/*check if the process has attached to userdev and increase reference count*/
	if (!IS_WRITE_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	ws.offset = *lpos;			/*fill in data to send*/
	ws.flags = filp->f_flags;

	size = 0;
	for(j=0;j<num;j++)			/*find total size of data*/
		size = size +iov[j].iov_len;
	ws.data.len = size;

	buf = (char*)kmalloc(size,GFP_KERNEL);
	if (buf == NULL) { remref(p); return ERR_MEMORY_FAULT; }
	offset = 0;
	for(j=0;j<num;j++)			/*copy all portions into single string for writing*/
	{
		if(copy_from_user(buf+offset, iov[j].iov_base, iov[j].iov_len))
		{
			kfree(buf);
			remref(p);
			return ERR_MEMORY_FAULT;
		}
		offset = offset + iov[j].iov_len;
	}
	ws.data.elements=buf;

	id = genID();				/*construct packet header*/
	head.opcode = OP_WRITE;
	head.request_id = id;
	len = write_send_to_char(&head,&ws,&packet);	/*transform to string*/
	kfree(buf);

	ret = send(packet,len,p);		/*send header and data packet*/
	kfree(packet);
	PRINT("Sent write packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);	/*wait and receive response*/
	PRINT("Received write packet\n");
	if(ret<0) { remref(p); return ret; }
	wr = write_recv_from_char(packet);	/*retreive result from response*/
	ret = wr->num_written;
	*lpos = wr->offset;

	kfree(packet);
	kfree(wr);
	remref(p);
	return ret;
}


/*
//check media change function of userdev
//call the media change check function of the driver process
//and return the result obtained
*/
int userdev_check_media_change (kdev_t dev)
{
    	int minor,i,len,id,ret;
	struct header head;
	struct check_media_change_send cs;
	struct check_media_change_recv *cr;
	char *packet;
	struct device_record *p;

	PRINT("Check Media Change USERDEV\n");
    	minor = MINOR(dev);
    	if (minor==0) return 0;		/*if minor = 0 do nothing*/

	read_lock(&userdev_rwlock);
	if ((i = blk_map[minor]) == -1) 	/*defined only for block devies*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);			/*check if the process has attached to userdev and increase reference count */
	if (!IS_CHECK_MEDIA_CHANGE_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	id = genID();				/*construct packet header*/
	head.opcode = OP_CHECK_MEDIA_CHANGE;
	head.request_id = id;
	len = check_media_change_send_to_char(&head,&cs,&packet);	/*send request for calling user process function*/

	ret = send(packet,len,p);		/*construct header and send packet*/
	kfree(packet);
	PRINT("Sent check media change packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);	/*wait and receive response packet*/
	PRINT("Received check media change packet\n");
	if(ret<0) { remref(p); return ret; }
	cr = check_media_change_recv_from_char(packet);

	ret = cr->flag;				/*return the flag obtained in response packet*/
	kfree(packet);
	kfree(cr);
	remref(p);
	return ret;
}


/*
//revalidate function of userdev
//call the revalidate function of the driver process and 
//return the result obtained
*/
int userdev_revalidate (kdev_t dev)
{
    	int minor,i,len,id,ret;
	struct header head;
	struct revalidate_send rs;
	struct revalidate_recv *rr;
	char *packet;
	struct device_record *p;

	PRINT("Revalidate USERDEV\n");
    	minor = MINOR(dev);
    	if (minor==0) return 0;		/*if minor=0 do nothing*/

	read_lock(&userdev_rwlock);
	if ((i = blk_map[minor]) == -1) 	/*defined only for block devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);			/*check if the process has attached to userdev and increase reference count */
	if (!IS_REVALIDATE_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	id = genID();			/*construct packet header*/
	head.opcode = OP_REVALIDATE;
	head.request_id = id;
	len = revalidate_send_to_char(&head,&rs,&packet);	/*send the request packet*/

	ret = send(packet,len,p);
	kfree(packet);
	PRINT("Sent revalidate packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);		/*wait and receive response packet*/
	PRINT("Received revalidate packet\n");
	if(ret<0) { remref(p); return ret; }
	rr = revalidate_recv_from_char(packet);		/*return result of the operation*/

	ret = rr->result;
	kfree(packet);
	kfree(rr);
	remref(p);
	return ret;
}


int userdev_mediactl (kdev_t dev, int op, int optarg)
{
    	int minor,i,len,id,ret;
	struct header head;
	struct mediactl_send ms;
	struct mediactl_recv *mr;
	char *packet;
	struct device_record *p;

	PRINT("Mediactl USERDEV\n");
    	minor = MINOR(dev);
    	if (minor==0) return 0;		/*if minor=0 do nothing*/

	read_lock(&userdev_rwlock);
	if ((i = blk_map[minor]) == -1) 	/*defined only for block devices*/
	{
		read_unlock(&userdev_rwlock);
		return ERR_DEVICE_NOT_ATTACHED;
	}
	p = attached_process[i];
	read_unlock(&userdev_rwlock);

	if (p == NULL) return ERR_DEVICE_NOT_ATTACHED;
	if (atomic_read(&p->detach_called)) return ERR_DEVICE_NOT_ATTACHED;
	addref(p);			/*check if the process has attached to userdev and increase reference count */
	if (!IS_MEDIACTL_SUPPORTED(p->func_mask)) { remref(p); return ERR_FUNC_NOT_SUPPORTED; }

	id = genID();			/*construct packet header*/
	head.opcode = OP_MEDIACTL;
	head.request_id = id;
	ms.op = op;
	ms.optarg = optarg;
	len = mediactl_send_to_char(&head,&ms,&packet);	/*send the request packet*/

	ret = send(packet,len,p);
	kfree(packet);
	PRINT("Sent mediactl packet\n");
	if(ret<0) { remref(p); return ret; }

	ret = receive(&head,&packet,p,id);		/*wait and receive response packet*/
	PRINT("Received mediactl packet\n");
	if(ret<0) { remref(p); return ret; }
	mr = mediactl_recv_from_char(packet);		/*return result of the operation*/

	ret = mr->result;
	kfree(packet);
	kfree(mr);
	remref(p);
	return ret;
}


/*
//request function for block devices that does the read to or write from device
//The function is passed a set of requests in form of a request queue, each of which has
//a list of buffer head structures that need to be operated upon.
//
//We unlock the io_request lock, unchain the list of buffer head structures from the first request
//and combine the whole list of requests on adjacent sectors into a single request, ie cluster the set of requests
//and ask the user process to do the required operation, passing in the assocated data such as
//type of operation, starting sector number, total length of data and combined data incase of write operation.
//The reply obtained is written back to each bufferhead structure in the list before proceeding to the next
//request structure.
//
//Once done with all in the first request, we again obtain the io_request_lock to remove the request
//from the request chain and proceed to next request if any.
//
//We use a device spinlock to protect against simultaneous modification of the buffer head list of a 
//request structure by a parallel invocation of userdev_request since io_request_lock has been dropped.
*/
void userdev_request(request_queue_t *q)
{
	struct device_record *p;
	int minor,i,id,len,ret;
	unsigned long sec, size;
	struct request *req;
	struct buffer_head *bh,*ptr;
	struct header head;
	struct request_send rs;
	struct request_recv *rr;
	char *packet,*temp,*tmp;

	PRINT("Userdev_Request BEGIN\n");

	/*obtain the minor number from the first request in the request queue*/
	minor = DEVICE_NR(blkdev_entry_next_request(&q->queue_head)->rq_dev);

	read_lock(&userdev_rwlock);
	if ((i = blk_map[minor]) == -1)	/*defined only for block devices*/
	{
		read_unlock(&userdev_rwlock);
		PRINT("ERR_DEVICE_NOT_ATTACED\n");
		goto error;
	}
	p = attached_process[i];	/*obtain the device record*/
	read_unlock(&userdev_rwlock);

	/*check if the process has attached to userdev and request function is supported */
	if (p == NULL) { PRINT("ERR_DEVICE_NOT_ATTACHED\n"); goto error; }
	if (atomic_read(&p->detach_called)) { PRINT("ERR_DEVICE_NOT_ATTACHED\n"); goto error; }
	if (!IS_REQUEST_SUPPORTED(p->func_mask)) { PRINT("ERR_FUNC_NOT_SUPPORTED\n"); goto error; }

	while(1)	/*scan each request in the request queue*/
	{
		INIT_REQUEST;	/*macro to do consistency checks on the queue and move to next request; return if request queue empty*/
		spin_unlock_irq(&io_request_lock);	/*release lock on request queue as not needed till queue is to be modified*/
		addref(p);				/*increase reference count */

		do	/*do till the list of buffer head structures becomes empty*/
		{
			spin_lock_irq(&p->bh_list_lock);	/*device lock to prevent simultaneous access to the list*/

			bh = CURRENT->bh;		/*detach the list of buffer_head structures pointed to by request structure*/
			if(bh == NULL) break;		/*just incase*/
			CURRENT->bh = NULL;

			/*update the fields in the request structure*/
			sec = CURRENT->nr_sectors;			/*obtain number of sectors in the cluster*/
			CURRENT->hard_nr_sectors = CURRENT->hard_nr_sectors - sec;
			CURRENT->nr_sectors = CURRENT->nr_sectors - sec;/*decrement number of sectors = cluster size*/
			CURRENT->sector = CURRENT->sector + sec;	/*increment starting sector number*/
			CURRENT->hard_sector = CURRENT->hard_sector + sec;

			spin_unlock_irq(&p->bh_list_lock);	/*release device lock*/

			/*fill in data to be sent in the packet from buffer head structures*/
			rs.command = CURRENT->cmd;		/*command*/
			rs.sector = bh->b_rsector;		/*starting sector of the cluster*/
			rs.clustersize = sec;			/*cluster size*/
			size = sec * userdev_hardsects[minor];
			rs.length = size;			/*length in bytes*/

			if(CURRENT->cmd == WRITE)	/*acumulate data from all write requests in the clusters*/
			{				/*requests in a cluster are for adjacent sectors so we can combine them*/
				temp = (char*)kmalloc(size, GFP_KERNEL);
				tmp = temp;
				ptr = bh;
				while(ptr != NULL)	/*copy date from each buffer head structure*/
				{
					mcopy(tmp,ptr->b_data,ptr->b_size);
					tmp = tmp + ptr->b_size;
					ptr = ptr->b_reqnext;
				}
				rs.data.len = size;	/*fill details in packet*/
				rs.data.elements = temp;
			}
			else
			{
				rs.data.len = 0;	/*nothing to send incase of read*/
				rs.data.elements = NULL;
			}
				
			PRINT("start sector %ld command %d size %ld cluster %d\n",bh->b_rsector, CURRENT->cmd, size, sec);

			id = genID();			/*construct packet header*/
			head.opcode = OP_REQUEST;
			head.request_id = id;
			len = request_send_to_char(&head,&rs,&packet);	/*serialize the request packet*/
			if(CURRENT->cmd == WRITE) kfree(temp);	/*free memory allocated for combining requests, incase of write operation*/

			ret = send(packet,len,p);
			kfree(packet);
			PRINT("Sent request packet\n");
			if(ret<0) 			/*if send failed*/
			{ 
				remref(p); 		/*decrement reference count*/
				PRINT("send error %d\n",ret); 
				while(bh != NULL)	/*intimate failure to all buffer head structures in the list*/
				{		
					ptr = bh;
					bh = bh->b_reqnext;
					ptr->b_reqnext = 0;	/*IMPORTANT: detach each structure from list,otherwise can lead to PANIC*/
					ptr->b_end_io(ptr, 0);	/*intimate failure*/
				}
				goto error; 		/*return error for all requests to follow*/
			}

			ret = receive(&head,&packet,p,id);	/*wait for response and retreive packet*/
			PRINT("Received request packet\n");
			if(ret<0) 			/*if receive failed*/
			{ 
				remref(p); 
				PRINT("receive error %d\n",ret); 
				while(bh != NULL)	/*intimate failure to all buffer head structures in the list*/
				{
					ptr = bh;
					bh = bh->b_reqnext;
					ptr->b_reqnext = 0;	/*IMPORTANT: to avoid kernel PANIC*/
					ptr->b_end_io(ptr, 0);	/*intimate failure*/
				}
				goto error; 
			}
			rr = request_recv_from_char(packet);	/*construct response*/

			ret = rr->result;		/*copy result and data to buffer head incase of read*/
			PRINT("Result = %d\n",ret);
			if(CURRENT->cmd == READ)	
			{
				if(ret && rr->data.len > 0)	/*distribute the data read to all the buffer head structures, one by one*/
				{
					size = rr->data.len;
					temp = rr->data.elements;
					while(bh != NULL)
					{
						if(size < bh->b_size) break;	/*in case short of data break*/
						mcopy(bh->b_data, temp, bh->b_size);
						size = size - bh->b_size;
						temp = temp + bh->b_size;

						ptr = bh;
						bh = bh->b_reqnext;
						ptr->b_reqnext = 0;
						ptr->b_end_io(ptr,ret);
					}
				}
				while(bh != NULL)	/*incase buffer head structures that remain, intimate failure for them*/
				{
					ptr = bh;
					bh = bh->b_reqnext;
					ptr->b_reqnext = 0;
					ptr->b_end_io(ptr, 0);	/*intimate failure for current structure*/
				}
			}
			else	/*incase of write*/
			{
				while(bh != NULL)
				{
					ptr = bh;
					bh = bh->b_reqnext;
					ptr->b_reqnext = 0;
					ptr->b_end_io(ptr,ret);	/*intimate result of operation*/
				}
			}

			kfree(packet);
			kfree(rr);
		}while(CURRENT->bh != NULL);	/*till no more buffer head structures exist in the list*/

		remref(p);	/*once done with list of a request decrement reference count*/
		spin_lock_irq(&io_request_lock);/*obtain io_request lock*/
		req = CURRENT;				
		blkdev_dequeue_request(req);	/*remove request from request queue*/
		end_that_request_last(req);	/*release the request back to the system*/
	}

	/*incase of an error we complete scanning the request queue by returning failure to all*/
	/*requests and their associated buffer_head lists left in the queue*/
error:
	PRINT("error in serving requests\n");
	while(1)
	{
		INIT_REQUEST;	/*get next unfullfilled request*/
		PRINT("terminating request\n");
		end_request(0);	/*intimate failure*/
	}
}


/*
//function to attach a driver process to userdev given the attach record and
//minor number of the device to attach to
*/
static int attach_device(struct attach_record *att_rec, int minor, int type)
{
	int i,ret;
	struct device_record *dev_rec;
	struct file *fp;
	char *name;
	struct blkdev_data data;
	
	dev_rec = (struct device_record*)kmalloc(sizeof(struct device_record),GFP_KERNEL);

	write_lock(&userdev_rwlock);
	for(i=0;i<MAX_USERDEV_PROCESSES;i++)	/*obtain a free userdev id*/
		if (attached_process[i] == NULL) break;
	
	if (i == MAX_USERDEV_PROCESSES)
	{
		write_unlock(&userdev_rwlock);
		ret =  ERR_TOO_MANY_PROCESSES;
		goto cleanup;
	}

	attached_process[i] = dev_rec;
	if(type == USERDEV_CHR)		/*set minor -- userdev id mapping based on type of device*/
		chr_map[minor] = i;
	else
		blk_map[minor] = i;
	write_unlock(&userdev_rwlock);
	
	dev_rec->type = type;		/*set type of device*/
	fp = fdtofp(att_rec->readfd);	/*obtain file structure from file descriptor as it is process independent*/
	if (fp == NULL)
	{
		ret = ERR_INVALID_FILE_DESCRIPTOR;
		goto cleanup;
	}
	dev_rec->readfp = fp;	

	/*keep reference to file operations for calling release during detach*/
	/*reference to file operations is static*/
	dev_rec->readops = fp->f_op;	

	/*store a reference to the dentry,*/
	/*for checking if the file structure has been deallocated due to a crash of user process*/
	/*and for removing the dentry in the detach operation*/
	dev_rec->read_dentry = fp->f_dentry;

	fp = fdtofp(att_rec->writefd);	/*do the same with the write file descriptor too*/
	if (fp == NULL)
	{
		ret = ERR_INVALID_FILE_DESCRIPTOR;
		goto cleanup;
	}
	dev_rec->writefp = fp;
	dev_rec->writeops = fp->f_op;		/*keep reference to file operations*/
	dev_rec->write_dentry = fp->f_dentry;	/*same as the read descriptor case*/

	if(att_rec->namelen < 0)
	{
		ret = ERR_INVALID_RECORD;
		goto cleanup;
	}				/*copy and store from user space, the device filename*/
	dev_rec->devicename=(char*)kmalloc(att_rec->namelen+1,GFP_KERNEL);		
	if(copy_from_user(dev_rec->devicename,att_rec->devicename,att_rec->namelen))
	{
		kfree(dev_rec->devicename);
		ret = ERR_MEMORY_FAULT;
		goto cleanup;
	}
	dev_rec->devicename[att_rec->namelen] = 0;

	dev_rec->minor = minor;		/*store minor number*/
	dev_rec->pid = current->pid;	/*store attaching process pid*/
	if(att_rec->num_ioctls < 0)
	{
		kfree(dev_rec->devicename);
		ret = ERR_INVALID_RECORD;
		goto cleanup;
	}
	dev_rec->num_ioctls = att_rec->num_ioctls;	/*copy and store the list of ioctls supported*/

	dev_rec->ioctls = (struct ioctl_element*)kmalloc(sizeof(struct ioctl_element)*att_rec->num_ioctls,GFP_KERNEL);
	if(copy_from_user(dev_rec->ioctls,att_rec->ioctls,sizeof(struct ioctl_element)*att_rec->num_ioctls))
	{
		kfree(dev_rec->ioctls);
		kfree(dev_rec->devicename);
		ret = ERR_MEMORY_FAULT;
		goto cleanup;
	}

	if(type == USERDEV_BLK)
	{
		if(att_rec->data)
		{
			if(copy_from_user(&data, att_rec->data, sizeof(struct blkdev_data)))
			{
				kfree(dev_rec->ioctls);
				kfree(dev_rec->devicename);
				ret = ERR_MEMORY_FAULT;
				goto cleanup;
			}
			userdev_sizes[minor] = data.devsize;
			userdev_blksizes[minor] = data.blksize;
			userdev_hardsects[minor] = data.sectsize;
			userdev_maxreadaheads[minor] = data.maxreadahead;
			userdev_maxsectors[minor] = data.maxsectors;
		}
		PRINT("device size = %d kilobytes\n",userdev_sizes[minor]);
		PRINT("block size = %d bytes\n",userdev_blksizes[minor]);
		PRINT("sector size = %d bytes\n",userdev_hardsects[minor]);
		PRINT("max read ahead size = %d pages\n",userdev_maxreadaheads[minor]);
		PRINT("max request size = %d sectors\n",userdev_maxsectors[minor]);
		register_disk(NULL, MKDEV(userdev_major,minor), 1, &userdev_bdops, userdev_sizes[minor]<<1);
	}

	dev_rec->func_mask = att_rec->func_mask;	/*store the function mask and initialise rest of the variables*/
	dev_rec->pending_requests = NULL;		/*pending requests list with read-write lock*/
	dev_rec->req_list_lock = RW_LOCK_UNLOCKED;

	init_waitqueue_head(&dev_rec->poll_inq);	/*input polling queue and spinlock*/
	spin_lock_init(&dev_rec->poll_in_lock);

	init_waitqueue_head(&dev_rec->poll_outq);	/*ouptut polling queue and spinlock*/
	spin_lock_init(&dev_rec->poll_out_lock);

	dev_rec->fasync_queue = NULL;			/*asynchronous I/O queue and spinlock*/
	spin_lock_init(&dev_rec->fasync_lock);

	spin_lock_init(&dev_rec->bh_list_lock);	/*buffer head list lock for block device request*/

	atomic_set(&dev_rec->read_flag, 0);	/*nobody reading actively*/
	atomic_set(&dev_rec->refcount, 0);	/*nobody has a reference to the device record*/
	atomic_set(&dev_rec->detach_called, 0);	/*request for detach not encountered*/
	
	name = (char*)kmalloc(15,GFP_KERNEL);
	sprintf(name,"dev%d",minor);	/*make an entry in proc file system under userdev proc root*/
	if(type == USERDEV_CHR)
		dev_rec->proc_entry = new_proc_entry((const char*)name, 0, userdev_chr_base, device_read_proc, dev_rec);
	else
		dev_rec->proc_entry = new_proc_entry((const char*)name, 0, userdev_blk_base, device_read_proc, dev_rec);
	kfree(name);
	
	atomic_set(&dev_rec->irq_no, -1);	/*no irq requested*/
	atomic_set(&dev_rec->sig_no, -1);	/*no signal number specified*/
	atomic_set(&dev_rec->dma_no, -1);	/*no dma channel*/
	dev_rec->dma_buffer = NULL;		/*no dma buffer*/
	dev_rec->dma_len = 0;

	/*The reference count of the dentry is increased so that it does not get deallocated*/
	/*along with the file structure when the driver process crashes and release is called*/
	/*on the file structure. The reference count is decreased and the dentry is removed in*/
	/*the detach operation by calling dput() function*/
	/*dentry and inode structures are retained so that we can check if the process has crashed*/
	/*and inode deallocation is under our control, especially when private data of the inode*/
	/*is being used as in the case of pipes, which is released in the pipe_release call*/
	atomic_inc(&dev_rec->readfp->f_dentry->d_count);
	atomic_inc(&dev_rec->writefp->f_dentry->d_count);

	/*open is called to indicate that an additional refernce is kept for the file structure*/
	/*so that a wake up call comes to a blocked operation when the driver process crashes*/
	dev_rec->readfp->f_op->open(dev_rec->readfp->f_dentry->d_inode, dev_rec->readfp);
	dev_rec->writefp->f_op->open(dev_rec->writefp->f_dentry->d_inode, dev_rec->writefp);

	MOD_INC_USE_COUNT;		/*increment userdev module use count*/
	return i;

cleanup:
	write_lock(&userdev_rwlock);
	kfree(dev_rec);
	attached_process[i] = NULL;
	if(type == USERDEV_CHR)		/*set minor -- userdev id mapping based on type of device*/
		chr_map[minor] = -1;
	else
		blk_map[minor] = -1;
	write_unlock(&userdev_rwlock);
	return ret;
}


/*function to detach from userdev given the detach record and the device minor*/
static int detach_device(struct detach_record *det_rec)
{
	struct device_record *dev_rec;
	int i;

	read_lock(&userdev_rwlock);
	dev_rec = attached_process[det_rec->id];
	read_unlock(&userdev_rwlock);

	atomic_set(&dev_rec->detach_called, 1);	/*register the request for detach, it may be deferred till reference count = 0*/

	PRINT("Readfp.inode = %ld\n",dev_rec->readfp->f_dentry->d_inode->i_ino);
	PRINT("Writefp.inode = %ld\n",dev_rec->writefp->f_dentry->d_inode->i_ino);
	PRINT("Device = %s\n",dev_rec->devicename);
	PRINT("Device type = %d\n",dev_rec->type);
	PRINT("Numioctls = %d\n",dev_rec->num_ioctls);
	PRINT("mask = %d\n",dev_rec->func_mask);
	PRINT("minor = %d\n",dev_rec->minor);
	PRINT("pid = %d\n",dev_rec->pid);
	PRINT("irq = %d\n", atomic_read(&dev_rec->irq_no));
	PRINT("signal = %d\n",atomic_read(&dev_rec->sig_no));
	PRINT("dma = %d\n", atomic_read(&dev_rec->dma_no));
	for(i=0;i<dev_rec->num_ioctls;i++)
		PRINT("ioctl %u, size = %d\n",dev_rec->ioctls[i].num,dev_rec->ioctls[i].max_length);
	
	userdev_wake_up_all(dev_rec);		/*wake up processes waiting for responses*/
	if(atomic_read(&dev_rec->refcount) == 0)	/*if reference count falls to 0 delete device record of the process*/
		free_device(det_rec->id);
	return 0;	
}


/*
//function to send a serialized packet to the corresponding driver process 
//through the write file structure stored with the
//process' device record identified by the minor number
*/
static int send(char *packet, int len, struct device_record *p)
{
	int i,ret;
	mm_segment_t fs;

	if ((p->writefp == NULL) || (p->writefp->f_op == NULL) || (p->writefp->f_op->write == NULL) ||
			(p->writefp->f_dentry != p->write_dentry)) 	/*check if file structure has been deallocated*/
	{
		PRINT("Write communication failure\n");
		atomic_set(&p->detach_called, 1);	/*check if the packet can be sent, otherwise register that the*/
		return ERR_COMM_FAILURE;	/*process needs to be detached at the earliest and return an error*/
	}

	fs = get_fs();		/*change the fs segment to point to the data segment since write file operation*/
	set_fs(get_ds());	/*requires data in user space while we pass it from kernel space*/
	
	ret = p->writefp->f_op->write(p->writefp, packet, len, &p->writefp->f_pos);
	if(ret != len)		/*write header using the file structure write operation*/
	{
		PRINT("Header+Packet write failed %d\n",ret);
		atomic_set(&p->detach_called, 1);	/*if write failed, set fs back to its original value and register a*/
		set_fs(fs);		/*request to detach the process*/
		return ERR_PROCESS_DETACHED;
	}

	set_fs(fs);		/*reset value of fs*/
	return ret;
}


/*function to add a wait_entry record to the list kept with the device record*/
static void add_wait_entry(struct device_record *rec, struct wait_entry *entry)
{
	struct wait_entry *ptr;
	
	if((rec==NULL)||(entry==NULL)) return;	/*if either is null return*/
	entry->next = NULL;

	write_lock(&rec->req_list_lock);
	ptr = rec->pending_requests;		/*get a reference to the list of pending requests*/
	if(ptr == NULL)
	{
		rec->pending_requests = entry;	/*if first entry then assign it as pending requests list*/
		write_unlock(&rec->req_list_lock);
		return;
	}
	
	while(ptr->next != NULL)		/*else append the entry to the end of the list*/
		ptr = ptr->next;
	ptr->next = entry;
	write_unlock(&rec->req_list_lock);
}


/*function to remove a wait_entry from the device's list of pending requests*/
static void del_wait_entry(struct device_record *rec, struct wait_entry *entry)
{
	struct wait_entry *ptr,*pqr;
	
	if((rec==NULL)||(entry==NULL)) return;	/*if either is null return*/

	write_lock(&rec->req_list_lock);
	ptr = rec->pending_requests;		/*obtain reference to list of pending requests*/
	if(ptr->req_id == entry->req_id)	/*incase its the first entry*/
	{
		rec->pending_requests = ptr->next;	/*update pending requests reference*/
		ptr->next = NULL;
		write_unlock(&rec->req_list_lock);
		return;
	}
	
	pqr = ptr;
	ptr = ptr->next;
	while(ptr!=NULL)		/*trace through the list and locate the entry*/
	{
		if (ptr->req_id == entry->req_id) break;
		pqr = ptr;
		ptr = ptr->next;
	}
	
	if(ptr!=NULL)			/*if found update pointers to remove it from the list*/
	{
		pqr->next = ptr->next;
		ptr->next = NULL;
	}
	write_unlock(&rec->req_list_lock);
}


/*
//function to wake up a process that is waiting for the response of a request
//identified by the request id and set the reference to the response packet
//and set the reason for waking it up
*/
static void userdev_wake_up_one(int id, char *pkt, int size, int why, struct device_record *rec) 	/*!!!! KERNEL DEPENDANT !!!! */
{
	struct wait_entry *entry;

	if(rec==NULL) return;
	PRINT("waking up %d\n",id);

	read_lock(&rec->req_list_lock);
	entry = rec->pending_requests;	/*search for the entry with the given request id in the pending requests list*/
	while(entry!=NULL)
	{
		if(entry->req_id == id) break;
		entry = entry->next;
	}
	if(entry == NULL)
	{
		read_unlock(&rec->req_list_lock);
		return;	/*if not found return*/
	}

	entry->reason=why;		/*set the reason for waking up, reference to response packet*/
	entry->data=pkt;
	entry->size=size;
			
	wake_up(&entry->bed);		/*wake up the associated process sleeping on the wait_entry*/
	read_unlock(&rec->req_list_lock);
}


/*
//function to wake up the first process in the pending requests list, to become
//the active reader, and to moniter and receive response packets
*/
static void userdev_wake_up_first(struct device_record* rec)	/*!!!! KERNEL DEPENDANT !!!! */
{
	struct wait_entry *entry;

	if(rec==NULL) return;
	PRINT("waking up first\n");

	read_lock(&rec->req_list_lock);
	entry = rec->pending_requests;	/*obtain first wait entry in pending requests list, that has not got its packet*/
	if(entry == NULL)
	{
		read_unlock(&rec->req_list_lock);
		return;
	}

	while(entry != NULL)
	{
		if(entry->reason != RECEIVED_DATA)	/*skip those that have got their responses*/
		{
			entry->reason = APPOINTED_READER;	/*set reason for waking up*/
			wake_up(&entry->bed);		/*wake up the corresponding process*/
			PRINT("Woke up %d\n",entry->req_id);
			break;
		}
		entry = entry->next;
	}
	read_unlock(&rec->req_list_lock);
}


/*
//function to wake up all processes that have pending requests for a device
//Generally called when the process detaches from userdev in case of error
//We store the reference to next entry before waking up an entry bcoz once woken
//up, the other thread might change the next entry reference, while
//removing the entry from the pending request list
*/
static void userdev_wake_up_all(struct device_record *rec)		/*!!!! KERNEL DEPENDANT !!!! */
{
	struct wait_entry *entry, *nextentry;

	if(rec == NULL) return;
	PRINT("waking up all\n");

	read_lock(&rec->req_list_lock);
	entry = rec->pending_requests;
	while(entry != NULL)	/*run through the list of pending requests*/
	{
		nextentry = entry->next;		/*store next entry reference before waking up*/
		if(entry->reason != RECEIVED_DATA)	/*ignore those that have already got their responses*/
		{
			entry->reason = END_CONNECTION;	/*set reason as end of connection*/
			PRINT("Waking up %d\n",entry->req_id);
			wake_up(&entry->bed);
		}
		entry = nextentry;	/*go to next entry*/
	}
	read_unlock(&rec->req_list_lock);
}


/*
//function to wait and receive the response packet from the user process
//A wait entry is constructed and if nobody is actively monitoring and reading
//the incoming stream, then the current process becomes the active reader and
//whenever a packet arrives, based on the packet id a process is woken up and
//the response packet is delivered to it. If the packet is for itself then the
//active process appoints another to take its role and returns from the
//function call. 
//In case somebody is already monitoring the incoming stream then the current
//process goes to sleep on the wait entry, to be woken up when its response
//packet arrives. When this happens the function call returns
//The function takes a pointer to header structure to be filled and a pointer
//to get back the response packet along with the minor number and the id of the
//expected response (equal to that of the request sent earlier)
*/
static int receive(struct header *head, char **packet, struct device_record *p, int id)	/*!!!! KERNEL DEPENDANT !!!! */
{
	int i,ret,ret1,ret2;
	mm_segment_t fs;
	char *pkt;
	struct wait_entry *we;

	if ((p->readfp == NULL) || (p->readfp->f_op == NULL) || (p->readfp->f_op->read == NULL) ||
			(p->readfp->f_dentry != p->read_dentry))	/*check if file structure has been deallocated*/
	{
		PRINT("Read communication failure\n");
		atomic_set(&p->detach_called, 1);	/*check if the read mechanism is in place, or else register*/
		return ERR_COMM_FAILURE;	/*a request to detach the process and return an error*/
	}

	if(current->sigpending != 0)		/*if a signal has been received send message to driver saying so*/
	{
		char str[]="signal received";
		userdev_message(p,SIGNAL_PENDING,id,sizeof(str),str);
	}
	
	we = (struct wait_entry*)kmalloc(sizeof(struct wait_entry),GFP_KERNEL);
	we->req_id = id;			/*initialise a wait entry structure*/
	init_waitqueue_head(&we->bed);
	we->reason = NO_REASON;

	while(atomic_read(&p->read_flag))	/*check read flag to see if a active reader exists*/
	{
		PRINT("Somebody reading so sleeping(%d)\n",id);
		add_wait_entry(p,we);		/*if so then add wait entry to list of pending requests for the device*/
		interruptible_sleep_on(&we->bed);	/*and sleep on the wait entry*/
		
		PRINT("Woken up %d due to %d\n",we->req_id,we->reason);
		del_wait_entry(p,we);			/*remove entry from the pending requests list*/

		if(we->reason == END_CONNECTION)	/*when woken up, find out the reason.*/
		{		
			kfree(we);			/*incase connection closed, then return error*/
			return ERR_PROCESS_DETACHED;
		}
		else if(we->reason == NO_REASON)	/*incase no reason specified, must have woken up due to signal*/
		{
			kfree(we);			/*in which case send a message saying so and return error*/
			{
				char str[]="signal received";	/*send message to driver process*/
				userdev_message(p,SIGNAL_PENDING,id,sizeof(str),str);
			}
			return ERR_SIGNAL_RECEIVED;
		}
		if(we->reason == RECEIVED_DATA) break;	/*incase received data then jump over, otherwise recheck read flag*/
	}						/*to see if we were woken up to become active reader*/

	if((!atomic_read(&p->read_flag)) && (we->reason != RECEIVED_DATA))
		/*if no active reader present and we have'nt received our response then*/
	{
		atomic_set(&p->read_flag, 1);	/*set the read flag*/
		do
		{
			fs = get_fs();		/*change the fs segment to point to the data segment since write file operation*/
			set_fs(get_ds());	/*requires data in user space while we pass it from kernel space */
	
			PRINT("Waiting for header(%d)\n",id);
			ret1 = p->readfp->f_op->read(p->readfp, (char*)head, 
					sizeof(struct header), &p->readfp->f_pos);	/*read header, will block*/
			PRINT("Received header %d\n",ret1);
			if (ret1 < 0)				/*if error occured then*/
			{
				set_fs(fs);			/*reset fs segment*/
				kfree(we);
				atomic_set(&p->read_flag, 0);			/*reset read flag*/
				userdev_wake_up_first(p);	/*wake up first process in pending list to become active reader*/
				{
					char str[]="signal received";	/*send message to driver process*/
					userdev_message(p,SIGNAL_PENDING,id,sizeof(str),str);
				}
				return ret1;
			}
			if(ret1 != sizeof(struct header))	/*if header incompletely read then*/
			{
				PRINT("Header read failed %d\n",ret1);
				atomic_set(&p->detach_called, 1);	/*register request for process detach*/
				kfree(we);
				set_fs(fs);			/*reset fs segment and return error*/
				return ERR_PROCESS_DETACHED;
			}

			if (head->packet_size < 0)	/*if packet size expected is errorneous*/
			{
				PRINT("Invalid packet size in header\n");
				set_fs(fs);		/*reset fs segment*/
				kfree(we);
				atomic_set(&p->read_flag, 0);	/*wake up first process in pending list and return error*/
				userdev_wake_up_first(p);
				return ERR_INVALID_HEADER;
			}

			pkt = (char*)kmalloc(head->packet_size+1,GFP_KERNEL);	/*allocate memory for incoming packet*/
			if (pkt == NULL) return ERR_MEMORY_FAULT;
			ret2 = p->readfp->f_op->read(p->readfp, pkt, head->packet_size, &p->readfp->f_pos);	/*read packet*/
			PRINT("Received packet %d\n",ret2);
			if (ret2 < 0)			/*if error occured*/
			{
				kfree(pkt);
				set_fs(fs);		/*reset fs segment*/
				kfree(we);
				atomic_set(&p->read_flag, 0);		/*reset read flag and wake up first process in pending list*/
				userdev_wake_up_first(p);
				{
					char str[]="signal received";	/*send message to driver process*/
					userdev_message(p,SIGNAL_PENDING,id,sizeof(str),str);
				}
				return ret2;
			}
			if(ret2 != head->packet_size)	/*if amount read != amount expected*/
			{
				PRINT("Packet read failed %d\n",ret2);
				kfree(pkt);
				atomic_set(&p->detach_called, 1);	/*register request for process detach*/
				set_fs(fs);		/*reset fs segment and return error*/
				kfree(we);
				return ERR_PROCESS_DETACHED;
			}

			set_fs(fs);			/*reset fs segment*/
			
			if(head->request_id != id)	/*if the received packet belongs to someone else*/
			{
				PRINT("Got some other's packet %d\n",head->request_id);
				userdev_wake_up_one(head->request_id,pkt,head->packet_size,
						RECEIVED_DATA,p);	/*wake him up giving the received packet*/
			}
			else
			{
				PRINT("Got reply for self\n");
				atomic_set(&p->read_flag, 0);		/*if for self, then set the packet into ones wait entry*/
				we->data = pkt;
				we->size = head->packet_size;
				we->reason = RECEIVED_DATA;
				userdev_wake_up_first(p);/*reset read flag and wakeup the first process in pending list for reading*/
			}
		}while(head->request_id != id);		/*do until you get your packet*/
	}

	*packet = we->data;	/*copy the reference to received packet into the function argument and return*/
	ret = we->size;
	kfree(we);
	PRINT("Returning from receive %d\n",id);
	return ret;
}


/*
//function to create a new proc file system entry under the given parent
//directory with the given name and mode. A pointer is also passed that 
//identifies the node and of which info is available at the node
*/
static struct proc_dir_entry *new_proc_entry(const char *name, mode_t mode, struct proc_dir_entry *parent, 
				int (*readproc)(char*, char**, off_t, int, int*, void*), void *ptr)
{						/*!!!! KERNEL DEPENDANT !!!! */
    	struct proc_dir_entry *ent;
			 
    	if (mode == S_IFDIR)			/*if directory then set mode as dr-xr-xr-x */
        	mode |= S_IRUGO | S_IXUGO;
    	else if (mode == 0)			/*else default mode for regular file, r--r--r-- */
        	mode = S_IFREG | S_IRUGO;
				
    	ent = create_proc_entry(name, mode, parent);	/*register proc entry under the given parent entry*/
	if (ent)
	{
		if(ptr) ent->data = ptr;
		if(readproc) ent->read_proc = readproc;
	}
    	return ent;
}


/*
//function to remove a given proc entry from the proc file sytem giving the
//parent's proc entry and the inode number of the proc entry
*/
static void del_proc_entry(char *name, struct proc_dir_entry *parent)
{						/*!!!! KERNEL DEPENDANT !!!! */
	remove_proc_entry(name,parent);
}


/*function to initialize and create the proc entry root of userdev*/
int userdev_proc_init()							/*!!!! KERNEL DEPENDANT !!!! */
{
    	userdev_chr_base = new_proc_entry(USERDEV_CHR_NAME, S_IFDIR, &proc_root, NULL, NULL);	/*userdev root proc entry*/
    	if (userdev_chr_base == NULL)
   	{
        	printk("Unable to initialise /proc/%s.\n",USERDEV_CHR_NAME);
        	return 0;
    	}
	else printk("/proc/%s initialised\n",USERDEV_CHR_NAME);

    	userdev_blk_base = new_proc_entry(USERDEV_BLK_NAME, S_IFDIR, &proc_root, NULL, NULL);	/*userdev root proc entry*/
    	if (userdev_blk_base == NULL)
   	{
        	printk("Unable to initialise /proc/%s.\n",USERDEV_BLK_NAME);
        	return 0;
    	}
	else printk("/proc/%s initialised\n",USERDEV_BLK_NAME);

	return 1;
}


/*function to remove the userdev root proc entry from the proc file system*/
void userdev_proc_cleanup()						/*!!!! KERNEL DEPENDANT !!!! */
{
    	if (userdev_chr_base)
   	{
		del_proc_entry(USERDEV_CHR_NAME, &proc_root);
        	userdev_chr_base = NULL;
		printk("/proc/%s uninitialised\n",USERDEV_CHR_NAME);
   	}

    	if (userdev_blk_base)
   	{
		del_proc_entry(USERDEV_BLK_NAME, &proc_root);
		userdev_blk_base = NULL;
		printk("/proc/%s uninitialised\n",USERDEV_BLK_NAME);
   	}
}


/*
//function to print the contents of a device record when the corresponding proc
//file system node is read. The pointer to the device record gets passed to the
//function as it had been set when the node was created
*/
static int device_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len = 0,i,id;
	struct device_record *dev = (struct device_record *)data;
	
	if (!dev) return len;
	len += sprintf(page+len,"Details of user level driver process attached to userdev\n");
	len += sprintf(page+len,"--------------------------------------------------------\n");

	len += sprintf(page+len,"device type = ");
	if(dev->type == USERDEV_CHR) len += sprintf(page+len,"CHAR\n");
	else len+= sprintf(page+len,"BLOCK\n");

	read_lock(&userdev_rwlock);
	id = process_index(dev->type, dev->minor);
	read_unlock(&userdev_rwlock);
	len += sprintf(page+len,"userdev id = %d\n",id);

	len += sprintf(page+len,"minor = %d\n",dev->minor);
	len += sprintf(page+len,"device name = %s\n",dev->devicename);
	len += sprintf(page+len,"process id = %ld\n",dev->pid);
	len += sprintf(page+len,"irq no: = %d\n",atomic_read(&dev->irq_no));
	len += sprintf(page+len,"signal no: = %d\n",atomic_read(&dev->sig_no));
	len += sprintf(page+len,"dma channel no: = %d\n",atomic_read(&dev->dma_no));

	len += sprintf(page+len,"ioctls available are:\n");
	for(i=0;i<dev->num_ioctls;i++)
		len += sprintf(page+len,"\t ioctl %u of max length %d\n",dev->ioctls[i].num,dev->ioctls[i].max_length);

	len += sprintf(page+len,"functions implemented are:\n");
	if(IS_READ_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint read(int id, char *data, unsigned int size, long long off);\n");
	if(IS_WRITE_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint write(int id, char *data, int len, long long off);\n"); 
	if(IS_POLL_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint poll(int id);\n"); 
	if(IS_IOCTL_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint ioctl(int id, int command, char* data);\n"); 
	if(IS_OPEN_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint open(int id,unsigned int flags);\n"); 
	if(IS_FLUSH_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint flush(int id);\n"); 
	if(IS_CLOSE_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint close(int id);\n"); 
	if(IS_FSYNC_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint fsync(int id);\n"); 
	if(IS_FASYNC_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint fasync(int id);\n"); 
	if(IS_CHECK_MEDIA_CHANGE_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint check_media_change(int id);\n"); 
	if(IS_REVALIDATE_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint revalidate(int id);\n"); 
	if(IS_MEDIACTL_SUPPORTED(dev->func_mask)) len += sprintf(page+len,"\tint mediactl(int id, int op, int optarg);\n"); 
	
	*start = 0;
	*eof = 1;
	return len;
}

