/*file contains implementation of the user level driver library*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <stdio.h>
# include <unistd.h>
# include <sys/stat.h>
# include <errno.h>
# include "usrdriv.h"
# include "common.h"    /*for data structures and functions shared with kernel code*/ 
# include "ioctls.h"	/*for list of userdev ioctl numbers*/

# define MAX_DRIVERS	10	/*maximum number of minor supported by any driver process*/

/*
//data structure to keep track of information of each attachment
//made by the driver process with userdev driver
*/
static struct __userdev_info
{
	int userdev_id;			/*userdev id, obtained as a result of a call to attach to userdev*/
	int userdev_minor;		/*minor number*/
	int userdev_readfd[2];		/*pair of pipe descriptors for communication from kernel to process*/
	int userdev_writefd[2];		/*pair of pipe descriptors for communication from process to kernel*/
	struct userdev_operations *uops;/*userdev operations supported*/
	int signo;			/*signal number for interrupts*/
	int fd0;
} *__ud_info[MAX_DRIVERS] = { [0 ... MAX_DRIVERS-1] = NULL };

static int __ud_count = 0;		/*number of minors attached to userdev*/
extern int errno;
char userdev_chr[] = USERDEV_CHARDEV;	/*character and block device filenames*/
char userdev_blk[] = USERDEV_BLOCKDEV;


/*function that returns the device filename based on the type of device*/
static inline char *getUserdevName(int type)
{
	return (type == USERDEV_CHAR) ? userdev_chr : userdev_blk;
}


/*
//function to construct userdev function mask from userdev operations structure
//to tell userdev the list of operations supported by the driver
*/
static int constructMask(struct userdev_operations *uop)
{
	int mask = 0;
	
	if (uop == NULL) return mask;
	if (uop->read != NULL) mask = mask | 0x8000;
	if (uop->write != NULL) mask = mask | 0x4000;
	if (uop->poll != NULL) mask = mask | 0x2000;
	if (uop->ioctl != NULL) mask = mask | 0x1000;
	if (uop->open != NULL) mask = mask | 0x0400;
	if (uop->flush != NULL) mask = mask | 0x0200;
	if (uop->close != NULL) mask = mask | 0x0100;
	if (uop->fsync != NULL) mask = mask | 0x0080;
	if (uop->fasync != NULL) mask = mask | 0x0040;
	if (uop->check_media_change != NULL) mask = mask | 0x0004;
	if (uop->revalidate != NULL) mask = mask | 0x0002;
	if (uop->mediactl != NULL) mask = mask | 0x0001;
	if (uop->request != NULL) mask = mask | 0x0008;
	return mask;
}


/*function to obtain the minor number from the filename*/
static int getMinor(char *fname)
{
	struct stat status;
	int ret;

	ret = stat(fname,&status);	/*fill in the status structure using stat system call*/
	if (ret<0) return ret;
	return (status.st_rdev & 0xff);	/*return only the minor number from the device field*/
}


/*
//function to attach with userdev driver
//The function arguments are, the device filename, ioctl data array giving the
//list of ioctls supported and their corresponding memory requirement, the
//number of ioctls and userdev operations structure containing pointers to
//functions implemented by the driver
*/
int userdev_attach(char *fname, struct userdev_ioctl_data *ioelm, int num_ioelms, struct userdev_operations *ops, void *data)
{
	int fp,ret,euid,minor;
	struct attach_record a_rec;
	struct __userdev_info *udinfo;
	
	euid = geteuid();	/*obtain effective uid of the process*/
	if (euid != 0) 		/*if not that of the root then print warning*/
		fprintf(stderr,"Warning: root privilages required!\n");
	if (ops == NULL)
	{
		fprintf(stderr,"Userdev Operations is NULL\n");
		errno = EINVAL;
		return -1;
	}

	minor = getMinor(fname);	/*obtain minor number of the device filename */
	if (minor < 0)
	{
		fprintf(stderr,"Invalid device\n");
		errno = ENODEV;
		return -1;
	}

	udinfo = (struct __userdev_info*)malloc(sizeof(struct __userdev_info));
	memset(udinfo,-1,sizeof(struct __userdev_info));
	if(udinfo == NULL)		/*construct a new userdev_info structure*/
	{
		fprintf(stderr,"Memory fault, lack of memory\n");
		errno = ENOMEM;
		return -1;
	}

    	if(pipe(udinfo->userdev_readfd) < 0)	/*obtain read pipe descriptors*/
	{
		fprintf(stderr,"Error occured during read pipe creation\n");
		free(udinfo);
		errno = EPIPE;
		return -1;
    	}
    	if(pipe(udinfo->userdev_writefd) < 0)	/*obtain write pipe descriptors*/
    	{
        	fprintf(stderr,"Error occured during write pipe creation\n");
		close(udinfo->userdev_readfd[0]);
		close(udinfo->userdev_readfd[1]);
		free(udinfo);
		errno = EPIPE;
		return -1;
    	}
				/*fill in the userdev attach_record*/
	a_rec.readfd = udinfo->userdev_writefd[0];	
    	a_rec.writefd = udinfo->userdev_readfd[1];
    	a_rec.devicename = fname;
    	a_rec.namelen = strlen(fname);
    	a_rec.func_mask = constructMask(ops);	/*construct the function mask from the userdev operations structure*/
    	a_rec.num_ioctls = num_ioelms;
    	a_rec.ioctls = (struct ioctl_elem*)ioelm; 
	a_rec.data = data;

	fp = open(getUserdevName(ops->devtype), O_RDWR);	/*open userdev(0) - minor 0*/
	if (fp < 0)
	{
		fprintf(stderr,"Cannot access userdev in read-write mode\n");
		close(udinfo->userdev_readfd[0]);
		close(udinfo->userdev_readfd[1]);
		close(udinfo->userdev_writefd[0]);
		close(udinfo->userdev_writefd[1]);
		free(udinfo);
		errno = EACCES;
		return -1;
	}
	
	ret = ioctl(fp,USERDEV_ATTACH,&a_rec);	/*attach to userdev using ioctl USERDEV_ATTACH passing the attach_record as data*/

	if (ret<0) 	/*if ioctl failed!*/
	{
		fprintf(stderr,"Userdev attach failed!!!\n");
		close(udinfo->userdev_readfd[0]);
		close(udinfo->userdev_readfd[1]);
		close(udinfo->userdev_writefd[0]);
		close(udinfo->userdev_writefd[1]);
		free(udinfo);
		errno = EIO;
		return ret;
	}		
			/*else fill in the rest of the fields of userdev_info structure*/
	udinfo->userdev_id = ret;
	udinfo->userdev_minor = minor;
	udinfo->uops = ops;
	udinfo->fd0 = fp;
	udinfo->signo = -1;
	__ud_info[__ud_count++] = udinfo;
	return ret;	/*return the userdev id to the caller*/
}


/*function to get the index into udinfo using the userdev id passed by user*/
static int get_userinfo_id(int id)
{
	int i;
	if (__ud_count == 0)	/*no one is attached to userdev from this process*/
	{
		fprintf(stderr,"No one's attached to userdev\n");
		errno = ESRCH;
		return -1;
	}

	for(i=0;i<__ud_count;i++)
		if((__ud_info[i]!=NULL) && (__ud_info[i]->userdev_id == id)) break;

	if(i == __ud_count)	/*no match return invalid id -1*/
	{
		fprintf(stderr,"Invalid id\n");
		errno = EINVAL;
		return -1;
	}
	return i;			/*return the matching id*/
}


/*function to detach from userdev given the id number*/
int userdev_detach(int id)
{
	int fp,ret,i;
	struct detach_record d_rec;
	
	i = get_userinfo_id(id);	/*get userinfo_id */
	if (i < 0) return i;

	fp = __ud_info[i]->fd0;
	d_rec.id = __ud_info[i]->userdev_id;
	ret = ioctl(fp,USERDEV_DETACH,&d_rec);	/*make ioctl USERDEV_DETACH passing userdev_id as argument*/
	if(ret == 0)	 			/*if success then free userdev_info structure*/
	{
		close(__ud_info[i]->userdev_readfd[0]);
		close(__ud_info[i]->userdev_readfd[1]);
		close(__ud_info[i]->userdev_writefd[0]);
		close(__ud_info[i]->userdev_writefd[1]);
		free(__ud_info[i]);
		__ud_info[i] = NULL;
	}

	close(fp);
	return ret;
}


/*
//function to send a ioctl to userdev(0) given the ioctl number and the device
//id number
*/
static int send_ioctl_userdev(int num, int id)
{
	int fp,ret;

	id = get_userinfo_id(id);	/*get userinfo_id */
	if (id < 0) return id;

	if (num==USERDEV_ATTACH || num==USERDEV_DETACH)	/*these are sent using attach and detach functions*/
	{
		fprintf(stderr,"Invalid ioctl number\n");
		errno = EINVAL;
		return -1;
	}

	fp = __ud_info[id]->fd0;
	ret = ioctl(fp,num,&__ud_info[id]->userdev_id);	/*send the requested ioctl*/
	return ret;
}


/*
//function to inform the arrival of data during input polling to the waitq
//associated with the driver for input polling
*/
int userdev_inform_poll_in(int id) 	{ return send_ioctl_userdev(USERDEV_POLL_IN, id); }

/*
//function to inform of arrival of data during output polling to the waitq
//associated with the driver for input polling
*/
int userdev_inform_poll_out(int id) 	{ return send_ioctl_userdev(USERDEV_POLL_OUT, id); }

/*
//function to inform occurance of asynchronous I/O to the send SIGIO to the
//process waiting for it.
*/
int userdev_inform_fasync_io(int id)	{ return send_ioctl_userdev(USERDEV_FASYNC_SIGNAL, id); }


/*function to free the irq of the device identified by the id number*/
int userdev_free_irq(int id)
{ 
	int ret;

	ret = send_ioctl_userdev(USERDEV_FREE_IRQ, id); 	/*ask userdev to free the irq of the given device*/
	if(ret==0)
	{
		struct sigaction act;
		sigset_t sset;

		id = get_userinfo_id(id);	/*must be valid as otherwise send_ioctl would have failed*/
    		sigemptyset(&sset);
	    	act.sa_handler = SIG_DFL;	/*set the signal disposition of the signal for interrupts on the irq to default*/
		act.sa_mask = sset;
	    	act.sa_flags = SA_SIGINFO;
	    	sigaction(__ud_info[id]->signo, &act, NULL);
		__ud_info[id]->signo = -1;
	}
	return ret;
}


/*
//function to request userdev for handling interrupts on the given irq number
//and send the given signal whenever an interrupt occurs
//The signal disposition of the signal is changed to the function passed as
//argument.
*/
int userdev_request_irq(int irq, int sig, void (*hdlr)(int, siginfo_t *, void *), int id)
{
	int fp,ret;
	struct irq_request_record rec;
	
	id = get_userinfo_id(id);	/*get userinfo id */
	if (id < 0) return id;

	fp = __ud_info[id]->fd0;
	
	rec.id = __ud_info[id]->userdev_id;	/*fill in irq request record*/
	rec.irqno = irq;
	rec.signo = sig;
	ret = ioctl(fp,USERDEV_REQUEST_IRQ,&rec);/*send ioctl to request userdev to handle interrupts on given irq and send sig inturn*/

	if(ret==0)	/*if ioctl was successful*/
	{
		struct sigaction act;
		sigset_t sset;

    		sigemptyset(&sset);	/*change signal disposition of the signal to the function passed as argument*/
	    	act.sa_handler = (void(*)(int))hdlr;
		act.sa_mask = sset;
	    	act.sa_flags = SA_SIGINFO;	/*set the option to have info passed from signal source(userdev passes minor number*/
	    	sigaction(sig, &act, NULL);	/*so that a single handler can handle interrupts from more than one device*/
		__ud_info[id]->signo = sig;
	}
	return ret;
}


/*function to request for the given dma channel for the device whose id is given*/
int userdev_request_dma(int dma, int id)
{
	int fp,ret;
	struct dma_request_record rec;
	
	id = get_userinfo_id(id);	/*get userinfo id */
	if (id < 0) return id;

	fp = __ud_info[id]->fd0;
	
	rec.id = __ud_info[id]->userdev_id;	/*fill in dma request record*/
	rec.channel = dma;
	ret = ioctl(fp,USERDEV_REQUEST_DMA,&rec);/*send ioctl to request userdev to request for dma channel*/

	return ret;
}


/*function to free the dma channel for the device whose is is given*/
int userdev_free_dma(int id)
{ 
	return send_ioctl_userdev(USERDEV_FREE_DMA, id); 	/*ask userdev to free the dma channel*/
}


/*function to start dma operation to the dma buffer of given size, for the device whose id is given*/
int userdev_start_dma(int mode, int count, char *buf, int id)
{
	int fp,ret;
	struct dma_start_record rec;
	
	id = get_userinfo_id(id);	/*get userinfo id */
	if (id < 0) return id;

	fp = __ud_info[id]->fd0;
	
	rec.id = __ud_info[id]->userdev_id;	/*fill in dma request record*/
	rec.mode = mode;
	rec.count = count;
	rec.buf = buf;
	ret = ioctl(fp,USERDEV_START_DMA,&rec);/*send ioctl to request userdev to request for dma channel*/

	return ret;
}


/*function to check the dma operation in progress, for the device whose is is given*/
/*the return value is 0 if opertion has completed, otherwise greater than 0 if in progress and negative incase of an error*/
int userdev_check_dma(int id)
{ 
	return send_ioctl_userdev(USERDEV_CHECK_DMA, id); 	/*ask userdev to check the dma operation on dma channel of device*/
}


/*function to copy data from dma buffer into a local user space buffer, a pointer to which is passed*/
int userdev_copy_dma(char *buf, int id)
{
	int fp,ret;
	struct dma_copy_record rec;
	
	id = get_userinfo_id(id);	/*get userinfo id */
	if (id < 0) return id;

	fp = __ud_info[id]->fd0;
	
	rec.id  = __ud_info[id]->userdev_id;	/*fill in dma copy record*/
	rec.buf = buf;
	ret = ioctl(fp,USERDEV_COPY_DMA,&rec);/*send ioctl to request userdev to copy data from dma buffer*/

	return ret;
}


/*function to enable the dma operation on the given device*/
int userdev_enable_dma(int id)
{ 
	return send_ioctl_userdev(USERDEV_ENABLE_DMA, id); 	/*ask userdev to enable the dma operation on dma channel of device*/
}


/*function to disable the dma operation on the given device*/
int userdev_disable_dma(int id)
{ 
	return send_ioctl_userdev(USERDEV_DISABLE_DMA, id); 	/*ask userdev to disable the dma operation on dma channel of device*/
}


/*
//function to start accepting requests from userdev and sending responses
//Basically select system call is used to parallelly read from more than one
//read pipe descriptors and which ever has data to be read is serviced in order
//in which attach was called.
//Request packet is read and based on the type of operation, the rest of the
//data in the packet is interpreted and the requested function of the device on
//whose pipe this request came is executed passing the arguments obtained from
//the request packet.
//The return value of the driver function is again packaged and sent as the
//response to the request via the write pipe of the device
//The function returns only when an error occurs and the select system call
//returns saying so.
*/
int userdev_start()
{
	int n,id,ret,usrid;
	struct header head;
	char *packet;
	fd_set rfds;
	int maxfd,retval,i;
	
	if (__ud_count == 0)	/*check if there is anyone attached to userdev*/
	{
		fprintf(stderr,"No one attached to userdev\n");
		errno = ESRCH;
		return -1;
	}

	do
	{
		FD_ZERO(&rfds);
		maxfd = 0;			/*construct the set of read file descriptors for select system call*/
		for(i=0;i<__ud_count;i++)	/*also find the max fd needed for select*/
			if (__ud_info[i] != NULL)
			{
				FD_SET(__ud_info[i]->userdev_readfd[0],&rfds);
				if (__ud_info[i]->userdev_readfd[0] > maxfd) maxfd = __ud_info[i]->userdev_readfd[0];
			}
        	retval = select(maxfd+1, &rfds, NULL, NULL, NULL);	/*monitor read pipes for data*/
        	if (retval <= 0)
			if(errno == EINTR) continue;	/*if select failed due to signal, ignore*/
			else break;			/*otherwise exit*/
		
		for(i=0;i<__ud_count;i++)	/*identify the pipes that have data to be read*/
			if ((__ud_info[i] != NULL) && (FD_ISSET(__ud_info[i]->userdev_readfd[0], &rfds)))
			{
				n = read(__ud_info[i]->userdev_readfd[0],(char*)&head,sizeof(head));	/*read header*/
				if (n != sizeof(head)) 	/*check for errors, if so continue with next pipe*/
				{
					fprintf(stderr,"Could not read message header sufficiently\n");
					errno = EPROTO;	/*incase header insufficiently read*/
					continue;
				}
				if (head.packet_size < 0)	/*incase header shows invalid packet size*/
				{
					fprintf(stderr,"Header shows invalid packet size\n");
					errno = EPROTO;
					continue;
				}

				packet = (char*)malloc(head.packet_size);
				n = read(__ud_info[i]->userdev_readfd[0],packet,head.packet_size);	/*read packet*/
				if (n < head.packet_size)	/*check for error*/
				{
					fprintf(stderr,"Could not read message packet sufficiently\n");
					errno = EPROTO;
					continue;
				}
				id = head.request_id;	/*note request id for sending response*/
				usrid = __ud_info[i]->userdev_id;

				/*if the requested function is supported, call the function. The calling*/
				/*function would be responsible for sending the response to the request*/
				/*which can be sent immediately before the function returns or deferred */
			    	/*for a later instant of time*/
				/*Incase of deferred response the driver must make a local copy of any */
				/*pointer data as it will be deallocated on return from function call*/

				switch(head.opcode)	/*based on the requested operation code*/
				{
					case OP_READ:	/*read operation*/
						{
							struct read_send *rs;

							rs = read_send_from_char(packet);	/*obtain size and offset from packet*/
							free(packet);

							if(__ud_info[i]->uops->read != NULL)	/*call userdev function read*/
								__ud_info[i]->uops->read(usrid,rs->size,rs->offset,rs->flags,id);
							free(rs);
							break;
						}

					case OP_WRITE:	/*write operation*/
						{
							struct write_send *ws;

							ws = write_send_from_char(packet);	/*obtain data and offset to write from*/
							
							if(__ud_info[i]->uops->write != NULL)	/*call userdev function write*/
								__ud_info[i]->uops->write(usrid,ws->data.elements,ws->data.len,
										ws->offset,ws->flags,id);
							free(packet);	/*delayed as data.elements points within packet*/
							free(ws);
							break;
						}
						
					case OP_IOCTL:	/*ioctl operation*/
						{
							struct ioctl_send *is;
							void *data;

							is = ioctl_send_from_char(packet);	/*obtain ioctl command and SET operation data*/

							if (is->data.len == 0) data = NULL;	/*if no data passed set NULL*/
							else data = (void*)is->data.elements;

							if(__ud_info[i]->uops->ioctl != NULL)	/*call userdev operation ioctl*/
								__ud_info[i]->uops->ioctl(usrid,is->command,data,is->data.len,id);
							free(packet);	/*delayed as data.elements points within packet*/
							free(is);
							break;
						}
						
					case OP_OPEN:	/*open operation*/
						{
							struct open_send *os;
							
							os = open_send_from_char(packet);	/*obtain open call flags*/
							free(packet);

							if(__ud_info[i]->uops->open != NULL)/*call userdev open operation*/
								__ud_info[i]->uops->open(usrid,os->flags,os->mode,id);
							free(os);
							break;
						}
						
					case OP_CLOSE:	/*close operation*/
						{
							struct close_send *cs;
							
							cs = close_send_from_char(packet);
							free(packet);
							
							if(__ud_info[i]->uops->close != NULL)	/*call close userdev operation*/
								__ud_info[i]->uops->close(usrid,id);
							free(cs);
							break;
						}
						
					case OP_FLUSH:	/*flush operation*/
						{
							struct flush_send *fs;
							
							fs = flush_send_from_char(packet);
							free(packet);
							
							if (__ud_info[i]->uops->flush != NULL)	/*call flush userdev operation*/
								__ud_info[i]->uops->flush(usrid,id);
							free(fs);
							break;
						}
						
					case OP_CHECK_MEDIA_CHANGE:	/*media change check operation*/
						{
							struct check_media_change_send *cs;
							
							cs = check_media_change_send_from_char(packet);
							free(packet);
							
								/*call check_media_change userdev operation*/
							if(__ud_info[i]->uops->check_media_change != NULL)
								__ud_info[i]->uops->check_media_change(usrid,id);
							free(cs);
							break;
						}
						
					case OP_REVALIDATE:	/*revalidate operation*/
						{
							struct revalidate_send *rs;
							
							rs = revalidate_send_from_char(packet);
							free(packet);
							
							if(__ud_info[i]->uops->revalidate != NULL)/*call revalidate userdev function*/ 
								__ud_info[i]->uops->revalidate(usrid,id);
							free(rs);
							break;
						}
						
					case OP_FSYNC:	/*fsync operation*/
						{
							struct fsync_send *fs;
							
							fs = fsync_send_from_char(packet);
							free(packet);
							
							if(__ud_info[i]->uops->fsync != NULL)	/*call fsync userdev function*/
								__ud_info[i]->uops->fsync(usrid,fs->datasync,id);
							free(fs);
							break;
						}
						
					case OP_FASYNC:	/*fasync operation*/
						{
							struct fasync_send *fs;
							
							fs = fasync_send_from_char(packet);
							free(packet);
							
							if(__ud_info[i]->uops->fasync != NULL)	/*call fasync userdev function*/
								__ud_info[i]->uops->fasync(usrid,id);
							free(fs);
							break;
						}
						
					case OP_POLL:	/*poll operation*/
						{
							struct poll_send *ps;
							
							ps = poll_send_from_char(packet);
							free(packet);
							
							if(__ud_info[i]->uops->poll != NULL)	/*call poll userdev operation*/
								__ud_info[i]->uops->poll(usrid,id);
							free(ps);
							break;
						}
	
					case OP_MEDIACTL:	/*mediactl operation*/
						{
							struct mediactl_send *ms;
							
							ms = mediactl_send_from_char(packet);
							free(packet);
							
							if(__ud_info[i]->uops->mediactl != NULL)/*call mediactl userdev function*/ 
								__ud_info[i]->uops->mediactl(usrid, ms->op, ms->optarg, id);
							free(ms);
							break;
						}
						
					case OP_MESSAGE:/*message operation*/
						{
							struct message_send *ms;

							ms = message_send_from_char(packet);

							if(__ud_info[i]->uops->message != NULL)	/*call message userdev operation*/
								__ud_info[i]->uops->message(usrid,ms->type,ms->reqid,ms->data,ms->size);
							free(packet);	/*delayed as data.elements points within packet*/
							free(ms);
							break;
						}					
					case OP_REQUEST:/*request operation*/
						{
							struct request_send *rs;
							void *data;

							rs = request_send_from_char(packet);

							if (rs->data.len == 0) data = NULL;	/*if no data passed set NULL*/
							else data = (void*)rs->data.elements;

							if(__ud_info[i]->uops->request != NULL)
								__ud_info[i]->uops->request(usrid, rs->command, rs->sector, rs->length, rs->clustersize, data, rs->data.len, id);
							free(packet);	/*delayed as data.elements points within packet*/
							free(rs);
							break;
						}
					default: 
							fprintf(stderr,"Invalid request received\n");
							errno = EBADRQC;
							break;
				}
			}
	}while(1);
	return -1;
}


/*
//Function that can be called by driver to send response to the request gathered.
//Depending on the operation code, the passed arguments are used to construct
//the response packet
*/
void userdev_send_response(int id, int reqid, int opcode, int retval, int size, void *data)
{
	struct header head;
	char *packet;
	int i,n;

	head.opcode = opcode;	/*construct header for reply with same id as request*/
	head.request_id = reqid;

	switch(opcode)
	{
		case OP_READ:
			{
				struct read_recv rr;
				rr.data.elements = (char*)data;
				rr.data.len = size;
				rr.offset = retval;	/*offset stored in retval*/
				n = read_recv_to_char(&head,&rr,&packet);	/*send back the result of read*/
				break;
			}
		case OP_WRITE:
			{
				struct write_recv wr;
				wr.num_written = size;
				wr.offset = retval;	/*offset in retval*/
				n = write_recv_to_char(&head,&wr,&packet);	/*send back the result of write*/
				break;
			}
		case OP_POLL:
			{
				struct poll_recv pr;
				pr.poll_result = retval;/*return result*/
				n = poll_recv_to_char(&head,&pr,&packet);
				break;
			}
		case OP_IOCTL:
			{
				struct ioctl_recv ir;
				ir.result = retval;	/*return back result along with data, if any*/
				ir.data.len = size;
				if (size == 0) ir.data.elements = NULL;
				else ir.data.elements = (char*)data;
				n=ioctl_recv_to_char(&head,&ir,&packet);
				break;
			}
		case OP_OPEN:
			{
				struct open_recv or;
				or.result = retval;	/*return result*/
				n = open_recv_to_char(&head,&or,&packet);
				break;
			}
		case OP_FLUSH:
			{
				struct flush_recv fr;
				fr.result = retval;	/*return result of the operation*/
				n = flush_recv_to_char(&head,&fr,&packet);
				break;
			}
		case OP_CLOSE:
			{
				struct close_recv cr;
				cr.result = retval;	/*return result*/
				n = close_recv_to_char(&head,&cr,&packet);
				break;
			}
		case OP_FSYNC:
			{
				struct fsync_recv fr;
				fr.result = retval;	/*return result*/
				n = fsync_recv_to_char(&head,&fr,&packet);
				break;
			}
		case OP_FASYNC:
			{
				struct fasync_recv fr;
				fr.result = retval;	/*return result*/
				n = fasync_recv_to_char(&head,&fr,&packet);
				break;
			}
		case OP_CHECK_MEDIA_CHANGE:
			{
				struct check_media_change_recv cr;
				cr.flag = retval;	/*return result*/
				n = check_media_change_recv_to_char(&head,&cr,&packet);
				break;
			}
		case OP_REVALIDATE:
			{
				struct revalidate_recv rr;
				rr.result = retval;	/*return result*/
				n = revalidate_recv_to_char(&head,&rr,&packet);
				break;
			}
		case OP_MEDIACTL:
			{
				struct mediactl_recv mr;
				mr.result = retval;	/*return result*/
				n = mediactl_recv_to_char(&head,&mr,&packet);
				break;
			}
		case OP_REQUEST:
			{
				struct request_recv rr;
				rr.result = retval;	/*return back result along with data, if any*/
				rr.data.len = size;
				if (size == 0) rr.data.elements = NULL;
				else rr.data.elements = (char*)data;
				n = request_recv_to_char(&head,&rr,&packet);
				break;
			}
		default:
			fprintf(stderr,"Invalid request id for response packet\n");
			return;
	}

	i = get_userinfo_id(id);
	write(__ud_info[i]->userdev_writefd[1],packet,n);
	free(packet);	/*write back response packet to the pipe*/
}
