/*
//file contains functions that implement a tight loop delay function for delays
//of a specified number of microseconds
*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <sys/time.h>
# include <unistd.h>

# define CALIB_ITER	1000000		/*number of iterations to be used to compute loops_per_usec*/
static unsigned long loops_per_usec;		/*result of loop calibration used for obtaining microsecond delays*/


/*function that contains loop used to obtain a certain amount of delay*/
static void delay_loop(unsigned long l)	/*called with number of times to loop*/
{
	unsigned long i;
	int a,b,c;
	for(i=0;i<l;i++)
	{
		a = b | 3456;
		b = c & a;
		c = a ^ b;
	}
}


/*function to calculate difference between two time values*/
static struct timeval *diff(struct timeval *t1, struct timeval *t2)
{
	static struct timeval tdiff;	/*return result in a static variable*/
	
	tdiff.tv_sec = t1->tv_sec - t2->tv_sec;
	tdiff.tv_usec = t1->tv_usec - t2->tv_usec;
	return &tdiff;
}


/*
//function to calibrate number of loops required for 1 microsecond delay
//This function needs to be called before delay function can be used
//The values loops_per_usec cannot determined ahead as it depends on processor
//speed of operation, load on the system etc
*/
void userdev_calibrate_delay()
{
	struct timeval tv1,tv2,*delta;
	unsigned long l,d;
	
	l = CALIB_ITER;
	gettimeofday(&tv1,NULL);	/*record current time*/
	delay_loop(l);			/*delay of l loops*/
	gettimeofday(&tv2,NULL);	/*record current time*/
	delta = diff(&tv2,&tv1);	/*find difference*/
	d = delta->tv_sec*1000000 + delta->tv_usec;
	
	gettimeofday(&tv1,NULL);	/*do the same as above for (2*l) loops*/
	delay_loop(2*l);
	gettimeofday(&tv2,NULL);
	delta = diff(&tv2,&tv1);

	/*subtract time for l loops from time for (2*l) loops to reduce errors*/
	/*reciprocal of the difference gives number of loops per microsecond*/
	loops_per_usec = l/((delta->tv_sec*1000000 + delta->tv_usec) - d);
}


/*function to be called to obtain a deay of given number of microseconds*/
void userdev_udelay(unsigned long usec)
{
	delay_loop(usec * loops_per_usec);	/* (delay * number of loops for 1usec delay) */
}
