/*file containing functions that provide the timer queue implementation at user level*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <pthread.h>
# include <sys/time.h>
# include <signal.h>
# include "timerq.h"

static struct userdev_timer_queue timerqueue = { NULL, PTHREAD_MUTEX_INITIALIZER , 1, PTHREAD_COND_INITIALIZER}; /*timer queue*/
static void (*init)();		/*pointer to initialization function set in init_timer_queue, the first function called by the thread*/
static int timerq_stop = 0;	/*flag when set stops the timer queue thread*/


/*function to add a task to the timer queue while keeping the queue sorted by expiry time*/
void userdev_add_timer(struct userdev_timer_task *tsk)
{
	struct userdev_timer_task *ptr;

	tsk->next = NULL;
	tsk->prev = NULL;

	pthread_mutex_lock(&timerqueue.mutex);	/*lock the timer queue*/
	ptr = timerqueue.first;
	if (ptr == NULL) 
	{
		timerqueue.first = tsk;		/*if timer queue is empty, assign task as the first one*/
		timerqueue.running = 1;		/*set flag to running and wake up run timer queue*/
		pthread_cond_signal(&timerqueue.run_cond);
	}
	else if(ptr->expires > tsk->expires)	/*else if new task has the lowest expiry then add to start of list*/
	{
		tsk->next = ptr;
		ptr->prev = tsk;
		timerqueue.first = tsk;
	}
	else	/*otherwise*/
	{
		while(ptr->next != NULL) 	/*skip all tasks that have lower expiry time*/
		{
			if (ptr->next->expires > tsk->expires) break;
			ptr=ptr->next;
		}
		tsk->next = ptr->next;		/*add the task to the timer queue and update pointers*/
		tsk->prev = ptr;
		if(tsk->next != NULL) tsk->next->prev = tsk;
		ptr->next = tsk;
	}
	pthread_mutex_unlock(&timerqueue.mutex);/*unlock timer queue*/
}


/*function to remove a task from the timer queue if it is present(has not been executed)*/
/*the function returns 1 if task was found in the timer queue and 0 otherwise*/
int userdev_del_timer(struct userdev_timer_task *tsk)
{
	int found = 0;

	pthread_mutex_lock(&timerqueue.mutex);	/*lock timer queue*/
	if(tsk->next != NULL) 			/*check if next task is present, if so update its previous pointer*/
	{
		tsk->next->prev = tsk->prev;
		found = 1;
	}
	if(tsk->prev != NULL) 			/*check if previous task is present, if so update its next pointer*/
	{
		tsk->prev->next = tsk->next;
		found = 1;
	}
	if(timerqueue.first == tsk) 		/*if task is the first one then update pointer in timer queue*/
	{
		timerqueue.first = tsk->next;
		found = 1;
	}
	pthread_mutex_unlock(&timerqueue.mutex);/*unlock timer queue*/

	tsk->next = tsk->prev = NULL;		/*reset both pointers of task*/
	return found;			
}


/*function to modify the expiry time of a task in the timer queue*/
/*modification is possible if task has not been completed and is in the timer queue*/
/*in which case 1 is returned and 0 is returned otherwise*/
int userdev_mod_timer(struct userdev_timer_task *tsk, unsigned long expires)
{
	int found;

	found = userdev_del_timer(tsk);	/*remove task from timer queue*/
	if (!found) return 0;

	tsk->expires = expires;	/*if found, change expiry time and add to timer queue*/
	userdev_add_timer(tsk);
	return 1;
}


/*function to check if the given timer task is pending, ie it it present in the timer queue*/
int userdev_timer_pending(struct userdev_timer_task *tsk)
{
        int found = 0;
        pthread_mutex_lock(&timerqueue.mutex);	/*lock the timer queue*/
        if((tsk->next != NULL) || (tsk->prev != NULL) || (timerqueue.first == tsk))
                found = 1;	/*found if either has a next task or previous task or is the only task in queue*/
        pthread_mutex_unlock(&timerqueue.mutex);/*unlock timer queue*/
        return found;
}


/*function that is the signal handler for SIGALRM*/
static void timer_queue_sig_hdlr(int sig)
{
}


/*function that returns the current time in 100th of second since epoch*/
/*to be used for setting expiry time of tasks*/
unsigned long userdev_get_jiffies()
{
        struct timeval tv;
        gettimeofday(&tv,NULL);
        return tv.tv_sec*100 + tv.tv_usec/10000;	/*convert sec:usec to jiffies*/
}


/*internal function that runs through the timer queue and executes all tasks*/
/*that have their time expired. The timer queue thread runs this function*/
static void* run_timer_queue (void* dummy)
{
	struct userdev_timer_task *ptr,*pqr;
	short expired;
        unsigned long timeout,currtime;

	if(init) init();	/*call the thread initialisation function*/
	while(!timerq_stop)		/*keep checking timer queue until timerq_stop is set*/
	{
		expired = 0;	/*flag that denotes if a task's timer expired*/
                currtime = userdev_get_jiffies();

		pthread_mutex_lock(&timerqueue.mutex);	/*lock the timer queue*/
		ptr = timerqueue.first;
		if ((ptr != NULL) && (currtime >= ptr->expires))	/*check if a task exists and if so whether it expired*/
		{
			timerqueue.first = ptr->next;	/*since the tasks are sorted by expiry time, we always see the first one only*/
			if(timerqueue.first != NULL) timerqueue.first->prev = NULL;
			expired = 1;			/*set expiry flag*/
		}
                if (ptr != NULL) timeout = ptr->expires;	/*note timeout time for first task if it exists*/
                else 	/*timerq is empty so wait till a timer task is assigned*/
		{
			timerqueue.running = 0;
			while(timerqueue.running == 0)	/*while loop as we need to recheck why we were woken up, it could be a false alarm*/
				pthread_cond_wait(&timerqueue.run_cond, &timerqueue.mutex);
		}
		pthread_mutex_unlock(&timerqueue.mutex);

		if(expired)		/*task's timer expired, so execute it and reset its pointers*/
		{
			ptr->next = ptr->prev = NULL;
			ptr->routine(ptr->data);
		}
                else 	/*if there exists a task that is pending but its timeout has not expired*/
                {
                        struct timeval tv,ti;
                        struct itimerval itv;
                        int delta;

                        delta = timeout - currtime;	/*find amount of time remaining*/
                        tv.tv_sec = delta/100;		/*convert jiffies to sec:usec*/
                        tv.tv_usec = (delta%100)*10000;
                        ti.tv_sec = 0;
                        ti.tv_usec = 0;
                        itv.it_value = tv;
                        itv.it_interval = ti;
                        setitimer(ITIMER_REAL,&itv,NULL);	/*install alarm to wake up when first task times out*/
                        usleep(100000);				/*sleep for maximum delay(100ms), to be woken in between by SIGALRM*/
                }
	}
	pthread_exit(0);
	return NULL;
}


/*function to set the timerq_stop flag inorder to stop the timer queue thread*/
void userdev_timer_queue_cleanup()
{
	timerq_stop = 1;
}


/*function to initialize timer queue mechanism*/
/*install signal handler for SIGALRM signal*/
/*create and start a detached thread for polling the timer queue and execute tasks in it*/
void userdev_init_timer_queue(void (*fn)())
{
	pthread_t thd;
	init = fn;	/*assign the argument as the initialisation function for the thread*/
        signal(SIGALRM,timer_queue_sig_hdlr);
	pthread_create(&thd,NULL,run_timer_queue,(void*)NULL);	/*create thread for polling timer queue*/
	pthread_detach(thd);
}
