/*file contains implementation of block device request queue*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <pthread.h>
# include "blk_req.h"

static pthread_mutex_t blk_req_mutex = PTHREAD_MUTEX_INITIALIZER;	/*mutex for access to request queue*/
struct userdev_blk_request *rq_head = NULL;     /*pointer to start of queue*/
struct userdev_blk_request *rq_tail = NULL;     /*pointer to end of queue*/

/*function to add information from userdev request into the request queue*/
struct userdev_blk_request* userdev_add_request(int devid, int command, long sector, int length, int clustersize, void *data, int size, int reqid)
{
	struct userdev_blk_request *req;

	req = (struct userdev_blk_request*)malloc(sizeof(struct userdev_blk_request));
	req->devid = devid;	/*create a request queue node and fill information*/
	req->command = command;
	req->sector = sector;
	req->length = length;			/*size of the request*/
	req->clustersize = clustersize;
	req->buffer = (void*)malloc(length);	/*allocate buffer of requested length*/
	req->buffer_cp = req->buffer;		/*keep copy of buffer pointer*/
	if(data != NULL) memcpy(req->buffer, data, size);
	req->size = size;			/*size of data, = length incase of write, = 0 incase of read*/
	req->reqid = reqid;
	req->errors = 0;
	req->next = NULL;

	/*initialize condition variable so that if is anyone waiting for completion of block transfer*/
	/*using pthread_cond_wait, he could be woken up when the transfer has completed*/
	pthread_cond_init(&req->done_cond, NULL);

	pthread_mutex_lock(&blk_req_mutex);	/*lock request queue*/
	if (QUEUE_EMPTY)	/*if empty then set new node as the first and last in queue*/
	{
		rq_head = req;
		rq_tail = req;
	}
	else			/*else append the new node to end of the queue*/
	{
		rq_tail->next = req;
		rq_tail = req;
	}
	pthread_mutex_unlock(&blk_req_mutex);	/*unlock request queue*/
	return req;
}


/*function to end the current request with the given result value and move on to the next request*/
void userdev_end_request(int res)
{
	struct userdev_blk_request *ptr;
	pthread_mutex_lock(&blk_req_mutex);	/*lock request queue*/
	ptr = rq_head;		/*update queue by moving to the next request*/
	rq_head = rq_head->next;
	pthread_mutex_unlock(&blk_req_mutex);	/*unlock request queue*/
	pthread_cond_signal(&ptr->done_cond);	/*signal completion of request*/
	free(ptr->buffer_cp);	/*free request buffer*/
	free(ptr);		/*free request*/
}


/*function to cleanup the request queue*/
void userdev_blk_requestq_cleanup()
{
	struct userdev_blk_request *ptr,*pqr;

	pthread_mutex_lock(&blk_req_mutex);	/*lock the request queue*/
	ptr = rq_head;

	while(ptr!=NULL)			/*run thru each request in the queue*/
	{
		pqr = ptr;
		ptr = ptr->next;		/*and free it*/
		free(pqr->buffer);
		free(pqr);
		pqr = NULL;
	}

	pthread_mutex_unlock(&blk_req_mutex);	/*unlock request queue*/
}
