/*
//file containing functions to probe a given parallel port and find out useful
//information about it such as name of the manufacturer, device class, model
//and other such information that device is ready to supply
*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <stdio.h>
# include <string.h>
# include <time.h>
# include <errno.h>
# include "parport.h"
# include "usrdriv.h"

#undef DEBUG_PROBE


/*function to read a nibble(4bits) status port*/
static inline int read_nibble(struct parport *port) 
{
	unsigned char i;
	i = parport_read_status(port)>>3;
	i &= ~8;
	if ((i & 0x10) == 0) i |= 8;
	return (i & 0x0f);
}


/*function to terminate read operation*/
static void read_terminate(struct parport *port)
{
	parport_write_control(port, (parport_read_control(port) & ~2) | 8);
		/* SelectIN high, AutoFeed low */
	if (parport_wait_peripheral(port, 0x80, 0)) 
		/* timeout, SelectIN high, Autofeed low */
		return;
	parport_write_control(port, parport_read_control(port) | 2);
		/* AutoFeed high */
	parport_wait_peripheral(port, 0x80, 0x80);
		/* no timeout possible, Autofeed low, SelectIN high */
	parport_write_control(port, (parport_read_control(port) & ~2) | 8);
}


/*
//function to read data of given length from the status port and place it in
//the buffer passed as argument.
*/
static long read_polled(struct parport *port, char *buf, unsigned long length)
{
	int i;
	char *temp=buf;
	unsigned int count = 0;
	unsigned char z=0;
	unsigned char Byte=0;
	unsigned long igiveupat=time(NULL)+5;

	for (i=0; ((long)(time(NULL) - igiveupat) < 0L); i++)
   	{
 			/* if(current->need_resched) schedule(); */
		parport_write_control(port, parport_read_control(port) | 2); 	/* AutoFeed high */
		if (parport_wait_peripheral(port, 0x40, 0))
	   	{
#ifdef DEBUG_PROBE
			/* Some peripherals just time out when they've sent all their data*/
			printf("%s: read1 timeout.\n", port->name);
#endif
			parport_write_control(port, parport_read_control(port) & ~2);
			break;
		}

		z = read_nibble(port);
		parport_write_control(port, parport_read_control(port) & ~2); 	/* AutoFeed low */
		if (parport_wait_peripheral(port, 0x40, 0x40))
	   	{
			printf("%s: read2 timeout.\n", port->name);
			break;
		}

		if ((i & 1) != 0)	/*convert nibbles read into bytes*/
	   	{
			Byte |= (z<<4);
			if (temp) 
				*(temp++) = Byte; 
			if (++count == length)
				temp = NULL;
				/* Does the error line indicate end of data? */
			if ((parport_read_status(port) & PARPORT_STATUS_ERROR) == PARPORT_STATUS_ERROR)
				break;
		}
		else Byte=z;
	}
	read_terminate(port);	/*terminate read*/
	return count; 
}


/*
//function to probe the given port and obtain information from it
//This is done by registering as a device for the given port and using the
//port's implementation functions to read the status register and obtain
//information from it
*/
int parport_probe(struct parport *port, char *buffer, int len)
{
	struct pardevice *dev = parport_register_device(port, "IEEE 1284 probe", 0);
		/*register with parport as device for the port*/
	int result = 0;

	if (!dev)
   	{
		fprintf(stderr,"%s: unable to register for probe.\n", port->name);
		return -EINVAL;
	}
		/*check if ieee1284 standard is met*/
	switch (parport_ieee1284_nibble_mode_ok(port, 4))
   	{
	case 2:
		/* HACK: wait 10ms because printer seems to ack wrong */
		usleep((HZ+99)/100 * 10 * 1000);	/*schedule_timeout((HZ/99)/100);*/
		result = read_polled(port, buffer, len);
		break;
	default:
		result = -EIO;
		break;
	}

	parport_unregister_device(dev);	/*unregister from parport*/
	return result;
}



static struct
{
	char *token;
	char *descr;
} classes[] = {
	{ "",        "Legacy device" },
	{ "PRINTER", "Printer" }, 
	{ "MODEM",   "Modem" },
	{ "NET",     "Network device" },
	{ "HDC",     "Hard disk" },
	{ "PCMCIA",  "PCMCIA" },
	{ "MEDIA",   "Multimedia device" },
	{ "FDC",     "Floppy disk" },
	{ "PORTS",   "Ports" },
	{ "SCANNER", "Scanner" },
	{ "DIGICAM", "Digital camera" },
	{ "",        "Unknown device" },
	{ "",        "Unspecified" }, 
	{ NULL,      NULL }
};


/*
//function to parse the string passed containing data supplied by the device
//when it was probed and fill the parport's probe_info component
*/
static void parse_data(struct parport *port, char *str)
{
	char *txt = (char*)malloc(strlen(str)+1);
	char *p = txt, *q; 
	int guessed_class = PARPORT_CLASS_UNSPEC;

	if (!txt)
   	{
		fprintf(stderr,"%s probe: memory squeeze\n", port->name);
		return;
	}
	strcpy(txt, str);	/*copy the string and parse it*/
	while (p)
   	{
		char *sep; 
		q = strchr(p, ';');	/*get a component*/
		if (q) *q = 0;
		sep = strchr(p, ':');	/*get LHS of the component*/
		if (sep)
	   	{
			char *u = p;
			*(sep++) = 0;	/*get RHS of the component*/
			while (*u)
		   	{
				*u = toupper(*u);
				u++;
			}		/*based on LHS assign RHS to various fields of probe_info structure*/
			if (!strcmp(p, "MFG") || !strcmp(p, "MANUFACTURER"))
		   	{
				if (port->probe_info.mfr)
					free (port->probe_info.mfr);
				port->probe_info.mfr = strdup(sep);
			}
		   	else if (!strcmp(p, "MDL") || !strcmp(p, "MODEL"))
		   	{
				if (port->probe_info.model)
					free (port->probe_info.model);
				port->probe_info.model = strdup(sep);
			}
		   	else if (!strcmp(p, "CLS") || !strcmp(p, "CLASS"))
		   	{
				int i;
				if (port->probe_info.class_name)
					free (port->probe_info.class_name);
				port->probe_info.class_name = strdup(sep);
				for (u = sep; *u; u++)
					*u = toupper(*u);
				for (i = 0; classes[i].token; i++)
			   	{
					if (!strcmp(classes[i].token, sep))
				   	{
						port->probe_info.class = i;
						goto rock_on;
					}
				}
				fprintf(stderr,"%s probe: warning, class '%s' not understood.\n", port->name, sep);
				port->probe_info.class = PARPORT_CLASS_OTHER;
			}
		   	else if (!strcmp(p, "CMD") || !strcmp(p, "COMMAND SET"))
		   	{
				if (port->probe_info.cmdset)
					free (port->probe_info.cmdset);
				port->probe_info.cmdset = strdup(sep);
					/* if it speaks printer language, it's probably a printer*/
				if (strstr(sep, "PJL") || strstr(sep, "PCL"))
					guessed_class = PARPORT_CLASS_PRINTER;
			}
		   	else if (!strcmp(p, "DES") || !strcmp(p, "DESCRIPTION"))
		   	{
				if (port->probe_info.description)
					free (port->probe_info.description);
				port->probe_info.description = strdup(sep);
			}
		}
rock_on:
		if (q) p = q+1; else p=NULL;	/*shift to next component*/
	}

		/* If the device didn't tell us its class, maybe we have managed to*/
	   	/*guess one from the things it did say. */
	if (port->probe_info.class == PARPORT_CLASS_UNSPEC)
		port->probe_info.class = guessed_class;

	free(txt);
}


/*function to print details of the probed information*/
static void pretty_print(struct parport *port)
{
	printf("%s: %s", port->name, classes[port->probe_info.class].descr);
	if (port->probe_info.class)
   	{
		printf(", %s %s", port->probe_info.mfr, port->probe_info.model);
	}
	printf("\n");
}


/*function to probe a parport port to get info from device attached to it*/
void parport_probe_one(struct parport *port)
{
	char *buffer = (char*)malloc(2048);
	int r;

	printf("parport probe_one %s\n",port->name);	/*initialize probe_info structure*/
	port->probe_info.model = strdup ("Unknown device");
	port->probe_info.mfr = strdup ("Unknown vendor");
	port->probe_info.description = port->probe_info.cmdset = NULL;
	port->probe_info.class = PARPORT_CLASS_UNSPEC;
	port->probe_info.class_name = NULL;

	if (!buffer)
   	{
		fprintf(stderr,"%s probe: Memory squeeze.\n", port->name);
		return;
	}

	r = parport_probe(port, buffer, 2047);	/*read probe information from device*/

	if (r < 0)
   	{
		printf("%s: no IEEE-1284 device present.\n",port->name);
		port->probe_info.class = PARPORT_CLASS_LEGACY;
	}
   	else if (r == 0)
   	{
		printf("%s: no ID data returned by device.\n",port->name);
	}
   	else
   	{
		buffer[r] = 0; 
#ifdef DEBUG_PROBE
		printf("%s id: %s\n", port->name, buffer+2);
#endif
		parse_data(port, buffer+2); 	/*parse the information obtained and fill probe_info*/
		pretty_print(port);
	}
	free(buffer);
}


/*function to intialize the parport_probe library module*/
int parport_probe_init(void)
{
	struct parport *p;
	
	for (p = parport_enumerate(); p; p = p->next) 
		parport_probe_one(p);			/*probe ports already registered*/
	parport_probe_hook = &parport_probe_one;	/*set probing hook for future port registrations*/
	return 0;
}


/*function to cleanup ie reset the probe hook*/
void parport_probe_cleanup(void)
{
	parport_probe_hook = NULL;
}
