/*file containing a user level implementation of floppy device driver*/
/**********************************************************************
    Copyright (C) 2002  Hari Krishna Vemuri

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    For any problems contact the author at hkglobalnet@yahoo.com
**********************************************************************/

# include <stdio.h>
# include <sys/types.h>
# include <pthread.h>
# include <signal.h>
# include <errno.h>
# include <fcntl.h>
# include <sys/stat.h>
# include "usrdriv.h"	/*user level device driver library*/

# define FLOPPY_SANITY_CHECK
# define DEBUGT 2
# define DCL_DEBUG 		/* debug disk change line */
# define FDPATCHES
# include "fdreg.h"
# include "fd.h"
# include <linux/fs.h>		/*for BLKGETSIZE ioctl*/
# include <linux/hdreg.h>

# undef PRINT_FUNCTION_ENTRY	/*if defined then function call log is printed*/
# ifdef PRINT_FUNCTION_ENTRY
# define PRINT(format, args...) printf(format,##args)
# else
# define PRINT(format, args...)	/*ignore*/
# endif
static int print_unex=1;	/* do print messages for unexpected interrupts */

/*
 * PS/2 floppies have much slower step rates than regular floppies.
 * It's been recommended that take about 1/4 of the default speed
 * in some more extreme cases.
 */
static int slow_floppy;

static int FLOPPY_IRQ=6;		/*irq number*/
static int FLOPPY_DMA=2;		/*dma channel number*/
static int FLOPPY_SIG=45;		/*signal corresponding to interrupt*/
static int userdev_id;			/*id returned by attach to userdev*/
static char devicename[] = "/dev/ufd";	/*device filename*/

/*we keep a map between userdev id and minor number so as to obtain type & drive numbers*/
static int minor_map[256] = { [0 ... 255] = -1};
static int get_minor(int id)
{
	int i;
	for(i=0;i<255;i++)
		if(minor_map[i] == id) return i;
	return -1;
}
#define MINOR(id) get_minor(id)

DECLARE_TASK_QUEUE(tq_immediate)		/*declare and initialize a task queue*/

/*The interrupt handler for floppy calls the function pointed to below global function pointer*/
/*which is assigned different values for different operations depending on the operation being*/
/*done on the floppy, so that a single handler works for interrupts on all operations*/
static void (*DEVICE_INTR)(void) = NULL;	/*pointer to function to be called on interrupt*/
#define SET_INTR(x) (DEVICE_INTR = (x))		/*macro to set above*/
#define CLEAR_INTR SET_INTR(NULL)		/*macro to reset above*/

#define NR_SECTORS (CURRENT->clustersize)

/* can use virtual DMA:
 * 0 = use of virtual DMA disallowed by config
 * 1 = use of virtual DMA prescribed by config
 * 2 = no virtual DMA preference configured.  By default try hard DMA,
 * but fall back on virtual DMA when not enough memory available */
static int can_use_virtual_dma=0;

/* use virtual DMA
 * 0 using hard DMA
 * 1 using virtual DMA
 * This variable is set to virtual when a DMA mem problem arises, and
 * reset back in floppy_grab_irq_and_dma.
 * It is not safe to reset it in other circumstances, because the floppy
 * driver may have several buffers in use at once, and we do currently not
 * record each buffers capabilities */
static int use_virtual_dma;

static unsigned short virtual_dma_port=0x3f0;
void floppy_interrupt(int sig, siginfo_t *info, void *ptr);	/*floppy interrupt signal handler*/
static int set_dor(int fdc, char mask, char data);
static void setioperms();					/*set io permissions for required ports*/
static void thread_init();					/*function to be executed when taskq & timerq start*/

/* the following is the mask of allowed drives. By default units 2 and
 * 3 of both floppy controllers are disabled, because switching on the
 * motor of these drives causes system hangs on some PCI computers. drive
 * 0 is the low bit (0x1), and drive 7 is the high bit (0x80). Bits are on if
 * a drive is allowed.
 * NOTE: This must come before we include the arch floppy header because
 *       some ports reference this variable from there. -Dave */
static int allowed_drive_mask = 0x33;

#include "floppy.h"

static int irqdma_allocated;

/*function to fallback on virtual dma incase dma memory allocation fails, ie addr is not allocated*/
static inline void fallback_on_nodma_alloc(char **addr, size_t l)
{
	PRINT("fallback on nodma\n");
#ifdef FLOPPY_CAN_FALLBACK_ON_NODMA
	if (*addr) return; /* we have the memory */
	if (can_use_virtual_dma != 2) return; /* no fallback allowed */
	printf("DMA memory shortage. Temporarily falling back on virtual DMA\n");
	*addr = (char *) nodma_mem_alloc(l);
#else
	return;
#endif
}

static unsigned long fake_change;
static int initialising=1;

static inline int TYPE(int x) 	/*extract type of the device from device id, ranges from 0 to 31*/
{
	return  (MINOR(x)>>2) & 0x1f;	
}
static inline int DRIVE(int x) 	/*extract drive number from device id, ranges from 0 to 7*/
{
	return (MINOR(x)&0x03) | ((MINOR(x)&0x80) >> 5);
}

/*follwoing bit operations have been replicated from kernel*/
static inline void set_bit(int nr, unsigned long *addr)
{
	*addr = (0x1 << nr) | *addr;
}
static inline void clear_bit(int nr, unsigned long *addr)
{
	*addr = (~(0x1 << nr)) & *addr;
}
static inline int test_bit(int nr, unsigned long *addr)
{
	return (0x1 << nr) & *addr;
}

#define ITYPE(x) (((x)>>2) & 0x1f)
#define TOMINOR(x) ((x & 3) | ((x & 4) << 5))
#define UNIT(x) ((x) & 0x03)		/* drive on fdc */
#define FDC(x) (((x) & 0x04) >> 2)  	/* fdc of drive */
#define REVDRIVE(fdc, unit) ((unit) + ((fdc) << 2))	/* reverse mapping from unit and fdc to drive */

#define DP (&drive_params[current_drive])
#define DRS (&drive_state[current_drive])
#define DRWE (&write_errors[current_drive])
#define FDCS (&fdc_state[fdc])
#define CLEARF(x) (clear_bit(x##_BIT, &DRS->flags))
#define SETF(x) (set_bit(x##_BIT, &DRS->flags))
#define TESTF(x) (test_bit(x##_BIT, &DRS->flags))

#define UDP (&drive_params[drive])
#define UDRS (&drive_state[drive])
#define UDRWE (&write_errors[drive])
#define UFDCS (&fdc_state[FDC(drive)])
#define UCLEARF(x) (clear_bit(x##_BIT, &UDRS->flags))
#define USETF(x) (set_bit(x##_BIT, &UDRS->flags))
#define UTESTF(x) (test_bit(x##_BIT, &UDRS->flags))

#define DPRINT(format, args...) printf("Floppy %d: " format, current_drive , ## args)

#define PH_HEAD(floppy,head) (((((floppy)->stretch & 2) >>1) ^ head) << 2)
#define STRETCH(floppy) ((floppy)->stretch & FD_STRETCH)

#define CLEARSTRUCT(x) memset((x), 0, sizeof(*(x)))

/* read/write */
#define COMMAND raw_cmd->cmd[0]
#define DR_SELECT raw_cmd->cmd[1]
#define TRACK raw_cmd->cmd[2]
#define HEAD raw_cmd->cmd[3]
#define SECTOR raw_cmd->cmd[4]
#define SIZECODE raw_cmd->cmd[5]
#define SECT_PER_TRACK raw_cmd->cmd[6]
#define GAP raw_cmd->cmd[7]
#define SIZECODE2 raw_cmd->cmd[8]
#define NR_RW 9

/* format */
#define F_SIZECODE raw_cmd->cmd[2]
#define F_SECT_PER_TRACK raw_cmd->cmd[3]
#define F_GAP raw_cmd->cmd[4]
#define F_FILL raw_cmd->cmd[5]
#define NR_F 6

/*
 * Maximum disk size (in kilobytes). This default is used whenever the
 * current disk size is unknown.
 * [Now it is rather a minimum]
 */
#define MAX_DISK_SIZE 4 /* 3984*/

/*globals used by 'result()*/
#define MAX_REPLIES 16
static unsigned char reply_buffer[MAX_REPLIES];
static int inr; 	/* size of reply buffer, when called from interrupt */
#define ST0 (reply_buffer[0])
#define ST1 (reply_buffer[1])
#define ST2 (reply_buffer[2])
#define ST3 (reply_buffer[0]) /* result of GETSTATUS */
#define R_TRACK (reply_buffer[3])
#define R_HEAD (reply_buffer[4])
#define R_SECTOR (reply_buffer[5])
#define R_SIZECODE (reply_buffer[6])

#define SEL_DLY (2*HZ/100)

/*this struct defines the different floppy drive types.*/
static struct {
	struct floppy_drive_params params;
	const char *name; /* name printed while booting */
} default_drive_params[]= {
/* NOTE: the time values in jiffies should be in msec!
 CMOS drive type
  |     Maximum data rate supported by drive type
  |     |   Head load time, msec
  |     |   |   Head unload time, msec (not used)
  |     |   |   |     Step rate interval, usec
  |     |   |   |     |       Time needed for spinup time (jiffies)
  |     |   |   |     |       |      Timeout for spinning down (jiffies)
  |     |   |   |     |       |      |   Spindown offset (where disk stops)
  |     |   |   |     |       |      |   |     Select delay
  |     |   |   |     |       |      |   |     |     RPS
  |     |   |   |     |       |      |   |     |     |    Max number of tracks
  |     |   |   |     |       |      |   |     |     |    |     Interrupt timeout
  |     |   |   |     |       |      |   |     |     |    |     |   Max nonintlv. sectors
  |     |   |   |     |       |      |   |     |     |    |     |   | -Max Errors- flags */
{{0,  500, 16, 16, 8000,    1*HZ, 3*HZ,  0, SEL_DLY, 5,  80, 3*HZ, 20, {3,1,2,0,2}, 0,
      0, { 7, 4, 8, 2, 1, 5, 3,10}, 3*HZ/2, 0 }, "unknown" },

{{1,  300, 16, 16, 8000,    1*HZ, 3*HZ,  0, SEL_DLY, 5,  40, 3*HZ, 17, {3,1,2,0,2}, 0,
      0, { 1, 0, 0, 0, 0, 0, 0, 0}, 3*HZ/2, 1 }, "360K PC" }, /*5 1/4 360 KB PC*/

{{2,  500, 16, 16, 6000, 4*HZ/10, 3*HZ, 14, SEL_DLY, 6,  83, 3*HZ, 17, {3,1,2,0,2}, 0,
      0, { 2, 5, 6,23,10,20,12, 0}, 3*HZ/2, 2 }, "1.2M" }, /*5 1/4 HD AT*/

{{3,  250, 16, 16, 3000,    1*HZ, 3*HZ,  0, SEL_DLY, 5,  83, 3*HZ, 20, {3,1,2,0,2}, 0,
      0, { 4,22,21,30, 3, 0, 0, 0}, 3*HZ/2, 4 }, "720k" }, /*3 1/2 DD*/

{{4,  500, 16, 16, 4000, 4*HZ/10, 3*HZ, 10, SEL_DLY, 5,  83, 3*HZ, 20, {3,1,2,0,2}, 0,
      0, { 7, 4,25,22,31,21,29,11}, 3*HZ/2, 7 }, "1.44M" }, /*3 1/2 HD*/

{{5, 1000, 15,  8, 3000, 4*HZ/10, 3*HZ, 10, SEL_DLY, 5,  83, 3*HZ, 40, {3,1,2,0,2}, 0,
      0, { 7, 8, 4,25,28,22,31,21}, 3*HZ/2, 8 }, "2.88M AMI BIOS" }, /*3 1/2 ED*/

{{6, 1000, 15,  8, 3000, 4*HZ/10, 3*HZ, 10, SEL_DLY, 5,  83, 3*HZ, 40, {3,1,2,0,2}, 0,
      0, { 7, 8, 4,25,28,22,31,21}, 3*HZ/2, 8 }, "2.88M" } /*3 1/2 ED*/
/*    |  --autodetected formats---    |      |      |
 *    read_track                      |      |    Name printed when booting
 *				      |     Native format
 *	            Frequency of disk change checks */
};

static struct floppy_drive_params drive_params[N_DRIVE];
static struct floppy_drive_struct drive_state[N_DRIVE];
static struct floppy_write_errors write_errors[N_DRIVE];
static struct floppy_raw_cmd *raw_cmd, default_raw_cmd;

/*
 * This struct defines the different floppy types.
 *
 * Bit 0 of 'stretch' tells if the tracks need to be doubled for some
 * types (e.g. 360kB diskette in 1.2MB drive, etc.).  Bit 1 of 'stretch'
 * tells if the disk is in Commodore 1581 format, which means side 0 sectors
 * are located on side 1 of the disk but with a side 0 ID, and vice-versa.
 * This is the same as the Sharp MZ-80 5.25" CP/M disk format, except that the
 * 1581's logical side 0 is on physical side 1, whereas the Sharp's logical
 * side 0 is on physical side 0 (but with the misnamed sector IDs).
 * 'stretch' should probably be renamed to something more general, like
 * 'options'.  Other parameters should be self-explanatory (see also
 * setfdprm(8)).
 */
/*
	    Size
	     |  Sectors per track
	     |  | Head
	     |  | |  Tracks
	     |  | |  | Stretch
	     |  | |  | |  Gap 1 size
	     |  | |  | |    |  Data rate, | 0x40 for perp
	     |  | |  | |    |    |  Spec1 (stepping rate, head unload
	     |  | |  | |    |    |    |    /fmt gap (gap2) */
static struct floppy_struct floppy_type[32] = {
	{    0, 0,0, 0,0,0x00,0x00,0x00,0x00,NULL    },	/*  0 no testing    */
	{  720, 9,2,40,0,0x2A,0x02,0xDF,0x50,"d360"  }, /*  1 360KB PC      */
	{ 2400,15,2,80,0,0x1B,0x00,0xDF,0x54,"h1200" },	/*  2 1.2MB AT      */
	{  720, 9,1,80,0,0x2A,0x02,0xDF,0x50,"D360"  },	/*  3 360KB SS 3.5" */
	{ 1440, 9,2,80,0,0x2A,0x02,0xDF,0x50,"D720"  },	/*  4 720KB 3.5"    */
	{  720, 9,2,40,1,0x23,0x01,0xDF,0x50,"h360"  },	/*  5 360KB AT      */
	{ 1440, 9,2,80,0,0x23,0x01,0xDF,0x50,"h720"  },	/*  6 720KB AT      */
	{ 2880,18,2,80,0,0x1B,0x00,0xCF,0x6C,"H1440" },	/*  7 1.44MB 3.5"   */
	{ 5760,36,2,80,0,0x1B,0x43,0xAF,0x54,"E2880" },	/*  8 2.88MB 3.5"   */
	{ 6240,39,2,80,0,0x1B,0x43,0xAF,0x28,"E3120" },	/*  9 3.12MB 3.5"   */

	{ 2880,18,2,80,0,0x25,0x00,0xDF,0x02,"h1440" }, /* 10 1.44MB 5.25"  */
	{ 3360,21,2,80,0,0x1C,0x00,0xCF,0x0C,"H1680" }, /* 11 1.68MB 3.5"   */
	{  820,10,2,41,1,0x25,0x01,0xDF,0x2E,"h410"  },	/* 12 410KB 5.25"   */
	{ 1640,10,2,82,0,0x25,0x02,0xDF,0x2E,"H820"  },	/* 13 820KB 3.5"    */
	{ 2952,18,2,82,0,0x25,0x00,0xDF,0x02,"h1476" },	/* 14 1.48MB 5.25"  */
	{ 3444,21,2,82,0,0x25,0x00,0xDF,0x0C,"H1722" },	/* 15 1.72MB 3.5"   */
	{  840,10,2,42,1,0x25,0x01,0xDF,0x2E,"h420"  },	/* 16 420KB 5.25"   */
	{ 1660,10,2,83,0,0x25,0x02,0xDF,0x2E,"H830"  },	/* 17 830KB 3.5"    */
	{ 2988,18,2,83,0,0x25,0x00,0xDF,0x02,"h1494" },	/* 18 1.49MB 5.25"  */
	{ 3486,21,2,83,0,0x25,0x00,0xDF,0x0C,"H1743" }, /* 19 1.74 MB 3.5"  */

	{ 1760,11,2,80,0,0x1C,0x09,0xCF,0x00,"h880"  }, /* 20 880KB 5.25"   */
	{ 2080,13,2,80,0,0x1C,0x01,0xCF,0x00,"D1040" }, /* 21 1.04MB 3.5"   */
	{ 2240,14,2,80,0,0x1C,0x19,0xCF,0x00,"D1120" }, /* 22 1.12MB 3.5"   */
	{ 3200,20,2,80,0,0x1C,0x20,0xCF,0x2C,"h1600" }, /* 23 1.6MB 5.25"   */
	{ 3520,22,2,80,0,0x1C,0x08,0xCF,0x2e,"H1760" }, /* 24 1.76MB 3.5"   */
	{ 3840,24,2,80,0,0x1C,0x20,0xCF,0x00,"H1920" }, /* 25 1.92MB 3.5"   */
	{ 6400,40,2,80,0,0x25,0x5B,0xCF,0x00,"E3200" }, /* 26 3.20MB 3.5"   */
	{ 7040,44,2,80,0,0x25,0x5B,0xCF,0x00,"E3520" }, /* 27 3.52MB 3.5"   */
	{ 7680,48,2,80,0,0x25,0x63,0xCF,0x00,"E3840" }, /* 28 3.84MB 3.5"   */

	{ 3680,23,2,80,0,0x1C,0x10,0xCF,0x00,"H1840" }, /* 29 1.84MB 3.5"   */
	{ 1600,10,2,80,0,0x25,0x02,0xDF,0x2E,"D800"  },	/* 30 800KB 3.5"    */
	{ 3200,20,2,80,0,0x1C,0x00,0xCF,0x2C,"H1600" }, /* 31 1.6MB 3.5"    */
};

#define	NUMBER(x)	(sizeof(x) / sizeof(*(x)))	/*number of entities*/
#define SECTSIZE 	(_FD_SECTSIZE(*floppy))

/* Auto-detection: Disk type used until the next media change occurs. */
static struct floppy_struct *current_type[N_DRIVE];

/*
 * User-provided type information. current_type points to
 * the respective entry of this array.
 */
static struct floppy_struct user_params[N_DRIVE];

/*Actually the info in arrays below needs to be replicated with info kept with kernel*/
/*But since everywhere we refer to the local array, this is not done. */
/*So generic ioctls handled by kernel would give only the initial value*/
static int floppy_sizes[256];		/*device size*/
static int floppy_blocksizes[256];	/*block size for each device*/

/*
 * The driver is trying to determine the correct media format
 * while probing is set. rw_interrupt() clears it after a
 * successful access.
 */
static int probing;

/* Synchronization of FDC access. */
#define FD_COMMAND_NONE -1
#define FD_COMMAND_ERROR 2
#define FD_COMMAND_OKAY 3

static volatile int command_status = FD_COMMAND_NONE;
static unsigned long fdc_busy;				/*flag set to denote that driver is locked*/
pthread_mutex_t fdc_mutex = PTHREAD_MUTEX_INITIALIZER;	/*mutex for accessing fdc_busy*/
pthread_cond_t fdc_cond = PTHREAD_COND_INITIALIZER;	/*condition variable for providing sleep-wakeup mechanism while locking*/
pthread_mutex_t command_mutex = PTHREAD_MUTEX_INITIALIZER;	/*dummy mutex for below*/
pthread_cond_t command_cond = PTHREAD_COND_INITIALIZER;	/*condition variable for provinding sleep-wakeup mechanism for wait_til_done*/

#define NO_SIGNAL(i) (!interruptible || !userdev_check_sig_list((i)))
#define CALL(x) if ((x) == -EINTR) return -EINTR
#define ECALL(x) if ((ret = (x))) return ret;
#define _WAIT(x,i,r) CALL(ret=wait_til_done((x),i,r))
#define WAIT(x,r) _WAIT((x),interruptible,r)
#define IWAIT(x,r) _WAIT((x),1,r)

/* Errors during formatting are counted here. */
static int format_errors;

/* Format request descriptor. */
static struct format_descr format_req;

/*
 * Rate is 0 for 500kb/s, 1 for 300kbps, 2 for 250kbps
 * Spec1 is 0xSH, where S is stepping rate (F=1ms, E=2ms, D=3ms etc),
 * H is head unload time (1=16ms, 2=32ms, etc)
 */

/*
 * Track buffer
 * Because these are written to by the DMA controller, they must
 * not contain a 64k byte boundary crossing, or data will be
 * corrupted/lost.
 */
static char *floppy_track_buffer;	/*buffer to hold data for a complete track*/
static int max_buffer_sectors;		/*maximum number of sectors that can be accomadated in above*/

static int *errors;
typedef void (*done_f)(int);

/*continuations, each operation has its own set of these*/
static struct cont_t {
	void (*interrupt)(void);/* this is called after the interrupt of the main command */
	void (*redo)(void); 	/* this is called to retry the operation */
	void (*error)(void); 	/* this is called to tally an error */
	done_f done; 		/* this is called to say if the operation has succeeded/failed */
} *cont;

/*forward declarations*/
static void floppy_ready(void);
static void floppy_start(void);
static void process_fd_request(void);
static void recalibrate_floppy(void);
static void floppy_shutdown(void);
static int check_floppy_change(int,int);
static int floppy_revalidate(int,int);
static int floppy_grab_irq_and_dma(void);
static void floppy_release_irq_and_dma(void);

/*
 * The "reset" variable should be tested whenever an interrupt is scheduled,
 * after the commands have been sent. This is to ensure that the driver doesn't
 * get wedged when the interrupt doesn't come because of a failed command.
 * reset doesn't need to be tested before sending commands, because
 * output_byte is automatically disabled when reset is set.
 */
#define CHECK_RESET { if (FDCS->reset){ reset_fdc(); return; } }
static void reset_fdc(void);

/*
 * These are global variables, as that's the easiest way to give
 * information to interrupts. They are the data used for the current
 * request.
 */
#define NO_TRACK -1
#define NEED_1_RECAL -2
#define NEED_2_RECAL -3

/*keeps track of number of times irq & dma are acquired(incremented) and released(decremented)*/
static int usage_count;	

/* buffer related variables */
static int buffer_track = -1;
static int buffer_drive = -1;
static int buffer_min = -1;
static int buffer_max = -1;

/* fdc(floppy disk controller) related variables, should end up in a struct */
static struct floppy_fdc_state fdc_state[N_FDC];
static int fdc; /* current fdc */

static struct floppy_struct *_floppy = floppy_type;
static unsigned char current_drive;
static long current_count_sectors;
static unsigned char sector_t; 		/* sector in track */
static unsigned char in_sector_offset;	/* offset within physical sector, expressed in units of 512 bytes */

#ifndef fd_eject
#define fd_eject(x) -EINVAL
#endif

#ifdef DEBUGT
static long unsigned debugtimer;
#endif

/*
 * Debugging
 * =========
 */
static inline void set_debugt(void)
{
#ifdef DEBUGT
	debugtimer = 0;		/* = userdev_get_jiffies(); removed to reduce context switching*/
#endif
}

static inline void debugt(const char *message)
{
#ifdef DEBUGT
	if (DP->flags & DEBUGT)
		/*printf("%s dtime=%lu\n", message, userdev_get_jiffies()-debugtimer);*/
		/*removed above for sake of efficiency*/
		printf("%s dtime=%lu\n", message, 0);
#endif
}

typedef void (*timeout_fn)(unsigned long);
static struct userdev_timer_task fd_timeout ={ routine: (timeout_fn) floppy_shutdown };

static const char *timeout_message;

#ifdef FLOPPY_SANITY_CHECK
static void is_alive(const char *message)
{
	/* this routine checks whether the floppy driver is "alive" */
	if (fdc_busy && command_status < 2 && !userdev_timer_pending(&fd_timeout))
		DPRINT("timeout handler died: %s\n",message);
}
#endif

#ifdef FLOPPY_SANITY_CHECK

#define OLOGSIZE 20

static void (*lasthandler)(void);
static unsigned long interruptjiffies;	/*time of last interrupt*/
static unsigned long resultjiffies;	/*time of last result*/
static int resultsize;
static unsigned long lastredo;		/*last time redo_fd_request was called*/

static struct output_log {		/*log for each output_byte call*/
	unsigned char data;
	unsigned char status;
	unsigned long jiffies;
} output_log[OLOGSIZE];

static int output_log_pos;		/*pointer for log*/
#endif

#define CURRENTD -1
#define MAXTIMEOUT -2

/*function to reschedule the timeout for the given drive passing the message to be printed at timeout*/
/*Whenever an operation is started a timeout is set before which it needs to complete*/
/*if timeout expires before the operation completes then floppy_shutdown function is called*/
/*to cancel the operation and reset the drive*/
static void reschedule_timeout(int drive, const char *message, int marg)
{
	PRINT("reschedule_timeout called with msg = %s\n",message);
	if (drive == CURRENTD)
		drive = current_drive;
	userdev_del_timer(&fd_timeout);		/*delete timeout timer*/
	if (drive < 0 || drive > N_DRIVE) 
	{
		fd_timeout.expires = userdev_get_jiffies() + 20UL*HZ;
		drive=0;
	} 
	else fd_timeout.expires = userdev_get_jiffies() + UDP->timeout + 1200;	/*1200 added due to delays at user level*/
	userdev_add_timer(&fd_timeout);		/*add timeout timer*/
	if (UDP->flags & FD_DEBUG)
	{
		DPRINT("reschedule timeout ");
		printf(message, marg);
		printf("\n");
	}
	timeout_message = message;	/*assign message to be printed when timeout occurs*/
}

static inline int maximum(int a, int b)
{
	if (a > b) return a;
	else return b;
}
#define INFBOUND(a,b) (a)=maximum((a),(b));

static inline int minimum(int a, int b)
{
	if (a < b) return a;
	else return b;
}
#define SUPBOUND(a,b) (a)=minimum((a),(b));


/*
 * Bottom half floppy driver.
 * ==========================
 *
 * This part of the file contains the code talking directly to the hardware,
 * and also the main service loop (seek-configure-spinup-command)
 */

/*
 * This routine is responsible for maintaining the FD_DISK_CHANGE flag,
 * and the last_checked date.
 *
 * last_checked is the date of the last check which showed 'no disk change'
 * FD_DISK_CHANGE is set under two conditions:
 * 1. The floppy has been changed after some i/o to that floppy already
 *    took place.
 * 2. No floppy disk is in the drive. This is done in order to ensure that
 *    requests are quickly flushed in case there is no disk in the drive. It
 *    follows that FD_DISK_CHANGE can only be cleared if there is a disk in
 *    the drive.
 *
 * For 1., maxblock is observed. Maxblock is 0 if no i/o has taken place yet.
 * For 2., FD_DISK_NEWCHANGE is watched. FD_DISK_NEWCHANGE is cleared on
 *  each seek. If a disk is present, the disk change line should also be
 *  cleared on each seek. Thus, if FD_DISK_NEWCHANGE is clear, but the disk
 *  change line is set, this means either that no disk is in the drive, or
 *  that it has been removed since the last seek.
 *
 * This means that we really have a third possibility too:
 *  The floppy has been changed after the last seek.
 */
static int disk_change(int drive)
{
	int fdc = FDC(drive);
	int jiffies = userdev_get_jiffies();

	PRINT("disk change\n");
#ifdef FLOPPY_SANITY_CHECK
	if (jiffies - UDRS->select_date < UDP->select_delay)
		DPRINT("WARNING disk change called early\n");

	if (!(FDCS->dor & (0x10 << UNIT(drive))) || (FDCS->dor & 3) != UNIT(drive) || fdc != FDC(drive))
	{
		DPRINT("probing disk change on unselected drive\n");
		DPRINT("drive=%d fdc=%d dor=%x\n",drive, FDC(drive),(unsigned int)FDCS->dor);
	}
#endif

#ifdef DCL_DEBUG
	if (UDP->flags & FD_DEBUG)
	{
		DPRINT("checking disk change line for drive %d\n",drive);
		DPRINT("jiffies=%lu\n", jiffies);
		DPRINT("disk change line=%x\n",fd_inb(FD_DIR)&0x80);
		DPRINT("flags=%lx\n",UDRS->flags);
	}
#endif
	if (UDP->flags & FD_BROKEN_DCL)
		return UTESTF(FD_DISK_CHANGED);

	if ((fd_inb(FD_DIR) ^ UDP->flags) & 0x80)
	{
		USETF(FD_VERIFY); /* verify write protection */
		if (UDRS->maxblock)
			USETF(FD_DISK_CHANGED); /* mark it changed */

		/* invalidate its geometry */
		if (UDRS->keep_data >= 0) 
		{
			if ((UDP->flags & FTD_MSG) && current_type[drive] != NULL)
				DPRINT("Disk type is undefined after disk change\n");
			current_type[drive] = NULL;
			floppy_sizes[TOMINOR(drive)] = MAX_DISK_SIZE;
		}

		/*USETF(FD_DISK_NEWCHANGE);*/
		return 1;
	} 
	else 
	{
		UDRS->last_checked=jiffies;
		UCLEARF(FD_DISK_NEWCHANGE);
	}
	return 0;
}

static inline int is_selected(int dor, int unit)
{
	return ((dor  & (0x10 << unit)) && (dor &3) == unit);
}

static int set_dor(int fdc, char mask, char data)
{
	register unsigned char drive, unit, newdor,olddor;

	PRINT("set_dor\n");
	if (FDCS->address == -1)
		return -1;

	olddor = FDCS->dor;
	newdor =  (olddor & mask) | data;
	if (newdor != olddor)
	{
		unit = olddor & 0x3;
		if (is_selected(olddor, unit) && !is_selected(newdor,unit))
		{
			drive = REVDRIVE(fdc,unit);
#ifdef DCL_DEBUG
			if (UDP->flags & FD_DEBUG)
				DPRINT("calling disk change from set_dor\n");
#endif
			disk_change(drive);
		}
		FDCS->dor = newdor;
		fd_outb(newdor, FD_DOR);

		unit = newdor & 0x3;
		if (!is_selected(olddor, unit) && is_selected(newdor,unit))
		{
			drive = REVDRIVE(fdc,unit);
			UDRS->select_date = userdev_get_jiffies();
		}
	}
	if (newdor & FLOPPY_MOTOR_MASK)
		floppy_grab_irq_and_dma();
	if (olddor & FLOPPY_MOTOR_MASK)
		floppy_release_irq_and_dma();
	return olddor;
}

static void twaddle(void)
{
	PRINT("twaddle\n");
	if (DP->select_delay) return;
	fd_outb(FDCS->dor & ~(0x10<<UNIT(current_drive)), FD_DOR);
	fd_outb(FDCS->dor, FD_DOR);
	DRS->select_date = userdev_get_jiffies();
}

/* reset all driver information about the current fdc. This is needed after
 * a reset, and after a raw command. */
static void reset_fdc_info(int mode)
{
	int drive;

	PRINT("reset_fdc_info\n");
	FDCS->spec1 = FDCS->spec2 = -1;
	FDCS->need_configure = 1;
	FDCS->perp_mode = 1;
	FDCS->rawcmd = 0;
	for (drive = 0; drive < N_DRIVE; drive++)
		if (FDC(drive) == fdc && (mode || UDRS->track != NEED_1_RECAL))
			UDRS->track = NEED_2_RECAL;
}

/* selects the fdc and drive, and enables the fdc's input/dma. */
static void set_fdc(int drive)
{
	PRINT("set_fdc\n");
	if (drive >= 0 && drive < N_DRIVE)
	{
		fdc = FDC(drive);
		current_drive = drive;
	}
	if (fdc != 1 && fdc != 0) 
	{
		printf("bad fdc value\n");
		return;
	}
	set_dor(fdc,~0,8);
#if N_FDC > 1
	set_dor(1-fdc, ~8, 0);
#endif
	if (FDCS->rawcmd == 2)
		reset_fdc_info(1);
	if (fd_inb(FD_STATUS) != STATUS_READY)
		FDCS->reset = 1;
}

# ifdef PRINT_FUNCTION_ENTRY
static int l=0;		/*level of lock_fdc, used for log*/
# endif

/* locks the driver */
static int _lock_fdc(int drive, int interruptible, int line, int reqid)
{
	PRINT("---------lock fdc--------- %d\n",++l);
	if (!usage_count)
	{
		printf("Trying to lock fdc while usage count=0 at line %d\n", line);
		return -1;
	}
	if(floppy_grab_irq_and_dma()==-1)
		return -EBUSY;

	pthread_mutex_lock(&fdc_mutex);		/*obtain mutex on fdc_mutex*/
	while(fdc_busy)				/*as long as driver is locked*/
	{
		pthread_cond_wait(&fdc_cond,&fdc_mutex);	/*wait for fdc_busy to be unset*/
		if(!NO_SIGNAL(reqid)) 		/*incase woken up not due to cond_signal, then return error*/
		{
			pthread_mutex_unlock(&fdc_mutex);
			return -EINTR;
		}
	}
	fdc_busy = 1;				/*out here when fdc_busy is unset, so set it now*/
	pthread_mutex_unlock(&fdc_mutex);	/*release mutex on fdc_mutex*/

	command_status = FD_COMMAND_NONE;

	reschedule_timeout(drive, "lock fdc", 0);
	set_fdc(drive);
	return 0;
}

#define lock_fdc(drive,interruptible,reqid) _lock_fdc(drive,interruptible, __LINE__,reqid)

#define LOCK_FDC(drive,interruptible,reqid) \
if (lock_fdc(drive,interruptible,reqid)) return -EINTR;


/* unlocks the driver */
static inline void unlock_fdc(void)
{
	PRINT("------------unlock_fdc----------- %d\n",l--);
	raw_cmd = 0;
	if (!fdc_busy)
		DPRINT("FDC access conflict!\n");

	if (DEVICE_INTR)
		DPRINT("device interrupt still active at FDC release: %p!\n",DEVICE_INTR);
	command_status = FD_COMMAND_NONE;
	userdev_del_timer(&fd_timeout);
	cont = NULL;

	floppy_release_irq_and_dma();
	pthread_mutex_lock(&fdc_mutex);	/*reset fdc_busy*/
	fdc_busy = 0;
	pthread_mutex_unlock(&fdc_mutex);
	pthread_cond_signal(&fdc_cond);	/*signal any thread blocked on lock_fdc*/
}

/* switches the motor off after a given timeout */
static void motor_off_callback(unsigned long nr)
{
	unsigned char mask = ~(0x10 << UNIT(nr));

	PRINT("motor callback\n");
	set_dor(FDC(nr), mask, 0);
}

static struct userdev_timer_task motor_off_timer[N_DRIVE] = {
	{ data: 0, routine: motor_off_callback },
	{ data: 1, routine: motor_off_callback },
	{ data: 2, routine: motor_off_callback },
	{ data: 3, routine: motor_off_callback },
	{ data: 4, routine: motor_off_callback },
	{ data: 5, routine: motor_off_callback },
	{ data: 6, routine: motor_off_callback },
	{ data: 7, routine: motor_off_callback }
};

/* schedules motor off */
static void floppy_off(unsigned int drive)
{
	unsigned long volatile delta;
	register int fdc=FDC(drive);

	PRINT("//////////////floppy_off\n");
	if (!(FDCS->dor & (0x10 << UNIT(drive)))) return;

	userdev_del_timer(motor_off_timer+drive);

	/* make spindle stop in a position which minimizes spinup time next time */
	if (UDP->rps)
	{
		int jiffies = userdev_get_jiffies();
		delta = jiffies - UDRS->first_read_date + HZ - UDP->spindown_offset;
		delta = ((delta * UDP->rps) % HZ) / UDP->rps;
		motor_off_timer[drive].expires = jiffies + UDP->spindown - delta;
	}
	userdev_add_timer(motor_off_timer+drive);
}

/*
 * cycle through all N_DRIVE floppy drives, for disk change testing.
 * stopping at current drive. This is done before any long operation, to
 * be sure to have up to date disk change information.
 */
static void scandrives(void)
{
	int i, drive, saved_drive;

	PRINT("scandrives\n");
	if (DP->select_delay) return;

	saved_drive = current_drive;
	for (i=0; i < N_DRIVE; i++)
	{
		drive = (saved_drive + i + 1) % N_DRIVE;
		if (UDRS->fd_ref == 0 || UDP->select_delay != 0)
			continue; /* skip closed drives */
		set_fdc(drive);
		if (!(set_dor(fdc, ~3, UNIT(drive) | (0x10 << UNIT(drive))) & (0x10 << UNIT(drive))))
			/* switch the motor off again, if it was off to begin with */
			set_dor(fdc, ~(0x10 << UNIT(drive)), 0);
	}
	set_fdc(saved_drive);
}

static void empty(void)
{
}

/*task that is enqueued for each task scheduled by schedule_bh function*/
static struct userdev_task floppy_tq = { sync:0 };

static void schedule_bh( void (*handler)(void*) )
{
	PRINT("schedule bh\n");
	floppy_tq.routine = (void *)(void *) handler;
	userdev_queue_task(&floppy_tq, &tq_immediate);
}

/*timer task used for delaying execution of a particular function*/
static struct userdev_timer_task fd_timer;

/*function called to cancel all activity currently being done by the driver*/
static void cancel_activity(void)
{
	PRINT("cancel activity\n");
	CLEAR_INTR;
	floppy_tq.routine = (void *)(void *) empty;
	userdev_del_timer(&fd_timer);
}

/* this function makes sure that the disk stays in the drive during the transfer */
static void fd_watchdog(void)
{
	PRINT("fd_watchdog\n");
#ifdef DCL_DEBUG
	if (DP->flags & FD_DEBUG)
		DPRINT("calling disk change from watchdog\n");
#endif

	if (disk_change(current_drive))
	{
		DPRINT("disk removed during i/o\n");
		cancel_activity();
		cont->done(0);
		reset_fdc();
	} 
	else 
	{
		userdev_del_timer(&fd_timer);
		fd_timer.routine = (timeout_fn) fd_watchdog;
		fd_timer.expires = userdev_get_jiffies() + HZ / 10;
		userdev_add_timer(&fd_timer);
	}
}

static void main_command_interrupt(void)
{
	PRINT("main command interrupt\n");
	userdev_del_timer(&fd_timer);
	cont->interrupt();
}

/* waits for a delay (spinup or select) to pass */
static int wait_for_completion(unsigned long delay, timeout_fn function)
{
	PRINT("wait for completion\n");
	if (FDCS->reset)
	{
		/* do the reset during sleep to win time if we don't need to sleep, it's a good occasion anyways */
		reset_fdc(); 
		return 1;
	}

	if ((signed) (userdev_get_jiffies() - delay) < 0)
	{
		userdev_del_timer(&fd_timer);
		fd_timer.routine = function;
		fd_timer.expires = delay;
		userdev_add_timer(&fd_timer);
		return 1;
	}
	return 0;
}

/*The below code has no significance in user level*/
static int hlt_disabled;
static pthread_mutex_t floppy_hlt_lock = PTHREAD_MUTEX_INITIALIZER;

static void floppy_disable_hlt(void)
{
	PRINT("floppy disable halt\n");
	pthread_mutex_lock(&floppy_hlt_lock);
	if (!hlt_disabled) 
		hlt_disabled = 1;
	pthread_mutex_unlock(&floppy_hlt_lock);
}

static void floppy_enable_hlt(void)
{
	PRINT("floppy enable halt\n");
	pthread_mutex_lock(&floppy_hlt_lock);
	if (hlt_disabled)
		hlt_disabled = 0;
	pthread_mutex_unlock(&floppy_hlt_lock);
}


static void setup_DMA(void)
{
	PRINT("setup DMA\n");

#ifdef FLOPPY_SANITY_CHECK
	if (raw_cmd->length == 0)
	{
		int i;

		printf("zero dma transfer size:");
		for (i=0; i < raw_cmd->cmd_count; i++)
			printf("%x,", raw_cmd->cmd[i]);
		printf("\n");
		cont->done(0);
		FDCS->reset = 1;
		return;
	}
#endif

	/*setup parameters for DMA operation*/
	if (fd_dma_setup(raw_cmd->kernel_data, raw_cmd->length, 
		(raw_cmd->flags & FD_RAW_READ)?DMA_READ : DMA_WRITE, FDCS->address,userdev_id) < 0) 
	{
		cont->done(0);
		FDCS->reset=1;
		return;
	}
	floppy_disable_hlt();
}

static void show_floppy(void);

/* waits until the fdc becomes ready */
static int wait_til_ready(void)
{
	int counter, status;

	PRINT("wait_til_ready\n");
	if (FDCS->reset) return -1;
	for (counter = 0; counter < 10000; counter++) 
	{
		status = fd_inb(FD_STATUS);		
		if (status & STATUS_READY)
			return status;
	}
	if (!initialising) 
	{
		DPRINT("Getstatus times out (%x) on fdc %d\n",status, fdc);
		show_floppy();
	}
	FDCS->reset = 1;
	return -1;
}

/* sends a command byte to the fdc */
static int output_byte(char byte)
{
	int status;

	PRINT("output byte\n");
	if ((status = wait_til_ready()) < 0) return -1;
	if ((status & (STATUS_READY|STATUS_DIR|STATUS_DMA)) == STATUS_READY)
	{
		fd_outb(byte,FD_DATA);
#ifdef FLOPPY_SANITY_CHECK
		output_log[output_log_pos].data = byte;
		output_log[output_log_pos].status = status;
		output_log[output_log_pos].jiffies = 0;	/*userdev_get_jiffies(); //very costly as done for each byte*/
		output_log_pos = (output_log_pos + 1) % OLOGSIZE;
#endif
		return 0;
	}
	FDCS->reset = 1;
	if (!initialising) 
	{
		DPRINT("Unable to send byte %x to FDC. Fdc=%x Status=%x\n",byte, fdc, status);
		show_floppy();
	}
	return -1;
}
#define LAST_OUT(x) if (output_byte(x)<0){ reset_fdc();return;}

/* gets the response from the fdc */
static int result(void)
{
	int i, status=0;

	PRINT("result\n");
	for(i=0; i < MAX_REPLIES; i++) 
	{
		if ((status = wait_til_ready()) < 0) break;
		status &= STATUS_DIR|STATUS_READY|STATUS_BUSY|STATUS_DMA;
		if ((status & ~STATUS_BUSY) == STATUS_READY)
		{
#ifdef FLOPPY_SANITY_CHECK
			resultjiffies = 0;/*userdev_get_jiffies(); //very costly*/
			resultsize = i;
#endif
			return i;
		}
		if (status == (STATUS_DIR|STATUS_READY|STATUS_BUSY))
			reply_buffer[i] = fd_inb(FD_DATA);
		else break;
	}
	if (!initialising) 
	{
		DPRINT("get result error. Fdc=%d Last status=%x Read bytes=%d\n",
		       fdc, status, i);
		show_floppy();
	}
	FDCS->reset = 1;
	return -1;
}

#define MORE_OUTPUT -2
/* does the fdc need more output? */
static int need_more_output(void)
{
	int status;
	PRINT("need more output\n");
	if ((status = wait_til_ready()) < 0) return -1;
	if ((status & (STATUS_READY|STATUS_DIR|STATUS_DMA)) == STATUS_READY)
		return MORE_OUTPUT;
	return result();
}

/* Set perpendicular mode as required, based on data rate, if supported.
 * 82077 Now tested. 1Mbps data rate only possible with 82077-1.*/
static inline void perpendicular_mode(void)
{
	unsigned char perp_mode;

	PRINT("perpendicular mode\n");
	if (raw_cmd->rate & 0x40)
	{
		switch(raw_cmd->rate & 3)
		{
			case 0:
				perp_mode=2;
				break;
			case 3:
				perp_mode=3;
				break;
			default:
				DPRINT("Invalid data rate for perpendicular mode!\n");
				cont->done(0);
				FDCS->reset = 1; /* convenient way to return to redo without to much hassle (deep stack et al. */
				return;
		}
	} 
	else perp_mode = 0;

	if (FDCS->perp_mode == perp_mode) return;
	if (FDCS->version >= FDC_82077_ORIG) 
	{
		output_byte(FD_PERPENDICULAR);
		output_byte(perp_mode);
		FDCS->perp_mode = perp_mode;
	} 
	else if (perp_mode) 
		DPRINT("perpendicular mode not supported by this FDC.\n");
} /* perpendicular_mode */

static int fifo_depth = 0xa;
static int no_fifo;

static int fdc_configure(void)
{
	/* Turn on FIFO */
	PRINT("fdc configure\n");
	output_byte(FD_CONFIGURE);
	if (need_more_output() != MORE_OUTPUT) return 0;
	output_byte(0);
	output_byte(0x10 | (no_fifo & 0x20) | (fifo_depth & 0xf));
	output_byte(0);	/* pre-compensation from track 0 upwards */
	return 1;
}	

#define NOMINAL_DTR 500

/* Issue a "SPECIFY" command to set the step rate time, head unload time,
 * head load time, and DMA disable flag to values needed by floppy.
 *
 * The value "dtr" is the data transfer rate in Kbps.  It is needed
 * to account for the data rate-based scaling done by the 82072 and 82077
 * FDC types.  This parameter is ignored for other types of FDCs (i.e. 8272a).
 *
 * Note that changing the data transfer rate has a (probably deleterious)
 * effect on the parameters subject to scaling for 82072/82077 FDCs, so
 * fdc_specify is called again after each data transfer rate change.
 *
 * srt: 1000 to 16000 in microseconds
 * hut: 16 to 240 milliseconds
 * hlt: 2 to 254 milliseconds
 *
 * These values are rounded up to the next highest available delay time.
 */
static void fdc_specify(void)
{
	unsigned char spec1, spec2;
	unsigned long srt, hlt, hut;
	unsigned long dtr = NOMINAL_DTR;
	unsigned long scale_dtr = NOMINAL_DTR;
	int hlt_max_code = 0x7f;
	int hut_max_code = 0xf;

	PRINT("fdc specify\n");
	if (FDCS->need_configure && FDCS->version >= FDC_82072A) 
	{
		fdc_configure();
		FDCS->need_configure = 0;
	}

	switch (raw_cmd->rate & 0x03) 
	{
		case 3:
			dtr = 1000;
			break;
		case 1:
			dtr = 300;
			if (FDCS->version >= FDC_82078) 
			{
				/* chose the default rate table, not the one where 1 = 2 Mbps */
				output_byte(FD_DRIVESPEC);
				if (need_more_output() == MORE_OUTPUT) 
				{
					output_byte(UNIT(current_drive));
					output_byte(0xc0);
				}
			}
			break;
		case 2:
			dtr = 250;
			break;
	}

	if (FDCS->version >= FDC_82072) 
	{
		scale_dtr = dtr;
		hlt_max_code = 0x00; /* 0==256msec*dtr0/dtr (not linear!) */
		hut_max_code = 0x0; /* 0==256msec*dtr0/dtr (not linear!) */
	}

	/* Convert step rate from microseconds to milliseconds and 4 bits */
	srt = 16 - (DP->srt*scale_dtr/1000 + NOMINAL_DTR - 1)/NOMINAL_DTR;
	if( slow_floppy ) srt = srt / 4;

	SUPBOUND(srt, 0xf);
	INFBOUND(srt, 0);

	hlt = (DP->hlt*scale_dtr/2 + NOMINAL_DTR - 1)/NOMINAL_DTR;
	if (hlt < 0x01)
		hlt = 0x01;
	else if (hlt > 0x7f)
		hlt = hlt_max_code;

	hut = (DP->hut*scale_dtr/16 + NOMINAL_DTR - 1)/NOMINAL_DTR;
	if (hut < 0x1)
		hut = 0x1;
	else if (hut > 0xf)
		hut = hut_max_code;

	spec1 = (srt << 4) | hut;
	spec2 = (hlt << 1) | (use_virtual_dma & 1);

	/* If these parameters did not change, just return with success */
	if (FDCS->spec1 != spec1 || FDCS->spec2 != spec2) 
	{
		/* Go ahead and set spec1 and spec2 */
		output_byte(FD_SPECIFY);
		output_byte(FDCS->spec1 = spec1);
		output_byte(FDCS->spec2 = spec2);
	}
} /* fdc_specify */

/* Set the FDC's data transfer rate on behalf of the specified drive.
 * NOTE: with 82072/82077 FDCs, changing the data rate requires a reissue
 * of the specify command (i.e. using the fdc_specify function).*/
static int fdc_dtr(void)
{
	PRINT("fdc dtr\n");
	/* If data rate not already set to desired value, set it. */
	if ((raw_cmd->rate & 3) == FDCS->dtr) return 0;

	/* Set dtr */
	fd_outb(raw_cmd->rate & 3, FD_DCR);

	/* TODO: some FDC/drive combinations (C&T 82C711 with TEAC 1.2MB)
	 * need a stabilization period of several milliseconds to be
	 * enforced after data rate changes before R/W operations.
	 * Pause 5 msec to avoid trouble. (Needs to be 2 jiffies)
	 */
	FDCS->dtr = raw_cmd->rate & 3;
	return(wait_for_completion(userdev_get_jiffies()+2UL*HZ/100, (timeout_fn) floppy_ready));
} /* fdc_dtr */

static inline void tell_sector(void)
{
	printf(": track %d, head %d, sector %d, size %d", R_TRACK, R_HEAD, R_SECTOR, R_SIZECODE);
} 


/*
 * OK, this error interpreting routine is called after a
 * DMA read/write has succeeded
 * or failed, so we check the results, and copy any buffers.
 */
static int interpret_errors(void)
{
	char bad;

	PRINT("interpret errors\n");
	if (inr != 7) 
	{
		DPRINT("-- FDC reply error");
		FDCS->reset = 1;
		return 1;
	}

	/* check IC to find cause of interrupt */
	switch (ST0 & ST0_INTR) 
	{
		case 0x40:	/* error occurred during command execution */
			if (ST1 & ST1_EOC)
				return 0; /* occurs with pseudo-DMA */
			bad = 1;
			if (ST1 & ST1_WP) 
			{
				DPRINT("Drive is write protected\n");
				CLEARF(FD_DISK_WRITABLE);
				cont->done(0);
				bad = 2;
			} 
			else if (ST1 & ST1_ND) 
				SETF(FD_NEED_TWADDLE);
			else if (ST1 & ST1_OR) 
			{
				if (DP->flags & FTD_MSG)
					DPRINT("Over/Underrun - retrying\n");
				bad = 0;
			}
			else if (*errors >= DP->max_errors.reporting)
			{
				DPRINT("");
				if (ST0 & ST0_ECE) 
				{
					printf("Recalibrate failed!");
				} 
				else if (ST2 & ST2_CRC) 
				{
					printf("data CRC error");
					tell_sector();
				} 
				else if (ST1 & ST1_CRC) 
				{
					printf("CRC error");
					tell_sector();
				} 
				else if ((ST1 & (ST1_MAM|ST1_ND)) || (ST2 & ST2_MAM)) 
				{
					if (!probing) 
					{
						printf("sector not found");
						tell_sector();
					} 
					else printf("probe failed...");
				} 
				else if (ST2 & ST2_WC) 
				{	/* seek error */
					printf("wrong cylinder");
				} 
				else if (ST2 & ST2_BC) 
				{	/* cylinder marked as bad */
					printf("bad cylinder");
				} 
				else 
				{
					printf("unknown error. ST[0..2] are: 0x%x 0x%x 0x%x", ST0, ST1, ST2);
					tell_sector();
				}
				printf("\n");

			}
			if (ST2 & ST2_WC || ST2 & ST2_BC) 	/* wrong cylinder => recal */
				DRS->track = NEED_2_RECAL;
			return bad;
		case 0x80: /* invalid command given */
			DPRINT("Invalid FDC command given!\n");
			cont->done(0);
			return 2;
		case 0xc0:
			DPRINT("Abnormal termination caused by polling\n");
			cont->error();
			return 2;
		default: /* (0) Normal command termination */
			return 0;
	}
}

/*
 * This routine is called when everything should be correctly set up
 * for the transfer (i.e. floppy motor is on, the correct floppy is
 * selected, and the head is sitting on the right track).*/
static void setup_rw_floppy(void)
{
	int i,r, flags,dflags;
	unsigned long ready_date;
	timeout_fn function;

	PRINT("setup rw floppy\n");
	flags = raw_cmd->flags;
	if (flags & (FD_RAW_READ | FD_RAW_WRITE))
		flags |= FD_RAW_INTR;

	if ((flags & FD_RAW_SPIN) && !(flags & FD_RAW_NO_MOTOR))
	{
		ready_date = DRS->spinup_date + DP->spinup;
		/* If spinup will take a long time, rerun scandrives
		 * again just before spinup completion. Beware that
		 * after scandrives, we must again wait for selection.*/
		if ((signed) (ready_date - userdev_get_jiffies()) > DP->select_delay)
		{
			ready_date -= DP->select_delay;
			function = (timeout_fn) floppy_start;
		} 
		else function = (timeout_fn) setup_rw_floppy;

		/* wait until the floppy is spinning fast enough */
		if (wait_for_completion(ready_date,function)) return;
	}
	dflags = DRS->flags;

	if ((flags & FD_RAW_READ) || (flags & FD_RAW_WRITE))
		setup_DMA();

	if (flags & FD_RAW_INTR)
		SET_INTR(main_command_interrupt);

	/*The following is set of commands sent to the floppy controller for*/
	/*each operation to be performed*/
	/*
	printf("command = %d\n",COMMAND);
	printf("dr_select = %d\n",DR_SELECT);
	printf("track = %d\n",TRACK);
	printf("head = %d\n",HEAD);
	printf("sector = %d\n",SECTOR);
	printf("sizecode = %d\n",SIZECODE);
	printf("sect per track = %d\n",SECT_PER_TRACK);
	printf("gap = %d\n",GAP);
	printf("sizecode2 = %d\n",SIZECODE2);
	*/

	r=0;
	for (i=0; i< raw_cmd->cmd_count; i++)
		r|=output_byte(raw_cmd->cmd[i]);

#ifdef DEBUGT
	debugt("rw_command: ");
#endif
	if (r)
	{
		cont->error();
		reset_fdc();
		return;
	}

	if (!(flags & FD_RAW_INTR))
	{
		inr = result();
		cont->interrupt();
	} 
	else if (flags & FD_RAW_NEED_DISK)
		fd_watchdog();
}

static int blind_seek;

/*
 * This is the routine called after every seek (or recalibrate) interrupt
 * from the floppy controller.
 */
static void seek_interrupt(void)
{
#ifdef DEBUGT
	debugt("seek interrupt:");
#endif
	PRINT("seek interrupt\n");
	if (inr != 2 || (ST0 & 0xF8) != 0x20) 
	{
		DPRINT("seek failed\n");
		DRS->track = NEED_2_RECAL;
		cont->error();
		cont->redo();
		return;
	}
	if (DRS->track >= 0 && DRS->track != ST1 && !blind_seek)
	{
		int jiffies = userdev_get_jiffies();
#ifdef DCL_DEBUG
		if (DP->flags & FD_DEBUG)
		{
			DPRINT("clearing NEWCHANGE flag because of effective seek\n");
			DPRINT("jiffies=%lu\n", jiffies);
		}
#endif
		CLEARF(FD_DISK_NEWCHANGE); /* effective seek */
		DRS->select_date = jiffies;
	}
	DRS->track = ST1;
	floppy_ready();
}

/*function to chekc if the disk is write protected*/
static void check_wp(void)
{
	PRINT("check wp\n");
	if (TESTF(FD_VERIFY)) 
	{
		/* check write protection */
		output_byte(FD_GETSTATUS);
		output_byte(UNIT(current_drive));
		if (result() != 1)
		{
			FDCS->reset = 1;
			return;
		}
		CLEARF(FD_VERIFY);
		CLEARF(FD_NEED_TWADDLE);
#ifdef DCL_DEBUG
		if (DP->flags & FD_DEBUG)
		{
			DPRINT("checking whether disk is write protected\n");
			DPRINT("wp=%x\n",ST3 & 0x40);
		}
#endif
		if (!(ST3  & 0x40))
			SETF(FD_DISK_WRITABLE);
		else
			CLEARF(FD_DISK_WRITABLE);
	}
}

/*function to perform seek operation*/
static void seek_floppy(void)
{
	int track;

	PRINT("seek floppy\n");
	blind_seek=0;

#ifdef DCL_DEBUG
	if (DP->flags & FD_DEBUG)
		DPRINT("calling disk change from seek\n");
#endif

	if (!TESTF(FD_DISK_NEWCHANGE) && disk_change(current_drive) && (raw_cmd->flags & FD_RAW_NEED_DISK))
	{
		/* the media changed flag should be cleared after the seek.
		 * If it isn't, this means that there is really no disk in
		 * the drive.*/
		SETF(FD_DISK_CHANGED);
		cont->done(0);
		cont->redo();
		return;
	}
	if (DRS->track <= NEED_1_RECAL)
	{
		recalibrate_floppy();
		return;
	} 
	else if (TESTF(FD_DISK_NEWCHANGE) && (raw_cmd->flags & FD_RAW_NEED_DISK) &&
		   (DRS->track <= NO_TRACK || DRS->track == raw_cmd->track)) 
	{
		/* we seek to clear the media-changed condition. Does anybody
		 * know a more elegant way, which works on all drives? */
		if (raw_cmd->track)
			track = raw_cmd->track - 1;
		else 
		{
			if (DP->flags & FD_SILENT_DCL_CLEAR)
			{
				set_dor(fdc, ~(0x10 << UNIT(current_drive)), 0);
				blind_seek = 1;
				raw_cmd->flags |= FD_RAW_NEED_SEEK;
			}
			track = 1;
		}
	} 
	else 
	{
		check_wp();
		if (raw_cmd->track != DRS->track && (raw_cmd->flags & FD_RAW_NEED_SEEK))
			track = raw_cmd->track;
		else 
		{
			setup_rw_floppy();
			return;
		}
	}

	SET_INTR(seek_interrupt);
	output_byte(FD_SEEK);
	output_byte(UNIT(current_drive));
	LAST_OUT(track);
#ifdef DEBUGT
	debugt("seek command:");
#endif
}

/*function that handles interrupt that arrives after floppy is recalibrated*/
static void recal_interrupt(void)
{
#ifdef DEBUGT
	debugt("recal interrupt:");
#endif
	PRINT("recal interrupt\n");
	if (inr != 2) FDCS->reset = 1;
	else if (ST0 & ST0_ECE) 
	{
	       	switch(DRS->track)
		{
			case NEED_1_RECAL:
#ifdef DEBUGT
				debugt("recal interrupt need 1 recal:");
#endif
				/* after a second recalibrate, we still haven't
				 * reached track 0. Probably no drive. Raise an
				 * error, as failing immediately might upset
				 * computers possessed by the Devil :-) */
				cont->error();
				cont->redo();
				return;
			case NEED_2_RECAL:
#ifdef DEBUGT
				debugt("recal interrupt need 2 recal:");
#endif
				/* If we already did a recalibrate,
				 * and we are not at track 0, this
				 * means we have moved. (The only way
				 * not to move at recalibration is to
				 * be already at track 0.) Clear the
				 * new change flag */
#ifdef DCL_DEBUG
				if (DP->flags & FD_DEBUG)
					DPRINT("clearing NEWCHANGE flag because of second recalibrate\n");
#endif

				CLEARF(FD_DISK_NEWCHANGE);
				DRS->select_date = userdev_get_jiffies();
				/* fall through */
			default:
#ifdef DEBUGT
				debugt("recal interrupt default:");
#endif
				/* Recalibrate moves the head by at
				 * most 80 steps. If after one
				 * recalibrate we don't have reached
				 * track 0, this might mean that we
				 * started beyond track 80.  Try
				 * again.  */
				DRS->track = NEED_1_RECAL;
				break;
		}
	} 
	else DRS->track = ST1;
	floppy_ready();
}

static void print_result(char *message, int inr)
{
	int i;

	DPRINT("%s ", message);
	if (inr >= 0)
		for (i=0; i<inr; i++)
			printf("repl[%d]=%x ", i, reply_buffer[i]);
	printf("\n");
}

/* interrupt handler. Note that this can be called externally on the Sparc */
void floppy_interrupt(int sig, siginfo_t *info, void *ptr)
{
	void (*handler)(void) = DEVICE_INTR;
	int do_print;
	unsigned long f;
	int devid;
	
	PRINT("floppy interrupt\n");
	if (info != NULL) devid = info->si_value.sival_int;
	else devid = userdev_id;

	lasthandler = handler;
	interruptjiffies = 0;	/*=userdev_get_jiffies();  //very costly*/

	fd_disable_dma(userdev_id);
	
	floppy_enable_hlt();
	CLEAR_INTR;
	if (fdc >= N_FDC || FDCS->address == -1)
	{
		/* we don't even know which FDC is the culprit */
		printf("DOR0=%x\n", fdc_state[0].dor);
		printf("floppy interrupt on bizarre fdc %d\n",fdc);
		printf("handler=%p\n", handler);
		is_alive("bizarre fdc");
		return;
	}

	FDCS->reset = 0;
	/* We have to clear the reset flag here, because apparently on boxes
	 * with level triggered interrupts (PS/2, Sparc, ...), it is needed to
	 * emit SENSEI's to clear the interrupt line. And FDCS->reset blocks the
	 * emission of the SENSEI's.
	 * It is OK to emit floppy commands because we are in an interrupt
	 * handler here, and thus we have to fear no interference of other
	 * activity.
	 */

	do_print = !handler && print_unex && !initialising;

	inr = result();
	if (do_print) print_result("unexpected interrupt", inr);
	if (inr == 0)
	{
		int max_sensei = 4;
		do 
		{
			output_byte(FD_SENSEI);
			inr = result();
			if (do_print) print_result("sensei", inr);
			max_sensei--;
		} while ((ST0 & 0x83) != UNIT(current_drive) && inr == 2 && max_sensei);
	}
	if (handler) /*schedule task pointer to DEVICE_INTR*/
		schedule_bh( (void *)(void *) handler);
	else
		FDCS->reset = 1;
	is_alive("normal interrupt end");
}

/*function to recalibrate the floppy*/
static void recalibrate_floppy(void)
{
#ifdef DEBUGT
	debugt("recalibrate floppy:");
#endif
	PRINT("recalibrate floppy\n");
	SET_INTR(recal_interrupt);
	output_byte(FD_RECALIBRATE);
	LAST_OUT(UNIT(current_drive));
}

/*Must do 4 FD_SENSEIs after reset because of ``drive polling''.*/
static void reset_interrupt(void)
{
#ifdef DEBUGT
	debugt("reset interrupt:");
#endif
	PRINT("reset interrupt\n");
	result();		/* get the status ready for set_fdc */
	if (FDCS->reset) 
	{
		printf("reset set in interrupt, calling %p\n", cont->error);
		cont->error(); /* a reset just after a reset. BAD! */
	}
	cont->redo();
}

/*
 * reset is done by pulling bit 2 of DOR low for a while (old FDCs),
 * or by setting the self clearing bit 7 of STATUS (newer FDCs)
 */
static void reset_fdc(void)
{
	unsigned long flags;
	
	PRINT("reset fdc\n");
	SET_INTR(reset_interrupt);
	FDCS->reset = 0;
	reset_fdc_info(0);

	/* Pseudo-DMA may intercept 'reset finished' interrupt.  */
	/* Irrelevant for systems with true DMA (i386).          */
	
	fd_disable_dma(userdev_id);

	if (FDCS->version >= FDC_82072A)
		fd_outb(0x80 | (FDCS->dtr &3), FD_STATUS);
	else 
	{
		fd_outb(FDCS->dor & ~0x04, FD_DOR);
		userdev_udelay(FD_RESET_DELAY);
		fd_outb(FDCS->dor, FD_DOR);
	}
}

static void show_floppy(void)
{
	int i;
	unsigned long jiffies = userdev_get_jiffies();

	printf("\n");
	printf("floppy driver state\n");
	printf("-------------------\n");
	printf("now=%lu last interrupt=%lu diff=%lu last called handler=%p\n",
	       	jiffies, interruptjiffies, jiffies-interruptjiffies, lasthandler);


#ifdef FLOPPY_SANITY_CHECK
	printf("timeout_message=%s\n", timeout_message);
	printf("last output bytes:\n");
	for (i=0; i < OLOGSIZE; i++)
		printf("%2x %2x %lu\n",output_log[(i+output_log_pos) % OLOGSIZE].data,
		       output_log[(i+output_log_pos) % OLOGSIZE].status, output_log[(i+output_log_pos) % OLOGSIZE].jiffies);
	printf("last result at %lu\n", resultjiffies);
	printf("last redo_fd_request at %lu\n", lastredo);
	for (i=0; i<resultsize; i++)
		printf("%2x ", reply_buffer[i]);
	printf("\n");
#endif

	printf("status=%x\n", fd_inb(FD_STATUS));
	printf("fdc_busy=%lu\n", fdc_busy);
	if (DEVICE_INTR)
		printf("DEVICE_INTR=%p\n", DEVICE_INTR);
	if (floppy_tq.sync)
		printf("floppy_tq.routine=%p\n", floppy_tq.routine);
	if (userdev_timer_pending(&fd_timer))
		printf("fd_timer.function=%p\n", fd_timer.routine);
	if (userdev_timer_pending(&fd_timeout))
	{
		printf("timer_function=%p\n",fd_timeout.routine);
		printf("expires=%lu\n",fd_timeout.expires-jiffies);
		printf("now=%lu\n",jiffies);
	}
	printf("cont=%p\n", cont);
	printf("CURRENT=%p\n", CURRENT);
	printf("command_status=%d\n", command_status);
	printf("\n");
}

/*floppy shutdown, called when fd_timeout timer expires*/
static void floppy_shutdown(void)
{
	unsigned long flags;
	
	PRINT("######floppy shutdown######\n");
	if (!initialising)
		show_floppy();
	cancel_activity();

	floppy_enable_hlt();
	
	fd_disable_dma(userdev_id);
	
	/* avoid dma going to a random drive after shutdown */

	if (!initialising)
		DPRINT("floppy timeout called\n");
	FDCS->reset = 1;
	if (cont)
	{
		cont->done(0);
		cont->redo(); /* this will recall reset when needed */
	} 
	else 
	{
		printf("no cont in shutdown!\n");
		process_fd_request();
	}
	is_alive("floppy shutdown");
}

/* start motor, check media-changed condition and write protection */
static int start_motor(void (*function)(void) )
{
	int mask, data;

	PRINT("///////////////////start motor\n");
	mask = 0xfc;
	data = UNIT(current_drive);
	if (!(raw_cmd->flags & FD_RAW_NO_MOTOR))
	{
		if (!(FDCS->dor & (0x10 << UNIT(current_drive))))
		{
			set_debugt();
			/* no read since this drive is running */
			DRS->first_read_date = 0;
			/* note motor start time if motor is not yet running */
			DRS->spinup_date = userdev_get_jiffies();
			data |= (0x10 << UNIT(current_drive));
		}
	} 
	else if (FDCS->dor & (0x10 << UNIT(current_drive)))
		mask &= ~(0x10 << UNIT(current_drive));

	/* starts motor and selects floppy */
	userdev_del_timer(motor_off_timer + current_drive);
	set_dor(fdc, mask, data);

	/* wait_for_completion also schedules reset if needed. */
	return(wait_for_completion(DRS->select_date+DP->select_delay, (timeout_fn)function));
}

/*function that readies the floppy for a subsequent operation*/
static void floppy_ready(void)
{
	PRINT("floppy ready\n");
	CHECK_RESET;
	if (start_motor(floppy_ready)) return;
	if (fdc_dtr()) return;

#ifdef DCL_DEBUG
	if (DP->flags & FD_DEBUG)
	{
		DPRINT("calling disk change from floppy_ready\n");
	}
#endif
	if (!(raw_cmd->flags & FD_RAW_NO_MOTOR) && disk_change(current_drive) && !DP->select_delay)
		twaddle(); /* this clears the dcl on certain drive/controller combinations */

#ifdef fd_chose_dma_mode
	if ((raw_cmd->flags & FD_RAW_READ) || (raw_cmd->flags & FD_RAW_WRITE))
		fd_chose_dma_mode(raw_cmd->kernel_data, raw_cmd->length);
#endif

	if (raw_cmd->flags & (FD_RAW_NEED_SEEK | FD_RAW_NEED_DISK))
	{
		perpendicular_mode();
		fdc_specify(); /* must be done here because of hut, hlt ... */
		seek_floppy();
	} 
	else 
	{
		if ((raw_cmd->flags & FD_RAW_READ) || (raw_cmd->flags & FD_RAW_WRITE))
			fdc_specify();
		setup_rw_floppy();
	}
}

/*function that starts any operation on the floppy*/
static void floppy_start(void)
{
	PRINT("floppy start\n");
	reschedule_timeout(CURRENTD, "floppy start", 0);
	scandrives();
#ifdef DCL_DEBUG
	if (DP->flags & FD_DEBUG)
		DPRINT("setting NEWCHANGE in floppy_start\n");
#endif
	SETF(FD_DISK_NEWCHANGE);
	floppy_ready();
}

/*
 * ========================================================================
 * here ends the bottom half. Exported routines are:
 * floppy_start, floppy_off, floppy_ready, lock_fdc, unlock_fdc, set_fdc,
 * start_motor, reset_fdc, reset_fdc_info, interpret_errors.
 * Initialization also uses output_byte, result, set_dor, floppy_interrupt
 * and set_dor.
 * ========================================================================
 */
/*
 * General purpose continuations.
 * ==============================
 */

/*function called to wakeup a thread blocked on wait_til_done*/
static void do_wakeup(void)
{
	PRINT("do wakeup\n");
	reschedule_timeout(MAXTIMEOUT, "do wakeup", 0);
	cont = 0;
	command_status += 2;
	pthread_cond_signal(&command_cond);
}

static struct cont_t wakeup_cont={
	empty,
	do_wakeup,
	empty,
	(done_f)empty
};


static struct cont_t intr_cont={
	empty,
	process_fd_request,
	empty,
	(done_f) empty
};

/*wait_til_done blocks till the passed function completes*/
static int wait_til_done(void (*handler)(void), int interruptible, int reqid)
{
	int ret;

	PRINT("wait til done\n");
	schedule_bh((void *)(void *)handler);	/*schedule the function on task queue*/

	pthread_mutex_lock(&command_mutex);	/*wait till command_status becomes >2*/
	while(command_status < 2 && NO_SIGNAL(reqid))	/*or a signal message arrives for the current request*/
		pthread_cond_wait(&command_cond, &command_mutex);
	pthread_mutex_unlock(&command_mutex);

	if (command_status < 2)		/*failure*/
	{
		cancel_activity();
		cont = &intr_cont;
		reset_fdc();
		return -EINTR;
	}

	if (FDCS->reset)
		command_status = FD_COMMAND_ERROR;
	if (command_status == FD_COMMAND_OKAY)
		ret=0;
	else
		ret=-EIO;
	command_status = FD_COMMAND_NONE;
	return ret;
}

static void generic_done(int result)
{
	command_status = result;
	cont = &wakeup_cont;
}

static void generic_success(void)
{
	cont->done(1);
}

static void generic_failure(void)
{
	cont->done(0);
}

static void success_and_wakeup(void)
{
	generic_success();
	cont->redo();
}


/*
 * formatting and rw support.
 * ==========================
 */
/*This function probes through the list of valid formats for the current drive*/
/*and returns the next valid one*/
static int next_valid_format(void)
{
	int probed_format;

	PRINT("next valid format\n");
	probed_format = DRS->probed_format;
	while(1)
	{
		if (probed_format >= 8 || !DP->autodetect[probed_format])
		{
			DRS->probed_format = 0;
			return 1;
		}
		if (floppy_type[DP->autodetect[probed_format]].sect)
		{
			DRS->probed_format = probed_format;
			return 0;
		}
		probed_format++;
	}
}

/*error routine called when an operation fails*/
/*based on the number of retries that can be made, given by the max_errors structure*/
/*the routine decides whether to signal a failure or not*/
static void bad_flp_intr(void)
{
	PRINT("bad flp intr\n");
	if (probing)
	{
		DRS->probed_format++;
		if (!next_valid_format()) return;
	}
	(*errors)++;
	INFBOUND(DRWE->badness, *errors);
	if (*errors > DP->max_errors.abort)
		cont->done(0);
	if (*errors > DP->max_errors.reset)
		FDCS->reset = 1;
	else if (*errors > DP->max_errors.recal)
		DRS->track = NEED_2_RECAL;
}

/*function to set the current floppy data depending on its type*/
static void set_floppy(int device)
{
	PRINT("set floppy\n");
	if (TYPE(device))
		_floppy = TYPE(device) + floppy_type;
	else
		_floppy = current_type[ DRIVE(device) ];
}

/*formatting support.*/
/*===================*/

/*function called in response to an interrupt after a format request*/
static void format_interrupt(void)
{
	PRINT("format interrupt\n");
	switch (interpret_errors())
	{
		case 1:
			cont->error();
		case 2:
			break;
		case 0:
			cont->done(1);
	}
	cont->redo();
}

#define CODE2SIZE (ssize = ((1 << SIZECODE) + 3) >> 2)
#define FM_MODE(x,y) ((y) & ~(((x)->rate & 0x80) >>1))
#define CT(x) ((x) | 0xc0)

/*function to setup the parameters for formating a given track*/
static void setup_format_params(int track)
{
	struct fparm {
		unsigned char track,head,sect,size;
	} *here = (struct fparm *)floppy_track_buffer;
	int il,n;
	int count,head_shift,track_shift;

	PRINT("setup format params\n");
	raw_cmd = &default_raw_cmd;
	raw_cmd->track = track;

	raw_cmd->flags = FD_RAW_WRITE | FD_RAW_INTR | FD_RAW_SPIN | FD_RAW_NEED_DISK | FD_RAW_NEED_SEEK;
	raw_cmd->rate = _floppy->rate & 0x43;
	raw_cmd->cmd_count = NR_F;
	COMMAND = FM_MODE(_floppy, FD_FORMAT);
	DR_SELECT = UNIT(current_drive) + PH_HEAD(_floppy,format_req.head);
	F_SIZECODE = FD_SIZECODE(_floppy);
	F_SECT_PER_TRACK = _floppy->sect << 2 >> F_SIZECODE;
	F_GAP = _floppy->fmt_gap;
	F_FILL = FD_FILL_BYTE;

	raw_cmd->kernel_data = floppy_track_buffer;
	raw_cmd->length = 4 * F_SECT_PER_TRACK;

	/* allow for about 30ms for data transport per track */
	head_shift  = (F_SECT_PER_TRACK + 5) / 6;

	/* a ``cylinder'' is two tracks plus a little stepping time */
	track_shift = 2 * head_shift + 3;

	/* position of logical sector 1 on this track */
	n = (track_shift * format_req.track + head_shift * format_req.head) % F_SECT_PER_TRACK;

	/* determine interleave */
	il = 1;
	if (_floppy->fmt_gap < 0x22) il++;

	/* initialize field */
	for (count = 0; count < F_SECT_PER_TRACK; ++count) 
	{
		here[count].track = format_req.track;
		here[count].head = format_req.head;
		here[count].sect = 0;
		here[count].size = F_SIZECODE;
	}
	/* place logical sectors */
	for (count = 1; count <= F_SECT_PER_TRACK; ++count) 
	{
		here[n].sect = count;
		n = (n+il) % F_SECT_PER_TRACK;
		if (here[n].sect) 
		{ /* sector busy, find next free sector */
			++n;
			if (n>= F_SECT_PER_TRACK) 
			{
				n-=F_SECT_PER_TRACK;
				while (here[n].sect) ++n;
			}
		}
	}
}

/*function that initiates the format operation*/
static void redo_format(void)
{
	PRINT("redo format\n");
	buffer_track = -1;
	setup_format_params(format_req.track << STRETCH(_floppy));
	floppy_start();
#ifdef DEBUGT
	debugt("queue format request");
#endif
}

static struct cont_t format_cont={
	format_interrupt,
	redo_format,
	bad_flp_intr,
	generic_done 
};

/*function for formating a track of the disk identified by the format descriptor structure*/
static int do_format(int device, struct format_descr *tmp_format_req, int reqid)
{
	int ret;
	int drive=DRIVE(device);

	PRINT("do format\n");
	LOCK_FDC(drive,1,reqid);
	set_floppy(device);
	if (!_floppy || _floppy->track > DP->tracks || tmp_format_req->track >= _floppy->track ||
	    tmp_format_req->head >= _floppy->head || (_floppy->sect << 2) % (1 <<  FD_SIZECODE(_floppy)) || !_floppy->fmt_gap) 
	{
		process_fd_request();
		return -EINVAL;
	}
	format_req = *tmp_format_req;
	format_errors = 0;
	cont = &format_cont;
	errors = &format_errors;
	IWAIT(redo_format,reqid);
	process_fd_request();
	return ret;
}

/*
 * Buffer read/write and support
 * =============================
 */

/*function called when a given request from buffer cache is satisfied*/
/*Can handle physical sectors which are smaller than a logical buffer */
static void request_done(int uptodate)
{
	int block,current_nr_sectors;
	unsigned long flags;

	PRINT("request done %d\n",uptodate);
	probing = 0;
	reschedule_timeout(MAXTIMEOUT, "request done %d", uptodate);

	if (QUEUE_EMPTY)
	{
		DPRINT("request list destroyed in floppy request done\n");
		return;
	}

	if (uptodate)
	{
		/* maintain values for invalidation on geometry change */
		block = current_count_sectors + CURRENT->sector;
		INFBOUND(DRS->maxblock, block);
		if (block > _floppy->sect) DRS->maxtrack = 1;

		/* unlock chained buffers */
		current_nr_sectors = NR_SECTORS;
		while (current_count_sectors && !QUEUE_EMPTY && current_count_sectors >= current_nr_sectors)
		{
			current_count_sectors -= current_nr_sectors;
			CURRENT->clustersize -= current_nr_sectors;
			CURRENT->sector += current_nr_sectors;
	
			if(CURRENT->reqid != -1)	/*if request is not a dummy one generated by revalidate*/
				userdev_send_response(CURRENT->devid, CURRENT->reqid, REQUEST_OP, 1, CURRENT->length, CURRENT->buffer_cp);
			/*use buffer_cp pointer which has not been moved while buffer pointer might have */
			/*been moved by multiple copy operations*/

			userdev_end_request(1);	/*remove request from request queue and send positive result*/
		}

		if (current_count_sectors && !QUEUE_EMPTY)
		{
			/* "unlock" last subsector */
			CURRENT->buffer += current_count_sectors <<9;
			CURRENT->clustersize -= current_count_sectors;
			CURRENT->sector += current_count_sectors;
			return;
		}
	} 
	else 
	{
		if (CURRENT->command == WRITE_REQ) 
		{
			/* record write error information */
			DRWE->write_errors++;
			if (DRWE->write_errors == 1) 
			{
				DRWE->first_error_sector = CURRENT->sector;
				DRWE->first_error_generation = DRS->generation;
			}
			DRWE->last_error_sector = CURRENT->sector;
			DRWE->last_error_generation = DRS->generation;
		}

		if(CURRENT->reqid != -1)	/*if request is not a dummy one generated by revalidate*/
			userdev_send_response(CURRENT->devid, CURRENT->reqid, REQUEST_OP, 0, CURRENT->length, CURRENT->buffer_cp);
		/*use buffer_cp pointer which has not been moved while buffer pointer might have */
		/*been moved by multiple copy operations*/

		userdev_end_request(0);	/*remove request and send negative result*/
	}
}

/*Interrupt handler evaluating the result of the r/w operation */
/*called by floppy interrupt function*/
static void rw_interrupt(void)
{
	int nr_sectors, ssize, eoc, heads;

	PRINT("rw interrupt\n");
	if (R_HEAD >= 2) 
	{
	    /* some Toshiba floppy controllers occasionnally seem to
	     * return bogus interrupts after read/write operations, which
	     * can be recognized by a bad head number (>= 2) */
	     return;
	}  

	if (!DRS->first_read_date)
		DRS->first_read_date = userdev_get_jiffies();

	nr_sectors = 0;
	CODE2SIZE;

	if (ST1 & ST1_EOC)
		eoc = 1;
	else
		eoc = 0;

	if (COMMAND & 0x80)
		heads = 2;
	else
		heads = 1;

	nr_sectors = (((R_TRACK-TRACK) * heads + R_HEAD-HEAD) * SECT_PER_TRACK + R_SECTOR-SECTOR + eoc) << SIZECODE >> 2;

#ifdef FLOPPY_SANITY_CHECK
	if (nr_sectors / ssize > (in_sector_offset + current_count_sectors + ssize - 1) / ssize) 
	{
		DPRINT("long rw: %x instead of %lx\n", nr_sectors, current_count_sectors);
		printf("rs=%d s=%d\n", R_SECTOR, SECTOR);
		printf("rh=%d h=%d\n", R_HEAD, HEAD);
		printf("rt=%d t=%d\n", R_TRACK, TRACK);
		printf("heads=%d eoc=%d\n", heads, eoc);
		printf("spt=%d st=%d ss=%d\n", SECT_PER_TRACK, sector_t, ssize);
		printf("in_sector_offset=%d\n", in_sector_offset);
	}
#endif

	nr_sectors -= in_sector_offset;
	INFBOUND(nr_sectors,0);
	SUPBOUND(current_count_sectors, nr_sectors);

	switch (interpret_errors())
	{
		case 2:
			cont->redo();
			return;
		case 1:
			if (!current_count_sectors)
			{
				cont->error();
				cont->redo();
				return;
			}
			break;
		case 0:
			if (!current_count_sectors)
			{
				cont->redo();
				return;
			}
			current_type[current_drive] = _floppy;
			floppy_sizes[TOMINOR(current_drive) ] = (_floppy->size+1)>>1;
			break;
	}

	if (probing) 
	{
		if (DP->flags & FTD_MSG)
			DPRINT("Auto-detected floppy type %s in fd%d\n",_floppy->name,current_drive);
		current_type[current_drive] = _floppy;
		floppy_sizes[TOMINOR(current_drive)] = (_floppy->size+1) >> 1;
		probing = 0;
	}

	if (CT(COMMAND) != FD_READ || raw_cmd->kernel_data == CURRENT->buffer)
	{
		/* transfer directly from buffer */
		cont->done(1);
	} 
	else if (CT(COMMAND) == FD_READ)
	{
		if(!use_virtual_dma) 	/*copy data from kernel buffer incase of read operation*/
			userdev_copy_dma(raw_cmd->kernel_data, userdev_id);
		buffer_track = raw_cmd->track;
		buffer_drive = current_drive;
		INFBOUND(buffer_max, nr_sectors + sector_t);
	}
	cont->redo();
}


/* Compute the maximal transfer size */
static int transfer_size(int ssize, int max_sector, int max_size)
{
	PRINT("transfer_size\n");
	SUPBOUND(max_sector, sector_t + max_size);

	/* alignment */
	max_sector -= (max_sector % _floppy->sect) % ssize;

	/* transfer size, beginning not aligned */
	current_count_sectors = max_sector - sector_t ;

	return max_sector;
}

/*
 * Move data from/to the track buffer to/from the buffer cache.
 */
static void copy_buffer(int ssize, int max_sector, int max_sector_2)
{
	int remaining; /* number of transferred 512-byte sectors */
	struct userdev_blk_request *br;
	char *buffer, *dma_buffer;
	int size;

	PRINT("copy buffer\n");
	max_sector = transfer_size(ssize, minimum(max_sector, max_sector_2), NR_SECTORS);

	if (current_count_sectors <= 0 && CT(COMMAND) == FD_WRITE && buffer_max > sector_t + NR_SECTORS)
		current_count_sectors = minimum(buffer_max - sector_t, NR_SECTORS);

	remaining = current_count_sectors << 9;
#ifdef FLOPPY_SANITY_CHECK
	if ((remaining >> 9) > NR_SECTORS  && CT(COMMAND) == FD_WRITE)
	{
		DPRINT("in copy buffer\n");
		printf("current_count_sectors=%ld\n", current_count_sectors);
		printf("remaining=%d\n", remaining >> 9);
		printf("CURRENT->nr_sectors=%ld\n",NR_SECTORS);
		printf("CURRENT->current_nr_sectors=%ld\n", CURRENT->length >> 9);
		printf("max_sector=%d\n", max_sector);
		printf("ssize=%d\n", ssize);
	}
#endif

	buffer_max = maximum(max_sector, buffer_max);

	dma_buffer = floppy_track_buffer + ((sector_t - buffer_min) << 9);

	br = CURRENT;	
	size = CURRENT->length;
	buffer = CURRENT->buffer;

	while (remaining > 0)
	{
		SUPBOUND(size, remaining);
#ifdef FLOPPY_SANITY_CHECK
		if (dma_buffer + size > floppy_track_buffer + (max_buffer_sectors << 10) ||
		    	dma_buffer < floppy_track_buffer)
		{
			DPRINT("buffer overrun in copy buffer %d\n",(int) ((floppy_track_buffer - dma_buffer) >>9));
			printf("sector_t=%d buffer_min=%d\n",sector_t, buffer_min);
			printf("current_count_sectors=%ld\n",current_count_sectors);
			if (CT(COMMAND) == FD_READ) printf("read\n");
			if (CT(COMMAND) == FD_READ) printf("write\n");
			break;
		}
#endif
		if (CT(COMMAND) == FD_READ)
			memcpy(buffer, dma_buffer, size);
		else
			memcpy(dma_buffer, buffer, size);
		remaining -= size;
		if (!remaining) break;

		dma_buffer += size;
		br = br->next;	
#ifdef FLOPPY_SANITY_CHECK
		if (!br) break;
#endif
		size = br->size;
		buffer = br->buffer;
	}
#ifdef FLOPPY_SANITY_CHECK
	if (remaining)
	{
		if (remaining > 0)
			max_sector -= remaining >> 9;
	}
#endif
}

/* work around a bug in pseudo DMA
 * (on some FDCs) pseudo DMA does not stop when the CPU stops
 * sending data.  Hence we need a different way to signal the
 * transfer length:  We use SECT_PER_TRACK.  Unfortunately, this
 * does not work with MT, hence we can only transfer one head at
 * a time
 */
static void virtualdmabug_workaround(void)
{
	int hard_sectors, end_sector;

	if(CT(COMMAND) == FD_WRITE) 
	{
		COMMAND &= ~0x80; /* switch off multiple track mode */

		hard_sectors = raw_cmd->length >> (7 + SIZECODE);
		end_sector = SECTOR + hard_sectors - 1;
#ifdef FLOPPY_SANITY_CHECK
		if(end_sector > SECT_PER_TRACK) 
		{
			printf("too many sectors %d > %d\n", end_sector, SECT_PER_TRACK);
			return;
		}
#endif
		SECT_PER_TRACK = end_sector; /* make sure SECT_PER_TRACK points to end of transfer */
	}
}

/*
 * Formulate a read/write request.
 * this routine decides where to load the data (directly to buffer, or to
 * tmp floppy area), how much data to load (the size of the buffer, the whole
 * track, or a single sector)
 * All floppy_track_buffer handling goes in here. If we ever add track buffer
 * allocation on the fly, it should be done here. No other part should need
 * modification.
 */
static int make_raw_rw_request(void)
{
	int aligned_sector_t;
	int max_sector, max_size, tracksize, ssize;

	PRINT("make raw rw request\n");
	if(max_buffer_sectors == 0) 
	{
		printf("VFS: Block I/O scheduled on unopened device\n");
		return 0;
	}

	set_fdc(DRIVE(CURRENT->devid));

	raw_cmd = &default_raw_cmd;
	raw_cmd->flags = FD_RAW_SPIN | FD_RAW_NEED_DISK | FD_RAW_NEED_DISK | FD_RAW_NEED_SEEK;
	raw_cmd->cmd_count = NR_RW;
	if (CURRENT->command == READ_REQ)
	{
		raw_cmd->flags |= FD_RAW_READ;
		COMMAND = FM_MODE(_floppy,FD_READ);
	} 
	else if (CURRENT->command == WRITE_REQ)
	{
		raw_cmd->flags |= FD_RAW_WRITE;
		COMMAND = FM_MODE(_floppy,FD_WRITE);
	} 
	else 
	{
		DPRINT("make_raw_rw_request: unknown command\n");
		return 0;
	}

	max_sector = _floppy->sect * _floppy->head;

	TRACK = CURRENT->sector / max_sector;
	sector_t = CURRENT->sector % max_sector;
	if (_floppy->track && TRACK >= _floppy->track) 
	{
		if((CURRENT->length >> 9) & 1)
		{
			current_count_sectors = 1;
			return 1;
		} 
		else return 0;
	}
	HEAD = sector_t / _floppy->sect;

	if (((_floppy->stretch & FD_SWAPSIDES) || TESTF(FD_NEED_TWADDLE)) && sector_t < _floppy->sect)
		max_sector = _floppy->sect;

	/* 2M disks have phantom sectors on the first track */
	if ((_floppy->rate & FD_2M) && (!TRACK) && (!HEAD))
	{
		max_sector = 2 * _floppy->sect / 3;
		if (sector_t >= max_sector)
		{
			current_count_sectors = minimum(_floppy->sect - sector_t, NR_SECTORS);
			return 1;
		}
		SIZECODE = 2;
	} 
	else SIZECODE = FD_SIZECODE(_floppy);
	raw_cmd->rate = _floppy->rate & 0x43;
	if ((_floppy->rate & FD_2M) && (TRACK || HEAD) && raw_cmd->rate == 2)
		raw_cmd->rate = 1;

	if (SIZECODE)
		SIZECODE2 = 0xff;
	else
		SIZECODE2 = 0x80;
	raw_cmd->track = TRACK << STRETCH(_floppy);
	DR_SELECT = UNIT(current_drive) + PH_HEAD(_floppy,HEAD);
	GAP = _floppy->gap;
	CODE2SIZE;
	SECT_PER_TRACK = _floppy->sect << 2 >> SIZECODE;
	SECTOR = ((sector_t % _floppy->sect) << 2 >> SIZECODE) + 1;

	/* tracksize describes the size which can be filled up with sectors of size ssize.*/
	tracksize = _floppy->sect - _floppy->sect % ssize;
	if (tracksize < _floppy->sect)
	{
		SECT_PER_TRACK ++;
		if (tracksize <= sector_t % _floppy->sect) SECTOR--;

		/* if we are beyond tracksize, fill up using smaller sectors */
		while (tracksize <= sector_t % _floppy->sect)
		{
			while(tracksize + ssize > _floppy->sect)
			{
				SIZECODE--;
				ssize >>= 1;
			}
			SECTOR++; SECT_PER_TRACK ++;
			tracksize += ssize;
		}
		max_sector = HEAD * _floppy->sect + tracksize;
	} 
	else if (!TRACK && !HEAD && !(_floppy->rate & FD_2M) && probing) 
	{
		max_sector = _floppy->sect;
	} 
	else if (!HEAD && CT(COMMAND) == FD_WRITE) 
	{
		/* for virtual DMA bug workaround */
		max_sector = _floppy->sect;
	}

	in_sector_offset = (sector_t % _floppy->sect) % ssize;
	aligned_sector_t = sector_t - in_sector_offset;
	max_size = NR_SECTORS;
	if ((raw_cmd->track == buffer_track) && (current_drive == buffer_drive) && 
			(sector_t >= buffer_min) && (sector_t < buffer_max)) 
	{
		/* data already in track buffer */
		if (CT(COMMAND) == FD_READ) 
		{
			copy_buffer(1, max_sector, buffer_max);
			return 1;
		}
	} 
	else if (in_sector_offset || NR_SECTORS < ssize)
	{
		if (CT(COMMAND) == FD_WRITE)
		{
			if (sector_t + NR_SECTORS > ssize && sector_t + NR_SECTORS < ssize + ssize)
				max_size = ssize + ssize;
			else
				max_size = ssize;
		}
		raw_cmd->flags &= ~FD_RAW_WRITE;
		raw_cmd->flags |= FD_RAW_READ;
		COMMAND = FM_MODE(_floppy,FD_READ);
	} 

	if (CT(COMMAND) == FD_READ)
		max_size = max_sector; /* unbounded */

	/* claim buffer track if needed */
	if (buffer_track != raw_cmd->track ||  	/* bad track */
	    buffer_drive != current_drive || 	/* bad drive */
	    sector_t > buffer_max || sector_t < buffer_min || ((CT(COMMAND) == FD_READ ||
	    (!in_sector_offset && NR_SECTORS >= ssize)) &&
	     	max_sector > 2 * max_buffer_sectors + buffer_min &&
	        max_size + sector_t > 2 * max_buffer_sectors + buffer_min)/* not enough space */)
	{
		buffer_track = -1;
		buffer_drive = current_drive;
		buffer_max = buffer_min = aligned_sector_t;
	}
	raw_cmd->kernel_data = floppy_track_buffer + ((aligned_sector_t-buffer_min)<<9);

	if (CT(COMMAND) == FD_WRITE)
	{
		/* copy write buffer to track buffer.
		 * if we get here, we know that the write
		 * is either aligned or the data already in the buffer
		 * (buffer will be overwritten) */

#ifdef FLOPPY_SANITY_CHECK
		if (in_sector_offset && buffer_track == -1)
			DPRINT("internal error offset !=0 on write\n");
#endif

		buffer_track = raw_cmd->track;
		buffer_drive = current_drive;
		copy_buffer(ssize, max_sector, 2*max_buffer_sectors+buffer_min);
	} 
	else transfer_size(ssize, max_sector, 2*max_buffer_sectors+buffer_min-aligned_sector_t);

	/* round up current_count_sectors to get dma xfer size */
	raw_cmd->length = in_sector_offset+current_count_sectors;
	raw_cmd->length = ((raw_cmd->length -1)|(ssize-1))+1;
	raw_cmd->length <<= 9;

#ifdef FLOPPY_SANITY_CHECK
	/*check_dma_crossing(raw_cmd->kernel_data, raw_cmd->length, "end of make_raw_request");*/
	if ((raw_cmd->length < current_count_sectors << 9) ||
	    (raw_cmd->kernel_data != CURRENT->buffer &&
	     	CT(COMMAND) == FD_WRITE && (aligned_sector_t + (raw_cmd->length >> 9) > buffer_max ||
	    aligned_sector_t < buffer_min)) || raw_cmd->length % (128 << SIZECODE) ||
	    raw_cmd->length <= 0 || current_count_sectors <= 0)
	{
		DPRINT("fractionary current count b=%lx s=%lx\n", raw_cmd->length, current_count_sectors);
		if (raw_cmd->kernel_data != CURRENT->buffer)
			printf("addr=%d, length=%ld\n",(int) ((raw_cmd->kernel_data - floppy_track_buffer) >> 9), current_count_sectors);
		printf("st=%d ast=%d mse=%d msi=%d\n", sector_t, aligned_sector_t, max_sector, max_size);
		printf("ssize=%x SIZECODE=%d\n", ssize, SIZECODE);
		printf("command=%x SECTOR=%d HEAD=%d, TRACK=%d\n",COMMAND, SECTOR, HEAD, TRACK);
		printf("buffer drive=%d\n", buffer_drive);
		printf("buffer track=%d\n", buffer_track);
		printf("buffer_min=%d\n", buffer_min);
		printf("buffer_max=%d\n", buffer_max);
		return 0;
	}

	if (raw_cmd->kernel_data != CURRENT->buffer)
	{
		if (raw_cmd->kernel_data < floppy_track_buffer || current_count_sectors < 0 || raw_cmd->length < 0 || 
			raw_cmd->kernel_data + raw_cmd->length > floppy_track_buffer + (max_buffer_sectors<<10))
		{
			DPRINT("buffer overrun in schedule dma\n");
			printf("sector_t=%d buffer_min=%d current_count=%ld\n",sector_t, buffer_min, raw_cmd->length >> 9);
			printf("current_count_sectors=%ld\n", current_count_sectors);
			if (CT(COMMAND) == FD_READ)
				printf("read\n");
			if (CT(COMMAND) == FD_READ)
				printf("write\n");
			return 0;
		}
	} 
	else if (raw_cmd->length > NR_SECTORS << 9 || current_count_sectors > NR_SECTORS)
	{
		DPRINT("buffer overrun in direct transfer\n");
		return 0;
	} 
	else if (raw_cmd->length < current_count_sectors << 9)
	{
		DPRINT("more sectors than bytes\n");
		printf("bytes=%ld\n", raw_cmd->length >> 9);
		printf("sectors=%ld\n", current_count_sectors);
	}
	if (raw_cmd->length == 0)
	{
		DPRINT("zero dma transfer attempted from make_raw_request\n");
		return 0;
	}
#endif

	virtualdmabug_workaround();
	return 2;
}

/*function that processes r/w requests in the request queue*/
static void redo_fd_request(void)
{
#define REPEAT {request_done(0); continue; }
	int device;
	int tmp;

	PRINT("redo fd request\n");
	lastredo = 0;	/*userdev_get_jiffies();//costly*/
	if (current_drive < N_DRIVE)
		floppy_off(current_drive);

	while(1)
	{
		if (QUEUE_EMPTY) 
		{
			CLEAR_INTR;
			unlock_fdc();
			/*additional call introduced to make sure floppy motor is switched off*/
			/*when request processing is completed --- HK */
			floppy_off(current_drive);
			return;
		}
		device = CURRENT->devid;
		set_fdc(DRIVE(device));
		reschedule_timeout(CURRENTD, "redo fd request", 0);

		set_floppy(device);
		raw_cmd = &default_raw_cmd;
		raw_cmd->flags = 0;
		if (start_motor(redo_fd_request)) return;
		disk_change(current_drive);
		if (test_bit(current_drive, &fake_change) || TESTF(FD_DISK_CHANGED))
		{
			DPRINT("disk absent or changed during operation\n");
			REPEAT;
		}
		if (!_floppy) 
		{ /* Autodetection */
			if (!probing)
			{
				DRS->probed_format = 0;
				if (next_valid_format())
				{
					DPRINT("no autodetectable formats\n");
					_floppy = NULL;
					REPEAT;
				}
			}
			probing = 1;
			_floppy = floppy_type+DP->autodetect[DRS->probed_format];
		} 
		else probing = 0;
		errors = &(CURRENT->errors);
		tmp = make_raw_rw_request();
		if (tmp < 2)
		{
			request_done(tmp);
			continue;
		}

		if (TESTF(FD_NEED_TWADDLE)) twaddle();
		schedule_bh( (void *)(void *) floppy_start);
#ifdef DEBUGT
		debugt("queue fd request");
#endif
		return;
	}
#undef REPEAT
}

static struct cont_t rw_cont={
	rw_interrupt,
	redo_fd_request,
	bad_flp_intr,
	request_done 
};

/*function to schedule processing of r/w request queue*/
static void process_fd_request(void)
{
	PRINT("process fd request\n");
	cont = &rw_cont;
	schedule_bh( (void *)(void *) redo_fd_request);
}

/*userdev request function for floppy*/
static void do_fd_request(int devid, int command, long sector, int length, int clustersize, void *data, int size, int reqid)
{
	struct userdev_blk_request *br;		/*add a new r/w request to the request queue*/

	PRINT("command %d, sector %d, length %u, clustersize %u, size %u\n",command,sector,length,clustersize,size);
	br = userdev_add_request(devid, command, sector, length, clustersize, data, size, reqid);

	PRINT("do fd request\n");
	if(max_buffer_sectors == 0) 
	{
		printf("VFS: do_fd_request called on non-open device\n");
		return;
	}

	if (usage_count == 0) 
	{
		printf("warning: usage count=0, CURRENT=%p exiting\n", CURRENT);
		printf("sect=%ld cmd=%d\n", CURRENT->sector, CURRENT->command);
		return;
	}
	if (fdc_busy)
	{
		/* fdc busy, this new request will be treated when the current one is done */
		is_alive("do fd request, old request running");
		return;
	}
	lock_fdc(MAXTIMEOUT,0,reqid);
	process_fd_request();		/*initiate processing of request queue*/
	is_alive("do fd request");
}

static struct cont_t poll_cont={
	success_and_wakeup,
	floppy_ready,
	generic_failure,
	generic_done 
};

/*function to poll the current drive*/
static int poll_drive(int interruptible, int flag, int reqid)
{
	int ret=0;
	PRINT("poll_drive\n");
	/* no auto-sense, just clear dcl */
	raw_cmd = &default_raw_cmd;
	raw_cmd->flags= flag;
	raw_cmd->track=0;
	raw_cmd->cmd_count=0;
	cont = &poll_cont;
#ifdef DCL_DEBUG
	if (DP->flags & FD_DEBUG)
		DPRINT("setting NEWCHANGE in poll_drive\n");
#endif
	SETF(FD_DISK_NEWCHANGE);
	WAIT(floppy_ready,reqid);
	return ret;
}

/*
 * User triggered reset
 * ====================
 */

static void reset_intr(void)
{
	printf("weird, reset interrupt called\n");
}

static struct cont_t reset_cont={
	reset_intr,
	success_and_wakeup,
	generic_failure,
	generic_done 
};

/*function to perform reset of floppy controller on request from user*/
static int user_reset_fdc(int drive, int arg, int interruptible, int reqid)
{
	int ret;

	PRINT("user reset fdc\n");
	ret=0;
	LOCK_FDC(drive,interruptible,reqid);
	if (arg == FD_RESET_ALWAYS)
		FDCS->reset=1;
	if (FDCS->reset)
	{
		cont = &reset_cont;
		WAIT(reset_fdc,reqid);
	}
	process_fd_request();
	return ret;
}

/*
 * Misc Ioctl's and support
 * ========================
 */
static inline int fd_copyout(void *param, const void *address, unsigned long size)
{
	return memcpy(param,address, size) ? 0: -EFAULT;
}

static inline int fd_copyin(void *param, void *address, unsigned long size)
{
	return memcpy(address, param, size) ? 0:-EFAULT;
}

#define _COPYOUT(x) (memcpy((void *)param, &(x), sizeof(x)) ? 0:-EFAULT)
#define _COPYIN(x) (memcpy(&(x), (void *)param, sizeof(x)) ? 0:-EFAULT)

#define COPYOUT(x) ECALL(_COPYOUT(x))
#define COPYIN(x) ECALL(_COPYIN(x))

/*function to obtain the drive name depending on the type and drive numbers*/
static inline const char *drive_name(int type, int drive)
{
	struct floppy_struct *floppy;

	if (type)
		floppy = floppy_type + type;
	else 
	{
		if (UDP->native_format)
			floppy = floppy_type + UDP->native_format;
		else
			return "(null)";
	}
	if (floppy->name)
		return floppy->name;
	else
		return "(null)";
}


/* raw commands */

/*function to obtain the result of the raw command operation*/
static void raw_cmd_done(int flag)
{
	int i;

	PRINT("raw cmd done\n");
	if (!flag) 
	{
		raw_cmd->flags |= FD_RAW_FAILURE;
		raw_cmd->flags |= FD_RAW_HARDFAILURE;
	} 
	else 
	{
		raw_cmd->reply_count = inr;
		if (raw_cmd->reply_count > MAX_REPLIES)
			raw_cmd->reply_count=0;
		for (i=0; i< raw_cmd->reply_count; i++)
			raw_cmd->reply[i] = reply_buffer[i];

		if (raw_cmd->flags & (FD_RAW_READ | FD_RAW_WRITE))
		{
			unsigned long flags;
			raw_cmd->length = fd_get_dma_residue(userdev_id);
		}
		
		if ((raw_cmd->flags & FD_RAW_SOFTFAILURE) && (!raw_cmd->reply_count || (raw_cmd->reply[0] & 0xc0)))
			raw_cmd->flags |= FD_RAW_FAILURE;

		if (disk_change(current_drive))
			raw_cmd->flags |= FD_RAW_DISK_CHANGE;
		else
			raw_cmd->flags &= ~FD_RAW_DISK_CHANGE;
		if (raw_cmd->flags & FD_RAW_NO_MOTOR_AFTER)
			motor_off_callback(current_drive);

		if (raw_cmd->next && (!(raw_cmd->flags & FD_RAW_FAILURE) || !(raw_cmd->flags & FD_RAW_STOP_IF_FAILURE)) &&
		   	((raw_cmd->flags & FD_RAW_FAILURE) || !(raw_cmd->flags &FD_RAW_STOP_IF_SUCCESS))) 
		{
			raw_cmd = raw_cmd->next;
			return;
		}
	}
	generic_done(flag);
}


static struct cont_t raw_cmd_cont={
	success_and_wakeup,
	floppy_start,
	generic_failure,
	raw_cmd_done
};

/*function to copy data from list of raw commands to ioctl parameter*/
static inline int raw_cmd_copyout(int cmd, char *param, struct floppy_raw_cmd *ptr)
{
	int ret;

	PRINT("raw cmd copyout\n");
	while(ptr) 
	{
		COPYOUT(*ptr);
		param += sizeof(struct floppy_raw_cmd);
		if ((ptr->flags & FD_RAW_READ) && ptr->buffer_length)
		{
			if (ptr->length>=0 && ptr->length<=ptr->buffer_length)
				ECALL(fd_copyout(ptr->data, ptr->kernel_data, ptr->buffer_length -  ptr->length));
		}
		ptr = ptr->next;
	}
	return 0;
}

/*function to free the list of raw commands constructed*/
static void raw_cmd_free(struct floppy_raw_cmd **ptr)
{
	struct floppy_raw_cmd *next,*this;

	PRINT("raw cmd free\n");
	this = *ptr;
	*ptr = 0;
	while(this) 
	{
		if (this->buffer_length) 
		{
			fd_dma_mem_free((unsigned long)this->kernel_data, this->buffer_length);
			this->buffer_length = 0;
		}
		next = this->next;
		free(this);
		this = next;
	}
}

/*function to copy from ioctl parameter to construct a list of raw commands*/
static inline int raw_cmd_copyin(int cmd, char *param, struct floppy_raw_cmd **rcmd)
{
	struct floppy_raw_cmd *ptr;
	int ret;
	int i;
	
	PRINT("raw cmd copyin\n");
	*rcmd = 0;
	while(1) 
	{
		ptr = (struct floppy_raw_cmd *)malloc(sizeof(struct floppy_raw_cmd));
		if (!ptr) return -ENOMEM;
		*rcmd = ptr;
		COPYIN(*ptr);
		ptr->next = 0;
		ptr->buffer_length = 0;
		param += sizeof(struct floppy_raw_cmd);
		if (ptr->cmd_count > 33)
			/* the command may now also take up the space
			 * initially intended for the reply & the
			 * reply count. Needed for long 82078 commands
			 * such as RESTORE, which takes ... 17 command
			 * bytes. Murphy's law #137: When you reserve
			 * 16 bytes for a structure, you'll one day
			 * discover that you really need 17...
			 */
			return -EINVAL;

		for (i=0; i< 16; i++)
			ptr->reply[i] = 0;
		ptr->resultcode = 0;
		ptr->kernel_data = 0;

		if (ptr->flags & (FD_RAW_READ | FD_RAW_WRITE)) 
		{
			if (ptr->length <= 0) return -EINVAL;
			ptr->kernel_data =(char*)fd_dma_mem_alloc(ptr->length);
			fallback_on_nodma_alloc(&ptr->kernel_data,ptr->length);
			if (!ptr->kernel_data) return -ENOMEM;
			ptr->buffer_length = ptr->length;
		}
		if (ptr->flags & FD_RAW_WRITE)
			ECALL(fd_copyin(ptr->data, ptr->kernel_data, ptr->length));
		rcmd = &(ptr->next);
		if (!(ptr->flags & FD_RAW_MORE)) return 0;
		ptr->rate &= 0x43;
	}
}

/*the function that perform the operation of sending raw commands to floppy*/
/*controller and obtaining the responses which are relayed back in the ioctl argument*/
/*this function is called in response to the ioctl on the driver with the same name*/
static int raw_cmd_ioctl(int cmd, void *param, int reqid)
{
	int drive, ret, ret2;
	struct floppy_raw_cmd *my_raw_cmd;

	PRINT("raw cmd ioctl\n");
	if (FDCS->rawcmd <= 1)
		FDCS->rawcmd = 1;
	for (drive= 0; drive < N_DRIVE; drive++)
	{
		if (FDC(drive) != fdc)
			continue;
		if (drive == current_drive)
		{
			if (UDRS->fd_ref > 1)
			{
				FDCS->rawcmd = 2;
				break;
			}
		} 
		else if (UDRS->fd_ref)
		{
			FDCS->rawcmd = 2;
			break;
		}
	}

	if (FDCS->reset) return -EIO;

	ret = raw_cmd_copyin(cmd, param, &my_raw_cmd);
	if (ret) 
	{
		raw_cmd_free(&my_raw_cmd);
		return ret;
	}

	raw_cmd = my_raw_cmd;
	cont = &raw_cmd_cont;
	ret=wait_til_done(floppy_start,1,reqid);
#ifdef DCL_DEBUG
	if (DP->flags & FD_DEBUG)
		DPRINT("calling disk change from raw_cmd ioctl\n");
#endif

	if (ret != -EINTR && FDCS->reset) ret = -EIO;

	DRS->track = NO_TRACK;

	ret2 = raw_cmd_copyout(cmd, param, my_raw_cmd);
	if (!ret) ret = ret2;
	raw_cmd_free(&my_raw_cmd);
	return ret;
}

/*function to check if the disk has changed, if so revalidate it*/
static int check_disk_change(int devid, int reqid)	/*reframed & coded function from fs/blk_dev.c */
{
	PRINT("check disk change\n");
	if(!check_floppy_change(devid,reqid)) return 0;
	floppy_revalidate(devid,reqid);
	return 1;
}

/*function to invalidate the given drive*/
static int invalidate_drive(int devid, int reqid)
{
	/* invalidate the buffer track to force a reread */
	PRINT("invalidate drive\n");
	set_bit(DRIVE(devid), &fake_change);
	process_fd_request();
	check_disk_change(devid,reqid);
	return 0;
}


static inline void clear_write_error(int drive)
{
	CLEARSTRUCT(UDRWE);
}

/*function to set the geometry of the floppy based on the floppy_struct structure passed*/
static inline int set_geometry(unsigned int cmd, struct floppy_struct *g, int drive, int type, int device, int reqid)
{
	int cnt;

	PRINT("set geometry\n");
	/* sanity checking for parameters.*/
	if (g->sect <= 0 || g->head <= 0 || g->track <= 0 || g->track > UDP->tracks>>STRETCH(g) ||
	    /* check if reserved bits are set */ (g->stretch&~(FD_STRETCH|FD_SWAPSIDES)) != 0)
		return -EINVAL;

	if (type)
	{
		LOCK_FDC(drive,1,reqid);
		for (cnt = 0; cnt < N_DRIVE; cnt++)
		{
			if (ITYPE(drive_state[cnt].fd_device) == type && drive_state[cnt].fd_ref)
				set_bit(drive, &fake_change);
		}
		floppy_type[type] = *g;
		floppy_type[type].name = "user format";
		for (cnt = type << 2; cnt < (type << 2) + 4; cnt++)
			floppy_sizes[cnt] = floppy_sizes[cnt+0x80] = (floppy_type[type].size+1)>>1;
		process_fd_request();
		for (cnt = 0; cnt < N_DRIVE; cnt++)
		{
			if (ITYPE(drive_state[cnt].fd_device) == type && drive_state[cnt].fd_ref)
				check_disk_change(drive_state[cnt].fd_device, reqid);
		}
	} 
	else 
	{
		LOCK_FDC(drive,1,reqid);
		if (cmd != FDDEFPRM)	/* notice a disk change immediately, else we lose our settings immediately*/
			CALL(poll_drive(1, FD_RAW_NEED_DISK, reqid));
		user_params[drive] = *g;
		if (buffer_drive == drive)
			SUPBOUND(buffer_max, user_params[drive].sect);
		current_type[drive] = &user_params[drive];
		floppy_sizes[drive] = (user_params[drive].size+1) >> 1;
		if (cmd == FDDEFPRM)
			DRS->keep_data = -1;
		else
			DRS->keep_data = 1;
		/* invalidation. Invalidate only when needed, i.e. when there are already sectors in the buffer cache whose number 
		 * will change. This is useful, because mtools often changes the geometry of the disk after looking at the boot block */
		if (DRS->maxblock > user_params[drive].sect || DRS->maxtrack)
			invalidate_drive(device,reqid);
		else
			process_fd_request();
	}
	return 0;
}

/* handle obsolete ioctl's */
static int ioctl_table[]= {
	FDCLRPRM,
	FDSETPRM,
	FDDEFPRM,
	FDGETPRM,
	FDMSGON,
	FDMSGOFF,
	FDFMTBEG,
	FDFMTTRK,
	FDFMTEND,
	FDSETEMSGTRESH,
	FDFLUSH,
	FDSETMAXERRS,
	FDGETMAXERRS,
	FDGETDRVTYP,
	FDSETDRVPRM,
	FDGETDRVPRM,
	FDGETDRVSTAT,
	FDPOLLDRVSTAT,
	FDRESET,
	FDGETFDCSTAT,
	FDWERRORCLR,
	FDWERRORGET,
	FDRAWCMD,
	FDEJECT,
	FDTWADDLE
};

static inline int normalize_ioctl(int *cmd, int *size)
{
	int i;

	PRINT("normalize ioctl\n");
	for (i=0; i < ARRAY_SIZE(ioctl_table); i++) 
	{
		if ((*cmd & 0xffff) == (ioctl_table[i] & 0xffff))
		{
			*size = _IOC_SIZE(*cmd);
			*cmd = ioctl_table[i];
			if (*size > _IOC_SIZE(*cmd)) 
			{
				printf("ioctl not yet supported\n");
				return -EFAULT;
			}
			return 0;
		}
	}
	return -EINVAL;
}

/*function to obtain the current floppy geometry*/
static int get_floppy_geometry(int drive, int type, struct floppy_struct **g, int reqid)
{
	PRINT("get geometry\n");
	if (type)
		*g = &floppy_type[type];
	else 
	{
		LOCK_FDC(drive,0,reqid);
		CALL(poll_drive(0,0,reqid));
		process_fd_request();		
		*g = current_type[drive];
	}
	if (!*g) return -ENODEV;
	return 0;
}

/*argument passed to ioctl thread*/
/*all paramters for ioctl function need to packaged into a single argument*/
struct fd_ioctl_arg {
	int devid;
	int cmd;
	void *param;
	int size;
	int reqid;
};

/*ioctl thread function*/
static void* fd_ioctl_thread(void *p)
{
#define OUT(c,x) case c: PRINT("ioctl OUT\n"); outparam = (const char *) (x); break
#define IN(c,x,tag) case c: PRINT("ioctl IN\n"); *(x) = inparam. tag ; ret = 0; goto send

	int i,drive,type;
	int ret;
	union inparam {
		struct floppy_struct g; /* geometry */
		struct format_descr f;
		struct floppy_max_errors max_errors;
		struct floppy_drive_params dp;
	} inparam; 		/* parameters coming from user space */
	const char *outparam; 	/* parameters passed back to user space */
	int devid;
	int cmd;
	void *param;
	int size;
	int reqid;
	struct fd_ioctl_arg *ptr = (struct fd_ioctl_arg*)p;

	PRINT("fd_ioctl thread\n");
	devid = ptr->devid;	/*obtain parameters for ioctl call*/
	cmd = ptr->cmd;
	param = ptr->param;
	size = ptr->size;
	reqid = ptr->reqid;

	type = TYPE(devid);
	drive = DRIVE(devid);

	/* convert compatibility eject ioctls into floppy eject ioctl.
	 * We do this in order to provide a means to eject floppy disks before
	 * installing the new fdutils package */
	if (cmd == 0x5309  || /* CDROMEJECT */ 
		cmd == 0x6470 /* SunOS floppy eject */) 
	{
		DPRINT("obsolete eject ioctl\n");
		DPRINT("please use floppycontrol --eject\n");
		cmd = FDEJECT;
	}

	/* generic block device ioctls */
	switch(cmd) 
	{
		struct floppy_struct *g;
		case HDIO_GETGEO:
		{
			struct hd_geometry loc;
			ret = get_floppy_geometry(drive, type, &g, reqid);
			if(ret) goto send;
			loc.heads = g->head;
			loc.sectors = g->sect;
			loc.cylinders = g->track;
			loc.start = 0;
			ret = _COPYOUT(loc);
			goto send;
		}

		case BLKGETSIZE:	
			ret = get_floppy_geometry(drive, type, &g, reqid);
			if(ret) goto send;
			memcpy((long *) param, &g->size, sizeof(g->size));
			ret = 0;
			goto send;

		/* BLKRRPART is not defined as floppies don't have partition tables */
	}

	/* convert the old style command into a new style command */
	if ((cmd & 0xff00) == 0x0200) 
	{
		ret = normalize_ioctl(&cmd, &size);
		if (ret) goto send;
	}
	else
	{
		ret = -EINVAL;
		goto send;
	}

	/* copyin */
	CLEARSTRUCT(&inparam);
	if (_IOC_DIR(cmd) & _IOC_WRITE)
	{
		ret = fd_copyin((void *)param, &inparam, size);
		if (ret) goto send;
	}

	switch (cmd) 
	{
		case FDEJECT:
			PRINT("ioctl FDEJECT\n");
			if (UDRS->fd_ref != 1)
			{
				ret = -EBUSY;	 /* somebody else has this drive open */
				goto send;
			}
			if (lock_fdc(drive,1,reqid)) 
			{
				ret = -EINTR;
				goto send;
			}

			/* do the actual eject. Fails on non-Sparc architectures */
			ret=fd_eject(UNIT(drive));

			USETF(FD_DISK_CHANGED);
			USETF(FD_VERIFY);
			process_fd_request();
			goto send;

		case FDCLRPRM:
			PRINT("ioctl FDCLRPRM\n");
			if (lock_fdc(drive,1,reqid)) 
			{
				ret = -EINTR;
				goto send;
			}
			current_type[drive] = NULL;
			floppy_sizes[drive] = MAX_DISK_SIZE;
			UDRS->keep_data = 0;
			ret = invalidate_drive(devid,reqid);
			goto send;

		case FDSETPRM:
		case FDDEFPRM:
			PRINT("ioctl FDSETPRM / FDDEFPRM\n");
			ret = set_geometry(cmd, &inparam.g, drive, type, devid, reqid);
			goto send;

		case FDGETPRM:
			PRINT("ioctl FDGETPRM\n");
			ret = get_floppy_geometry(drive, type, (struct floppy_struct**)&outparam, reqid);
			if (ret) goto send;
			break;

		case FDMSGON:
			PRINT("ioctl FDMSGON\n");
			UDP->flags |= FTD_MSG;
			ret = 0;
			goto send;

		case FDMSGOFF:
			PRINT("ioctl FDMSGOFF\n");
			UDP->flags &= ~FTD_MSG;
			ret = 0;
			goto send;

		case FDFMTBEG:
			PRINT("ioctl FDFMTBEG\n");
			if (lock_fdc(drive,1,reqid)) 
			{
				ret = -EINTR;
				goto send;
			}
			ret = poll_drive(1, FD_RAW_NEED_DISK, reqid);
			if (ret == -EINTR) goto send;
			ret = UDRS->flags;
			process_fd_request();
			if (ret & FD_VERIFY) ret = -ENODEV;
			else if (!(ret & FD_DISK_WRITABLE)) ret = -EROFS;
			else ret = 0;
			goto send;

		case FDFMTTRK:
			PRINT("ioctl FDFMTTRK\n");
			if (UDRS->fd_ref != 1) ret = -EBUSY;
			else ret = do_format(devid, &inparam.f,reqid);
			goto send;

		case FDFMTEND:
		case FDFLUSH:
			PRINT("ioctl FDFMTEND / FDFLUSH\n");
			if (lock_fdc(drive,1,reqid)) 
			{
				ret = -EINTR;
				goto send;
			}
			ret = invalidate_drive(devid,reqid);
			goto send;

		case FDSETEMSGTRESH:
			PRINT("ioctl FDSETEMSGTRESH\n");
			UDP->max_errors.reporting = (unsigned short) (*((unsigned int*)param) & 0x0f);
			ret = 0;
			goto send;

		OUT(FDGETMAXERRS, &UDP->max_errors);
		IN(FDSETMAXERRS, &UDP->max_errors, max_errors);

		case FDGETDRVTYP:
			PRINT("ioctl FDGETDRVTYP\n");
			outparam = drive_name(type,drive);
			SUPBOUND(size,strlen(outparam)+1);
			break;

		IN(FDSETDRVPRM, UDP, dp);
		OUT(FDGETDRVPRM, UDP);

		case FDPOLLDRVSTAT:
			PRINT("ioctl FDPOLLDRVSTAT\n");
			if (lock_fdc(drive,1,reqid)) 
			{
				ret = -EINTR;
				goto send;
			}
			ret = poll_drive(1, FD_RAW_NEED_DISK, reqid);
			if (ret == -EINTR) goto send;
			process_fd_request();
			/* fall through */

	       	OUT(FDGETDRVSTAT, UDRS);

		case FDRESET:
			PRINT("ioctl FDRESET\n");
			ret = user_reset_fdc(drive, (int)param, 1, reqid);
			goto send;

		OUT(FDGETFDCSTAT,UFDCS);

		case FDWERRORCLR:
			PRINT("ioctl FDWERRORCLR\n");
			CLEARSTRUCT(UDRWE);
			ret = 0;
			goto send;

		OUT(FDWERRORGET,UDRWE);

		case FDRAWCMD:
			PRINT("ioctl FDRAWCMD\n");
		/*the following has been commented as user level floppy driver*/
		/*does not support passing a pointer in the ioctl structure*/
		/*which is required for this ioctl to pass data to be read/written to disk*/
		/*
			if (type) 
			{
				ret = -EINVAL;
				goto send;
			}
			if (lock_fdc(drive,1,reqid)) 
			{
				ret = -EINTR;
				goto send;
			}
			set_floppy(devid);
			i = raw_cmd_ioctl(cmd,(void *) param, reqid);
			if(i == -EINTR) { ret = i; goto send;}
			process_fd_request();
			ret = i;
		*/
			ret = -EINVAL;
			goto send;

		case FDTWADDLE:
			PRINT("ioctl FDTWADDLE\n");
			if (lock_fdc(drive,1,reqid)) 
			{
				ret = -EINTR;
				goto send;
			}
			twaddle();
			process_fd_request();
			ret = 0;
			goto send;

		default:
			PRINT("ioctl unknown\n");
			ret = -EINVAL;
			goto send;
	}

	if (_IOC_DIR(cmd) & _IOC_READ)
		ret = fd_copyout((void *)param, outparam, size);
	else
		ret = 0;

send: 	userdev_send_response(devid, reqid, IOCTL_OP, ret, size, param);
	if(ptr->param) free(ptr->param);	/*free param*/
	free(ptr);				/*free thread argument*/
	pthread_exit(0);			/*die...*/
	return NULL;
#undef OUT
#undef IN
}

/*userdev ops, ioctl function for floppy driver*/
static void fd_ioctl(int devid, int cmd, void* param, int size, int reqid)
{
	pthread_t ioctl_thread;
	struct fd_ioctl_arg *p;

	PRINT("fd ioctl\n");
	p = (struct fd_ioctl_arg*)malloc(sizeof(struct fd_ioctl_arg));
	p->devid = devid;	/*construct the ioctl thread argument*/
	p->cmd = cmd;
	if (param)
	{
		p->param = (void*)malloc(size);
		memcpy(p->param,param,size);
	}
	else p->param = NULL;
	p->size = size;
	p->reqid = reqid;

	pthread_create(&ioctl_thread,NULL,fd_ioctl_thread,(void*)p);
	pthread_detach(ioctl_thread);	/*create a detached thread that will send response*/
}

/*function to read the floppy type from the CMOS and to initialize floppy drive params*/
static void config_types(void)
{
	int first=1;
	int drive;

	PRINT("config_types\n");
	/* read drive info out of physical CMOS */
	drive=0;
	if (!UDP->cmos)
		UDP->cmos = FLOPPY0_TYPE;
	PRINT("CMOS read %u\n", UDP->cmos);
	drive=1;
	if (!UDP->cmos && FLOPPY1_TYPE)
		UDP->cmos = FLOPPY1_TYPE;

	/* additional physical CMOS drive detection should go here */

	for (drive=0; drive < N_DRIVE; drive++)
	{
		unsigned int type = UDP->cmos;
		struct floppy_drive_params *params;
		const char *name = NULL;
		static char temparea[32];

		if (type < NUMBER(default_drive_params)) 
		{
			params = &default_drive_params[type].params;
			if (type) 
			{
				name = default_drive_params[type].name;
				allowed_drive_mask |= 1 << drive;
			}
		} 
		else 
		{
			params = &default_drive_params[0].params;
			sprintf(temparea, "unknown type %d (usb?)", type);
			name = temparea;
		}
		if (name) 
		{
			const char * prepend = ",";
			if (first) 
			{
				prepend = "Floppy drive(s):";
				first = 0;
			}
			printf("%s fd%d is %s", prepend, drive, name);
		}
		*UDP = *params;	/*assign floppy drive params*/
	}
	if (!first) printf("\n");
}

/*function to close the given device*/
static void floppy_release(int devid)
{
	int drive = DRIVE(devid);

	PRINT("floppy release\n");
	if (UDRS->fd_ref < 0) UDRS->fd_ref=0;
	else if (!UDRS->fd_ref--) 
	{
		DPRINT("floppy_release with fd_ref == 0\n");
		UDRS->fd_ref = 0;
	}
	floppy_release_irq_and_dma();
}

/*userdev ops release function to close the given device*/
static void fd_release(int devid, int reqid)
{
	PRINT("fd release\n");
	floppy_release(devid);	/*call floppy close and send response*/
	userdev_send_response(devid, reqid, CLOSE_OP, 0, 0, NULL);
}

/*open thread argument that packages all the open call parameters*/
struct fd_open_arg{
	int devid;
	unsigned int flags;
	mode_t mode;
	int reqid;
};

#define RETERR(x) do{ floppy_release(devid); userdev_send_response(devid,reqid, OPEN_OP,-(x),0,NULL); return; }while(0)

/*
 * floppy_open check for aliasing (/dev/fd0 can be the same as
 * /dev/PS0 etc), and disallows simultaneous access to the same
 * drive with different device numbers.
 */

/*floppy open thread function*/
static void* floppy_open(void *p)
{
	int drive;
	int old_dev;
	int try;
	char *tmp;
	int ret;
	struct fd_open_arg *ptr = (struct fd_open_arg*)p;

	/*unpackage open call arguments from thread function argument*/
	int devid = ptr->devid;
	unsigned int flags = ptr->flags;
	mode_t mode = ptr->mode;
	int reqid = ptr->reqid;
	
	PRINT("floppy open\n");
	drive = DRIVE(devid);
	if (drive >= N_DRIVE || !(allowed_drive_mask & (1 << drive)) || fdc_state[FDC(drive)].version == FDC_NONE)
	{
		ret = -ENXIO;
		goto send;
	}

	if (TYPE(devid) >= NUMBER(floppy_type))
	{
		ret = -ENXIO;
		goto send;
	}

	old_dev = UDRS->fd_device;
	if (UDRS->fd_ref && old_dev != devid)
	{
		ret = -EBUSY;
		goto send;
	}

	if (!UDRS->fd_ref && (UDP->flags & FD_BROKEN_DCL))
	{
		USETF(FD_DISK_CHANGED);
		USETF(FD_VERIFY);
	}

	if (UDRS->fd_ref == -1 || (UDRS->fd_ref && (flags & O_EXCL)))
	{
		ret = -EBUSY;
		goto send;
	}

	if (floppy_grab_irq_and_dma())
	{
		ret = -EBUSY;
		goto send;
	}

	if (flags & O_EXCL)
		UDRS->fd_ref = -1;
	else
		UDRS->fd_ref++;

	/*Allocate track buffer for read/write operations*/
	if (!floppy_track_buffer)
	{
		/* if opening an ED drive, reserve a big buffer, else reserve a small one */
		if ((UDP->cmos == 6) || (UDP->cmos == 5))
			try = 64; /* Only 48 actually useful */
		else
			try = 32; /* Only 24 actually useful */

		tmp=(char *)fd_dma_mem_alloc(1024 * try);
		if (!tmp && !floppy_track_buffer) 
		{
			try >>= 1; /* buffer only one side */
			INFBOUND(try, 16);
			tmp= (char *)fd_dma_mem_alloc(1024*try);
		}
		if (!tmp && !floppy_track_buffer) 
			fallback_on_nodma_alloc(&tmp, 2048 * try);
		if (!tmp && !floppy_track_buffer) 
		{
			DPRINT("Unable to allocate DMA memory\n");
			RETERR(ENXIO);
		}
		if (floppy_track_buffer) 
		{
			if (tmp)
				fd_dma_mem_free((unsigned long)tmp,try*1024);
		}
		else 
		{
			buffer_min = buffer_max = -1;
			floppy_track_buffer = tmp;
			max_buffer_sectors = try;
		}
	}

	UDRS->fd_device = devid;
	if (old_dev != -1 && old_dev != devid) 
	{
		if (buffer_drive == drive) buffer_track = -1;
		/*The following function call to invalidate buffer cache buffers when*/
		/*a new floppy device is opened is ignored when operating from user level*/
		/*invalidate_buffers(MKDEV(FLOPPY_MAJOR,old_dev));*/
	}

	if (UFDCS->rawcmd == 1) UFDCS->rawcmd = 2;

	if (flags & O_NDELAY)
	{
		ret = 0;
		goto send;
	}

	if (mode & 3) 
	{
		UDRS->last_checked = 0;
		check_disk_change(devid, reqid);
		if (UTESTF(FD_DISK_CHANGED)) RETERR(ENXIO);
	}
	if ((mode & 2) && !(UTESTF(FD_DISK_WRITABLE)))
		RETERR(EROFS);
	ret = 0;

send: 	userdev_send_response(devid, reqid, OPEN_OP, ret, 0, NULL);
	free(ptr);	/*free thread function argument*/
	pthread_exit(0);/*end thread*/
	return NULL;
#undef RETERR
}

/*userdev ops open function*/
static void fd_open(int devid, unsigned int flags, mode_t mode, int reqid)
{
	struct fd_open_arg *p;
	pthread_t open_thread;

	PRINT("fd open\n");
	p = (struct fd_open_arg*)malloc(sizeof(struct fd_open_arg));
	p->devid = devid;	/*package the function arguments into a the open thread argument*/
	p->flags = flags;
	p->mode = mode;
	p->reqid = reqid;
	pthread_create(&open_thread, NULL, floppy_open, (void*)p);
	pthread_detach(open_thread);	/*create a detached thread that would send the response*/
}

/*Check if the disk has been changed or if a change has been faked.*/
static int check_floppy_change(int devid, int reqid)
{
	int drive = DRIVE(devid);

	PRINT("check floppy change\n");
	if (UTESTF(FD_DISK_CHANGED) || UTESTF(FD_VERIFY)) return 1;

	if (UDP->checkfreq < (int)(userdev_get_jiffies() - UDRS->last_checked)) 
	{
		if(floppy_grab_irq_and_dma()) return 1;
		lock_fdc(drive,0,reqid);
		poll_drive(0,0,reqid);
		process_fd_request();
		floppy_release_irq_and_dma();
	}

	if (UTESTF(FD_DISK_CHANGED) || UTESTF(FD_VERIFY) || test_bit(drive, &fake_change) || 
	   (!TYPE(devid) && !current_type[drive]))
		return 1;
	return 0;
}

/*floppy change thread argument that packages arguments of check_floppy change function*/
struct floppy_change_arg {
	int devid;
	int reqid;
};

/*check floppy change thread*/
static void* check_floppy_change_thread(void *p)
{
	struct floppy_change_arg *ptr = (struct floppy_change_arg*)p;
	/*call check floppy change and send back the response*/
	int ret = check_floppy_change(ptr->devid, ptr->reqid);
	userdev_send_response(ptr->devid, ptr->reqid, CHECK_MEDIA_CHANGE_OP, ret, 0, NULL);
	free(ptr);	/*free thread argument*/
	pthread_exit(0);/*end the thread*/
}

/*userdev ops check media change function*/
static void fd_check_change(int devid, int reqid)
{
	pthread_t check_thread;
	struct floppy_change_arg *p;

	PRINT("fd check change\n");
	p = (struct floppy_change_arg*)malloc(sizeof(struct floppy_change_arg));
	p->devid = devid;	/*package arguments*/
	p->reqid = reqid;
	pthread_create(&check_thread, NULL, check_floppy_change_thread, (void*)p);
	pthread_detach(check_thread);	/*create detached thread that would send response*/
}


/* revalidate the floppy disk, i.e. trigger format autodetection by reading
 * the bootblock (block 0). "Autodetection" is also needed to check whether
 * there is a disk in the drive at all... Thus we also do it for fixed
 * geometry formats */
static int floppy_revalidate(int devid, int reqid)
{
#define NO_GEOM (!current_type[drive] && !TYPE(devid))
	struct userdev_blk_request *br;
	pthread_mutex_t waitmutex = PTHREAD_MUTEX_INITIALIZER;
	int drive=DRIVE(devid);
	int cf;

	PRINT("floppy revalidate\n");
	if (UTESTF(FD_DISK_CHANGED) || UTESTF(FD_VERIFY) || test_bit(drive, &fake_change) || NO_GEOM)
	{
		if(usage_count == 0) 
		{
			printf("VFS: revalidate called on non-open device.\n");
			return -EFAULT;
		}
		lock_fdc(drive,0,reqid);
		cf = UTESTF(FD_DISK_CHANGED) || UTESTF(FD_VERIFY);
		if (!(cf || test_bit(drive, &fake_change) || NO_GEOM))
		{
			process_fd_request(); /*already done by another thread*/
			return 0;
		}
		UDRS->maxblock = 0;
		UDRS->maxtrack = 0;
		if (buffer_drive == drive)
			buffer_track = -1;
		clear_bit(drive, &fake_change);
		UCLEARF(FD_DISK_CHANGED);
		if (cf)	UDRS->generation++;
		if (NO_GEOM)
		{
			/* auto-sensing by adding a dumy request(reqid=-1) in request queue and processing it*/
			int size = floppy_blocksizes[MINOR(devid)];
			if (!size) size = 1024;
			/*deviceid, operation, sector no:, size, clustersize, data, datasize, dummy reqid*/
			br = userdev_add_request(devid, READ_REQ, 0, size, size>>9, NULL, 0, -1);/*add a dummy request into request queue*/
			process_fd_request();
			pthread_mutex_lock(&waitmutex);			/*lock dummy waitmutex*/
			pthread_cond_wait(&br->done_cond, &waitmutex);	/*block till request completed (pthread_cond_signal called)*/
			pthread_mutex_unlock(&waitmutex);		/*unlock dummy waitmutex*/
			return 0;
		}
		if (cf) poll_drive(0, FD_RAW_NEED_DISK,reqid);
		process_fd_request();
	}
	return 0;
}

/*floppy revalidate thread arg, structurally same as that of floppy change thread arg*/
typedef struct floppy_change_arg floppy_revalid_arg;

/*floppy revalidate thread*/
static void* fd_revalidate_thread(void *p)
{
	floppy_revalid_arg *ptr = (floppy_revalid_arg*)p;
	int ret = floppy_revalidate(ptr->devid, ptr->reqid);
	userdev_send_response(ptr->devid, ptr->reqid, REVALIDATE_OP, ret, 0, NULL);
	free(ptr);	/*free function argument*/
	pthread_exit(0);/*end thread*/
}

/*userdev ops, revalidate function for floppy driver*/
static void fd_revalidate(int devid, int reqid)
{
	floppy_revalid_arg *p;
	pthread_t revalid_thread;

	PRINT("fd revalidate\n");
	p = ( floppy_revalid_arg*)malloc(sizeof(floppy_revalid_arg));
	p->devid = devid;	/*package arguments*/
	p->reqid = reqid;
	pthread_create(&revalid_thread, NULL, fd_revalidate_thread, (void*)p);
	pthread_detach(revalid_thread);	/*create a detached thread to send response*/
}

/*userdev ops, message function to receive signal pending message*/
static void fd_message(int devid, int type, int reqid, void *data, int size)
{
	PRINT("fd message\n");
	if(type == SIGNAL_PENDING) userdev_add_sig_list(reqid);	/*add to signal message list*/
	pthread_cond_signal(&command_cond);/*wake up any blocked thread so that the operation can be aborted*/
	/*when the cond signal breaks the condition variable wait, a check is made if a signal is pending*/
	/*for the operation (using request id) and the operation is aborted*/
}

/*userdev operations for floppy driver*/
static struct userdev_operations floppy_uops = {
	devtype:		USERDEV_BLOCK,
	open:			fd_open,
	close:			fd_release,
	ioctl:			fd_ioctl,
	check_media_change:	fd_check_change,
	revalidate:		fd_revalidate,
	message:		fd_message,
	request:		do_fd_request
};

/*ioctls supported by floppy driver, list for userdev_attach*/
# define FD_NUM_IOCTLS 25
static struct userdev_ioctl_data ioelm[FD_NUM_IOCTLS] = {
	{FDEJECT,0},
	{FDCLRPRM,0},
	{FDSETPRM,sizeof(struct floppy_struct)},
	{FDDEFPRM,sizeof(struct floppy_struct)},
	{FDGETPRM,sizeof(struct floppy_struct)},
	{FDMSGON,0},
	{FDMSGOFF,0},
	{FDFMTBEG,0},
	{FDFMTTRK,sizeof(struct format_descr)},
	{FDFMTEND,0},
	{FDFLUSH,0},
	{FDSETEMSGTRESH,sizeof(int)},
	{FDGETMAXERRS,sizeof(struct floppy_max_errors)},
	{FDSETMAXERRS,sizeof(struct floppy_max_errors)},
	{FDGETDRVTYP,sizeof(floppy_drive_name)},
	{FDSETDRVPRM,sizeof(struct floppy_drive_params)},
	{FDGETDRVPRM,sizeof(struct floppy_drive_params)},
	{FDPOLLDRVSTAT,sizeof(struct floppy_drive_struct)},
	{FDGETDRVSTAT,sizeof(struct floppy_drive_struct)},
	{FDRESET,0},
	{FDGETFDCSTAT,sizeof(struct floppy_fdc_state)},
	{FDWERRORCLR,0},
	{FDWERRORGET,sizeof(struct floppy_write_errors)},
	{FDRAWCMD,sizeof(struct floppy_raw_cmd)},
	{FDTWADDLE,0}
};


/* Floppy Driver initialization */
/* =============================*/

/* Determine the floppy disk controller type */
static char get_fdc_version(void)
{
	int r;

	PRINT("get fdc version\n");
	output_byte(FD_DUMPREGS);	/* 82072 and better know DUMPREGS */
	if (FDCS->reset)
		return FDC_NONE;
	if ((r = result()) <= 0x00)
		return FDC_NONE;	/* No FDC present ??? */
	if ((r==1) && (reply_buffer[0] == 0x80))
	{
		printf("FDC %d is an 8272A\n",fdc);
		return FDC_8272A;	/* 8272a/765 don't know DUMPREGS */
	}
	if (r != 10) 
	{
		printf("FDC %d init: DUMPREGS: unexpected return of %d bytes.\n",
		       fdc, r);
		return FDC_UNKNOWN;
	}

	if (!fdc_configure()) 
	{
		printf("FDC %d is an 82072\n",fdc);
		return FDC_82072;      	/* 82072 doesn't know CONFIGURE */
	}

	output_byte(FD_PERPENDICULAR);
	if (need_more_output() == MORE_OUTPUT) 
	{
		output_byte(0);
	} 
	else 
	{
		printf("FDC %d is an 82072A\n", fdc);
		return FDC_82072A;	/* 82072A as found on Sparcs. */
	}

	output_byte(FD_UNLOCK);
	r = result();
	if ((r == 1) && (reply_buffer[0] == 0x80))
	{
		printf("FDC %d is a pre-1991 82077\n", fdc);
		return FDC_82077_ORIG;	/* Pre-1991 82077, doesn't know 
					 * LOCK/UNLOCK */
	}
	if ((r != 1) || (reply_buffer[0] != 0x00)) 
	{
		printf("FDC %d init: UNLOCK: unexpected return of %d bytes.\n",
		       fdc, r);
		return FDC_UNKNOWN;
	}
	output_byte(FD_PARTID);
	r = result();
	if (r != 1) 
	{
		printf("FDC %d init: PARTID: unexpected return of %d bytes.\n",
		       fdc, r);
		return FDC_UNKNOWN;
	}
	if (reply_buffer[0] == 0x80) 
	{
		printf("FDC %d is a post-1991 82077\n",fdc);
		return FDC_82077;	/* Revised 82077AA passes all the tests */
	}
	switch (reply_buffer[0] >> 5) 
	{
		case 0x0:
			/* Either a 82078-1 or a 82078SL running at 5Volt */
			printf("FDC %d is an 82078.\n",fdc);
			return FDC_82078;
		case 0x1:
			printf("FDC %d is a 44pin 82078\n",fdc);
			return FDC_82078;
		case 0x2:
			printf("FDC %d is a S82078B\n", fdc);
			return FDC_S82078B;
		case 0x3:
			printf("FDC %d is a National Semiconductor PC87306\n", fdc);
			return FDC_87306;
		default:
			printf("FDC %d init: 82078 variant with unknown PARTID=%d.\n",
			       fdc, reply_buffer[0] >> 5);
			return FDC_82078_UNKN;
	}
}

/* lilo configuration functions, called for command line configuration of driver parameters*/

static void floppy_set_flags(int *ints,int param, int param2)
{
	int i;

	PRINT("floppy set flags\n");
	for (i=0; i < ARRAY_SIZE(default_drive_params); i++)
	{
		if (param)
			default_drive_params[i].params.flags |= param2;
		else
			default_drive_params[i].params.flags &= ~param2;
	}
	DPRINT("%s flag 0x%x\n", param2 ? "Setting" : "Clearing", param);
}

static void daring(int *ints,int param, int param2)
{
	int i;

	PRINT("daring\n");
	for (i=0; i < ARRAY_SIZE(default_drive_params); i++)
	{
		if (param)
		{
			default_drive_params[i].params.select_delay = 0;
			default_drive_params[i].params.flags |= FD_SILENT_DCL_CLEAR;
		} 
		else 
		{
			default_drive_params[i].params.select_delay = 2*HZ/100;
			default_drive_params[i].params.flags &= ~FD_SILENT_DCL_CLEAR;
		}
	}
	DPRINT("Assuming %s floppy hardware\n", param ? "standard" : "broken");
}

static void set_cmos(int *ints, int dummy, int dummy2)
{
	int current_drive=0;

	PRINT("set cmos\n");
	if (ints[0] != 2)
	{
		DPRINT("wrong number of parameters for CMOS\n");
		return;
	}
	current_drive = ints[1];
	if (current_drive < 0 || current_drive >= 8)
	{
		DPRINT("bad drive for set_cmos\n");
		return;
	}
	if (current_drive >= 4 && !FDC2)
		FDC2 = 0x370;
	DP->cmos = ints[2];
	DPRINT("setting CMOS code to %d\n", ints[2]);
}


static int have_no_fdc= -EIO;

/*cleanup function of the driver to release all resourses and exit gracefully*/
void floppy_cleanup()
{
	int ret;
	PRINT("floppy cleanup pid=%d\n",getpid());
	userdev_task_queue_cleanup(&tq_immediate);	/*set stop flag so that the task queue thread stops*/
	userdev_timer_queue_cleanup();	/*set stop flag for timer queue thread to stop it when it checks the flag next*/

	fdc = 0;
	userdev_reset_ioperm(FD_STATUS);
	userdev_reset_ioperm(FD_DOR);
	userdev_reset_ioperm(FD_DCR);
	userdev_reset_ioperm(FD_DIR);
	userdev_reset_ioperm(FD_DATA);
	if (FDCS->address != -1) 
	{
 		userdev_release_ioregion(FDCS->address, 6);
 		userdev_release_ioregion(FDCS->address+7, 1);
	}

	fdc = 1;
	userdev_reset_ioperm(FD_STATUS);
	userdev_reset_ioperm(FD_DOR);
	userdev_reset_ioperm(FD_DCR);
	userdev_reset_ioperm(FD_DIR);
	userdev_reset_ioperm(FD_DATA);
	if (FDCS->address != -1) 
	{
 		userdev_release_ioregion(FDCS->address, 6);
 		userdev_release_ioregion(FDCS->address+7, 1);
	}

	ret = userdev_detach(userdev_id);	/*detach from userdev, frees irq and dma*/
	if(ret) printf("Detach from userdev failed !!!!\n");
	else printf("Detached from userdev.\n");
}

/*function to initialise the floppy driver*/
int floppy_init(void)
{
	int i,unit,drive,ret;
	struct userdev_blk_data data;
	struct stat status;

	PRINT("floppy init\n");
	raw_cmd = NULL;

        for(i=0;i<256;i++)	/*initialize floppy_sizes array*/
		if (ITYPE(i))
			floppy_sizes[i] = (floppy_type[ITYPE(i)].size+1) >> 1;
		else
			floppy_sizes[i] = MAX_DISK_SIZE;

	/* Here we register with userdev for only one device, instead of for all 256*/
	/*define the various size parameters of the device*/
	data.devsize = floppy_sizes[7];			/*device size*/
	data.blksize = floppy_blocksizes[7] = 1024;	/* block size in bytes*/
	data.sectsize = 512;				/* sector size in bytes*/
	data.maxreadahead = 31;				/*max read ahead*/
	data.maxsectors = 255;				/*max number of sectors in each request*/

	if ((userdev_id = userdev_attach(devicename, ioelm, FD_NUM_IOCTLS, &floppy_uops, &data)) < 0)
	{
		printf("Unable to attach to userdev\n");
		return -EBUSY;
	}
	else printf("Attached to userdev with id = %d\n",userdev_id);

	/*obtain minor number from device filename used for attach*/
	ret = stat(devicename,&status);
	if(ret < 0)
	{
		printf("Could not detect minor number of device\n");
		userdev_detach(userdev_id);
		return -EBUSY;
	}

	/*here since we cannot have minor zero and only minor zero allows auto configuration*/
	/*we need to assign zero in the minor map so that for each device registered the configuration*/
	/*will be read from the device instead of being pre-determined*/

	/*minor_map[status.st_rdev & 0xff] = userdev_id;*/
	minor_map[0] = userdev_id;

	config_types();

	for (i = 0; i < N_FDC; i++)
	{
		fdc = i;
		CLEARSTRUCT(FDCS);
		FDCS->dtr = -1;
		FDCS->dor = 0x4;
#ifdef __sparc__
		/*sparcs don't have a DOR reset which we can fall back on to*/
		FDCS->version = FDC_82072A;
#endif
	}

	use_virtual_dma = can_use_virtual_dma & 1;
	fdc_state[0].address = FDC1;
	if (fdc_state[0].address == -1)
	{
		userdev_del_timer(&fd_timeout);
		userdev_detach(userdev_id);
		return -ENODEV;
	}

#if N_FDC > 1
	fdc_state[1].address = FDC2;
#endif
	setioperms();			/*obtain io permissions for all ports*/
	userdev_init_timer_queue(thread_init);	/*initialise and start timer queue*/
	userdev_run_task_queue(&tq_immediate,thread_init);	/*start the task queue thread*/
	reschedule_timeout(MAXTIMEOUT, "floppy init", MAXTIMEOUT);

	fdc = 0; /* reset fdc in case of unexpected interrupt */
	if (floppy_grab_irq_and_dma())
	{
		userdev_del_timer(&fd_timeout);
		userdev_detach(userdev_id);
		return -EBUSY;
	}

	/* initialise drive state */
	for (drive = 0; drive < N_DRIVE; drive++)
	{
		CLEARSTRUCT(UDRS);
		CLEARSTRUCT(UDRWE);
		USETF(FD_DISK_NEWCHANGE);
		USETF(FD_DISK_CHANGED);
		USETF(FD_VERIFY);
		UDRS->fd_device = -1;
		floppy_track_buffer = NULL;
		max_buffer_sectors = 0;
	}

	for (i = 0; i < N_FDC; i++)
	{
		fdc = i;
		FDCS->driver_version = FD_DRIVER_VERSION;
		for (unit=0; unit<4; unit++)
			FDCS->track[unit] = 0;
		if (FDCS->address == -1)
			continue;
		FDCS->rawcmd = 2;
		if (user_reset_fdc(-1,FD_RESET_ALWAYS,0, -1))	/*dummy reqid=-1 so that search in pending signal list will be empty*/
		{
 			/* free ioports reserved by floppy_grab_irq_and_dma() */
 			userdev_release_ioregion(FDCS->address, 6);
 			userdev_release_ioregion(FDCS->address+7, 1);
			FDCS->address = -1;
			FDCS->version = FDC_NONE;
			continue;
		}
		/* Try to determine the floppy controller type */
		FDCS->version = get_fdc_version();
		if (FDCS->version == FDC_NONE)
		{
 			/* free ioports reserved by floppy_grab_irq_and_dma() */
 			userdev_release_ioregion(FDCS->address, 6);
 			userdev_release_ioregion(FDCS->address+7, 1);
			FDCS->address = -1;
			continue;
		}
		if (can_use_virtual_dma == 2 && FDCS->version < FDC_82072A)
			can_use_virtual_dma = 0;

		have_no_fdc = 0;
		/* Not all FDCs seem to be able to handle the version command
		 * properly, so force a reset for the standard FDC clones,
		 * to avoid interrupt garbage.
		 */
		user_reset_fdc(-1,FD_RESET_ALWAYS,0, -1);
	}
	fdc=0;
	userdev_del_timer(&fd_timeout);
	current_drive = 0;
	floppy_release_irq_and_dma();
	initialising=0;
	if (have_no_fdc)
	{
		DPRINT("no floppy controllers found\n");
		floppy_tq.routine = (void *)(void *) empty;
		if (usage_count) floppy_release_irq_and_dma();
		userdev_blk_requestq_cleanup();
		floppy_cleanup();
	}

	return have_no_fdc;
}

/*mutex for access to usage_count*/
static pthread_mutex_t floppy_usage_lock = PTHREAD_MUTEX_INITIALIZER;

/*function to obtain irq, dma channel and io address space for the floppy*/
/*A call to the function  while holding these resources just increments the usage count*/
static int floppy_grab_irq_and_dma(void)
{
	PRINT("floppy_grab_irq_dma");
	pthread_mutex_lock(&floppy_usage_lock);
	if (usage_count++)
	{
		PRINT(" %d\n",usage_count);
		pthread_mutex_unlock(&floppy_usage_lock);
		return 0;
	}
	PRINT(" %d\n",usage_count);
	pthread_mutex_unlock(&floppy_usage_lock);

	if (fd_request_irq(userdev_id)) 
	{
		DPRINT("Unable to grab IRQ%d for the floppy driver\n",FLOPPY_IRQ);
		pthread_mutex_lock(&floppy_usage_lock);
		usage_count--;
		pthread_mutex_unlock(&floppy_usage_lock);
		return -1;
	}
	if (fd_request_dma(userdev_id)) 
	{
		DPRINT("Unable to grab DMA%d for the floppy driver\n",FLOPPY_DMA);
		fd_free_irq(userdev_id);
		pthread_mutex_lock(&floppy_usage_lock);
		usage_count--;
		pthread_mutex_unlock(&floppy_usage_lock);
		return -1;
	}

	for (fdc=0; fdc< N_FDC; fdc++)
	{
		if (FDCS->address != -1)
		{
			if (userdev_request_ioregion(FDCS->address, 6, "floppy") < 0) 
			{
				DPRINT("Floppy io-port 0x%04lx in use\n", FDCS->address);
				goto cleanup1;
			}
			if (userdev_request_ioregion(FDCS->address + 7, 1, "floppy DIR") < 0) 
			{
				DPRINT("Floppy io-port 0x%04lx in use\n", FDCS->address + 7);
				goto cleanup2;
			}
			/* address + 6 is reserved, and may be taken by IDE.
			 * Unfortunately, Adaptec doesn't know this :-(, */
		}
	}
	
	for (fdc=0; fdc< N_FDC; fdc++)
	{
		if (FDCS->address != -1)
		{
			reset_fdc_info(1);
			fd_outb(FDCS->dor, FD_DOR);
		}
	}

	fdc = 0;
	set_dor(0, ~0, 8);  /* avoid immediate interrupt */
	for (fdc = 0; fdc < N_FDC; fdc++)
		if (FDCS->address != -1)
			fd_outb(FDCS->dor, FD_DOR);

	/*The driver will try and free resources and relies on us to know if they were allocated or not.*/
	fdc = 0;
	irqdma_allocated = 1;
	return 0;

cleanup2:
	userdev_release_ioregion(FDCS->address, 6);
cleanup1:
	fd_free_irq(userdev_id);
	fd_free_dma(userdev_id);
	while(--fdc >= 0) 
	{
		userdev_release_ioregion(FDCS->address, 6);
		userdev_release_ioregion(FDCS->address + 7, 1);
	}
	pthread_mutex_lock(&floppy_usage_lock);
	usage_count--;
	pthread_mutex_unlock(&floppy_usage_lock);
	return -1;
}

/*function to release irq, dma channel and io address space when the usage count drops to zero*/
static void floppy_release_irq_and_dma(void)
{
	int old_fdc;
#ifdef FLOPPY_SANITY_CHECK
#ifndef __sparc__
	int drive;
#endif
#endif
	long tmpsize;
	unsigned long tmpaddr;
	unsigned long flags;

	PRINT("floppy_release dma & irq");
	pthread_mutex_lock(&floppy_usage_lock);
	if (--usage_count)
	{
		PRINT(" %d\n",usage_count);
		pthread_mutex_unlock(&floppy_usage_lock);
		return;
	}
	PRINT(" %d\n",usage_count);
	pthread_mutex_unlock(&floppy_usage_lock);

	if(irqdma_allocated)
	{
		fd_free_dma(userdev_id);
		fd_free_irq(userdev_id);
		irqdma_allocated=0;
	}
	set_dor(0, ~0, 8);
#if N_FDC > 1
	set_dor(1, ~8, 0);
#endif
	floppy_enable_hlt();
	
	if (floppy_track_buffer && max_buffer_sectors) 
	{
		tmpsize = max_buffer_sectors*1024;
		tmpaddr = (unsigned long)floppy_track_buffer;
		floppy_track_buffer = NULL;
		max_buffer_sectors = 0;
		buffer_min = buffer_max = -1;
		fd_dma_mem_free(tmpaddr, tmpsize);
	}

#ifdef FLOPPY_SANITY_CHECK
#ifndef __sparc__
	for (drive=0; drive < N_FDC * 4; drive++)
		if (userdev_timer_pending(motor_off_timer + drive))
			printf("motor off timer %d still active\n", drive);
#endif

	if (userdev_timer_pending(&fd_timeout))
		printf("floppy timer still active:%s\n", timeout_message);
	if (userdev_timer_pending(&fd_timer))
		printf("auxiliary floppy timer still active\n");
	if (floppy_tq.sync)
		printf("task queue still active\n");
#endif
	old_fdc = fdc;
	for (fdc = 0; fdc < N_FDC; fdc++)
		if (FDCS->address != -1) 
		{
			userdev_release_ioregion(FDCS->address, 6);
			userdev_release_ioregion(FDCS->address+7, 1);
		}
	fdc = old_fdc;
}

/*table of parameters that can be configured using command-line arguments*/
static struct param_table {
	const char *name;
	void (*fn)(int *ints, int param, int param2);
	int *var;
	int def_param;
	int param2;
} config_params[]={
	{ "allowed_drive_mask", 0, &allowed_drive_mask, 0xff, 0}, /* obsolete */
	{ "all_drives", 0, &allowed_drive_mask, 0xff, 0 }, /* obsolete */
	{ "asus_pci", 0, &allowed_drive_mask, 0x33, 0},

	{ "irq", 0, &FLOPPY_IRQ, 6, 0 },
	{ "dma", 0, &FLOPPY_DMA, 2, 0 },

	{ "daring", daring, 0, 1, 0},

	{ "two_fdc",  0, &FDC2, 0x370, 0 },
	{ "one_fdc", 0, &FDC2, 0, 0 },

	{ "thinkpad", floppy_set_flags, 0, 1, FD_INVERTED_DCL },
	{ "broken_dcl", floppy_set_flags, 0, 1, FD_BROKEN_DCL },
	{ "messages", floppy_set_flags, 0, 1, FTD_MSG },
	{ "silent_dcl_clear", floppy_set_flags, 0, 1, FD_SILENT_DCL_CLEAR },
	{ "debug", floppy_set_flags, 0, 1, FD_DEBUG },

	{ "nodma", 0, &can_use_virtual_dma, 1, 0 },
	{ "omnibook", 0, &can_use_virtual_dma, 1, 0 },
	{ "yesdma", 0, &can_use_virtual_dma, 0, 0 },

	{ "fifo_depth", 0, &fifo_depth, 0xa, 0 },
	{ "nofifo", 0, &no_fifo, 0x20, 0 },
	{ "usefifo", 0, &no_fifo, 0, 0 },

	{ "cmos", set_cmos, 0, 0, 0 },
	{ "slow", 0, &slow_floppy, 1, 0 },

	{ "unexpected_interrupts", 0, &print_unex, 1, 0 },
	{ "no_unexpected_interrupts", 0, &print_unex, 0, 0 },
	{ "L40SX", 0, &print_unex, 0, 0 }
};


/*
 * simple_strtoul - convert a string to an unsigned long
 * @cp: The start of the string
 * @endp: A pointer to the end of the parsed string will be placed here
 * @base: The number base to use
 */
static unsigned long simple_strtoul(const char *cp,char **endp,unsigned int base)
{
        unsigned long result = 0,value;

        if (!base) 
	{
                base = 10;
                if (*cp == '0') 
		{
                        base = 8;
                        cp++;
                        if ((*cp == 'x') && isxdigit(cp[1])) 
			{
                                cp++;
                                base = 16;
                        }
                }
        }
        while (isxdigit(*cp) && (value = isdigit(*cp) ? *cp-'0' : toupper(*cp)-'A'+10) < base) 
	{
                result = result*base + value;
                cp++;
        }
        if (endp)
                *endp = (char *)cp;
        return result;
}


/*
 * simple_strtol - convert a string to a signed long
 * @cp: The start of the string
 * @endp: A pointer to the end of the parsed string will be placed here
 * @base: The number base to use
 */
static long simple_strtol(const char *cp,char **endp,unsigned int base)
{
        if(*cp=='-') return -simple_strtoul(cp+1,endp,base);
        return simple_strtoul(cp,endp,base);
}


/*
 *get_option - Parse integer from an option string
 *@str: option string
 *@pint: (output) integer value parsed from @str
 *Read an int from an option string; if available accept a subsequent comma as well.
 *Return values:
 *      0 : no int in string
 *      1 : int found, no subsequent comma
 *      2 : int found including a subsequent comma
 */
static int get_option (char **str, int *pint)
{
        char *cur = *str;

        if (!cur || !(*cur)) return 0;
        *pint = simple_strtol (cur, str, 0);
        if (cur == *str) return 0;
        if (**str == ',') 
	{
                (*str)++;
                return 2;
        }
        return 1;
}


/*
 *get_options - Parse a string into a list of integers
 *@str: String to be parsed
 *@nints: size of integer array
 *@ints: integer array
 *This function parses a string containing a comma-separated
 *list of integers.  The parse halts when the array is
 *full, or when no more numbers can be retrieved from the string
 *Return value is the character in the string which caused
 *the parse to end (typically a null terminator, if @str is
 *completely parseable).
 */
static char *get_options (char *str, int nints, int *ints)
{
        int res, i = 1;

        while (i < nints) 
	{
                res = get_option (&str, ints + i);
                if (res == 0) break;
                i++;
                if (res == 1) break;
        }
        ints[0] = i - 1;
        return (str);
}

/*function to assign user-specified values to the floppy drive parameters*/
/*by parsing the option string to obtain the option and the value to be assigned*/
/*followed by lookup in the table above and calling a function or assigning the value*/
/*to a matching entry in the table. The matching is done using parameter name*/
static int floppy_setup(char *str)
{
	int i;
	int param;
	int ints[11];

	PRINT("floppy setup\n");
	str = get_options(str,ARRAY_SIZE(ints),ints);
	if (str) 
	{
		for (i=0; i< ARRAY_SIZE(config_params); i++)
			if (strcmp(str,config_params[i].name) == 0)
			{
				if (ints[0])
					param = ints[1];
				else
					param = config_params[i].def_param;
				if (config_params[i].fn)
					config_params[i].fn(ints,param,config_params[i].param2);
				if (config_params[i].var) 
				{
					DPRINT("%s=%d\n", str, param);
					*config_params[i].var = param;
				}
				return 1;
			}
	}
	if (str)
	{
		DPRINT("unknown floppy option [%s]\n", str);
		DPRINT("allowed options are:");
		for (i=0; i< ARRAY_SIZE(config_params); i++)
			printf(" %s",config_params[i].name);
		printf("\n");
		printf("Usage: floppy \"val1,option1 val2,option2 ...\"\n");
	}
	else DPRINT("botched floppy option\n");
	return 0;
}

/*function to parse the command line string to obtain the list of options specified*/
/*and to call floppy_setup to configure the correponding floppy drive parameter*/
/*options are given as : "val1,option1 val2,option2 ...." */
static void parse_floppy_cfg_string(char *cfg)
{
	char *ptr;

	while(*cfg)
	{
		for(ptr = cfg;*cfg && *cfg != ' ' && *cfg != '\t'; cfg++);
		if (*cfg)
		{
			*cfg = '\0';
			cfg++;
		}
		if (*ptr) floppy_setup(ptr);
	}
}

/*function to set the io permissions for all the ports that would be accessed*/
/*For each thread we need to again set the io permissions as in linux each thread*/
/*is implemented as a separate process and the permissions are not inhereited*/
static void setioperms()
{
	int tmp = fdc;
	fdc = 0;	/*floppy drive controller 0*/
	if(userdev_set_ioperm(FD_STATUS))
		printf("could not obtain io permisssion for %d\n",FD_STATUS);
	if(userdev_set_ioperm(FD_DOR))
		printf("could not obtain io permisssion for %d\n",FD_DOR);
	if(userdev_set_ioperm(FD_DCR))
		printf("could not obtain io permisssion for %d\n",FD_DCR);
	if(userdev_set_ioperm(FD_DIR))
		printf("could not obtain io permisssion for %d\n",FD_DIR);
	if(userdev_set_ioperm(FD_DATA))
		printf("could not obtain io permisssion for %d\n",FD_DATA);
# if N_FDC > 1
	fdc = 1;	/*floppy drive controller 1*/
	if(userdev_set_ioperm(FD_STATUS))
		printf("could not obtain io permisssion for %d\n",FD_STATUS);
	if(userdev_set_ioperm(FD_DOR))
		printf("could not obtain io permisssion for %d\n",FD_DOR);
	if(userdev_set_ioperm(FD_DCR))
		printf("could not obtain io permisssion for %d\n",FD_DCR);
	if(userdev_set_ioperm(FD_DIR))
		printf("could not obtain io permisssion for %d\n",FD_DIR);
	if(userdev_set_ioperm(FD_DATA))
		printf("could not obtain io permisssion for %d\n",FD_DATA);
# endif
	fdc = tmp;
}

/*function that is passed as the initialisation function for timer queue and task queue*/
/*This function gets called at start of each of the threads*/
/*The function masks the SIGINT signal on each of them and calls setioperms*/
/*SIGINT is masked so that only the main thread would receive the SIGINT signal*/
/*which is used to end the driver process*/
void thread_init()
{
	sigset_t set;
	sigemptyset(&set);
	sigaddset(&set,SIGINT);	/*mask SIGINT signal*/
	pthread_sigmask(SIG_BLOCK,&set,NULL);
	setioperms();		/*set io permissions*/
}

/*signal handler for SIGINT signal, used to end the driver process*/
void signal_handler(int sig)
{
	floppy_cleanup();	/*cleanup floppy driver*/
	exit(0);		/*end driver process*/
}


int main(int argc, char *argv[])
{
	signal(SIGINT,signal_handler);	/*install signal handler for SIGINT to end driver*/
	if(argc > 1) parse_floppy_cfg_string(argv[1]);	/*parse command line argument*/
	if(floppy_init() < 0)				/*initialise floppy driver*/
	{
		printf("Floppy initialization failed!!!\n");
		exit(1);
	}
	else printf("Floppy driver initialized\n");

	/*Start accepting requests from kernel*/
	if(userdev_start() < 0) printf("Cannot start accepting requests!!!\n");

	/*Incase of a failure, cleanup*/
	floppy_cleanup();
}
