/*
    SRNDT
    
    A tool for the miroMEDIA Surround Dolby Pro Logic decoder.

    Copyright (C) 1998-2001 Oliver Gantz <Oliver.Gantz@ePost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ----
    3D-Phonic(R) is a registered trademark of VICTOR COMPANY OF JAPAN, LIMITED.
    DOLBY and PRO LOGIC are trademarks of Dolby Laboratories Licensing Corporation.
    miro(R) is a registered trademark of miro Computer Products AG.
*/


#define SRNDT_VERSION 1.0


#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>

#include <srnd.h>


static char devname[128] = "/dev/srnd";



void srnd_ioctl(int number, int arg)
{
	int fd;

	if ((fd = open(devname, O_RDWR)) == -1) {
		perror(devname);
		exit(-1);
	}

	if (ioctl(fd, number, arg) == -1) {
		perror(devname);
		close(fd);
		exit(-1);
	}

	if (close(fd) == -1) {
		perror(devname);
		exit(-1);
	}
}


void srnd_ioctl_perc(int number, char *arg, int max)
{
	int narg;
	
	narg = atoi(arg) * max / 100;

	if (narg < 0)
		narg = 0;
	if (narg > max)
		narg = max;

	srnd_ioctl(number, narg);
}



void print_version (void)
{
	int fd, v;

	fprintf(stderr, "srndt %.2f (SRND driver ", SRNDT_VERSION);

	if ((fd = open(devname, O_RDWR)) != -1) {
		ioctl(fd, SRNDVERSION, &v);
		fprintf(stderr, "%d.%d", v >> 8, v & 0xff);
		close(fd);
	}
	else
		fprintf(stderr, "not found");
	
	fprintf(stderr, ")\n");
}


void print_help (void)
{
	fprintf(stderr, "Usage: srndt [-Vh] [-d device] [--device=device] [--help] [--version]\n" \
	                "       command [arg...]\n");
	exit(1);
}


int main (int argc, char *argv[])
{
	int c;

	while (1) {
		int option_index = 0;
		static struct option long_options[] = {
			{ "device", 1, 0, 'd' },
			{ "help", 0, 0, 'h' },
			{ "version", 0, 0, 'V' },
			{ 0, 0, 0, 0 }
		};

		c = getopt_long(argc, argv, "Vd:h", long_options, &option_index);

		if (c == -1)
			break;
		
		switch (c) {
			case 'd':
				strcpy(devname, optarg);
				break;
			case 'h':
				print_help();
				break;
			case 'V':
				print_version();
				break;
			default:
				print_help();
		}
	}

	if (optind < argc) {
		if (strcmp(argv[optind],"reset") == 0) {
			srnd_ioctl(SRNDRESET, 0);
		} else
		if (strcmp(argv[optind],"volume") == 0) {
			optind++;
			if (optind == argc)
				print_help();
			srnd_ioctl_perc(SRNDDAC_VOL, argv[optind++], 16);
		} else
		if (strcmp(argv[optind],"bypass") == 0) {
			optind++;
			if (optind == argc)
				print_help();
			if (strcmp(argv[optind],"off") == 0) {
				srnd_ioctl(SRNDBYPASS, 0);
			} else
			if (strcmp(argv[optind],"on") == 0) {
				srnd_ioctl(SRNDBYPASS, 1);
			} else {
				fprintf(stderr, "%s: illegal command arg -- %s\n", argv[0], argv[optind]);
				print_help();
			}
		} else
		if (strcmp(argv[optind],"dolby") == 0) {
			optind++;
			if (optind == argc)
				print_help();
			if (strcmp(argv[optind],"normal") == 0) {
				optind++;
				if ((argc-optind) < 4)
					print_help();
				srnd_ioctl(SRNDDBPL_INIT, 0);
				srnd_ioctl(SRNDDBPL_MODE, 0);
				srnd_ioctl_perc(SRNDDBPL_VOL_LR, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_VOL_C, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_VOL_S, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_DELAY, argv[optind], 15);
			} else
			if (strcmp(argv[optind],"wide") == 0) {
				optind++;
				if ((argc-optind) < 4)
					print_help();
				srnd_ioctl(SRNDDBPL_INIT, 0);
				srnd_ioctl(SRNDDBPL_MODE, 1);
				srnd_ioctl_perc(SRNDDBPL_VOL_LR, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_VOL_C, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_VOL_S, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_DELAY, argv[optind], 15);
			} else
			if (strcmp(argv[optind],"phantom") == 0) {
				optind++;
				if ((argc-optind) < 3)
					print_help();
				srnd_ioctl(SRNDDBPL_INIT, 0);
				srnd_ioctl(SRNDDBPL_MODE, 2);
				srnd_ioctl_perc(SRNDDBPL_VOL_LR, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_VOL_S, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_DELAY, argv[optind], 15);
			} else
			if (strcmp(argv[optind],"3") == 0) {
				optind++;
				if ((argc-optind) < 3)
					print_help();
				srnd_ioctl(SRNDDBPL_INIT, 0);
				srnd_ioctl(SRNDDBPL_MODE, 3);
				srnd_ioctl_perc(SRNDDBPL_VOL_LR, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_VOL_C, argv[optind++], 16);
				srnd_ioctl_perc(SRNDDBPL_DELAY, argv[optind], 15);
			} else {
				fprintf(stderr, "%s: illegal command arg -- %s\n", argv[0], argv[optind]);
				print_help();
			}
		} else
		if (strcmp(argv[optind],"3d-phonic") == 0) {
			optind++;
			if ((optind-argc) < 2)
				print_help();
			if (strcmp(argv[optind],"standard") == 0) {
				optind++;
				srnd_ioctl(SRND3DPHON_INIT, 0);
				srnd_ioctl(SRND3DPHON_MODE, 0);
				srnd_ioctl_perc(SRND3DPHON_LEVEL, argv[optind], 15);
			} else
			if (strcmp(argv[optind],"music") == 0) {
				optind++;
				srnd_ioctl(SRND3DPHON_INIT, 0);
				srnd_ioctl(SRND3DPHON_MODE, 1);
				srnd_ioctl_perc(SRND3DPHON_LEVEL, argv[optind], 15);
			} else
			if (strcmp(argv[optind],"theater") == 0) {
				optind++;
				srnd_ioctl(SRND3DPHON_INIT, 0);
				srnd_ioctl(SRND3DPHON_MODE, 2);
				srnd_ioctl_perc(SRND3DPHON_LEVEL, argv[optind], 15);
			} else
			if (strcmp(argv[optind],"strong") == 0) {
				optind++;
				srnd_ioctl(SRND3DPHON_INIT, 0);
				srnd_ioctl(SRND3DPHON_MODE, 3);
				srnd_ioctl_perc(SRND3DPHON_LEVEL, argv[optind], 15);
			} else
			if (strcmp(argv[optind],"theater1") == 0) {
				optind++;
				srnd_ioctl(SRND3DPHON_INIT, 0);
				srnd_ioctl(SRND3DPHON_MODE, 4);
				srnd_ioctl_perc(SRND3DPHON_LEVEL, argv[optind], 15);
			} else
			if (strcmp(argv[optind],"theater2") == 0) {
				optind++;
				srnd_ioctl(SRND3DPHON_INIT, 0);
				srnd_ioctl(SRND3DPHON_MODE, 5);
				srnd_ioctl_perc(SRND3DPHON_LEVEL, argv[optind], 15);
			}
			else {
				fprintf(stderr, "%s: illegal command arg -- %s\n", argv[0], argv[optind]);
				print_help();
			}
		} else
		if (strcmp(argv[optind],"through") == 0) {
			srnd_ioctl(SRNDTHROUGH_INIT, 0);
		} else
		if (strcmp(argv[optind],"noise") == 0) {
			optind++;
			if (optind == argc)
				print_help();
			if (strcmp(argv[optind],"off") == 0) {
				srnd_ioctl(SRNDNOISE_INIT, 0);
				srnd_ioctl(SRNDNOISE_MODE, 0);
			} else
			if (strcmp(argv[optind],"left") == 0) {
				optind++;
				if ((argc-optind) < 1)
					print_help();
				srnd_ioctl(SRNDNOISE_INIT, 0);
				srnd_ioctl(SRNDNOISE_MODE, 1);
				srnd_ioctl_perc(SRNDNOISE_LEVEL, argv[optind], 9);
			} else
			if (strcmp(argv[optind],"center") == 0) {
				optind++;
				if ((argc-optind) < 1)
					print_help();
				srnd_ioctl(SRNDNOISE_INIT, 0);
				srnd_ioctl(SRNDNOISE_MODE, 2);
				srnd_ioctl_perc(SRNDNOISE_LEVEL, argv[optind], 9);
			} else
			if (strcmp(argv[optind],"right") == 0) {
				optind++;
				if ((argc-optind) < 1)
					print_help();
				srnd_ioctl(SRNDNOISE_INIT, 0);
				srnd_ioctl(SRNDNOISE_MODE, 3);
				srnd_ioctl_perc(SRNDNOISE_LEVEL, argv[optind], 9);
			} else
			if (strcmp(argv[optind],"surround") == 0) {
				optind++;
				if ((argc-optind) < 1)
					print_help();
				srnd_ioctl(SRNDNOISE_INIT, 0);
				srnd_ioctl(SRNDNOISE_MODE, 4);
				srnd_ioctl_perc(SRNDNOISE_LEVEL, argv[optind], 9);
			} else {
				fprintf(stderr, "%s: illegal command arg -- %s\n", argv[0], argv[optind]);
				print_help();
			}
		}
		else {
			fprintf(stderr, "%s: illegal command -- %s\n", argv[0], argv[optind]);
			print_help();
		}
	}

	return 0;
}
