/*
    SRND
    
    A driver for the miroMEDIA Surround Dolby Pro Logic decoder.

    Copyright (C) 1998-2001 Oliver Gantz <Oliver.Gantz@ePost.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    ----
    3D-Phonic(R) is a registered trademark of VICTOR COMPANY OF JAPAN, LIMITED.
    DOLBY and PRO LOGIC are trademarks of Dolby Laboratories Licensing Corporation.
    miro(R) is a registered trademark of miro Computer Products AG.
*/

#ifdef MODULE
#include <linux/module.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#include <linux/version.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/fcntl.h>
#include <linux/delay.h>
#include <asm/segment.h>
#include <asm/io.h>
#include <asm/system.h>

#if LINUX_VERSION_CODE >= 0x020200
#include <asm/uaccess.h>
#endif

#ifdef CONFIG_ISAPNP
#include <linux/isapnp.h>
#endif

#include "srnd.h"


#if LINUX_VERSION_CODE < 0x020000
#error Kernel versions prior to 2.0.0 are not supported by SRND.
#endif

#if (LINUX_VERSION_CODE >= 0x020100) && (LINUX_VERSION_CODE < 0x020200)
#error Developer kernel versions 2.1.x are not supported by SRND.
#error Please update to a stable kernel version 2.2.x.
#endif

#if (LINUX_VERSION_CODE >= 0x020300) && (LINUX_VERSION_CODE < 0x020400)
#error Developer kernel versions 2.3.x are not supported by SRND.
#error Please update to a stable kernel version 2.4.x.
#endif

#if (LINUX_VERSION_CODE >= 0x020000) && (LINUX_VERSION_CODE < 0x020100)
#define SRND_LINUX 20
#endif

#if (LINUX_VERSION_CODE >= 0x020200) && (LINUX_VERSION_CODE < 0x020300)
#define SRND_LINUX 22
#endif

#if LINUX_VERSION_CODE >= 0x020300
#define SRND_LINUX 24
#endif

#if LINUX_VERSION_CODE >= 0x020500
#warning Kernel versions higher than 2.4.x are not yet tested with SRND.
#endif




/* Flags for driver status */
#define SRND_EXIST 0x0001
#define SRND_BUSY  0x0002


/* Macro for I/O port access */
#define SRND_OUT(x) outb_p((x),devices[minor].base)


/* Control data for program memory download */
static unsigned char control_data[12] = {
	0x84, 0x10, 0x41, 0x81, 0x23, 0x80, 0x30, 0xff,
	0x00, 0x00, 0x20, 0x00
};

/* DSP software to clear DSP memory */
static unsigned char clear_data[48] = {
	0x3a, 0x27, 0x27, 0x27, 0xba, 0x27, 0x27, 0xa7,
	0x27, 0xa7, 0x04, 0x27, 0x55, 0x27, 0x05, 0x27,
	0xb6, 0x07, 0x27, 0x27, 0xb6, 0x27, 0x27, 0x27,
	0x8d, 0x33, 0x01, 0x27, 0x16, 0xe3, 0x27, 0x27,
	0x27, 0x27, 0x05, 0x27, 0x03, 0x26, 0x05, 0x27,
	0x27, 0xa7, 0x04, 0x27, 0x14, 0x27, 0x27, 0x63
};

/* DSP software for Dolby Pro Logic mode */
static unsigned char dolbyprologic_data[116] = {
	0x14, 0x27, 0x27, 0x23, 0x14, 0x27, 0x27, 0x2f,
	0x76, 0x27, 0x23, 0x27, 0x56, 0x01, 0x27, 0xe7,
	0xa5, 0x27, 0x05, 0x37, 0x27, 0xa7, 0x84, 0x37,
	0x7a, 0x01, 0x58, 0xd8, 0x14, 0x27, 0x27, 0xe3,
	0x1c, 0x27, 0x23, 0x87, 0x76, 0x0f, 0x23, 0xdd,
	0x1c, 0x27, 0xa3, 0x4d, 0xa5, 0x27, 0x07, 0x2d,
	0x94, 0x26, 0x27, 0xe1, 0xb6, 0x0f, 0xa3, 0xaa,
	0xa5, 0x27, 0x07, 0x43, 0x94, 0x26, 0x27, 0xa5,
	0xb6, 0x0f, 0x23, 0x58, 0xb6, 0x07, 0x23, 0x40,
	0xa5, 0x27, 0x07, 0x01, 0x94, 0x26, 0x27, 0xe5,
	0xb6, 0x07, 0x23, 0x2a, 0x1c, 0x27, 0xa3, 0xd1,
	0x5a, 0x27, 0x05, 0x27, 0x76, 0x4b, 0x03, 0x3a,
	0xa5, 0x27, 0x07, 0x01, 0x94, 0x26, 0x27, 0xa9,
	0x76, 0x4b, 0x03, 0x68, 0x1c, 0x27, 0xa3, 0xbe,
	0x5a, 0x27, 0x05, 0x27
};

/* DSP software for noise generator */
static unsigned char noise_data[200] = {
	0x14, 0x27, 0x27, 0xcf, 0x76, 0xdd, 0x15, 0xe7,
	0x91, 0x9f, 0x04, 0x27, 0x33, 0x45, 0x21, 0x63,
	0x33, 0xcf, 0x29, 0xa7, 0x5f, 0x23, 0x29, 0x27,
	0x5f, 0x27, 0x29, 0x23, 0x5f, 0x27, 0x29, 0x67,
	0x5f, 0x27, 0x29, 0xe7, 0x27, 0x23, 0x35, 0x1a,
	0x27, 0x23, 0x25, 0xa3, 0x27, 0x23, 0x35, 0x9a,
	0x27, 0x23, 0x35, 0x5e, 0x27, 0x23, 0x35, 0xba,
	0xa5, 0x27, 0x05, 0x37, 0x94, 0x27, 0x27, 0xe9,
	0xa5, 0x27, 0x07, 0x27, 0x94, 0x27, 0x27, 0x65,
	0x27, 0x23, 0x25, 0x1a, 0x14, 0x27, 0x27, 0xe9,
	0xa5, 0x27, 0x07, 0xa7, 0x94, 0x27, 0x27, 0x2d,
	0x27, 0x23, 0x25, 0x5e, 0x14, 0x27, 0x27, 0xe9,
	0xa5, 0x27, 0x07, 0x23, 0x94, 0x27, 0x27, 0x6d,
	0x27, 0x23, 0x25, 0x9a, 0x14, 0x27, 0x27, 0xe9,
	0xa5, 0x27, 0x07, 0xa3, 0x94, 0x27, 0x27, 0xe9,
	0x27, 0x23, 0x25, 0xba, 0xb6, 0x07, 0x23, 0x91,
	0x27, 0x27, 0x05, 0x27, 0x33, 0x0b, 0x31, 0x7e,
	0x5f, 0x8f, 0x20, 0xfe, 0x24, 0x4f, 0x21, 0xfa,
	0x5f, 0x27, 0x21, 0x34, 0x5f, 0x27, 0x21, 0xba,
	0x1c, 0x27, 0xa3, 0x2b, 0x27, 0x6f, 0x35, 0x1a,
	0x27, 0x6f, 0x34, 0x9a, 0x27, 0xef, 0x35, 0x5e,
	0x27, 0xef, 0x34, 0xde, 0x27, 0xc3, 0x04, 0x27,
	0x5a, 0x27, 0x05, 0x27, 0x76, 0xa7, 0x20, 0xad,
	0x56, 0x2d, 0x27, 0x05, 0xa5, 0x27, 0x05, 0x37,
	0x27, 0xa7, 0x84, 0x37, 0x14, 0x27, 0x67, 0x6c
};

/* Additional data for noise generator */
static unsigned char coes_data[21] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0x00, 0x00, 0x00
};

/* DSP software for through mode */
static unsigned char through_data[64] = {
	0x14, 0x27, 0x27, 0x23, 0x14, 0x27, 0x27, 0xe3,
	0x76, 0x27, 0x23, 0xd4, 0x56, 0x2f, 0x27, 0xe7,
	0xa5, 0x27, 0x05, 0x37, 0x27, 0xa7, 0x84, 0x37,
	0x14, 0x27, 0x67, 0x6c, 0x27, 0x6f, 0x25, 0x14,
	0x27, 0x6f, 0x24, 0x1c, 0x76, 0x27, 0x27, 0x27,
	0x1c, 0x27, 0x67, 0x1b, 0x91, 0x27, 0x05, 0x27,
	0x27, 0xad, 0x05, 0x27, 0x27, 0xad, 0x04, 0x27,
	0x27, 0xc3, 0x04, 0x27, 0x5a, 0x27, 0x05, 0x27
};


/* Structure for driver status of each Surround board */
struct srnd_dev {
	unsigned int base;		/* I/O port	 		*/
	int flags;			/* Status flags			*/
	unsigned char bpflag;		/* Bypass flag (0x00 or 0x80)	*/
	unsigned int last_dac1_volume;	/* Last volume of DAC 1		*/
	unsigned int last_dac2_volume;	/* Last volume of DAC 2		*/
	int dac1_muted;			/* True, if DAC 1 muted		*/
	int dac2_muted;			/* True, if DAC 2 muted		*/
};


/* Minor device number of currently accessed Surround board */
static unsigned int minor;

/* Status table for all Surround boards */
static struct srnd_dev devices[SRND_MAXDEV];



/* -------------------------- I2C related stuff -------------------------- */

/* Start I2C transmission */
static void i2c_start (void)
{
	SRND_OUT(devices[minor].bpflag | 0x1f);	/* clk=1, data=1 */
	SRND_OUT(devices[minor].bpflag | 0x1d);	/* clk=1, data=0 */
	SRND_OUT(devices[minor].bpflag | 0x1c);	/* clk=0, data=0 */
}

/* Stop I2C transmission */
static void i2c_stop (void)
{
	SRND_OUT(devices[minor].bpflag | 0x1c);	/* clk=0, data=0 */
	SRND_OUT(devices[minor].bpflag | 0x1d);	/* clk=1, data=0 */
	SRND_OUT(devices[minor].bpflag | 0x1f);	/* clk=1, data=1 */
}

/* Put I2C bus into initial state */
static void i2c_init (void)
{
	int i;

	SRND_OUT(devices[minor].bpflag | 0x1e);	/* clk=0, data=1 */
	for (i=0; i < 4; i++)
		i2c_stop();
}

/* Write single byte to I2C bus */
static void i2c_putbyte (unsigned char val)
{
	int i;
	unsigned char bit;
	
	for (i=0; i < 8; i++) {
		bit = (unsigned char)(val & 0x80) >> 6;
		SRND_OUT(devices[minor].bpflag | 0x1c | bit);	/* clk=0, data=x */
		SRND_OUT(devices[minor].bpflag | 0x1d | bit);	/* clk=1, data=x */
		SRND_OUT(devices[minor].bpflag | 0x1c | bit);	/* clk=0, data=x */
		val <<= 1;
	}
	SRND_OUT(devices[minor].bpflag | 0x1e);		/* clk=0, data=1 */
}

/* Wait for acknowledge on I2C bus */
static int i2c_getack (void)
{
	int i=0, timeout=0;

	SRND_OUT(devices[minor].bpflag | 0x1f);	/* clk=1, data=1 */

	while (((inb_p(devices[minor].base) & 0x02) == 0x02) && !timeout)
		timeout = (++i == 100);

	SRND_OUT(devices[minor].bpflag | 0x1d);	/* clk=1, data=0 */
	SRND_OUT(devices[minor].bpflag | 0x1c);	/* clk=0, data=0 */

#ifdef SRND_DEBUG
	if (timeout)
		printk(KERN_DEBUG "srnd i2c_getack timeout\n");
#endif

	return timeout;

}

/* Write one data block to I2C bus */
static void i2c_write (unsigned char address, unsigned char command, unsigned char data[], int size)
{
	int i, rep=0, timeout;

	do {
		i2c_start();
		i2c_putbyte(address);
		if ((timeout = i2c_getack())) {
			i2c_stop();
			continue;
		}
		i2c_putbyte(command);
		if ((timeout = i2c_getack())) {
			i2c_stop();
			continue;
		}
		for (i=0; i < size; i++) {
			i2c_putbyte(data[i]);
			if ((timeout = i2c_getack()))
				break;
		}
		i2c_stop();
	} while (timeout && (++rep < 5));

#ifdef SRND_DEBUG
	if (rep==5)
		printk(KERN_DEBUG "srnd i2c_write timeout\n");
#endif
}

/* Write two successive data blocks to I2C bus */
static void i2c_write2 (unsigned char address, unsigned char command, unsigned char data1[], int size1, unsigned char data2[], int size2)
{
	int i, rep=0, timeout;

	do {
		i2c_start();
		i2c_putbyte(address);
		if ((timeout = i2c_getack())) {
			i2c_stop();
			continue;
		}
		i2c_putbyte(command);
		if ((timeout = i2c_getack())) {
			i2c_stop();
			continue;
		}
		for (i=0; i < size1; i++) {
			i2c_putbyte(data1[i]);
			if ((timeout = i2c_getack()))
				break;
		}
		if (timeout) {
			i2c_stop();
			continue;
		}
		for (i=0; i < size2; i++) {
			i2c_putbyte(data2[i]);
			if ((timeout = i2c_getack()))
				break;
		}
		i2c_stop();
	} while (timeout && (++rep < 5));

#ifdef SRND_DEBUG
	if (rep==5)
		printk(KERN_DEBUG "srnd i2c_write2 timeout\n");
#endif
}
	




/* -------------------------- DAC related stuff -------------------------- */

/* Write data to DAC 1 and/or DAC 2 */
static void dac_write (unsigned char which, unsigned short value)
{
	int i;
	unsigned char bit;

	for (i=0; i < 16; i++) {	/* clock bits 0 to 15 */
		bit = (unsigned char)((value & 0x01) << 2);
		SRND_OUT(devices[minor].bpflag | 0x1a | bit);
		udelay(100);
		SRND_OUT(devices[minor].bpflag | 0x1b | bit);
		udelay(100);
		value >>= 1;
	}

	bit = (unsigned char)(0x1a ^ ((which & 0x03) << 3));

	SRND_OUT(devices[minor].bpflag | bit);		/* drop the latch */
	udelay(100);
	SRND_OUT(devices[minor].bpflag | bit | 0x01);	/* clock it       */
	udelay(100);
	SRND_OUT(devices[minor].bpflag | 0x1f);	/* get back       */
	udelay(100);
}


/* Set volume of both DACs */
static int dac_volume (unsigned int level)
{
	if (level > 16)
		return -EINVAL;

	dac_write(0x03, (unsigned short)level << 6);

	devices[minor].last_dac1_volume = devices[minor].last_dac2_volume = level;
	devices[minor].dac1_muted = devices[minor].dac2_muted = 0;

	return 0;
}

/* Set volume of DAC 1 */
static int dac1_volume (unsigned int level)
{
	if (level > 16)
		return -EINVAL;

	dac_write(0x01, (unsigned short)level << 6);

	devices[minor].last_dac1_volume = level;
	devices[minor].dac1_muted = 0;

	return 0;
}

/* Set volume of DAC 2 */
static int dac2_volume (unsigned int level)
{
	if (level > 16)
		return -EINVAL;

	dac_write(0x02, (unsigned short)level << 6);

	devices[minor].last_dac2_volume = level;
	devices[minor].dac2_muted = 0;

	return 0;
}

/* Mute DAC 1 and/or DAC 2 internally without changing any flags */
static void dac_mute_temp (unsigned char which)
{
	dac_write(which, 0x1000);
}

/* Unmute DAC 1 and/or DAC 2 internally without changing any flags */
static void dac_unmute_temp (void)
{
	if (!devices[minor].dac1_muted)
		dac1_volume(devices[minor].last_dac1_volume);
	if (!devices[minor].dac2_muted)
		dac2_volume(devices[minor].last_dac2_volume);
}	

/* Mute or unmute DAC 1 and/or DAC 2 */
static void dac_mute (unsigned char which)
{
	dac_mute_temp(which);

	devices[minor].dac1_muted = (which & 0x01) != 0;
	devices[minor].dac2_muted = (which & 0x02) != 0;

	dac_unmute_temp();
}



/* Modify DSP data memory */
static void cmem_update (unsigned short addr, long data)
{
	unsigned char dat[5];

	dat[0] = (unsigned char)(addr >> 8);		/* address high */
	dat[1] = (unsigned char)(addr & 0xff);		/* address low  */
	dat[2] = (unsigned char)((data >> 16) & 0xff);	/* data high    */
	dat[3] = (unsigned char)((data >> 8) & 0xff);	/* data middle  */
	dat[4] = (unsigned char)(data & 0xff);		/* data low     */
	i2c_write(0x10, 0x03, dat, 5);
}


/* Download DSP software to program memory */
static void pmem_down (unsigned char data[], int size)
{
	i2c_write2(0x10, 0x8c, control_data, sizeof(control_data), data, size);
}
	

/* Reset Surround board */
static void reset (void)
{
	devices[minor].bpflag = 0x00;
	i2c_init();

	pmem_down(clear_data, sizeof(clear_data));
	udelay(1000);		/* required to clear the memory */

	dac_volume(16);
	dac_write(0x03, 0x7002);

	SRND_OUT(0x1e);
	SRND_OUT(0x1f);
}
	


/* Switch bypass function on or off */
static void bypass (int on)
{
	devices[minor].bpflag = on ? 0x00 : 0x80;
	SRND_OUT(devices[minor].bpflag | 0x1e);
	SRND_OUT(devices[minor].bpflag | 0x1f);
}


/* Preset Dolby Pro Logic mode (must be done for Dolby Pro Logic and 3D-Phonic) */
static void dolbyprologic_preset (void)
{
	static unsigned short addr[11] = {
		0x02f, 0x015, 0x016, 0x003,
		0x004, 0x005, 0x006, 0x007,
		0x019, 0x01a, 0x01b
	};
	static long data[11] = {
		0x23201d, 0x0039b9, 0x0031b9, 0x072148,
		0x000000, 0x000000, 0x072148, 0x072148,
		0x000000, 0x000000, 0x000000
	};

	int i;

	pmem_down(dolbyprologic_data, sizeof(dolbyprologic_data));

	for (i=0; i < 11; i++)
		cmem_update(addr[i], data[i]);
}


/* Initialize Dolby Pro Logic mode */
static void dolbyprologic_init (void)
{
	dac_mute_temp(0x03);
	dolbyprologic_preset();
	dac_unmute_temp();
}
	
/* Setup a specific Dolby Pro Logic mode */
static int dolbyprologic_mode (unsigned int mode)
{
	dac_mute_temp(0x03);
	switch (mode) {
		case 0:		/* Normal */
			cmem_update(0x019, 0x000000);
			cmem_update(0x01a, 0x000000);
			cmem_update(0x01b, 0x000000);
			break;
		case 1:		/* Wide */
			cmem_update(0x019, 0x000000);
			cmem_update(0x01a, 0x000001);
			cmem_update(0x01b, 0x000000);
			break;
		case 2:		/* Phantom */
			cmem_update(0x019, 0x000001);
			cmem_update(0x01a, 0x000000);
			cmem_update(0x01b, 0x000000);
			break;
		case 3:		/* Dolby 3 Stereo */
			cmem_update(0x019, 0x000000);
			cmem_update(0x01a, 0x000000);
			cmem_update(0x01b, 0x000001);
			break;
		default:
			return -EINVAL;
	}
	dac_unmute_temp();

	return 0;
}

/* Set volume of left and right speaker */
static int dolbyprologic_volume_lr (unsigned int level)
{
	if (level > 16)
		return -EINVAL;

	cmem_update(0x003, (long)level << 15);

	return 0;
}

/* Set volume of center speaker */
static int dolbyprologic_volume_c (unsigned int level)
{
	if (level > 16)
		return -EINVAL;

	cmem_update(0x006, (long)level << 15);

	return 0;
}

/* Set volume of surround speakers */
static int dolbyprologic_volume_s (unsigned int level)
{
	if (level > 16)
		return -EINVAL;

	cmem_update(0x007, (long)level << 15);

	return 0;
}

/* Set delay time between front and rear speakers */
static int dolbyprologic_delay (unsigned int level)
{
	static long data1[16] = {
		0x003900, 0x003960, 0x003970, 0x003980,
		0x003990, 0x0039a0, 0x0039b0, 0x0039c0,
		0x0039d0, 0x0039e0, 0x0039f0, 0x003a00,
		0x003a10, 0x003a20, 0x003a30, 0x003a40
	};
	static long data2[16] = {
		0x003100, 0x003160, 0x003170, 0x003180,
		0x003190, 0x0031a0, 0x0031b0, 0x0031c0,
		0x0031d0, 0x0031e0, 0x0031f0, 0x003200,
		0x003210, 0x003220, 0x003230, 0x003240
	};

	if (level > 15)
		return -EINVAL;

	cmem_update(0x015, data1[level]);
	cmem_update(0x016, data2[level]);

	return 0;
}

/* Preset 3D-Phonic mode */
static void dddphonic_preset (void)
{
	static unsigned short addr[7] = {
		0x004, 0x005, 0x006, 0x007,
		0x019, 0x01a, 0x01b
	};
	static long data[7] = {
		0x043f40, 0x0e39ea, 0x000000, 0x000000,
		0x000000, 0x000001, 0x000000
	};

	int i;

	dolbyprologic_preset();

	for (i=0; i < 7; i++)
		cmem_update(addr[i], data[i]);
}

/* Set 3D-Phonic standard mode */
static void dddphonic_standard (void)
{
	static unsigned short addr[6] = {
		0x026, 0x00a, 0x00b, 0x00c,
		0x00d, 0x00e
	};
	static long data[6] = {
		0x000000, 0x000000, 0x080000, 0x000000,
		0x030000, 0x060000
	};

	int i;

	for (i=0; i < 6; i++)
		cmem_update(addr[i], data[i]);
}

/* Set 3D-Phonic music/sports mode */
static void dddphonic_music (void)
{
	static unsigned short addr[7] = {
		0x026, 0x00a, 0x00b, 0x00c,
		0x00d, 0x00e, 0x032
	};
	static long data[7] = {
		0x000000, 0x028000, 0x050000, 0x028000,
		0x030000, 0x060000, 0x000000
	};

	int i;

	for (i=0; i < 7; i++)
		cmem_update(addr[i], data[i]);
}

/* Set 3D-Phonic theater mode */
static void dddphonic_theater (void)
{
	static unsigned short addr[7] = {
		0x026, 0x00a, 0x00b, 0x00c,
		0x00d, 0x00e, 0x032
	};
	static long data[7] = {
		0x000000, 0x028000, 0x050000, 0x028000,
		0x030000, 0x060000, 0x000001
	};

	int i;

	for (i=0; i < 7; i++)
		cmem_update(addr[i], data[i]);
}

/* Set 3D-Phonic strong mode */
static void dddphonic_strong (void)
{
	static unsigned short addr[6] = {
		0x026, 0x00a, 0x00b, 0x00c,
		0x00d, 0x00e
	};
	static long data[6] = {
		0x000001, 0x000000, 0x080000, 0x008000,
		0x000000, 0x080000
	};

	int i;

	for (i=0; i < 6; i++)
		cmem_update(addr[i], data[i]);
}

/* Set 3D-Phonic strong theater 1 mode */
static void dddphonic_strong_th1 (void)
{
	static unsigned short addr[7] = {
		0x026, 0x00a, 0x00b, 0x00c,
		0x00d, 0x00e, 0x032
	};
	static long data[7] = {
		0x000001, 0x028000, 0x050000, 0x028000,
		0x000000, 0x080000, 0x000000
	};

	int i;

	for (i=0; i < 7; i++)
		cmem_update(addr[i], data[i]);
}

/* Set 3D-Phonic strong theater 2 mode */
static void dddphonic_strong_th2 (void)
{
	static unsigned short addr[7] = {
		0x026, 0x00a, 0x00b, 0x00c,
		0x00d, 0x00e, 0x032
	};
	static long data[7] = {
		0x000001, 0x028000, 0x050000, 0x028000,
		0x000000, 0x080000, 0x000001
	};

	int i;

	for (i=0; i < 7; i++)
		cmem_update(addr[i], data[i]);
}


/* Setup a specific 3D-Phonic mode */
static int dddphonic_mode (unsigned int mode)
{
	dac_mute_temp(0x03);

	dddphonic_preset();
	switch (mode) {
		case 0:
			dddphonic_standard();
			break;
		case 1:
			dddphonic_music();
			break;
		case 2:
			dddphonic_theater();
			break;
		case 3:
			dddphonic_strong();
			break;
		case 4:
			dddphonic_strong_th1();
			break;
		case 5:
			dddphonic_strong_th2();
			break;
		default:
			dac_unmute_temp();
			return -EINVAL;
	}
	dac_unmute_temp();

	return 0;
}

/* Set 3D-Phonic effect level */
static int dddphonic_level (unsigned int level)
{
	static long data[16] = {
		0x02f2e7, 0x037bc3, 0x03e892, 0x0462a9,
		0x04eba5, 0x058558, 0x0631cc, 0x06f34c,
		0x07cc67, 0x08c0c0, 0x09d152, 0x0b03fe,
		0x0c5c16, 0x0dde29, 0x0f8f59, 0x117563
	};

	if (level > 15)
		return -EINVAL;

	dac_mute_temp(0x03);
	cmem_update(0x009, data[level]);
	dac_unmute_temp();

	return 0;
}


/* Initialize through mode */
static void through_init (void)
{
	dac_mute_temp(0x03);
	pmem_down(through_data, sizeof(through_data));
	cmem_update(0x000, 0x04026e);
	dac_unmute_temp();
}


/* Initialize noise generator */
static void noise_init (void)
{
	dac_mute_temp(0x03);
	pmem_down(noise_data, sizeof(noise_data));
	i2c_write(0x10, 0x0d, coes_data, sizeof(coes_data));
	i2c_write(0x10, 0x00, (unsigned char*)NULL, 0);
	dac_unmute_temp();
}
	
/* Setup a specific noise generator mode */
static int noise_mode (unsigned int mode)
{
	unsigned short addr;

	if (mode > 4)
		return -EINVAL;

	if (mode)
		cmem_update(0x006, 0x000001);	/* enable  */
	else
		cmem_update(0x006, 0x000000);	/* disable */

	for (addr=0x000; addr < 0x004; addr++)
		if ((addr+1) == mode)
			cmem_update(addr, 0x000001);
		else
			cmem_update(addr, 0x000000);

	return 0;
}

/* Set noise generator level */
static int noise_level (unsigned int level)
{
	static long data[10] = {
		0x000000, 0x00e38e, 0x01c71c, 0x02aaaa,
		0x038e38, 0x0471c6, 0x055554, 0x0638e2,
		0x071c70, 0x07fffe
	};

	if (level > 9)
		return -EINVAL;

	cmem_update(0x005, data[level]);

	return 0;
}





static int srnd_ioctl (struct inode * inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	int retval = 0;

	minor = MINOR(inode->i_rdev);

#ifdef SRND_DEBUG
	printk(KERN_DEBUG "srnd%u: ioctl(%u)\n", minor, cmd);
#endif

	if (minor >= SRND_MAXDEV)
		return -ENODEV;
	if (!(devices[minor].flags & SRND_EXIST))
		return -ENODEV;

	switch(cmd) {
		case SRNDRESET:
			reset();
			break;

		case SRNDVERSION:
			put_user(SRND_VERSION_CODE, (int *)arg);
			break;

		case SRNDDAC_VOL:
			retval = dac_volume((unsigned int)arg);
			break;
		case SRNDDAC1_VOL:
			retval = dac1_volume((unsigned int)arg);
			break;
		case SRNDDAC2_VOL:
			retval = dac2_volume((unsigned int)arg);
			break;
		case SRNDDAC_MUTE:
			dac_mute((unsigned char)(arg & 0xff));
			break;

		case SRNDBYPASS:
			bypass((int)arg);
			break;

		case SRNDDBPL_INIT:
			dolbyprologic_init();
			break;
		case SRNDDBPL_MODE:
			retval = dolbyprologic_mode((unsigned int)arg);
			break;
		case SRNDDBPL_VOL_LR:
			retval = dolbyprologic_volume_lr((unsigned int)arg);
			break;
		case SRNDDBPL_VOL_C:
			retval = dolbyprologic_volume_c((unsigned int)arg);
			break;
		case SRNDDBPL_VOL_S:
			retval = dolbyprologic_volume_s((unsigned int)arg);
			break;
		case SRNDDBPL_DELAY:
			retval = dolbyprologic_delay((unsigned int)arg);
			break;

		case SRND3DPHON_INIT:
			dddphonic_mode(0);
			break;
		case SRND3DPHON_MODE:
			retval = dddphonic_mode((unsigned int)arg);
			break;
		case SRND3DPHON_LEVEL:
			retval = dddphonic_level((unsigned int)arg);
			break;

		case SRNDTHROUGH_INIT:
			through_init();
			break;

		case SRNDNOISE_INIT:
			noise_init();
			break;
		case SRNDNOISE_MODE:
			retval = noise_mode((unsigned int)arg);
			break;
		case SRNDNOISE_LEVEL:
			retval = noise_level((unsigned int)arg);
			break;
		default:
			retval = -EINVAL;
	}

	return retval;
}


static int srnd_open (struct inode *inode, struct file *file)
{
	minor = MINOR(inode->i_rdev);

#ifdef SRND_DEBUG
	printk(KERN_DEBUG "srnd%u: open\n", minor);
#endif

	if (minor >= SRND_MAXDEV)
		return -ENXIO;
	if (!(devices[minor].flags & SRND_EXIST))
		return -ENXIO;
	if (devices[minor].flags & SRND_BUSY)
		return -EBUSY;

	MOD_INC_USE_COUNT;

	devices[minor].flags |= SRND_BUSY;

	return 0;
}


#if SRND_LINUX == 20
static void srnd_release (struct inode *inode, struct file *file)
#else
static int srnd_release (struct inode *inode, struct file *file)
#endif
{
	minor = MINOR(inode->i_rdev);

#ifdef SRND_DEBUG
	printk(KERN_DEBUG "srnd%u: release\n", minor);
#endif

	if (minor >= SRND_MAXDEV)
#if SRND_LINUX == 20
		return;
#else
		return -ENXIO;
#endif

	devices[minor].flags &= ~SRND_BUSY;

	MOD_DEC_USE_COUNT;

#if (SRND_LINUX == 22) || (SRND_LINUX == 24)
	return 0;
#endif
}

#if (SRND_LINUX == 24)
static struct file_operations srnd_fops = {
	owner:THIS_MODULE,
	ioctl:srnd_ioctl,
	open:srnd_open,
	release:srnd_release
};
#else
static struct file_operations srnd_fops = {
	NULL,		/* llseek		*/
	NULL,		/* read			*/
	NULL,		/* write		*/
	NULL,		/* readdir		*/
	NULL,		/* poll			*/
	srnd_ioctl,
	NULL,		/* mmap			*/
	srnd_open,
#if (SRND_LINUX == 22)
	NULL,		/* flush		*/
#endif
	srnd_release,
	NULL,		/* fsync		*/
	NULL		/* fasync		*/
#if (SRND_LINUX == 20) || (SRND_LINUX == 22)
	,
	NULL,		/* check_media_change	*/
	NULL		/* revalidate		*/
#endif
#if (SRND_LINUX == 22)
	,
	NULL		/* lock			*/
#endif
};
#endif

int srnd_dev_init (unsigned int minorno, unsigned int io)
{
	static char id[6] = "srnd0";

	id[4] = '0' | minorno;

#if (SRND_LINUX == 24)
	if (!request_region(io, 1, "srnd")) {
#ifdef SRND_DEBUG
		printk(KERN_DEBUG "%s: unable to get I/O port 0x%x\n", id, io);
#endif
		return -EBUSY;
	}
#else
	if (check_region(io, 1) < 0) {
#ifdef SRND_DEBUG
		printk(KERN_DEBUG "%s: unable to get I/O port 0x%x\n", id, io);
#endif
		return -EIO;
	}
	request_region(io, 1, id);
#endif

	outb_p(0xfd, io);
	if (inb_p(io) != 0xfd) {
#ifdef SRND_DEBUG
		printk(KERN_DEBUG "%s: no miroMEDIA Surround at 0x%x\n", id, io);
#endif
		release_region(io, 1);
		return -EIO;
	}

	printk(KERN_INFO "%s: miroMEDIA Surround at 0x%x\n", id, io);

	minor = minorno;

	devices[minor].base = io;
	devices[minor].flags = SRND_EXIST;

	reset();

	return 0;
}




#ifdef CONFIG_ISAPNP

static int srnd_pnp_probe (void)
{
	struct pci_dev *dev = NULL;
	unsigned int min = 0;
	int count = 0;

	while ((dev = isapnp_find_dev(NULL, ISAPNP_VENDOR('M','I','R'), ISAPNP_FUNCTION(0x0100), dev))) {
		if (min >= SRND_MAXDEV)
			break;

		devices[min].base = 0;
		devices[min].flags = 0;

		if (dev->prepare(dev) < 0) {				/* -EAGAIN */
#ifdef SRND_DEBUG
			printk(KERN_DEBUG "srnd%d: isapnp prepare failed\n", min);
#endif
			continue;
		}
		if (!(dev->resource[0].flags & IORESOURCE_IO)) {	/* -ENODEV */
#ifdef SRND_DEBUG
			printk(KERN_DEBUG "srnd%d: no isapnp io resource found\n", min);
#endif
			continue;
		}
		if (dev->activate(dev) < 0) {				/* -ENOMEM */
			printk(KERN_ERR "srnd%d: isapnp activate failed\n", min);
			continue;
		}
		
		if (!srnd_dev_init(min, dev->resource[0].start))
			count++;

		min++;
	}

	return count;
}

void srnd_pnp_deactivate (void)
{
	struct pci_dev *dev = NULL;
	unsigned int min = 0;

	while ((dev = isapnp_find_dev(NULL, ISAPNP_VENDOR('M','I','R'), ISAPNP_FUNCTION(0x0100), dev))) {
		if (min >= SRND_MAXDEV)
			break;

		if (dev->deactivate(dev) < 0) {
			printk(KERN_ERR "srnd%d: isapnp deactivate failed\n", min);
			continue;		/* -ENOMEM	*/
		}

		min++;
	}

}

#else

int srnd_dev_probe (unsigned int minorno, unsigned int io)
{
	unsigned int base, i;

	if (io)
		return srnd_dev_init(minorno, io);

	for (base=0x300; base <= 0x380; base += 0x10) {
		for (i=0; i < minorno; i++)
			if (devices[i].base == base)
				break;
		if (i == minorno)
			if (!srnd_dev_init(minorno, base))
				return 0;
	}

	return -EIO;
}

static int srnd_nonpnp_probe (int io[])
{
	unsigned int min = 0;
	int count = 0;

	for (min=0; min < SRND_MAXDEV; min++) {
		if (io[min] == -1)
			break;

		devices[min].base = 0;
		devices[min].flags = 0;

		if (!srnd_dev_probe(min, (unsigned int)(io[min])))
			count++;
	}

	return count;
}

#endif


int srnd_init (int io[])
{
	int count;

	if (register_chrdev(SRND_MAJOR, "srnd", &srnd_fops)) {
		printk(KERN_ERR "srnd: unable to get major %u\n", SRND_MAJOR);
		return 0;
	}

#ifdef CONFIG_ISAPNP
	count = srnd_pnp_probe();
#else
	count = srnd_nonpnp_probe(io);
#endif

	if (!count) {
		unregister_chrdev(SRND_MAJOR, "srnd");
#ifdef CONFIG_ISAPNP
		srnd_pnp_deactivate();
#endif
	}

	return count;
}


#ifdef MODULE

static int io[SRND_MAXDEV] = { SRND_DEFIO, [1 ... SRND_MAXDEV-1] = -1 };

#if (SRND_LINUX == 22) || (SRND_LINUX == 24)
MODULE_PARM(io, "1-" __MODULE_STRING(SRND_MAXDEV) "i");
#if (SRND_LINUX == 24)
MODULE_PARM_DESC(io, "miroMEDIA Surround I/O port");
#endif
#endif


/*
static int __init init_module (void)
*/
int init_module (void)
{
	return srnd_init(io) ? 0 : -ENODEV;
}

/*
static void __exit finit (void)
*/
void cleanup_module (void)
{
	if (MOD_IN_USE) {
#ifdef SRND_DEBUG
	    printk(KERN_DEBUG "srnd: busy - cleanup delayed\n");
#endif
	    return;
	}

	unregister_chrdev(SRND_MAJOR, "srnd");

	for (minor=0; minor < SRND_MAXDEV; minor++)
		if (devices[minor].flags)
			release_region(devices[minor].base, 1);

#ifdef CONFIG_ISAPNP
	srnd_pnp_deactivate();
#endif
}
#endif /* of #ifdef MODULE */
