#include "eeprog.h"

int debug_flag;
int parport_address;

char *strsave(char *s) /* save char array s somewhere */
{
char *p;

p = malloc(strlen(s) +  1);
if(p) strcpy(p, s);
return(p);
}/* end function strsave */


int main(int argc, char **argv)
{
int a, b, c, i, j;
unsigned char input_buffer[4096];
unsigned char output_buffer[4096];
FILE *ifptr;
FILE *ofptr;
char *input_filename;
char *output_filename;
int input_bin_file_flag;
int input_hex_file_flag;
int output_file_flag;
int eeprom_length;
int hex_out_flag;
int hex_dump_flag;

/* defaults */
input_bin_file_flag = 0;
input_hex_file_flag = 0;
output_file_flag = 0;
parport_address = 0x278;
eeprom_length = 2048;
hex_out_flag = 0;
hex_dump_flag = 0;
/* end defaults */

/*select con io 1 byte at the time*/
setbuf(stdout,NULL);
setbuf(stdin,NULL);

/* identify */
printf("\nPanteltje (c) eeprog-%s EEPROM programmer for 24c16\n\n",\
VERSION);
while(1)
	{
	a = getopt(argc, argv, "di:j:hl:o:p:xy");
	if(a == -1) break;
	switch(a)
		{
        case 'd': /* debug */
			debug_flag = 1;
			break;
		case 'h':	/* help */
			fprintf(stdout,\
"Usage:\n\
eeprog [-d] [-h] [-i input filename] [-l eeprom length] [-p parport address]
       [-o output filename] [-x] [-y]\n\
\t-d  print debug messages (o.a. function calls and arguments).\n\
\t-h  help (this help).\n\
\t-i  input filemame binary format.\n\
\t-j  input filename hex format\n\
\t-l  eeprom length in decimal (default 2048).\n\
\t-p  par port address in hex (default 0x378).\n\
\t-o  output filename.\n\
\t-x  output in Intel hex8 format.\n\
\t-y  output as hexdump.
");
			exit(0);	
			break;
		case 'i':	/* input filename */
			input_filename = strsave(optarg);
			if(! input_filename)
				{
				printf("Could not allocate space for input_filename\n");

				exit(1);
				}
			input_bin_file_flag = 1;
			break;
		case 'j':	/* input filename */
			input_filename = strsave(optarg);
			if(! input_filename)
				{
				printf("Could not allocate space for input_filename\n");

				exit(1);
				}
			input_hex_file_flag = 1;
			break;
		case 'l':	/* eeprom length */
			eeprom_length = atoi(optarg);
			if(eeprom_length <= 0)
				{
				printf("invalid eeprom length %d\n", eeprom_length);

				exit(1);
				}
			break;
		case 'o':	/* output filename */
			output_filename = strsave(optarg);
			if(! output_filename)
				{
				printf("Could not allocate space for output_filename\n");

				exit(1);
				}
			output_file_flag = 1;
			break;
		case 'p':	/* par port address */
			sscanf(optarg, "%x", &parport_address);
			break;
		case 'x':	/* output in Intel hex 8 format */
			hex_out_flag = 1;
			break;
		case 'y':	/* oytput as hex dump */
			hex_dump_flag = 1;
			break;
		case '?':
			if (isprint(optopt) )
 				{
 				fprintf(stdout, "Unknown option `-%c'.\n", optopt);
 				}
			else
				{
				fprintf(stdout, "Unknown option character `\\x%x'.\n",\
				optopt);
				}
			exit(1);
			break;			
		default:
			exit(1);
			break;
		}/* end switch a */
	}/* end while getopt() */

if( (! input_bin_file_flag) &&\
(! input_hex_file_flag) &&\
(! output_file_flag) )
	{
	printf(\
	"No input or output file specified, aborting, try eeprog -h for help.\n");

	exit(1);
	}

/* get io permissions */
a = iopl(3);
if(a != 0)
	{
	perror("iopl():");
	printf("Could not get access to io ports\n");

	exit(1);
	}

/* terminate any iic command in progress */
iicstop();

if(input_bin_file_flag)
	{
	/* open input file */
	ifptr = fopen(input_filename, "r");
	if(! ifptr)
		{
		fprintf(stdout, "Could not open file %s for read\n",\
		input_filename);

		exit(1);
		}
	printf("Loading input file %s\n", input_filename);
	/* load input file in buffer */
	for(i = 0; i < eeprom_length; i++)
		{
		while(1)/* error re read */
			{
			c = getc(ifptr);
			if(! ferror(ifptr) ) break;
			perror("getc():");
			}/* end while error re read */	

		if(feof(ifptr) ) break;

		input_buffer[i] = c;
		} /* end for all bytes to buffer */

	fclose(ifptr);

	printf("Programming %d bytes\n", eeprom_length);
	if(! write_eeprom(eeprom_length, input_buffer) ) exit(1);
	} /* end if input_bin_file_flag */
else if(input_hex_file_flag)
	{
	/* open input file */
	ifptr = fopen(input_filename, "r");
	if(! ifptr)
		{
		fprintf(stdout, "Could not open file %s for read\n",\
		input_filename);

		exit(1);
		}
	printf("Loading input file %s\n", input_filename);


	/* load input file in buffer */

	load_hex_file(ifptr, input_buffer, eeprom_length);

	fclose(ifptr);

for(i = 0; i < 40; i++)
	{
	printf("WAS i=%04x d=%02x\n", i, input_buffer[i]);
	}

	printf("Programming %d bytes\n", eeprom_length);
	if(! write_eeprom(eeprom_length, input_buffer) ) exit(1);
	} /* end if input_hex_file_flag */

/* read eeprom */
printf("Reading %d bytes from eeprom\n", eeprom_length);
if(! read_eeprom(eeprom_length, output_buffer) ) exit(1);

/* if did write, verify eeprom against data in buffer */
if(input_bin_file_flag || input_hex_file_flag)
	{
	printf("Verifying %d bytes\n", eeprom_length);
	for(i = 0; i < eeprom_length; i++)
		{
		if(output_buffer[i] != input_buffer[i])
			{
			printf("Verify failed address=%04x org=%02x eeprom=%02x\n",\
			i, input_buffer[i], output_buffer[i]);
						
			exit(1);
			}
		} /* end for i */

	printf("Verify OK\n");
	} /* end if input_file_flag */

if(output_file_flag)
	{
	/* open output file */
	ofptr = fopen(output_filename, "w");
	if(! ofptr)
		{
		fprintf(stdout, "Could not open file %s for write\n",\
		output_filename);

		exit(1);
		}

	if(hex_dump_flag)
		{
		printf("Writing hex dump output file %s\n", output_filename);
		i = 0;
		while(1)
			{
			a = eeprom_length - i;
			if(a <= 0) break;

			if(a > 8) a = 8;					

			/* print address */
			fprintf(ofptr, "%04x ", i);

			/* word in hex */
			for(j = 0; j < a; j++)
				{
				fprintf(ofptr, "%02x ",output_buffer[i + j]);
				} /* end for j */
			fprintf(ofptr, " ");

			for(j = 0; j < a; j++)
				{
				/* byte as char */
				b = output_buffer[i + j] & 0xff;
				if( (b > 31) && (b < 128) ) fprintf(ofptr, "%c", b);
				else fprintf(ofptr, " ");
				} /* end for j */

			fprintf(ofptr, "\n");

			/* next position */
			i += a; 

			} /* end for i */
		} /* end if hex_dump_flag */

	else if(hex_out_flag)
		{
		printf("Writing Intel hex 8 output file %s\n", output_filename);
		i = 0;
		while(1)
			{
			/* write intel hex record */
			j = eeprom_length - i;
			if(j > 8) j = 8;					
			
			/* j will be 0 to 7 */
			write_hex_record(\
				ofptr,\
				j,					/* record length */
				i,					/* dest address */
				&output_buffer[i]	/* source data */
				);
			
			/* allow for zero record as EOF */
			if(j <= 0) break;

			/* next position */
			i += j; 
			} /* end for i */

		} /* end if Intel hex 8 file output */
	else
		{
		printf("Writing binary output file %s\n", output_filename);
		for(i = 0; i < eeprom_length; i++)
			{
			fprintf(ofptr, "%c", output_buffer[i]);
			}
		} /* end if binary file output */

	/* close file */
	fclose(ofptr);
	} /* end if output_file_flag */

exit(0);
} /* end function main */


int write_eeprom(int length, unsigned char *buffer)
{
int i;
int ctrl_byte;
int word_address;
int word_address_low;
int word_address_high;
int read_notwrite;
unsigned char obuffer[2048];
unsigned char data;
int er_flag;

if(debug_flag)
	{
	fprintf(stdout, "write_eeprom(): arg length=%d buffer=%lu\n",\
	length, buffer);
	}

/* paramater check */
if(length <= 0)  return 0;
if(! buffer) return 0;

/* enable error reporting in iicsnd() */
er_flag = 1;
read_notwrite = 0;
for(i = 0; i < length; i++)
	{
	if( (i % 10) == 0) show_progress();
	word_address = i;
	word_address_high = (word_address & 0xff00) >> 8;
	word_address_low = word_address & 0xff;

	if(debug_flag)
		{
		printf("write_eeprom(): word_address_high=%d\n",\
		word_address_high);
		}

	/* 1 0 1 0 b2 b1 b0 r/!w */
	ctrl_byte = 0; 
	ctrl_byte = (10 << 4) | (word_address_high << 1) | read_notwrite;

	data = buffer[i];

	iicstart();

	/* send */
	obuffer[0] = word_address_low;
	obuffer[1] = data;

	/* write current address */
	if(! iicsnd(ctrl_byte, 2, obuffer, er_flag) )
		{
		printf("write_eeprom(): failed\n");

		return 0;
		}

	/* start program cycle */
	iicstop();

	/* poll if programing complete for this location (if ack happens) */
	while(1)
		{
		/* send start condition */
		iicstart();

		/* disable error reporting in iicsnd(), (would list all not ack) */
		er_flag = 0;

		/* send control byte with r/!w = zero */
		if( iicsnd(ctrl_byte, 0, obuffer, er_flag) ) break; /* ack OK */
		}

	} /* end for all bytes in length */

return 1;
} /* end function write_eeprom */


int read_eeprom(int length, unsigned char *buffer)
{
int i;
int ctrl_byte;
int word_address;
int word_address_low;
int word_address_high;
int read_notwrite;
unsigned char obuffer[10];
unsigned char data;
int er_flag;

if(debug_flag)
	{
	fprintf(stdout,\
	"read_eeprom(): arg length=%d buffer=%lu\n",\
	length, buffer);
	}

/* parameter check */
if(length <= 0)  return 0;
if(! buffer) return 0;

/* enable error report in iicsnd() */
er_flag = 1;

for(i = 0; i < length; i++)
	{
	if( (i % 10) == 0) show_progress();
	word_address = i;
	word_address_low = word_address & 0xff;
	word_address_high = (word_address & 0xff00) >> 8;

	iicstart();

	/* select write mode */
	read_notwrite = 0;
	ctrl_byte = 0;
	ctrl_byte = (10 << 4) | (word_address_high << 1) | read_notwrite;
	obuffer[0] = word_address_low;

	/* write address */
	if(! iicsnd(ctrl_byte, 1, obuffer, er_flag) )
		{
		printf("read_eeprom(): write start address failed\n");

		return 0;
		}

	/* switch to read mode */
	iicstart();

	read_notwrite = 1;
	ctrl_byte = 0;
	ctrl_byte = (10 << 4) | (word_address_high << 1) | read_notwrite;
	/* read current address */
	if(! iicrcv(ctrl_byte, 1, obuffer) )
		{
		printf("read_eeprom(): failed\n");

		return 0;
		}

	if(debug_flag)
		{
		printf("read_eeprom(): address=%04x data=%02x\n",\
		i, obuffer[0]);
		}

	/* store in buffer */
	buffer[i] = obuffer[0];

	iicstop();
	} /* end for all bytes in length */

return 1;
} /* end function read_eeprom */


int show_progress()
{
static int ap;
int c;

if(ap == 0) c = '|';
if(ap == 1) c = '/';
if(ap == 2) c = '-';
if(ap == 3)
	{
	c = '\\';
	ap = 0;
	}

printf("%c\r", c);

ap++;

return 1;
} /* end function show_progress */

