/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* sound.c  --  Functions to handle sound output. */


#include "common.h"
#include "misc.h"
#include "log.h"
#include "conf.h"


#if HAVE_LIBMIKMOD


#include "sound.h"


#define MAX_LEN_CHILDCOMMAND 256

MODULE *module = NULL;
gint filedes[2];
gchar *modname;

gboolean loop_music_forever = YES;
gboolean sound_music_stopped = NO;


/******************************************************************************
 Initialize sound.
******************************************************************************/
gint sound_init(void)
{
 MikMod_RegisterAllDrivers();
 MikMod_RegisterAllLoaders();

 md_mode |= DMODE_SOFT_MUSIC;

 /* FIXME This doesn't work with libmikmod 3.1.6 and older. */
 if (MikMod_Init(NULL) != 0)
 {
  g_warning("Could not initialize sound: %s", MikMod_strerror(MikMod_errno));
  return -1;
 }
 return 0;
}

/******************************************************************************
 Shutdown sound.
******************************************************************************/
void sound_shutdown(void)
{
 MikMod_Exit();
}

/******************************************************************************
 Load a MOD.
******************************************************************************/
gint sound_load_module(const gchar *filename)
{
 if ((module = Player_Load(filename, 128, 0)) == NULL)
 {
  g_warning("Could not load module %s: %s", filename,
            MikMod_strerror(MikMod_errno));
  sound_shutdown();
  return -1;
 }
 return 0;
}

/******************************************************************************
 Free a MOD.
******************************************************************************/
void sound_free_module(void)
{
 Player_Free(module);
 module = NULL;
}

/******************************************************************************
 Play a MOD.
******************************************************************************/
gint sound_play_module(const gchar *filename)
{
 gint n;
 pid_t pid;
 gchar buf[MAX_LEN_CHILDCOMMAND+1];
 gboolean shutdown_child = NO;

 
 if (pipe(filedes) < 0)
 {
  log_perror("sound_play_module:pipe");
  return -1;
 }

 if (fcntl(filedes[0], F_SETFL, O_NONBLOCK) < 0)  /* Set pipe nonblocking. */
 {
  log_perror("sound_play_module:fnctl");
  return -1;
 }

 if (fcntl(filedes[1], F_SETFL, O_NONBLOCK) < 0)  /* Set pipe nonblocking. */
 {
  log_perror("sound_play_module:fnctl");
  return -1;
 }

 switch (pid = fork())
 {
  /***************************************************************************/
  case 0:  /* Child. */
  {
   parent = NO;

   /* Close the write-end of the pipe. */
   if (close(filedes[1]) < 0)
   {
    log_perror("sound_play_module:close");
    return -1;
   }

   if (sound_init() < 0)
    exit(EXIT_FAILURE);

   modname = g_strdup(filename);

   if (sound_load_module((CHAR *)modname) < 0)
    exit(EXIT_FAILURE);

   Player_Start(module);

   while (!shutdown_child)
   {

    if (Player_Active() != 0)
    {
     usleep(10000);
     MikMod_Update();
    }
    else  /* Play the same MOD again. */
    {
     if (loop_music_forever && !sound_music_stopped)
     {
      Player_Stop();

      if (module != NULL)
       sound_free_module();

      if (sound_load_module((CHAR *)modname) < 0)
       exit(EXIT_FAILURE);

      Player_Start(module);
     }
    }

    if ((n = read(filedes[0], buf, MAX_LEN_CHILDCOMMAND)) < 0)
    {     
     if (errno != EAGAIN)
     {    
      log_perror("sound_play_module:read");
      return -1;
     }
    }
    else
    {
     /************************************************************************/
     if (g_strcasecmp(buf, "pause") == 0)
      if (!Player_Paused())
       Player_TogglePause();

     /************************************************************************/
     if (g_strcasecmp(buf, "continue") == 0)
      if (Player_Paused())
       Player_TogglePause();

     /************************************************************************/
     if (g_strcasecmp(buf, "togglepause") == 0)
      Player_TogglePause();

     /************************************************************************/
     if (g_strcasecmp(buf, "play") == 0)
     {
      Player_Stop();

      if (module != NULL)
       sound_free_module();

      /* strncpy(modname, (gchar *)&buf[5], MAX_LEN_MODNAME); */
      modname = g_strdup((gchar *)&buf[5]);

      if (sound_load_module((CHAR *)modname) < 0)
       exit(EXIT_FAILURE);

      Player_Start(module);
      sound_music_stopped = NO; 
     }

     /************************************************************************/
     if (g_strcasecmp(buf, "stop") == 0)
     {
      Player_Stop();
      sound_free_module();
      sound_music_stopped = YES; 
     }

     /************************************************************************/
     if (g_strcasecmp(buf, "quit") == 0)
      shutdown_child = YES;

    }
   }

   Player_Stop();

   if (module != NULL)
    sound_free_module();

   sound_shutdown();

   exit(EXIT_SUCCESS);
  }
  break;

  /***************************************************************************/
  case -1:      /* error */
  {
   log_perror("sound_play_module:fork");
   return -1;
  }
  break;

  /***************************************************************************/
  default: /* parent */
   child_pid = pid;

   /* close the read-end of the pipe */
   if (close(filedes[0]) < 0)
   {
    log_perror("sound_play_module:close");
    return -1;
   }

  break;
 }

 return 0;
}

/******************************************************************************
 ...
******************************************************************************/
gint sound_send_command_to_child(const gchar *message)
{
 if (write(filedes[1], message, MAX_LEN_CHILDCOMMAND) < 0)
 {
  log_perror("sound_send_command_to_child:write");
  return -1;
 }
 return 0;
}

#endif /* HAVE_LIBMIKMOD */

