/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* servercmds.c  --  Functions to handle server commands. */


#include "common.h"
#include "misc.h"
#include "log.h"
#include "goods.h"
#include "nc.h"
#include "net.h"
#include "players.h"
#include "transporters.h"
#include "towns.h"
#include "input.h"

#include "servercmds.h"


/******************************************************************************
 Check if 'message' contains a server-command. If it does, handle the command.
******************************************************************************/
gint check_for_servercmds(const gchar *message)
{
 gchar *command;
 GPtrArray *ptr_array;

 if ((ptr_array = tokenize(message)) == NULL)
 {
  g_warning("check_for_servercmds: tokenize() failed.");
  return -1;
 }

 if ((command = (gchar *)g_ptr_array_index(ptr_array, 0)) == NULL)
 {
  (void) g_ptr_array_free(ptr_array, TRUE);
  g_warning("check_for_servercmds: Command is NULL.");
  return -1;
 }

 (void) g_ptr_array_remove_index(ptr_array, 0); /* Remove the command. */


 if (g_strcasecmp(command, "chat") == 0)
  servercmd_chat(ptr_array);

 if (g_strcasecmp(command, "good_amount_update") == 0)
  servercmd_good_amount_update(ptr_array);

 if (g_strcasecmp(command, "good_price_update") == 0)
  servercmd_good_price_update(ptr_array);

 if (g_strcasecmp(command, "player_new") == 0)
  servercmd_player_new(ptr_array);

 if (g_strcasecmp(command, "player_remove") == 0)
  servercmd_player_remove(ptr_array);

 if (g_strcasecmp(command, "player_logged_in") == 0)
  servercmd_player_logged_in(ptr_array);

 if (g_strcasecmp(command, "player_logged_out") == 0)
  servercmd_player_logged_out(ptr_array);

 if (g_strcasecmp(command, "transporter_arrived") == 0)
  servercmd_transporter_arrived(ptr_array);

 if (g_strcasecmp(command, "transporter_left") == 0)
  servercmd_transporter_left(ptr_array);


 if (g_strcasecmp(command, "shutdown") == 0)
 {
  /* FIXME nhtsclient shouldn't shut down, but only disconnect and cleanup. */
  g_warning("Server is shutting down. Quitting too.");
  g_main_quit(mainloop);
  g_main_destroy(mainloop);
 }


 (void) g_ptr_array_free(ptr_array, TRUE);


 if (in_trade_screen)
  trade_screen_changed = YES;

 return 0;
}

/******************************************************************************
 ...
******************************************************************************/
gint servercmd_init(GPtrArray *args)
{
 guint counter = 1;
 guint i, j, id;
 guint number_of_towns, number_of_players, number_of_transporters;
 guint number_of_goods_in_transporter, number_of_goods_in_town;
 guint number_of_towns_where_player_has_transporters;

 town_t *town;
 player_t *player;
 gchar *name;


 my_player->id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);


 /*****************************************************************************
  Goods.
 *****************************************************************************/
 number_of_goods = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 for (i=0; i<number_of_goods; i++)
 {
  id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

  g_snprintf((gchar *)&goodnames[i][0], MAX_LEN_GOODNAME, "%s",
             (const gchar *)g_ptr_array_index(args, counter++));
 }

 /*****************************************************************************
  Towns.
 *****************************************************************************/
 number_of_towns = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 for (i=0; i<number_of_towns; i++)
 {
  good_t *good;

  id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);
  name = g_strdup(g_ptr_array_index(args, counter++));

  town = town_new(id, name, 0, 0, NULL);
  towns = g_list_append(towns, town);

  /* Create all goods in the town. */
  /* FIXME Should not rely on the IDs to be 0..number_of_goods-1. */
  for (j=0; j<number_of_goods; j++)
  {
   good = good_new(j, 0, 0);
   town->goods = g_list_append(town->goods, good);
  }

  g_free(name);
 }

 /*****************************************************************************
  Players.
 *****************************************************************************/
 number_of_players = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 for (i=0; i<number_of_players; i++)
 {
  gboolean logged_in;

  id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);
  name = g_strdup(g_ptr_array_index(args, counter++));
  logged_in = (gboolean)strtol(g_ptr_array_index(args, counter++), NULL, 0);

  if (player_find_by_id(id) != NULL) /* Is this is our player? */
  {
   my_player->logged_in = logged_in;
  }
  else
  {
   player = player_new(id, name, "", logged_in, 0, 0, NULL, NULL);
   player_add(player);
  }

  g_free(name);
 }

 /*****************************************************************************
  Player age and money.
 *****************************************************************************/
 my_player->age = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 my_player->money = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 /*****************************************************************************
  Transporters.
 *****************************************************************************/
 number_of_transporters = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 for (i=0; i<number_of_transporters; i++)
 {
  glong money;
  glong capacity;
  guint location;
  transporter_t *transporter;
  GList *goods = NULL;

  id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);
  name = g_strdup(g_ptr_array_index(args, counter++));
  money = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);
  capacity = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);
  location = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

  /****************************************************************************
   Goods in transporter.
  ****************************************************************************/
  number_of_goods_in_transporter = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

  for (j=0; j<number_of_goods_in_transporter; j++)
  {
   guint good_id;
   glong amount;
   good_t *good;

   good_id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);
   amount = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);
 
   good = good_new(good_id, amount, 0);
   goods = g_list_append(goods, good);
  }

  transporter = transporter_new(id, name, money, capacity, location, goods);
  player_add_transporter(my_player, transporter);

  g_free(name);
 }

 /*****************************************************************************
  Towns where the player has stationed one or more transporters.
 *****************************************************************************/
 number_of_towns_where_player_has_transporters = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 for (i=0; i<number_of_towns_where_player_has_transporters; i++)
 {
  id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

  town = town_find_by_id(id);

  town->money = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);
  town->inhabitants = (gulong)strtol(g_ptr_array_index(args, counter++), NULL, 0);

  /****************************************************************************
   Goods in town.
  ****************************************************************************/
  number_of_goods_in_town = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

  for (j=0; j<number_of_goods_in_town; j++)
  {
   guint good_id;
   good_t *good;

   good_id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

   good = good_find_by_id(town->goods, good_id);

   good->amount = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);;
   good->price = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);
  }

 }

 display_main_screen();

 return 0;
}

/******************************************************************************
 ...
******************************************************************************/
gint servercmd_chat(GPtrArray *args)
{
 const gchar *sender, *message;
 guint sender_id;
 player_t *player;
 
 sender = g_ptr_array_index(args, 0);
 message = g_ptr_array_index(args, 1);

 sender_id = (guint)strtol(sender, NULL, 0);

 if ((player = player_find_by_id(sender_id)) == NULL)
 {
  /* Sender does not exist. */
  return -1;
 }

 /* print_chat_message("%s: %s", player->name, message); */
 print_chat_message(message);

 return 0;
}

/******************************************************************************
 Update the amount of the given good in the given town.
******************************************************************************/
gint servercmd_good_amount_update(GPtrArray *args)
{
 guint town_id, good_id;
 town_t *town;
 good_t *good;

 /* TODO Add checks. */

 town_id = (guint)strtol(g_ptr_array_index(args, 0), NULL, 0);
 good_id = (guint)strtol(g_ptr_array_index(args, 1), NULL, 0);

 town = town_find_by_id(town_id);
 good = good_find_by_id(town->goods, good_id);

 good->amount = (glong)strtol(g_ptr_array_index(args, 2), NULL, 0);

 return 0;
}

/******************************************************************************
 Update the price of the given good in the given town.
******************************************************************************/
gint servercmd_good_price_update(GPtrArray *args)
{
 guint town_id, good_id;
 town_t *town;
 good_t *good;

 /* TODO Add checks. */

 town_id = (guint)strtol(g_ptr_array_index(args, 0), NULL, 0);
 good_id = (guint)strtol(g_ptr_array_index(args, 1), NULL, 0);

 town = town_find_by_id(town_id);
 good = good_find_by_id(town->goods, good_id);

 good->price = (glong)strtol(g_ptr_array_index(args, 2), NULL, 0);

 return 0;
}

/******************************************************************************
 The player with the given ID has just been created.
******************************************************************************/
gint servercmd_player_new(GPtrArray *args)
{
 guint player_id;
 gchar *playername;
 player_t *player;

 player_id = (guint)strtol(g_ptr_array_index(args, 0), NULL, 0);
 playername = g_ptr_array_index(args, 1);

 player = player_new(player_id, playername, "", FALSE, 0, 0, NULL, NULL);

 player_add(player);

 return 0;
}

/******************************************************************************
 The player with the given ID has just been removed.
******************************************************************************/
gint servercmd_player_remove(GPtrArray *args)
{
 guint player_id;
 player_t *player;

 player_id = (guint)strtol(g_ptr_array_index(args, 0), NULL, 0);

 player = player_find_by_id(player_id);

 player_free(player);
 player_remove(player);

 /* TODO Logout this player? */

 /* TODO More stuff needs to be done (?) */

 return 0;
}

/******************************************************************************
 The given player logged in.
******************************************************************************/
gint servercmd_player_logged_in(GPtrArray *args)
{
 guint player_id;
 player_t *player;

 player_id = (guint)strtol(g_ptr_array_index(args, 0), NULL, 0);

 player = player_find_by_id(player_id);

 player->logged_in = YES;

 return 0;
}

/******************************************************************************
 The given player logged out.
******************************************************************************/
gint servercmd_player_logged_out(GPtrArray *args)
{
 guint player_id;
 player_t *player;

 player_id = (guint)strtol(g_ptr_array_index(args, 0), NULL, 0);

 player = player_find_by_id(player_id);

 player->logged_in = NO;

 return 0;
}

/******************************************************************************
 A transporter has arrived in a town where you have a transporter of your own.
******************************************************************************/
gint servercmd_transporter_arrived(GPtrArray *args)
{
 guint transporter_id, player_id, town_id;
 transporter_t *transporter;
 player_t *player;
 town_t *town;

 transporter_id = (guint)strtol(g_ptr_array_index(args, 0), NULL, 0);
 player_id = (guint)strtol(g_ptr_array_index(args, 1), NULL, 0);
 town_id = (guint)strtol(g_ptr_array_index(args, 2), NULL, 0);

 /* TODO */

 return 0;
}

/******************************************************************************
 A transporter has left a town where you have a transporter of your own.
******************************************************************************/
gint servercmd_transporter_left(GPtrArray *args)
{
 guint transporter_id, player_id;
 transporter_t *transporter;
 player_t *player;

 transporter_id = (guint)strtol(g_ptr_array_index(args, 0), NULL, 0);
 player_id = (guint)strtol(g_ptr_array_index(args, 1), NULL, 0);

 /* TODO */

 return 0;
}

/******************************************************************************
 ...
******************************************************************************/
gint servercmd_town_info(GPtrArray *args)
{
 guint counter = 1;
 guint i, town_id, good_id, number_of_goods_in_town;
 town_t *town;
 good_t *good;

 town_id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 town = town_find_by_id(town_id);

 town->money = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);
 town->inhabitants = (gulong)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 /****************************************************************************
  Goods in town.
 ****************************************************************************/
 number_of_goods_in_town = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

 for (i=0; i<number_of_goods_in_town; i++)
 {
  good_id = (guint)strtol(g_ptr_array_index(args, counter++), NULL, 0);

  good = good_find_by_id(town->goods, good_id);

  good->amount = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);
  good->price = (glong)strtol(g_ptr_array_index(args, counter++), NULL, 0);
 }

 return 0;
}

