/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* protocol.c  --  htsprotocol implementation. */


#include "common.h"
#include "log.h"
#include "net.h"
#include "misc.h"
#include "nc.h"
#include "servercmds.h"

#include "protocol.h"


/******************************************************************************
 Returns the reply-code from the server.
******************************************************************************/
gint protocol_get_reply_code(const gchar *reply)
{
 gchar *reply_code_string = g_malloc(3+1);
 guint reply_code;

 g_snprintf(reply_code_string, 3+1, "%s", reply);

 if (!is_number_or_digit(reply_code_string))
  return -1;

 reply_code = (gint)strtol(reply_code_string, NULL, 0);

 g_free(reply_code_string);

 return reply_code;
}

/******************************************************************************
 Create a new player.
******************************************************************************/
gint protocol_addplayer(const gchar *playername, const gchar *password)
{
 gchar *server_reply;
 guint reply_code;

 net_send_line_to_server("addplayer \"%s\" \"%s\"", playername, password);

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);
 g_free(server_reply);

 if (reply_code == 200)
  return 0;

 return -1;
}

/******************************************************************************
 Login to the server.
******************************************************************************/
gint protocol_login(const gchar *playername, const gchar *password)
{
 gchar *server_reply;
 guint reply_code;

 net_send_line_to_server("login \"%s\" \"%s\"", playername, password);

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);
 g_free(server_reply);

 if (reply_code == 160)
 {
  if ((server_reply = net_read_line_from_server()) == NULL)
   return -1;

  reply_code = protocol_get_reply_code(server_reply);

  if (reply_code == 260)
  {
   GPtrArray *ptr_array;

   if ((ptr_array = tokenize(server_reply)) == NULL)
   {
    g_warning("protocol_login: tokenize() failed.");
    return -1;
   }

   servercmd_init(ptr_array);

   g_free(server_reply);
   my_player->logged_in = YES;
   return 0;
  }

  g_free(server_reply);
  return -1;
 }

 return -1;
}

/******************************************************************************
 Logout from the server.
******************************************************************************/
gint protocol_logout(void)
{
 gchar *server_reply;
 guint reply_code;
 GList *g;
 transporter_t *transporter;
 town_t *town;

 net_send_line_to_server("logout");

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);
 g_free(server_reply);

 if (reply_code == 200)
 {
  my_player->logged_in = NO;

  /* FIXME */

  /* Free Transporters */
  if ((g = g_list_first(my_player->transporters)) != NULL)
  {
   do
   {
    transporter = g->data;
    good_free_all(transporter->goods);
   } while ((g = g_list_next(g)) != NULL);
  }
  transporter_free_all(my_player->transporters);
  my_player->transporters = NULL;


  /* FIXME */

  /* Free Towns */
  if ((g = g_list_first(towns)) != NULL)
  {
   do
   {
    town = g->data;
    good_free_all(town->goods);
   } while ((g = g_list_next(g)) != NULL);
  }
  town_free_all();
  towns = NULL;

  player_free(my_player);

  return 0;
 }

 return -1;
}

/******************************************************************************
 Gain admin permissions.
******************************************************************************/
gint protocol_adminlogin(const gchar *password)
{
 gchar *server_reply;
 guint reply_code;

 net_send_line_to_server("adminlogin \"%s\"", password);

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);
 g_free(server_reply);

 if (reply_code == 200)
 {
  my_player->logged_in_as_admin = YES;
  return 0;
 }

 return -1;
}

/******************************************************************************
 Get the htsprotocol version the server uses.
******************************************************************************/
gint protocol_protocol(void)
{
 gchar *server_reply;
 guint reply_code;

 net_send_line_to_server("protocol");

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);

 if (reply_code == 210)
 {
  g_snprintf((gchar *)&server_protocol, 5+1, "%s", (gchar *)&server_reply[4]);
  g_free(server_reply);
  return 0;
 }

 g_free(server_reply);
 return -1;
}

/******************************************************************************
 Buy goods from the town where the given transporter is located.
******************************************************************************/
gint protocol_buy_from_town_to_transporter(guint transporter, guint what,
                                           glong amount)
{
 gchar *server_reply;
 guint reply_code;

 net_send_line_to_server("buy_from_town_to_transporter %u %u %lu", transporter,
                         what, amount);

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);
 g_free(server_reply);

 if (reply_code == 200)
  return 0;

 return -1;
}

/******************************************************************************
 Sell goods from a transporter to the town where the transporter is located.
******************************************************************************/
gint protocol_sell_from_transporter_to_town(guint transporter, guint what,
                                            glong amount)
{
 gchar *server_reply;
 guint reply_code;

 net_send_line_to_server("sell_from_transporter_to_town %u %u %lu",
                         transporter, what, amount);

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);
 g_free(server_reply);

 if (reply_code == 200)
  return 0;

 return -1;
}

/******************************************************************************
 Drive the given transporter to the destination town.
******************************************************************************/
gint protocol_transporter_drive(guint transporter, guint destination)
{
 gchar *server_reply;
 guint reply_code;

 net_send_line_to_server("transporter_drive %u %u", transporter, destination);

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);
 g_free(server_reply);

 if (reply_code == 160)
 {
  if ((server_reply = net_read_line_from_server()) == NULL)
   return -1;

  reply_code = protocol_get_reply_code(server_reply);

  if (reply_code == 260)
  {
   GPtrArray *ptr_array;

   if ((ptr_array = tokenize(server_reply)) == NULL)
   {
    g_warning("protocol_transporter_drive: tokenize() failed.");
    return -1;
   }

   servercmd_town_info(ptr_array);

   g_free(server_reply);
   return 0;
  }

  g_free(server_reply);
  return -1;
 }

 return -1;
}

/******************************************************************************
 Disconnect from the server.
******************************************************************************/
gint protocol_quit(void)
{
 gchar *server_reply;
 guint reply_code;

 net_send_line_to_server("quit");

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);

 if (reply_code == 221)
 {
  net_close_connection_to_server();

  g_snprintf((gchar *)&server_protocol, 5+1, "%s", "<?>  ");

  /* display_statusbar2(); */
  display_main_screen();
  
  g_free(server_reply);
  return 0;
 }

 g_free(server_reply);
 return -1;
}

