/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* net.c  --  Networking code. */


#include "common.h"
#include "log.h"
#include "protocol.h"
#include "misc.h"
#include "nc.h"
#include "conf.h"
#include "servercmds.h"
#include "file.h"
#include "input.h"

#include "net.h"


GMainLoop *mainloop;

GIOChannel *server_channel;

gint server_event_source_id;
gint input_keys_main_menu_event_source_id;


/******************************************************************************
 Create the main event loop, which manages all available event sources.
******************************************************************************/
void net_main_loop(void)
{
 mainloop = g_main_new(FALSE);

 input_keys_main_menu_event_source_id = g_timeout_add(100,
                                                      input_keys_main_menu,
                                                      NULL);

 g_main_run(mainloop);
}

/******************************************************************************
 Create an internet stream socket and return its descriptor.
******************************************************************************/
gint net_create_socket(const gchar *ip, gushort port)
{
 gint sockfd;
 gint flags;
 struct sockaddr_in server_addr;

 if ((sockfd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
 {
  log_perror("net_create_socket:socket");
  return -1;
 }

 memset(&server_addr, 0, sizeof(struct sockaddr_in));
 server_addr.sin_family = AF_INET;
 server_addr.sin_port = g_htons(port);
 server_addr.sin_addr.s_addr = inet_addr(ip);

 if (connect(sockfd, &server_addr, (socklen_t)sizeof(struct sockaddr_in)) < 0)
 {
  log_perror("net_create_socket:connect");
  return -1;
 }

 if ((flags = fcntl(sockfd, F_GETFL, 0)) < 0)
 {
  log_perror("net_create_socket:fcntl");
  /* if (close(sockfd) < 0)
   log_perror("net_create_socket:close");
  return -1; */
 }

 flags |= O_NONBLOCK; /* Set socket nonblocking. */

 if (fcntl(sockfd, F_SETFL, flags) < 0)
 {
  log_perror("net_create_socket:fcntl");
  /* if (close(sockfd) < 0)
   log_perror("net_create_socket:close");
  return -1; */
 }

 /* Set owner of the socket. */
 if (fcntl(sockfd, F_SETOWN, getpid()) < 0)
 {
  log_perror("net_create_socket:fcntl");
  /* if (close(sockfd) < 0)
   log_perror("net_create_socket:close");
  return -1; */
 }

 return sockfd;
}

/******************************************************************************
 Connect to the server.
******************************************************************************/
gint net_connect_to_server(const gchar *hostname, guint port)
{
 gint sockfd;
 gchar *server_reply;
 guint reply_code;

 if ((sockfd = net_create_socket(hostname, port)) < 0)
 {
  g_warning("Could not create the server socket.");
  return -1;
 }

 server_channel = g_io_channel_unix_new(sockfd);

 server_event_source_id = g_io_add_watch(server_channel, G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL, net_handle_server_message, NULL);

 if ((server_reply = net_read_line_from_server()) == NULL)
  return -1;

 reply_code = protocol_get_reply_code(server_reply);
 g_free(server_reply);

 if (reply_code == 220)
 {
  connected = YES;
  return 0;
 }

 return -1;
}

/******************************************************************************
 Close the connection to the server.
******************************************************************************/
void net_close_connection_to_server(void)
{
 g_message("Closing connection to server.");

 g_io_channel_close(server_channel);
 g_io_channel_unref(server_channel);

 connected = NO;
 my_player->logged_in = NO; /* We aren't connected, so we can't be logged in. */

 display_main_screen(); /* ? */
}

/******************************************************************************
 Read a line of data from the server.
******************************************************************************/
gchar *net_read_line_from_server(void)
{
 gchar *buf;
 guint bytes_read;
 guint count = 0;
 gchar c;
 GIOError error;
 guint size = 40;

 /* TODO Check whether we can read without blocking. */

 buf = g_malloc(size);

 while (TRUE)
 {
  bytes_read = 0;

net_read_line_from_server_again:
  error = g_io_channel_read(server_channel, &c, 1, &bytes_read);

  if (error == G_IO_ERROR_AGAIN)
   goto net_read_line_from_server_again;

  if ((error == G_IO_ERROR_INVAL) || (error == G_IO_ERROR_UNKNOWN))
  {
   g_warning("net_read_line_from_server: g_io_channel_read() error.");
   net_close_connection_to_server();
   return NULL;
  }

  if (bytes_read == 0) /* EOF. */
  {
   if (count == 0)
   {
    g_warning("net_read_line_from_server: Didn't read any data.");
    g_free(buf);
    return NULL;
   }
   else
    break;
  }

  buf[count] = c;
  count++;

  if (c == '\n')
  {
   buf[count] = '\0';
   break;
  }

  if (count+1 >= size)
  {
   size *= 2;
   buf = g_realloc(buf, size);
  }

 }

 g_message("Received from server: %s", g_strchomp(buf));

 return buf;
}

/******************************************************************************
 Send a line of data to the server.
******************************************************************************/
void net_send_line_to_server(const gchar *fmt, ...)
{
 va_list args;
 gchar *tmp, *message;

 /* TODO Check whether we can write without blocking. */

 va_start(args, fmt);
 tmp = g_strdup_vprintf(fmt, args);
 va_end(args);

 message = g_strdup_printf("%s\n", tmp);
 g_free(tmp);

 if (file_write_line(server_channel, message) < 0)
 {
  g_warning("Error writing to server. Aborting.");
  net_close_connection_to_server();
  g_free(message);
  return;
 }

 g_message("Sent to server: %s", g_strchomp(message));

 g_free(message);
}

/******************************************************************************
 Handle a message from the server.
******************************************************************************/
gboolean net_handle_server_message(GIOChannel *source, GIOCondition condition,
                                   gpointer data)
{
 if (condition == G_IO_IN)
 {
  gchar *message;

  if ((message = net_read_line_from_server()) == NULL)
  {
   net_close_connection_to_server();
   return FALSE;
  }

  check_for_servercmds(message);

  g_free(message);
 }

 if (condition == G_IO_PRI)
 {
  g_warning("net_handle_server_message: G_IO_PRI.");
  /* TODO */
 }

 if (condition == G_IO_ERR)
 {
  g_warning("net_handle_server_message: G_IO_ERR.");
  /* TODO */
 }

 if (condition == G_IO_HUP)
 {
  g_warning("Lost connection to server.");

  net_close_connection_to_server();

  return FALSE;
 }

 if (condition == G_IO_NVAL)
 {
  g_warning("net_handle_server_message: G_IO_NVAL.");
  /* TODO */
 }

 return TRUE;
}

/******************************************************************************
 Return the IP of the given hostname.
******************************************************************************/
gchar *net_resolve_hostname(const gchar *hostname)
{
 struct hostent *hostdata;

 if ((hostdata = gethostbyname(hostname)) == NULL)
 {
  g_warning("net_resolve_hostname:gethostbyname: %s(%d)",
            g_strerror(h_errno), h_errno); /* Does this really work? */
  /* herror("net_resolve_hostname:gethostbyname"); */
  return NULL;
 }

 return g_strdup(inet_ntoa(*((struct in_addr *)hostdata->h_addr)));
}

