/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* nc.c  --  Ncurses-related code. */


#include "common.h"
#include "log.h"
#include "goods.h"
#include "misc.h"
#include "players.h"
#include "net.h"
#include "colors.h"
#include "transporters.h"
#include "goods.h"
#include "trade.h"
#include "input.h"

#include "nc.h"


#define ABOUT_BOX_MESSAGE "\
                         "PACKAGE" "VERSION"\n\
   a ncurses-based client for the trading game Holsham Traders\n\
                   released on "NHTSCLIENT_RELEASEDATE"\n\n\
   Copyright (C) 1999-2001 Uwe Hermann <uh1763@hermann-uwe.de>\n\n\
                 http://nhtsclient.sourceforge.net\n\n\
 This program is covered by the GNU General Public License (GPL)"


#define HELP_BOX_MESSAGE "\
 "PACKAGE" keys:\n\
\n\
  c:  Connect               1,2,3,4: Play song 1/2/3/4\n\
  C:  Connect as            m: Pause/continue music\n\
  d:  Disconnect            s: Stop music\n\
  a:  Add player\n\
  l:  Login\n\
  L:  Adminlogin\n\
  o:  Logout\n\
  t:  Trade screen\n\
  p:  Players screen\n\
\n\n\n\
  A:  About "PACKAGE"\n\
  ?:  Show this help\n\
  q:  Quit                                   PRESS ANY KEY."


static gint menu_fetch_input(WINDOW *win);

/******************************************************************************
 ...
******************************************************************************/
void init_ncurses(void)
{
 (void) initscr();            /* Initialize the curses library. */
 (void) keypad(stdscr, TRUE); /* Enable keyboard mapping. */
 (void) nonl();               /* Tell curses not to do NL->CR/NL on output. */
 (void) cbreak();             /* Take input chars at once, don't wait for \n. */
 (void) noecho();             /* Don't echo input. */
 (void) curs_set(INVISIBLE);  /* Make cursor invisible. */

 (void) timeout(1);

 colors_init();
 colors_set_defaults();
}

/******************************************************************************
 Quit ncurses, hopefully without leaving any mess behind.
******************************************************************************/
void shutdown_ncurses(void)
{
 (void) standend();
 (void) clear();
 (void) refresh();
 (void) curs_set(VISIBLE);

 if (endwin() == ERR)
  g_warning("endwin() failed.");
}

/******************************************************************************
 ...
******************************************************************************/
void display_main_screen(void)
{
 (void) clear();

 display_statusbar1(LOCATION_MAIN_SCREEN);
 display_statusbar2(LOCATION_MAIN_SCREEN); 

 if (my_player->logged_in)
 {
  (void) mvprintw(2, 50, "Your name: %s", my_player->name);
  (void) mvprintw(3, 50, "Your age: %d", my_player->age);
  (void) mvprintw(4, 50, "Your money: %ld", my_player->money);
  (void) mvprintw(6, 50, "Your transporters: %d",
                  g_list_length(my_player->transporters));
 }

 (void) refresh();
}

/******************************************************************************
 Display the upper statusbar.
******************************************************************************/
void display_statusbar1(guint location)
{
 gint y = 0;
 gint maxx, maxy;

 (void) xmvclrtoeol(y, 0, colors.statusbar1);

 getmaxyx(stdscr, maxy, maxx);

 if (location == LOCATION_MAIN_SCREEN)
 {
  (void) xmvaddstr(y, 0, colors.statusbar1, ""PACKAGE" "VERSION"");
  (void) xmvaddstr(y, maxx-23, colors.statusbar1, "A:About  ?:Help  q:Quit");
 }

 if (location == LOCATION_TRADE_SCREEN)
  (void) xmvaddstr(y, 0, colors.statusbar1, "n:Next Transporter  ENTER:Buy/Sell  SPACE:Switch mode  d:Drive  q:Back");

 if (location == LOCATION_PLAYERS_SCREEN)
  (void) xmvaddstr(y, 0, colors.statusbar1, "Any key:Back");

}

/******************************************************************************
 Display the lower statusbar.
******************************************************************************/
void display_statusbar2(guint location)
{
 gint y = 23;

 (void) xmvclrtoeol(y, 0, colors.statusbar2);

 if (location == LOCATION_MAIN_SCREEN)
 {
  xmvaddstr(y, 0, colors.statusbar2,
            "[ ] Connected --- [ ] Logged in --- Protocol:");

  if (connected)
   xmvaddstr(y, 1, colors.statusbar2, "x");

  if (my_player->logged_in)
   xmvaddstr(y, 19, colors.statusbar2, "x");

  xmvaddstr(y, 46, colors.statusbar2, server_protocol);
 }

 if (location == LOCATION_TRADE_SCREEN)
  xmvprintw(y, 0, colors.statusbar2, "Money: %lu", my_player->money);

 if (location == LOCATION_PLAYERS_SCREEN)
  xmvprintw(y, 0, colors.statusbar2, "Players: %u", g_list_length(players));
}

/******************************************************************************
 ...
******************************************************************************/
void display_trade_screen(void)
{
 display_statusbar1(LOCATION_TRADE_SCREEN);
 display_statusbar2(LOCATION_TRADE_SCREEN);
 
 mvaddstr(2, 1, "Good            Town   Transporter Price");

 drawbox(stdscr, 0, 1, 42, 20);
 
 (void) move(3, 1);
 (void) hline(ACS_HLINE, 41);

 (void) mvaddch(3, 0, ACS_LTEE);
 (void) mvaddch(3, 42, ACS_RTEE);
 
 (void) mvaddch(1, 16, ACS_TTEE);
 (void) mvaddch(2, 16, ACS_VLINE);
 (void) mvaddch(3, 16, ACS_BTEE);

 (void) mvaddch(1, 23, ACS_TTEE);
 (void) mvaddch(2, 23, ACS_VLINE);
 (void) mvaddch(3, 23, ACS_BTEE);

 (void) mvaddch(1, 35, ACS_TTEE);
 (void) mvaddch(2, 35, ACS_VLINE);
 (void) mvaddch(3, 35, ACS_BTEE);
}

/******************************************************************************
 ...
******************************************************************************/
void print_transporter_info(transporter_t *transporter)
{
 town_t *town = town_find_by_id(transporter->location);

 (void) mvprintw(4, 50, "Transporter: %s", transporter->name);
 (void) mvprintw(5, 50, " * Money: %ld", transporter->money);
 (void) mvprintw(6, 50, " * Load: %d/%d",
                 transporter->capacity - transporter_free_capacity(transporter),                 transporter->capacity);
 (void) mvprintw(8, 50, "Town: %s", town->name);
}

/******************************************************************************
 ...
******************************************************************************/
void print_goods(GList *goods, transporter_t *transporter, gint highlighted)
{
 gint i, j = 0;
 GList *g;
 good_t *good_town, *good_transporter;
 town_t *town;
 chtype col = colors.normal;

 (void) clear();

 display_trade_screen();
 print_transporter_info(transporter);

 town = town_find_by_id(transporter->location);

 g = g_list_first(goods);
 for (i=0; i<(gint)g_list_length(goods); i++)
 {
  good_town = good_find_by_id(town->goods, (guint)GPOINTER_TO_UINT(g->data)); 
  good_transporter = good_find_by_id(transporter->goods,
                                     (guint)GPOINTER_TO_UINT(g->data));

  if ((good_town->amount != 0) || (good_transporter->amount != 0))
  {

   if (highlighted == j)
   {
    if (trade_buymode)
     col = colors.buy;
    else
     col = colors.sell;
   }

   xmvaddstr(4+j, 1, col, "                                         ");

   xmvaddstr(4+j, 1, col, goodnames[good_town->id]);

   xmvprintw(4+j, 1+MAX_LEN_GOODNAME+1, col, "%ld", good_town->amount);

   xmvprintw(4+j, 1+MAX_LEN_GOODNAME+1+6+1, col, "%ld",
             good_transporter->amount);

   xmvprintw(4+j, 1+MAX_LEN_GOODNAME+1+6+1+11+1, col, "%ld", good_town->price);

   if (highlighted == j)
    col = colors.normal;

   j++;
  }

  g = g_list_next(g);
 }

 (void) refresh();
}

/******************************************************************************
 Fill the given rectangle with the given character.
******************************************************************************/
void wfillbox(WINDOW *win, gint x1, gint y1, gint x2, gint y2, chtype ch)
{
 gint i, j;

 for (i=x1; i<=x2; i++)
  for (j=y1; j<=y2; j++)
   mvwaddch(win, j, i, ' ' | ch);
}

/******************************************************************************
 Wrapper for wfillbox().
******************************************************************************/
void fillbox(gint x1, gint y1, gint x2, gint y2, chtype ch)
{
 wfillbox(stdscr, x1, y1, x2, y2, ch);
}

/******************************************************************************
 ...
******************************************************************************/
void drawbox(WINDOW *win, gint x1, gint y1, gint x2, gint y2)
{
 (void) wmove(win, y1, x1);
 (void) whline(win, ACS_HLINE, x2-x1);

 (void) wmove(win, y2, x1);
 (void) whline(win, ACS_HLINE, x2-x1);

 (void) wmove(win, y1, x1);
 (void) wvline(win, ACS_VLINE, y2-y1);

 (void) wmove(win, y1, x2);
 (void) wvline(win, ACS_VLINE, y2-y1);

 (void) mvwaddch(win, y1, x1, ACS_ULCORNER);
 (void) mvwaddch(win, y1, x2, ACS_URCORNER);
 (void) mvwaddch(win, y2, x1, ACS_LLCORNER);
 (void) mvwaddch(win, y2, x2, ACS_LRCORNER);
}

/******************************************************************************
 Display the about box.
******************************************************************************/
void about_box(void)
{
 WINDOW *win = newwin(11, 65, 5, 7);
 PANEL *panel = new_panel(win);

 wfillbox(win, 0, 0, 65, 11, colors.about_box);
 xmvwaddstr(win, 1, 0, colors.about_box, ABOUT_BOX_MESSAGE);
 xbox(win, colors.about_box, ACS_VLINE, ACS_HLINE);

 update_panels();
 doupdate();

 input_wait_for_keypress(win);

 (void) del_panel(panel);
 (void) delwin(win);
}

/******************************************************************************
 Display the help box.
******************************************************************************/
void help_box(void)
{
 WINDOW *win = newwin(19, 60, 2, 10);
 PANEL *panel = new_panel(win);

 wfillbox(win, 0, 0, 60, 19, colors.help_box);
 xmvwaddstr(win, 1, 0, colors.help_box, HELP_BOX_MESSAGE);
 xbox(win, colors.help_box, ACS_VLINE, ACS_HLINE);

 update_panels();
 doupdate();

 input_wait_for_keypress(win);

 (void) del_panel(panel);
 (void) delwin(win);
}

/******************************************************************************
 ...
******************************************************************************/
void print_chat_message(const gchar *message)
{
 /* TODO Improve. This is just here for testing purposes. */
 (void) move(21, 0);
 (void) clrtoeol();
 (void) addstr(message);
 (void) refresh();
}

/******************************************************************************
 ...
******************************************************************************/
void print_log_message(const gchar *fmt, ...)
{
 va_list args;
 gchar *message;

 va_start(args, fmt);
 message = g_strdup_vprintf(fmt, args);
 va_end(args);

 /* TODO Improve. This is just here for testing purposes. */

 (void) move(20, 0);
 (void) clrtoeol();
 (void) addstr(message);
 (void) refresh();

 g_free(message);
}

/******************************************************************************
 Wrapper for mvwaddstr().
******************************************************************************/
gint xmvwaddstr(WINDOW *win, gint y, gint x, chtype ch, const gchar *message)
{
 gint return_value;
 chtype old_background = getbkgd(win);

 wbkgdset(win, ch);
 return_value = mvwaddstr(win, y, x, message);
 wbkgdset(win, old_background);

 return return_value;
}

/******************************************************************************
 Wrapper for xmvwaddstr().
******************************************************************************/
gint xmvaddstr(gint y, gint x, chtype ch, const gchar *message)
{
 return xmvwaddstr(stdscr, y, x, ch, message);
}

/******************************************************************************
 Wrapper for wclrtoeol().
******************************************************************************/
gint xwclrtoeol(WINDOW *win, chtype ch)
{
 gint return_value;
 chtype old_background = getbkgd(win);

 wbkgdset(win, ch);
 return_value = wclrtoeol(win);
 wbkgdset(win, old_background);

 return return_value;
}

/******************************************************************************
 Wrapper for xwclrtoeol().
******************************************************************************/
gint xclrtoeol(chtype ch)
{
 return xwclrtoeol(stdscr, ch);
}

/******************************************************************************
 Wrapper for xclrtoeol().
******************************************************************************/
gint xmvclrtoeol(gint y, gint x, chtype ch)
{
 (void) move(y, x);
 return xclrtoeol(ch);
}

/******************************************************************************
 Wrapper for mvprintw().
******************************************************************************/
gint xmvprintw(gint y, gint x, chtype ch, const gchar *fmt, ...)
{
 va_list args;
 gchar *message;
 gint return_value;
 chtype old_background = getbkgd(stdscr);

 va_start(args, fmt);
 message = g_strdup_vprintf(fmt, args);
 va_end(args);

 bkgdset(ch);
 return_value = mvaddstr(y, x, message);
 bkgdset(old_background);

 g_free(message);

 return return_value;
}

/******************************************************************************
 Wrapper for box().
******************************************************************************/
gint xbox(WINDOW *win, chtype ch, chtype verch, chtype horch)
{
 gint return_value;
 chtype old_background = getbkgd(win);

 wbkgdset(win, ch);
 return_value = box(win, verch, horch);
 wbkgdset(win, old_background);

 return return_value;
}

/******************************************************************************
 ...
******************************************************************************/
static gint menu_fetch_input(WINDOW *win)
{
 chtype ch = input_wait_for_keypress(win);

 if (ch == '\r' || ch == '\n' || ch == KEY_ENTER)
  return MAX_COMMAND+1;
 else if (ch == KEY_DOWN)
  return REQ_NEXT_ITEM;
 else if (ch == KEY_UP)
  return REQ_PREV_ITEM;
 else if (ch == KEY_HOME)
  return REQ_FIRST_ITEM;
 else if (ch == KEY_END)
  return REQ_LAST_ITEM;
 else
  return(ch);
}

/******************************************************************************
 Generic menu routine which supports scrolling.
******************************************************************************/
guint generic_menu(GPtrArray *ptr_array, chtype ch, const gchar *title,
                   chtype ch_title, chtype ch_fore, chtype ch_back,
                   const gchar *mark)
{
 gint c;
 MENU *menu;
 ITEM *items[100]; /* FIXME */
 ITEM **ip = items;
 WINDOW *win, *win2;
 PANEL *panel;
 gint rows, cols;
 guint i, return_value;


 for (i=0; i<ptr_array->len; i++)
  *ip++ = new_item(g_ptr_array_index(ptr_array, i), NULL);
 
 *ip = NULL; /* NULL-terminate the pointer array. */

 menu = new_menu(items);

 (void) set_menu_mark(menu, mark);

 (void) set_menu_format(menu, ptr_array->len <= 16 ? ptr_array->len : 16, 1);
 scale_menu(menu, &rows, &cols);

 /* (void) set_menu_fore(menu, ch_fore);
 (void) set_menu_back(menu, ch_back); */
 /* (void) set_menu_grey(menu, ch_grey); */

 /* TODO Place window in the middle of the screen. */
 win = newwin(rows+4, cols+2, 5, 30);

 /* win = newwin(2+CLAMP(ptr_array->len, 0, 19),
                 2+MAX_LEN_COL, 4, (gint)((78-MAX_LEN_COL)/2)); */

 set_menu_win(menu, win);

 /* wfillbox(win, 0, 0, cols+2, rows+4, ch); */
 /* xbox(win, ch, ACS_VLINE, ACS_HLINE); */

 box(win, ACS_VLINE, ACS_HLINE);

 /* TODO Check whether or not the title is too long. */

 /* (void) xmvwaddstr(win, ch_title, 1, 1, title); */
 (void) mvwaddstr(win, 1, 1, title);
 (void) mvwhline(win, 2, 1, ACS_HLINE, cols);
 (void) mvwaddch(win, 2, 0, ACS_LTEE);
 (void) mvwaddch(win, 2, cols+1, ACS_RTEE);

 win2 = derwin(win, rows, cols, 3, 1);
 set_menu_sub(menu, win2);

 panel = new_panel(win);

 post_menu(menu);

 update_panels();
 (void) doupdate();

 /* wrefresh(win); */

 while ((c = menu_driver(menu, menu_fetch_input(win))) != E_UNKNOWN_COMMAND)
 {
  /* if (c == E_REQUEST_DENIED)
   beep(); */
 }

 return_value = item_index(current_item(menu));

 unpost_menu(menu);

 (void) del_panel(panel);
 (void) delwin(win2);
 (void) delwin(win);

 free_menu(menu);

 for (i=0; i<ptr_array->len; i++)
  free_item(items[i]);

 return return_value;
}

