/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* misc.c  --  Miscellaneous code. */


#include "common.h"
#include "log.h"
#include "nc.h"
#include "sound.h"
#include "conf.h"
#include "net.h"

#include "misc.h"


gboolean connected = NO;

gboolean parent = YES;
pid_t child_pid;

gchar server_protocol[5+1] = "<?>  ";


static RETSIGTYPE signal_handler_exit(gint signum);
static RETSIGTYPE signal_handler_child(gint signum);

/******************************************************************************
 ...
******************************************************************************/
void setup_signal_handling(void)
{
 if (signal(SIGTERM, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGTERM, SIG_IGN);
 if (signal(SIGINT, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGINT, SIG_IGN);
 if (signal(SIGQUIT, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGQUIT, SIG_IGN);
 if (signal(SIGABRT, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGABRT, SIG_IGN);
 if (signal(SIGHUP, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGHUP, SIG_IGN);

 if (signal(SIGCHLD, signal_handler_child) == SIG_IGN)
  (void) signal(SIGCHLD, SIG_IGN);

 (void) signal(SIGPIPE, SIG_IGN);
}

/******************************************************************************
 ...
******************************************************************************/
static RETSIGTYPE signal_handler_exit(gint signum)
{
 g_message("Signal \"%s\" (%d) received. Shutting down.", g_strsignal(signum),
           signum);
 g_main_quit(mainloop);
 g_main_destroy(mainloop);
}

/******************************************************************************
 ...
******************************************************************************/
static RETSIGTYPE signal_handler_child(gint signum)
{
 g_message("Signal \"%s\" (%d) received. Waiting for child to terminate.",
           g_strsignal(signum), signum);

 if (waitpid(child_pid, NULL, WNOHANG) != child_pid)
  log_perror("signal_handler_child:waitpid");
}

/******************************************************************************
 Check if the given string contains a number or a digit.
******************************************************************************/
gboolean is_number_or_digit(const gchar *string)
{
 guint i;

 for (i=0; i<strlen(string); i++)
 {
  if (!isdigit(string[i]))
   return FALSE;
 }

 return TRUE;
}

/******************************************************************************
 Split a string into several tokens.
******************************************************************************/
GPtrArray *tokenize(const gchar *message)
{
 GTokenType token;
 GScanner *scanner;
 GPtrArray *ptr_array;

 ptr_array = g_ptr_array_new();
 scanner = g_scanner_new(NULL);

 scanner->config->cset_identifier_first =
 (G_CSET_a_2_z G_CSET_A_2_Z G_CSET_DIGITS);
 scanner->config->scan_identifier_1char = TRUE;
 /* TODO Enable/disable more options? */

 g_scanner_input_text(scanner, message, strlen(message));
 scanner->input_name = "tokenize";

 while (TRUE)
 {
  token = g_scanner_get_next_token(scanner);

  if (token == G_TOKEN_ERROR)
  {
   g_scanner_unexp_token(scanner, token, NULL, NULL, NULL, NULL, TRUE);
   (void) g_ptr_array_free(ptr_array, TRUE);
   g_scanner_destroy(scanner);
   return NULL;
  }

  if (token == G_TOKEN_EOF)
   break;

  /* TODO Check for more return values? */

  g_ptr_array_add(ptr_array, (gpointer)g_strdup(scanner->value.v_string));
 }

 g_scanner_destroy(scanner);

 return ptr_array;
}

