/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* input.c  --  Input routines. */


#include "common.h"
#include "misc.h"
#include "net.h"
#include "protocol.h"
#include "dialogs.h"
#include "conf.h"
#include "sound.h"
#include "log.h"
#include "players.h"
#include "nc.h"
#include "sound.h"
#include "trade.h"
#include "colors.h"
#include "towns.h"

#include "input.h"


gboolean in_trade_screen = NO;
gboolean trade_screen_changed = NO;

guint transporter_id = 0;

gint input_keys_trade_menu_event_source_id;


static void input_key_connect_as(void);
static void input_key_logout(void);
static void input_key_connect(void);
static void input_key_disconnect(void);
static void input_key_login(void);
static void input_key_adminlogin(void);
static void input_key_addplayer(void);
static void input_key_trade_screen(void);

#if HAVE_LIBMIKMOD
static void input_key_music(chtype key);
#endif /* HAVE_LIBMIKMOD */

static void input_key_players_screen(void);


/******************************************************************************
 ...
******************************************************************************/
gboolean input_keys_main_menu(gpointer data)
{
 chtype ch;

 /* TODO Set timeout to 1? */

 switch ((gint)(ch = getch()))
 {
  /*** quit ******************************************************************/
  case 'q':
   
   if ((connected) && (protocol_quit() < 0))
   {
    g_warning("Quit failed. Forcing disconnect.");
    net_close_connection_to_server();
   }

   g_main_quit(mainloop);
   g_main_destroy(mainloop);
  break;

  /*** connect ***************************************************************/
  case 'c': input_key_connect(); break;

  /*** disconnect ************************************************************/
  case 'd': input_key_disconnect(); break;

  /*** login *****************************************************************/
  case 'l': input_key_login(); break;

  /*** adminlogin ************************************************************/
  case 'L': input_key_adminlogin(); break;

  /*** addplayer *************************************************************/
  case 'a': input_key_addplayer(); break;

  /*** trade screen **********************************************************/
  case 't': input_key_trade_screen(); break;

  /*** players screen ********************************************************/
  case 'p': input_key_players_screen(); break;

  /*** logout ****************************************************************/
  case 'o': input_key_logout(); break;

  /*** about-box *************************************************************/
  case 'A': (void) about_box(); break;

  /*** connect as ************************************************************/
  case 'C': input_key_connect_as(); break;

  /*** help-box **************************************************************/
  case '?': help_box(); break;

  /*** music keys ************************************************************/
#if HAVE_LIBMIKMOD
  case '1':
  case '2':
  case '3':
  case 'm':
  case 's':
   input_key_music(ch);
  break;
#endif 

 }

 return TRUE;
}

/******************************************************************************
 ...
******************************************************************************/
static void input_key_addplayer(void)
{
 if (!connected)
 {
  print_log_message("You're not connected.");
  return;
 }

 dialog_playername_password(my_player->name, my_player->password,
                            colors.dialog_addplayer);

 if (protocol_addplayer(my_player->name, my_player->password) < 0)
  print_log_message("Player could not be created.");
 else
  print_log_message("Player successfully created.");
}

/******************************************************************************
 ...
******************************************************************************/
static void input_key_adminlogin(void)
{
 if (!connected)
 {
  print_log_message("You're not connected.");
  return;
 }

 if (my_player->logged_in_as_admin)
 {
  print_log_message("Already logged in as admin.");
  return;
 }

 /* TODO */
}

/******************************************************************************
 ...
******************************************************************************/
static void input_key_login(void)
{
 if (!connected)
 {
  print_log_message("You're not connected.");
  return;
 }

 if (my_player->logged_in)
 {
  print_log_message("You're already logged in.");
  return;
 }

 dialog_playername_password(my_player->name, my_player->password,
                            colors.dialog_login);

 if (protocol_login(my_player->name, my_player->password) < 0)
 {
  print_log_message("Login failed.");
  return;
 }

 /* display_statusbar2(); */
 display_main_screen();
}

/******************************************************************************
 ...
******************************************************************************/
static void input_key_disconnect(void)
{
 if (!connected)
 {
  print_log_message("You're not connected.");
  return;
 }

 if (protocol_quit() < 0)
 {
  g_warning("Quit failed. Forcing disconnect.");
  net_close_connection_to_server();
  return;
 }

 g_snprintf((gchar *)&server_protocol, 5+1, "%s", "<?>  ");
 /* display_statusbar2(); */
 display_main_screen();
}

/******************************************************************************
 ...
******************************************************************************/
static void input_key_connect(void)
{
 gchar *hostname;

 if (connected)
 {
  print_log_message("You're already connected.");
  return;
 }

 if ((hostname = net_resolve_hostname(conf.host)) == NULL)
  hostname = g_strdup(conf.host);

 if (net_connect_to_server(hostname, conf.port) < 0)
 {
  print_log_message("Connect failed.");
  g_free(hostname);
  return;
 }

 g_free(hostname);

 (void) protocol_protocol();

 /* display_statusbar2(); */
 display_main_screen();

 /* FIXME This is a quick hack. */
 if (strncmp((gchar *)&server_protocol, PROTOCOL_VERSION, 5) != 0)
 {
  print_log_message("Server uses htsprotocol %s, but we need %s.",
                    (gchar *)&server_protocol, PROTOCOL_VERSION);
 }

}

/******************************************************************************
 ...
******************************************************************************/
static void input_key_logout(void)
{
 if (!connected)
 {
  print_log_message("You're not connected.");
  return;
 }

 if (!my_player->logged_in)
 {
  print_log_message("You're not logged in.");
  return;
 }

 if (protocol_logout() < 0)
 {
  print_log_message("Logout failed.");
  return;
 }

 /* display_statusbar2(); */
 display_main_screen();
}

/******************************************************************************
 ...
******************************************************************************/
static void input_key_connect_as(void)
{
 gchar *hostname;
 gchar *servername = g_malloc(MAX_LEN_SERVERNAME+1);
 gushort port;

 if (connected)
 {
  print_log_message("Already connected.");
  return;
 }

 dialog_connect(servername, &port);

 if ((hostname = net_resolve_hostname(servername)) == NULL)
  hostname = servername;

 if (net_connect_to_server(hostname, port) < 0)
 {
  print_log_message("Connect failed.");
  g_free(hostname);
  return;
 }

 (void) protocol_protocol();
 /* display_statusbar2(); */
 display_main_screen();

 g_free(servername);
 g_free(hostname);
}

/******************************************************************************
 ...
******************************************************************************/
static void input_key_trade_screen(void)
{
 transporter_t *transporter;
 town_t *town;
 gint highlighted = 0;
 guint what;
 GList *goods;
 chtype ch;

 if (!my_player->logged_in)
 {
  print_log_message("Not logged in.");
  return;
 }

 in_trade_screen = YES;

 transporter = transporter_find_by_id(my_player, transporter_id);

 town = town_find_by_id(transporter->location);
 goods = good_in_town_or_transporter(town, transporter);

 what = GPOINTER_TO_UINT(g_list_nth_data(goods, (guint)highlighted));

 print_goods(goods, transporter, highlighted);
 /* TODO What happens if there are no goods? */

 do
 {
  trade_screen_changed = NO;

  switch ((gint)(ch = input_wait_for_keypress(stdscr)))
  {
   case ' ':
    trade_flip_buymode();
    print_goods(goods, transporter, highlighted);
   break;

   case 13:
    if (trade_buymode)
     trade_buy_from_town_to_transporter(transporter, what, 1);
    else
     trade_sell_from_transporter_to_town(transporter, what, 1);

    print_goods(goods, transporter, highlighted);
   break;

   case KEY_DOWN:
    if (highlighted+1 <= (gint)g_list_length(goods)-1)
    {
     highlighted++;
     what = GPOINTER_TO_UINT(g_list_nth_data(goods, (guint)highlighted));
     print_goods(goods, transporter, highlighted);
    }
   break;

   case KEY_UP:
    if (highlighted-1 >= 0)
    {
     highlighted--;
     what = GPOINTER_TO_UINT(g_list_nth_data(goods, (guint)highlighted));
     print_goods(goods, transporter, highlighted);
    }
   break;

   case 'n':
    if ((transporter_find_by_id(my_player, transporter_id+1)) == NULL)
     transporter_id = 0;
    else
     transporter_id++;

    transporter = transporter_find_by_id(my_player, transporter_id);
    town = town_find_by_id(transporter->location);
    goods = good_in_town_or_transporter(town, transporter);
    what = GPOINTER_TO_UINT(g_list_nth_data(goods, (guint)highlighted));
    print_goods(goods, transporter, highlighted);
   break;

   case 'd':
    {
     guint i;
     GList *g;
     town_t *town;
     guint destination;
     GPtrArray *ptr_array = g_ptr_array_new();

     if (!connected)
     {
      print_log_message("You're not connected.");
      return;
     }

     /* TODO Use towns_known_to_player = get_towns_known_to_player(); */

     g = g_list_first(towns);
     for (i=0; i<g_list_length(towns); i++)
     {
      town = g->data;

      /* TODO Do not display the town where the transporter is located or
         don't allow that town to be selected. */
      /* if (!(town->id == transporter->location)) */
      
      g_ptr_array_add(ptr_array, (gpointer)town->name);

      g = g_list_next(g);
     }

     destination = generic_menu(ptr_array,
                                colors_get_pair("black", "blue"), 
                                "Towns", colors_get_pair("red", "blue"),
                                colors_get_pair("cyan", "blue"),
                                colors_get_pair("black", "blue"), NULL);

     g_ptr_array_free(ptr_array, FALSE);


     if (protocol_transporter_drive(transporter->id, destination) >= 0)
      transporter_drive(my_player, transporter->id, destination);

     /* TODO */

     town = town_find_by_id(transporter->location);

     g_list_free(goods);
     goods = good_in_town_or_transporter(town, transporter);

     highlighted = 0;
     what = GPOINTER_TO_UINT(g_list_nth_data(goods, (guint)highlighted));

     print_goods(goods, transporter, highlighted);
    }
   break;
  }

  if (trade_screen_changed)
   print_goods(goods, transporter, highlighted);

 } while (ch != 'q');

 in_trade_screen = NO;

 display_main_screen();
}

/******************************************************************************
 ...
******************************************************************************/
#if HAVE_LIBMIKMOD
static void input_key_music(chtype key)
{
 if (conf.music != NO)
 {
  switch ((gint)key)
  {
   case '1': sound_send_command_to_child("play "SONG_HOLSHAM1); break;
   case '2': sound_send_command_to_child("play "SONG_HOLSHAM2); break;
   case '3': sound_send_command_to_child("play "SONG_ONTHEFAIR); break;
   case 'm': sound_send_command_to_child("togglepause"); break;
   case 's': sound_send_command_to_child("stop"); break;
  }
 }
}
#endif /* HAVE_LIBMIKMOD */

/******************************************************************************
 ...
******************************************************************************/
static void input_key_players_screen(void)
{
 GList *g;
 guint i;
 player_t *player;

 if (!my_player->logged_in)
 {
  print_log_message("Not logged in.");
  return;
 }

 (void) clear();

 display_statusbar1(LOCATION_PLAYERS_SCREEN);
 display_statusbar2(LOCATION_PLAYERS_SCREEN);

 mvaddstr(2, 0, "List of players in this game:");


 g = g_list_first(players);

 for (i=0; i<(guint)g_list_length(players); i++)
 {
  player = g->data;

  mvaddstr(i+4, 0, player->name);

  if (player->logged_in)
   xmvaddstr(i+4, 20, colors_get_pair("green", "black"), "[Logged in]");
  else
   xmvaddstr(i+4, 20, colors_get_pair("red", "black"), "[Not logged in]");

  g = g_list_next(g);
 }

 mvaddstr(i+4+2, 0, "Press any key.");

 (void) refresh();

 (void) input_wait_for_keypress(stdscr);

 display_main_screen();
}

/******************************************************************************
 Wait for a keypress while the mainloop is still running.
******************************************************************************/
chtype input_wait_for_keypress(WINDOW *win)
{
 chtype ch;
 GMainLoop *mainloop_wait_for_keypress = g_main_new(FALSE);

 /* Note: This function sets the window's getch() timeout to 1. 
    This is no problem, though, as we want it to be 1 in the game anyways... */

 (void) wtimeout(win, 1);
 (void) keypad(win, TRUE);

 while ((ch = wgetch(win)) == ERR)  /* Wait for keypress. */
  while (g_main_iteration(FALSE))
   ;

 g_main_quit(mainloop_wait_for_keypress);
 g_main_destroy(mainloop_wait_for_keypress);

 return ch;
}

