/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* file.c  --  File-handling routines. */


#include "common.h"
#include "log.h"

#include "file.h"


static gint file_mkdir(const gchar *dirname);

/******************************************************************************
 Create the given directory if it doesn't already exist.
******************************************************************************/
static gint file_mkdir(const gchar *dirname)
{
 if (!g_file_test(dirname, G_FILE_TEST_EXISTS))
 {
  if (mkdir(dirname, DIRMODE) < 0)
  {
   log_perror("file_mkdir:mkdir");
   return -1;
  }
  return 0;
 }

 if (!g_file_test(dirname, G_FILE_TEST_IS_DIR))
 {
  g_warning("Error. \"%s\" is not a directory.", dirname);
  return -1;
 }

 return 0;
}

/******************************************************************************
 Checks if 'filename' contains only characters, digits, dots and underscores.
******************************************************************************/
gboolean file_valid_name(const gchar *filename)
{
 size_t i;

 for (i=0; i<strlen(filename); i++)
 {
  if (!isalnum(filename[i]) && (filename[i] != '_') && (filename[i] != '.')
      && (filename[i] != '/'))
   return FALSE;
 }

 return TRUE;
}

/******************************************************************************
 Write a line of data to the given channel.
******************************************************************************/
gint file_write_line(GIOChannel *channel, const gchar *message)
{
 GIOError error;
 guint bytes_written;
 guint len = strlen(message);

file_write_line_again:
 error = g_io_channel_write(channel, message, len, &bytes_written);

 if ((error == G_IO_ERROR_AGAIN) || (bytes_written != strlen(message)))
 {
  len -= bytes_written;
  message += bytes_written;
  goto file_write_line_again;
 }

 if ((error == G_IO_ERROR_INVAL) || (error == G_IO_ERROR_UNKNOWN))
  return -1;

 return 0;
}

/******************************************************************************
 Read a line of data from the given channel.
******************************************************************************/
gchar *file_read_line(GIOChannel *channel)
{
 gchar *buf;
 guint bytes_read;
 guint count = 0;
 gchar c;
 GIOError error;
 gulong size = 40;

 buf = g_malloc(size);

 while (TRUE)
 {
  bytes_read = 0;

file_read_line_again:
  error = g_io_channel_read(channel, &c, 1, &bytes_read);

  if (error == G_IO_ERROR_AGAIN)
   goto file_read_line_again;

  if ((error == G_IO_ERROR_INVAL) || (error == G_IO_ERROR_UNKNOWN))
  {
   g_warning("file_read_line: g_io_channel_read() failed.");
   g_free(buf);
   return NULL;
  }

  if (bytes_read == 0) /* EOF. */
  {
   if (count == 0) /* We didn't read anything, yet. */
   {
    g_free(buf);
    return NULL;
   }
   else
    break;
  }

  buf[count] = c;
  count++;

  if (c == '\n')
  {
   buf[count] = '\0';
   break;
  }

  if (count+1 >= size)
  {
   size *= 2;
   buf = g_realloc(buf, size);
  }

 }

 return buf;
}

