/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* conf.c  --  Functions to handle configuration issues. */


#include "common.h"
#include "misc.h"
#include "defaults.h"
#include "log.h"
#include "colors.h"
#include "cmdline.h"

#include "conf.h"


enum
{
 SYMBOL_HOST             = G_TOKEN_LAST + 1,
 SYMBOL_PORT             = G_TOKEN_LAST + 2,
 SYMBOL_MUSIC            = G_TOKEN_LAST + 3,
 SYMBOL_SOUND            = G_TOKEN_LAST + 4,
 SYMBOL_COLOR            = G_TOKEN_LAST + 5,
 SYMBOL_LOGFILE          = G_TOKEN_LAST + 6,
 SYMBOL_DATADIR          = G_TOKEN_LAST + 7
};

static const struct
{
 const gchar *name;
 guint token;
} symbols[] =
{
 { "host",             SYMBOL_HOST },
 { "port",             SYMBOL_PORT },
 { "music",            SYMBOL_MUSIC },
 { "sound",            SYMBOL_SOUND },
 { "color",            SYMBOL_COLOR },
 { "logfile",          SYMBOL_LOGFILE },
 { "datadir",          SYMBOL_DATADIR },
 { NULL,               0 },
};

static const guint n_symbols = sizeof(symbols) / sizeof(symbols[0]);

static guint conf_parse_symbol(GScanner *scanner);
static gint conf_handle_colors(GScanner *scanner);

conf_t conf;


/******************************************************************************
 Configuration initialization.
******************************************************************************/
void conf_init(void)
{
 conf.host          = g_strdup(DEFAULT_HOST);
 conf.port          = DEFAULT_PORT;
 conf.music         = DEFAULT_MUSIC;
 conf.sound         = DEFAULT_SOUND;

 /* Initialize colors (?) */ 

 conf.configfile    = g_strdup(NHTSCLIENT_CONFIGFILE); /* !!! */
 conf.logfile       = g_strdup(NHTSCLIENT_LOGFILE);
 conf.datadir       = g_strdup(NHTSCLIENT_DATADIR);
}

/******************************************************************************
 Parse configuration file.
******************************************************************************/
gint conf_parse(void)
{
 guint i;
 gint fd;
 guint expected_token;
 GScanner *scanner = g_scanner_new(NULL);

 /* scanner->config->numbers_2_int = TRUE; */
 scanner->config->symbol_2_token = TRUE;

 if ((fd = open(conf.configfile, O_RDONLY)) < 0)
 {
  log_perror("conf_parse:open");
  g_scanner_destroy(scanner);
  return -1;
 }

 g_scanner_input_file(scanner, fd);
 scanner->input_name = "configfile";

 /* Load symbols into the scanner. */
 for (i = 0; i < n_symbols - 1; i++)
  g_scanner_scope_add_symbol(scanner, 0, symbols[i].name,
                             GINT_TO_POINTER(symbols[i].token));

 do
 {
  expected_token = conf_parse_symbol(scanner);
  g_scanner_peek_next_token(scanner);
 }
 while (expected_token == G_TOKEN_NONE && scanner->next_token != G_TOKEN_EOF
        && scanner->next_token != G_TOKEN_ERROR);

 if (expected_token != G_TOKEN_NONE)
 {
  g_scanner_unexp_token(scanner, expected_token, NULL, "symbol", NULL, NULL,
                        TRUE);
  g_scanner_destroy(scanner);
  return -1;
 }

 g_scanner_destroy(scanner);

 if (close(fd) < 0)
  log_perror("conf_parse:close");

 return 0;
}

/******************************************************************************
 ...
******************************************************************************/
static guint conf_parse_symbol(GScanner *scanner)
{
 guint symbol;

 /* Expect a valid symbol. */
 g_scanner_get_next_token(scanner);
 symbol = scanner->token;

 if (symbol < SYMBOL_HOST || symbol > SYMBOL_DATADIR)
  return G_TOKEN_SYMBOL;

 if (symbol == SYMBOL_COLOR)  /* Handle the 'color' keyword differently. */
 {
  (void) conf_handle_colors(scanner);
  return G_TOKEN_NONE;
 }

 /* Expect '='. */
 g_scanner_get_next_token(scanner);

 if (scanner->token != '=')
  return G_TOKEN_EQUAL_SIGN;

 g_scanner_get_next_token(scanner);

 switch (symbol)
 {
  case SYMBOL_HOST:
   if (cmdline.host == NO)
   {
    g_free(conf.host);
    conf.host = g_strdup(scanner->value.v_string);
   }
  break;

  case SYMBOL_PORT:
   if (cmdline.port == NO)
    conf.port = scanner->value.v_int;
  break;

  case SYMBOL_MUSIC:
   if (cmdline.music == NO)
   {
    if ((scanner->token != G_TOKEN_IDENTIFIER))
     return G_TOKEN_IDENTIFIER;

    if (g_strcasecmp(scanner->value.v_string, "no") == 0)
     conf.music = NO;
    else if (g_strcasecmp(scanner->value.v_string, "yes") == 0)
     conf.music = YES;
    else
     return G_TOKEN_IDENTIFIER;
   }
  break;

  case SYMBOL_SOUND:
   if (cmdline.sound == NO)
   {
    if ((scanner->token != G_TOKEN_IDENTIFIER))
     return G_TOKEN_IDENTIFIER;

    if (g_strcasecmp(scanner->value.v_string, "no") == 0)
     conf.sound = NO;
    else if (g_strcasecmp(scanner->value.v_string, "yes") == 0)
     conf.sound = YES;
    else
     return G_TOKEN_IDENTIFIER;
   }
  break;

  case SYMBOL_LOGFILE:
   if (cmdline.logfile == NO)
   {
    g_free(conf.logfile);
    conf.logfile = g_strdup(scanner->value.v_string);
   }
  break;

  case SYMBOL_DATADIR:
   if (cmdline.datadir == NO)
   {
    g_free(conf.datadir);
    conf.datadir = g_strdup(scanner->value.v_string);
   }
  break;
 }

 return G_TOKEN_NONE;
}

/******************************************************************************
 ...
******************************************************************************/
static gint conf_handle_colors(GScanner *scanner)
{
 const struct object_t
 {
  const gchar *name;
  chtype *color;
 } objects[] =
 {
  { "normal",              (chtype *)&colors.normal },
  { "statusbar1",          (chtype *)&colors.statusbar1 },
  { "statusbar2",          (chtype *)&colors.statusbar2 },
  { "about_box",           (chtype *)&colors.about_box },
  { "help_box",            (chtype *)&colors.help_box },
  { "dialog_connect",      (chtype *)&colors.dialog_connect },
  { "dialog_addplayer",    (chtype *)&colors.dialog_addplayer },
  { "dialog_login",        (chtype *)&colors.dialog_login },
  { "buy",                 (chtype *)&colors.buy },
  { "sell",                (chtype *)&colors.sell },
  { "terminal_background", (chtype *)&colors.terminal_background },
  { NULL,                  0 },
 };

 chtype *color;
 struct object_t *i;

 g_scanner_get_next_token(scanner); /* Get object to apply color to. */

 for (i = objects; i->name != NULL; i++)
 {
  if (g_strcasecmp(i->name, scanner->value.v_string) == 0)
  {
   color = i->color;
   break;
  }
 }

 /* TODO Output error if object is unknown. */

 g_scanner_get_next_token(scanner);  /* Get foreground color. */
 g_scanner_peek_next_token(scanner); /* Peek background color. */

 *color = colors_get_pair(scanner->value.v_string,
                          scanner->next_value.v_string);

 /* Get background color. This is only here to advance the scanner pointer. */
 g_scanner_get_next_token(scanner);

 return 0;
}

