/******************************************************************************
 nhtsclient - a ncurses-based client for the trading game Holsham Traders
 Copyright (C) 1999-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* cmdline.c  --  Commandline-options handling. */


#include "common.h"
#include "getopt.h"
#include "conf.h"
#include "file.h"
#include "misc.h"

#include "cmdline.h"


cmdline_t cmdline = {NO, NO, NO, NO, NO, NO, NO};


/******************************************************************************
 Parse the command-line arguments.
******************************************************************************/
void parse_arguments(gint argc, gchar *argv[])
{
 gint c;

 while (TRUE)
 {
  gint option_index = 0;

  static struct option long_options[] =
  {
   {"host",       required_argument, NULL, (gint)'H'},
   {"port",       required_argument, NULL, (gint)'p'},
   {"configfile", required_argument, NULL, (gint)'c'},
   {"logfile",    required_argument, NULL, (gint)'l'},
   {"datadir",    required_argument, NULL, (gint)'d'},
   {"nomusic",    no_argument,       NULL, (gint)'m'},
   {"nosound",    no_argument,       NULL, (gint)'s'},
   {"help",       no_argument,       NULL, (gint)'h'},
   {"version",    no_argument,       NULL, (gint)'V'},
   {NULL,         0,                 NULL, 0}
  };
 
  c = getopt_long(argc, argv, "H:p:c:l:d:mshV", long_options, &option_index);

  if (c == -1) /* End of the options. */
   break;

  switch (c)
  {
   /**************************************************************************/
   case 'H':  /* Host. */
    g_free(conf.host);
    conf.host = g_strdup(optarg);
    cmdline.host = YES;
   break;

   /**************************************************************************/
   case 'p':  /* Port. */
    if (!is_number_or_digit(optarg))
    {
     g_printerr("Invalid port.\n");
     exit(EXIT_FAILURE);
    }

    conf.port = (gushort)strtol(optarg, NULL, 0);
    cmdline.port = YES;
   break;

   /**************************************************************************/
   case 'c':  /* Configfile. */
    if (!file_valid_name(optarg))
    {
     g_printerr("Invalid filename '%s'. Aborting.\n", optarg);
     exit(EXIT_FAILURE);
    }

    /* TODO Check whether optarg is a file. */

    g_free(conf.configfile);
    conf.configfile = g_strdup(optarg);
    cmdline.configfile = YES;
   break;

   /**************************************************************************/
   case 'l':  /* Logfile. */
    if (!file_valid_name(optarg))
    {
     g_printerr("Invalid filename '%s'. Aborting.\n", optarg);
     exit(EXIT_FAILURE);
    }

    /* TODO Check whether optarg is a file. */

    g_free(conf.logfile);
    conf.logfile = g_strdup(optarg);
    cmdline.logfile = YES;
   break;

   /**************************************************************************/
   case 'd':  /* Datadir. */
    if (!file_valid_name(optarg))
    {
     g_printerr("Invalid directory-name '%s'. Aborting.\n", optarg);
     exit(EXIT_FAILURE);
    }

    /* TODO Check whether optarg is a directory. */

    g_free(conf.datadir);
    conf.datadir = g_strdup(optarg);
    cmdline.datadir = YES;
   break;

   /**************************************************************************/
   case 'm': /* Don't play music. */
    conf.music = NO;
    cmdline.music = YES;
   break;

   /**************************************************************************/
   case 's': /* Don't output sound. */
    conf.sound = NO;
    cmdline.sound = YES;
   break;

   /**************************************************************************/
   case 'h': /* Print a help message. */
    g_print("%s", USAGE);
    exit(EXIT_SUCCESS);
   break;

   /**************************************************************************/
   case 'V': /* Print version. */
    g_print("%s %s\n", PACKAGE, VERSION);
    exit(EXIT_SUCCESS);
   break;

   /**************************************************************************/
   case '?': /* An ambiguous match or an extraneous parameter. */
    exit(EXIT_FAILURE);
   break;
  }

 }
}

