/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* transporters.c  --  Functions to handle transporters. */


#include "common.h"
#include "players.h"
#include "towns.h"
#include "goods.h"

#include "transporters.h"


static guint transporter_get_next_free_id(player_t *player);

/******************************************************************************
 Create a new transporter.
******************************************************************************/
transporter_t *transporter_new(guint id, const gchar *name, glong money,
                               glong capacity, guint location)
{
 guint i;
 good_t *good;
 transporter_t *transporter = g_malloc(sizeof(transporter_t));

 transporter->id = id;
 g_snprintf(transporter->name, MAX_LEN_TRANSPORTERNAME+1, "%s", name);
 transporter->money = money;
 transporter->capacity = capacity;
 transporter->location = location;
 transporter->goods = NULL;

 for (i=0; i<number_of_goods; i++)
 {
  good = good_new(i, 0, 0);
  transporter_add_good(transporter, good);
 }

 return transporter;
}

/******************************************************************************
 Free the memory used by a transporter.
******************************************************************************/
void transporter_free(transporter_t *transporter)
{
 g_free(transporter);
}

/******************************************************************************
 Free the memory used by all transporters in the list.
******************************************************************************/
void transporter_free_all(GList *transporters)
{
 GList *g;
 transporter_t *transporter;

 if ((g = g_list_first(transporters)) == NULL)
 {
  g_list_free(transporters);
  return;
 }

 do
 {
  transporter = g->data;
  transporter_free(transporter);
 } while ((g = g_list_next(g)) != NULL);

 g_list_free(transporters);
}

/******************************************************************************
 Create the transporters a player has got in the beginning of the game.
******************************************************************************/
GList *transporter_at_beginning_of_game(void)
{
 GList *transporters = NULL;
 transporter_t *transporter;

 transporter = transporter_new(0, "Transporter 1", 2000, 40, HOME_TOWN);
 transporters = g_list_append(transporters, transporter);

 transporter = transporter_new(1, "Transporter 2", 1400, 19, HOME_TOWN);
 transporters = g_list_append(transporters, transporter);

 return transporters;
}

/******************************************************************************
 Find the transporter with the given ID.
******************************************************************************/
transporter_t *transporter_find_by_id(player_t *player, guint id)
{
 GList *g;
 transporter_t *transporter;

 if ((g = g_list_first(player->transporters)) == NULL)
  return NULL; /* Transporter not found. */

 do
 {
  transporter = g->data;

  if (transporter->id == id)
   return transporter;

 } while ((g = g_list_next(g)) != NULL);

 return NULL; /* Transporter not found. */
}

/******************************************************************************
 Drive the transporter to the given destination town.
******************************************************************************/
gint transporter_drive(player_t *player, guint id, guint destination)
{
 transporter_t *transporter;

 if ((transporter = transporter_find_by_id(player, id)) == NULL)
  return -1; /* Transporter not found. Driving aborted. */

 transporter->location = destination;

 return 0;
}

/******************************************************************************
 Add a good to a transporter.
******************************************************************************/
void transporter_add_good(transporter_t *transporter, good_t *good)
{
 transporter->goods = g_list_append(transporter->goods, good);
}

/******************************************************************************
 Get the free capacity of a transporter.
******************************************************************************/
glong transporter_free_capacity(transporter_t *transporter)
{
 GList *g;
 good_t *good;
 glong free_capacity = transporter->capacity;

 if ((g = g_list_first(transporter->goods)) == NULL)
  return free_capacity; /* No goods in transporter. */

 do
 {
  good = g->data;
  free_capacity -= good->amount;
 } while ((g = g_list_next(g)) != NULL);

 return free_capacity;
}

/******************************************************************************
 Set the name of the transporter.
******************************************************************************/
void transporter_set_name(transporter_t *transporter, const gchar *name)
{
 g_snprintf(transporter->name, MAX_LEN_TRANSPORTERNAME+1, "%s", name);
}

/******************************************************************************
 Return the next free ID for a transporter of a certain player.
******************************************************************************/
static guint transporter_get_next_free_id(player_t *player)
{
 GList *g;
 guint id = 0;
 transporter_t *transporter;

 if ((g = g_list_first(player->transporters)) == NULL)
  return 0; /* No transporters found. */

 do
 {
  transporter = g->data;

  if (transporter->id > id)
   id = transporter->id;

 } while ((g = g_list_next(player->transporters)) != NULL);

 return id + 1;
}

