/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* towns.c  --  Functions to handle towns. */


#include "common.h"
#include "goods.h"
#include "misc.h"
#include "log.h"
#include "file.h"
#include "conf.h"

#include "towns.h"


GList *towns = NULL;

/******************************************************************************
 Create a new town.
******************************************************************************/
town_t *town_new(guint id, const gchar *name, glong money, gulong inhabitants)
{
 guint i;
 good_t *good;
 town_t *town = g_malloc(sizeof(town_t));

 town->id = id;
 g_snprintf(town->name, MAX_LEN_TOWNNAME+1, "%s", name);
 town->money = money;
 town->inhabitants = inhabitants;
 town->goods = NULL;

 for (i=0; i<number_of_goods; i++)
 {
  good = good_new(i, 0, 0);
  town_add_good(town, good);
 }

 return town;
}

/******************************************************************************
 Create all towns listed in townnames.dat.
******************************************************************************/
gint town_new_all(void)
{
 gint fd;
 guint i;
 gchar *line;
 GIOChannel *channel;
 guint number_of_towns;
 town_t *town;
 gchar *datafile = g_strdup_printf("%s/townnames.dat", conf.datadir);

 if ((fd = open(datafile, O_RDONLY)) < 0)
 {
  log_perror("town_new_all:open");
  g_free(datafile);
  return -1;
 }

 g_free(datafile);

 channel = g_io_channel_unix_new(fd);

 if ((line = file_read_line(channel)) == NULL)
 {
  g_warning("town_new_all: file_read_line() failed.");
  return -1;
 }

 if (!is_number_or_digit(g_strchomp(line)))
 {
  g_warning("Corrupt townnames.dat.");
  g_free(line);
  return -1;
 }

 number_of_towns = (guint)strtol(line, NULL, 0);
 g_free(line);


 /* Read all the townnames, one per line. */
 for (i=0; i<number_of_towns; i++)
 {
  if ((line = file_read_line(channel)) == NULL)
  {
   g_warning("town_new_all: file_read_line() failed.");
   return -1;
  }

  town = town_new(i, g_strchomp(line),
                  (glong)g_random_int_range(50000, 300000),
                  (gulong)g_random_int_range(1000, 2000000));
  town_add(town);

  g_free(line);
 }

 g_io_channel_close(channel);
 g_io_channel_unref(channel);

 return 0;
}

/******************************************************************************
 Free the memory used by a town.
******************************************************************************/
void town_free(town_t *town)
{
 g_free(town);
}

/******************************************************************************
 Free the memory used by all towns.
******************************************************************************/
void town_free_all(void)
{
 GList *g;
 town_t *town;

 if ((g = g_list_first(towns)) == NULL)
 {
  g_list_free(towns);
  return;
 }

 do
 {
  town = g->data;
  town_free(town);
 } while ((g = g_list_next(g)) != NULL);

 g_list_free(towns);
}

/******************************************************************************
 Find the town with the given ID.
******************************************************************************/
town_t *town_find_by_id(guint id)
{
 GList *g;
 town_t *town;

 if ((g = g_list_first(towns)) == NULL)
  return NULL; /* Town not found. */

 do
 {
  town = g->data;

  if (town->id == id)
   return town;

 } while ((g = g_list_next(g)) != NULL);

 return NULL; /* Town not found. */
}

/******************************************************************************
 Add a town to the list of towns.
******************************************************************************/
void town_add(town_t *town)
{
 towns = g_list_append(towns, town);
}

/******************************************************************************
 Add a good to a town.
******************************************************************************/
void town_add_good(town_t *town, good_t *good)
{
 town->goods = g_list_append(town->goods, good);
}

/******************************************************************************
 Add the goods which are in the towns in the beginning of the game.
******************************************************************************/
void town_add_goods_at_beginning_of_game(void)
{
 guint i;
 GList *g; 
 town_t *town;
 good_t *good;

 if ((g = g_list_first(towns)) == NULL)  /* Should never happen. */
  return;

 do
 {
  town = g->data;

  for (i=0; i<number_of_goods; i++)
  {
   good = good_find_by_id(town->goods, i);

   /* Only about 1/4th of all goods are available in each town. */
   if (g_random_int_range(0, 100) < 25)
    good->amount = (glong)g_random_int_range(0, 900);
   else
    good->amount = 0;

   good->price = (glong)g_random_int_range(5, 150);
  }

 } while ((g = g_list_next(g)) != NULL);

}

/******************************************************************************
 Return a list of towns, containing towns where a certain player has stationed
 one or more transporters.
******************************************************************************/
GList *town_where_player_has_transporters(player_t *player)
{
 GList *g; 
 town_t *town;
 GList *town_list = NULL;
 transporter_t *transporter;

 if ((g = g_list_first(player->transporters)) == NULL)
  return NULL; /* No transporter found. */

 do
 {
  transporter = g->data;

  town = town_find_by_id(transporter->location); /* Should not fail. */

  if (g_list_find(town_list, town) == NULL) /* Prevent duplicate entries. */
   town_list = g_list_append(town_list, town);

 } while ((g = g_list_next(g)) != NULL);

 return town_list;
}

