/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* servercmds.c  --  Commands sent by the server. */


#include "common.h"
#include "connections.h"
#include "goods.h"
#include "net.h"
#include "towns.h"
#include "players.h"
#include "log.h"

#include "servercmds.h"


/******************************************************************************
 Send all data a client needs at login-time.
******************************************************************************/
void send_init_data(connection_t *connection)
{
 GString *init_string = g_string_new(NULL);
 guint i, j, number_of_transporters, number_of_towns, number_of_players;
 guint number_of_goods_in_transporter, number_of_goods_in_town;
 transporter_t *transporter;
 good_t *good;
 town_t *town;
 player_t *player;
 GList *g, *g2, *town_list, *player_list;


 /*****************************************************************************
  Player ID.
 *****************************************************************************/
 g_string_sprintfa(init_string, "260 %u ", connection->pl->id);

 /*****************************************************************************
  Goods.
 *****************************************************************************/
 g_string_sprintfa(init_string, "%u ", number_of_goods);

 for (i=0; i<number_of_goods; i++)
  g_string_sprintfa(init_string, "%u \"%s\" ", i, goodnames[i]);

 /*****************************************************************************
  Towns.
 *****************************************************************************/
 number_of_towns = g_list_length(towns);
 g_string_sprintfa(init_string, "%u ", number_of_towns);

 g = g_list_first(towns);

 for (i=0; i<number_of_towns; i++)
 {
  town = g->data;

  g_string_sprintfa(init_string, "%u \"%s\" ", town->id, town->name);

  g = g_list_next(g); 
 }

 /*****************************************************************************
  Players.
 *****************************************************************************/
 player_list = player_all_except_this_one(player);
 number_of_players = g_list_length(player_list);
 g_string_sprintfa(init_string, "%u ", number_of_players);

 g = g_list_first(player_list);

 for (i=0; i<number_of_players; i++)
 {
  player = g->data;

  g_string_sprintfa(init_string, "%u \"%s\" %u ", player->id, player->name,
                    player->logged_in);

  g = g_list_next(g); 
 }

 /*****************************************************************************
  Player age and money.
 *****************************************************************************/
 g_string_sprintfa(init_string, "%u %ld ", connection->pl->age,
                   connection->pl->money);

 /*****************************************************************************
  Transporters.
 *****************************************************************************/
 number_of_transporters = g_list_length(connection->pl->transporters);
 g_string_sprintfa(init_string, "%u ", number_of_transporters);

 g = g_list_first(connection->pl->transporters);

 for (i=0; i<number_of_transporters; i++)
 {
  transporter = g->data;

  g_string_sprintfa(init_string, "%u \"%s\" %ld %lu %u ", transporter->id,
                    transporter->name, transporter->money,
                    transporter->capacity, transporter->location);

  /****************************************************************************
   Goods in transporter.
  ****************************************************************************/
  number_of_goods_in_transporter = g_list_length(transporter->goods);
  g_string_sprintfa(init_string, "%u ", number_of_goods_in_transporter);

  g2 = g_list_first(transporter->goods);

  for (j=0; j<number_of_goods_in_transporter; j++)
  {
   good = g2->data;
   g_string_sprintfa(init_string, "%u %lu ", good->id, good->amount);
   g2 = g_list_next(g2);
  }

  g = g_list_next(g); 
 }

 /*****************************************************************************
  Towns where the player has stationed one or more transporters.
 *****************************************************************************/
 town_list = town_where_player_has_transporters(connection->pl);
 number_of_towns = g_list_length(town_list);
 g_string_sprintfa(init_string, "%u ", number_of_towns);

 g = g_list_first(town_list);

 for (i=0; i<number_of_towns; i++)
 {
  town = g->data;

  g_string_sprintfa(init_string, "%u %ld %lu ", town->id, town->money,
                    town->inhabitants);

  /****************************************************************************
   Goods in town.
  ****************************************************************************/
  number_of_goods_in_town = g_list_length(town->goods);
  g_string_sprintfa(init_string, "%u ", number_of_goods_in_town);

  g2 = g_list_first(town->goods);

  for (j=0; j<number_of_goods_in_town; j++)
  {
   good = g2->data;

   g_string_sprintfa(init_string, "%u %lu %lu ", good->id, good->amount,
                     good->price);

   g2 = g_list_next(g2);
  }
 
  g = g_list_next(g); 
 }


 g_string_truncate(init_string, init_string->len-1); /* Remove last comma. */

 net_send_to_client(connection, "160 %u", init_string->len+1);
 net_send_to_client(connection, init_string->str);

 g_string_free(init_string, TRUE);
}

/******************************************************************************
 Tell all other clients who have a transporter in the given town, that the
 amount of a certain good in the town has changed.
******************************************************************************/
void send_good_amount_update(player_t *player, town_t *town, good_t *good)
{
 net_broadcast_to_players(player_with_transporters_in_same_town(player, town),
                          "good_amount_update %u %u %lu", town->id, good->id,
                          good->amount);
}

/******************************************************************************
 Tell all other clients who have a transporter in the given town, that the
 price of a certain good in the town has changed.
******************************************************************************/
void send_good_price_update(player_t *player, town_t *town, good_t *good)
{
 net_broadcast_to_players(player_with_transporters_in_same_town(player, town),
                          "good_price_update %u %u %lu", town->id, good->id,
                          good->price);
}

/******************************************************************************
 Send town information every time a transporter of a player arrives in
 the given town.
******************************************************************************/
void send_town_info(connection_t *connection, town_t *town)
{
 GString *init_string = g_string_new(NULL);
 guint i, number_of_goods_in_town;
 GList *g;
 good_t *good;

 /****************************************************************************
  Town ID, money and inhabitants.
 ****************************************************************************/
 g_string_sprintf(init_string, "260 %u %ld %lu ", town->id, town->money,
                  town->inhabitants);

 /****************************************************************************
  Goods in town.
 ****************************************************************************/
 number_of_goods_in_town = g_list_length(town->goods);
 g_string_sprintfa(init_string, "%u ", number_of_goods_in_town);

 g = g_list_first(town->goods);

 for (i=0; i<number_of_goods_in_town; i++)
 {
  good = g->data;
  g_string_sprintfa(init_string, "%u %ld %ld ", good->id, good->amount,
                    good->price);
  g = g_list_next(g);
 }

 /* Remove last comma. */
 g_string_truncate(init_string, (guint)init_string->len-1);

 net_send_to_client(connection, "160 %u", init_string->len+1);
 net_send_to_client(connection, init_string->str);

 g_string_free(init_string, TRUE);
}

/******************************************************************************
 Send info to all other players that a new player has been created.
******************************************************************************/
void send_player_new(player_t *player)
{
 net_broadcast_to_players(player_all_except_this_one(player),
                          "player_new %u \"%s\"", player->id,
                          player->name);
}

/******************************************************************************
 Send info to all other players that the given player has been removed.
******************************************************************************/
void send_player_remove(player_t *player)
{
 net_broadcast_to_players(player_all_except_this_one(player),
                          "player_remove %u", player->id);
}

/******************************************************************************
 Send info to all other players that a certain player has logged in.
******************************************************************************/
void send_player_logged_in(player_t *player)
{
 net_broadcast_to_players(player_all_except_this_one(player),
                          "player_logged_in %u", player->id);
}

/******************************************************************************
 Send info to all other players that a certain player has logged out.
******************************************************************************/
void send_player_logged_out(player_t *player)
{
 if (player != NULL)
 {
  net_broadcast_to_players(player_all_except_this_one(player),
                           "player_logged_out %u", player->id);
 }
}

/******************************************************************************
 Send info that the game is paused now.
******************************************************************************/
void send_pause(player_t *player)
{
 net_broadcast_to_players(player_all_except_this_one(player), "pause");
}

/******************************************************************************
 Send info that the game continues.
******************************************************************************/
void send_continue(player_t *player)
{
 net_broadcast_to_players(player_all_except_this_one(player), "continue");
}

/******************************************************************************
 Send info to all other connections that the server will be shut down.
******************************************************************************/
void send_shutdown(connection_t *connection)
{
 net_broadcast_to_connections(connection_all_except_this_one(connection),
                              "shutdown");
}

/******************************************************************************
 Send info that the connection will be closed because the login-timeout of
 the client has expired.
******************************************************************************/
void send_login_timeout(connection_t *connection)
{
 net_send_to_client(connection, "login_timeout"); 
}

/******************************************************************************
 Send a chat message to one or more players.
******************************************************************************/
void send_chat_message(guint sender, GList *player_list, const gchar *message)
{
 net_broadcast_to_players(player_list, "chat %u \"%s\"", sender, message);
}

/******************************************************************************
 Send a message to all players who have a transporter in the given town, that
 the given transporter of the given player arrived there.
******************************************************************************/
void send_transporter_arrived(player_t *player, town_t *town,
                              guint transporter_id)
{
 GList *player_list = player_with_transporters_in_same_town(player, town);

 net_broadcast_to_players(player_list, "transporter_arrived %u %u %u",
                          transporter_id, player->id, town->id);
}

/******************************************************************************
 Send a message to all players who have a transporter in the given town, that
 the given transporter of the given player has left that town.
******************************************************************************/
void send_transporter_left(player_t *player, town_t *town,
                           guint transporter_id)
{
 GList *player_list = player_with_transporters_in_same_town(player, town);

 net_broadcast_to_players(player_list, "transporter_left %u %u %u",
                          transporter_id, player->id, town->id);
}

