/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* protocol.c  --  htsprotocol implementation. */


#include "common.h"
#include "log.h"
#include "connections.h"
#include "misc.h"
#include "file.h"
#include "conf.h"
#include "players.h"
#include "savegame.h"
#include "net.h"
#include "servercmds.h"
#include "transporters.h"
#include "towns.h"
#include "trade.h"

#include "protocol.h"


#define PROTOCOL_HELP "\
htsserver "VERSION" -- Copyright (C) 1998-2001 Uwe Hermann\n\
\n\
Available commands:\n\
\n\
addplayer [playername] [password]        help\n\
delplayer [id|playername] [password]     quit\n\
login [playername] [password]            chat [id|ALL] [message]\n\
logout                                   transporter_drive [id] [destination]\n\
adminlogin [adminpassword]               transporter_set_name [id] [name]\n\
adminlogout                              reconfig\n\
save [name]                              togglepause\n\
load [name]                              shutdown\n\
passwd [id] [newpassword]                \n\
version                                  \n\
protocol                                 \n\
\n\
buy_from_town_to_transporter [id] [what] [amount]\n\
sell_from_transporter_to_town [id] [what] [amount]"


#define PROTOCOL_VERSION "0.4.6"  /* htsprotocol version. */


/* Command handler functions. */
static void h_addplayer(connection_t *connection, GPtrArray *arguments);
static void h_delplayer(connection_t *connection, GPtrArray *arguments);

static void h_login(connection_t *connection, GPtrArray *arguments);
static void h_logout(connection_t *connection, GPtrArray *arguments);

static void h_adminlogin(connection_t *connection, GPtrArray *arguments);
static void h_adminlogout(connection_t *connection, GPtrArray *arguments);

static void h_save(connection_t *connection, GPtrArray *arguments);
static void h_load(connection_t *connection, GPtrArray *arguments);

static void h_buy_from_town_to_transporter(connection_t *connection,
                                           GPtrArray *arguments);
static void h_sell_from_transporter_to_town(connection_t *connection,
                                            GPtrArray *arguments);

static void h_passwd(connection_t *connection, GPtrArray *arguments);
static void h_version(connection_t *connection, GPtrArray *arguments);
static void h_protocol(connection_t *connection, GPtrArray *arguments);
static void h_help(connection_t *connection, GPtrArray *arguments);
static void h_quit(connection_t *connection, GPtrArray *arguments);
static void h_chat(connection_t *connection, GPtrArray *arguments);
static void h_transporter_drive(connection_t *connection,
                                GPtrArray *arguments);
static void h_transporter_set_name(connection_t *connection,
                                   GPtrArray *arguments);
static void h_reconfig(connection_t *connection, GPtrArray *arguments);
static void h_togglepause(connection_t *connection, GPtrArray *arguments);
static void h_shutdown(connection_t *connection, GPtrArray *arguments);


typedef struct command_t
{
 const gchar *name;
 guint argc;
 void (*handler)(connection_t *connection, GPtrArray *arguments);
} command_t;


/* This array holds all available commands, their number of arguments and the
   handler function to be executed upon receipt of the command. */
static command_t commands[] =
{
 { "addplayer",                     2, h_addplayer },
 { "delplayer",                     2, h_delplayer },

 { "login",                         2, h_login },
 { "logout",                        0, h_logout },

 { "adminlogin",                    1, h_adminlogin },
 { "adminlogout",                   0, h_adminlogout },

 { "save",                          1, h_save },
 { "load",                          1, h_load },

 { "buy_from_town_to_transporter",  3, h_buy_from_town_to_transporter },
 { "sell_from_transporter_to_town", 3, h_sell_from_transporter_to_town },

 { "passwd",                        2, h_passwd },
 { "version",                       0, h_version },
 { "protocol",                      0, h_protocol },
 { "help",                          0, h_help },
 { "quit",                          0, h_quit },
 { "chat",                          2, h_chat },
 { "transporter_drive",             2, h_transporter_drive },
 { "transporter_set_name",          2, h_transporter_set_name },
 { "reconfig",                      0, h_reconfig },
 { "togglepause",                   0, h_togglepause },
 { "shutdown",                      0, h_shutdown },

 { NULL,                            0, NULL }
};


/******************************************************************************
 Parse the message from the given connection and take the necessary actions.
******************************************************************************/
gint protocol_parse_message(connection_t *connection, const gchar *message)
{
 command_t *cmds;
 gchar *command;
 GPtrArray *ptr_array;

 if ((ptr_array = tokenize(message)) == NULL)
 {
  g_warning("protocol_parse_message: tokenize() failed.");
  return -1;  
 }

 if ((command = (gchar *)g_ptr_array_index(ptr_array, 0)) == NULL)
 {
  (void) g_ptr_array_free(ptr_array, TRUE);
  g_warning("protocol_parse_message: Command is NULL.");
  return -1;
 }

 /* Remove the command from the pointer array. Only the arguments remain. */
 command = g_ptr_array_remove_index(ptr_array, 0);

 cmds = commands;
 while (cmds->name != NULL)
 {
  if (g_strcasecmp(cmds->name, command) == 0)
  {
   if (ptr_array->len == cmds->argc)
    cmds->handler(connection, ptr_array);
   else
    net_send_to_client(connection, "500 Syntax error.");

   break;
  }

  cmds++;
 }

 if (cmds->name == NULL)
  net_send_to_client(connection, "500 Unknown command.");

 g_free(command);
 (void) g_ptr_array_free(ptr_array, TRUE);

 return 0;
}

/******************************************************************************
 Create a new player.
******************************************************************************/
static void h_addplayer(connection_t *connection, GPtrArray *arguments)
{
 const gchar *playername = (const gchar *)g_ptr_array_index(arguments, 0);
 const gchar *password = (const gchar *)g_ptr_array_index(arguments, 1);
 player_t *player;

 if (!player_has_valid_name(playername))
 {
  net_send_to_client(connection, "500 Invalid playername.");
  return;
 }
 
 /* TODO Check for valid password. */

 if (player_exists(playername))
 {
  net_send_to_client(connection, "532 Player already exists.");
  return;
 }

 player = player_new(playername, password, NO, START_AGE, START_MONEY,
                     transporter_at_beginning_of_game(), NULL);
 player_add(player);

 net_send_to_client(connection, "200 New player successfully created.");

 send_player_new(player); 
}

/******************************************************************************
 Login to the server.
******************************************************************************/
static void h_login(connection_t *connection, GPtrArray *arguments)
{
 player_t *player;
 connection_t *otherconn;
 const gchar *playername = (const gchar *)g_ptr_array_index(arguments, 0);
 const gchar *password = (const gchar *)g_ptr_array_index(arguments, 1);


 if (logged_in(connection))
 {
  net_send_to_client(connection,
                     "535 You are already logged in as \"%s\" (ID: %u).",
                     connection->pl->name, connection->pl->id);
  return;
 }

 if ((player = player_find_by_name(playername)) == NULL)
 {
  net_send_to_client(connection, "533 Player not found.");
  return;
 }

 if ((otherconn = connection_find_by_player(player)) != NULL)
 {
  net_send_to_client(connection, "535 \"%s\" (ID: %u) is already logged in.",
                     player->name, player->id);
  return;
 }

 if (player_has_valid_password(player, password) == NO)
 {
  net_send_to_client(connection, "534 Wrong password.");
  return;
 }

 connection->pl = player;

 player->logged_in = YES;

 send_init_data(connection); /* Login successful. */
 send_player_logged_in(player); 
}

/******************************************************************************
 Gain admin permissions.
******************************************************************************/
static void h_adminlogin(connection_t *connection, GPtrArray *arguments)
{
 const gchar *password = (const gchar *)g_ptr_array_index(arguments, 0);

 if (logged_in_as_admin(connection))
 {
  net_send_to_client(connection, "536 You already have admin permissions.");
  return;
 }

 if (strncmp(password, conf.adminpassword, MAX_LEN_ADMINPASSWORD) == 0)
 {
  connection->is_admin = YES;
  net_send_to_client(connection, "200 Admin permissions granted.");
 }
 else
 {
  connection->is_admin = NO;
  net_send_to_client(connection, "534 Wrong password.");
 }
}

/******************************************************************************
 Change the password of a player.
******************************************************************************/
static void h_passwd(connection_t *connection, GPtrArray *arguments)
{
 guint player_id = (guint)strtol((const gchar *)g_ptr_array_index(arguments, 0), NULL, 0);
 const gchar *password = (const gchar *)g_ptr_array_index(arguments, 1);
 player_t *player;

 
 if (!is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 0)))
 {
  net_send_to_client(connection, "500 Syntax error.");
  return;
 }

 if ((player = player_find_by_id(player_id)) == NULL)
 {
  net_send_to_client(connection, "533 Player not found.");
  return;
 }

 /* TODO Check whether the password is valid. */

 if (logged_in_as_admin(connection))
 {
  g_snprintf(player->password, MAX_LEN_PLAYERPASSWORD+1, "%s", password);
  net_send_to_client(connection, "200 Password successfully changed.");
  return;
 }

 if (!logged_in(connection))
 {
  net_send_to_client(connection, "530 Not logged in.");
  return;
 }

 /* Normal users can only change their own password. */
 if (connection->pl != player)
 {
  net_send_to_client(connection, "531 You do not have admin permissions.");
  return;
 }

 g_snprintf(player->password, MAX_LEN_PLAYERPASSWORD+1, "%s", password);

 net_send_to_client(connection, "200 Password successfully changed.");
}

/******************************************************************************
 Sends a message to the destination player. If the destination player is 'ALL',
 the message is sent to all players.
******************************************************************************/
static void h_chat(connection_t *connection, GPtrArray *arguments)
{
 player_t *player;
 GList *player_list = NULL;
 connection_t *otherconn;
 const gchar *receiver = (const gchar *)g_ptr_array_index(arguments, 0);
 const gchar *message = (const gchar *)g_ptr_array_index(arguments, 1);
 guint receiver_id = (guint)strtol(receiver, NULL, 0);

 
 if (!logged_in(connection))
 {
  net_send_to_client(connection, "530 Not logged in.");
  return;
 }

 if (g_strcasecmp(receiver, "ALL") == 0)
 { 
  send_chat_message(connection->pl->id, players, message);
  return;
 }

 if (!is_number_or_digit(receiver))
 {
  net_send_to_client(connection, "500 Syntax error.");
  return;
 }
 
 if ((player = player_find_by_id(receiver_id)) == NULL)
 {
  net_send_to_client(connection, "533 Player not found.");
  return;
 }

 if ((otherconn = connection_find_by_player(player)) == NULL)
 {
  net_send_to_client(connection, "520 Player not connected.");
  return;
 }

 player_list = g_list_append(player_list, player);

 send_chat_message(connection->pl->id, player_list, message);

 g_list_free(player_list);
}

/******************************************************************************
 Send the htsserver version.
******************************************************************************/
static void h_version(connection_t *connection, GPtrArray *arguments)
{
 net_send_to_client(connection, "210 %s", VERSION);
}

/******************************************************************************
 Send the htsprotocol version.
******************************************************************************/
static void h_protocol(connection_t *connection, GPtrArray *arguments)
{
 net_send_to_client(connection, "210 %s", PROTOCOL_VERSION);
}

/******************************************************************************
 Drive the given transporter to the destination town.
******************************************************************************/
static void h_transporter_drive(connection_t *connection, GPtrArray *arguments)
{
 guint id = (guint)strtol((const gchar *)g_ptr_array_index(arguments, 0), NULL, 0);
 guint destination = (gint)strtol((const gchar *)g_ptr_array_index(arguments, 1), NULL, 0);
 town_t *town;
 transporter_t *transporter;


 if (!is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 0)) || !is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 1)))
 {
  net_send_to_client(connection, "500 Syntax error.");
  return;
 }

 if (!logged_in(connection))
 {
  net_send_to_client(connection, "530 Not logged in.");
  return;
 }

 if ((transporter = transporter_find_by_id(connection->pl, id)) == NULL)
 {
  net_send_to_client(connection, "551 Transporter doesn't exist.");
  return;
 }

 if ((town = town_find_by_id(destination)) == NULL)
 {
  net_send_to_client(connection, "550 Destination town does not exist.");
  return;
 }

 if (transporter->location == destination)
 {
  net_send_to_client(connection, "556 Transporter is already in that town.");
  return;
 }


 (void) transporter_drive(connection->pl, id, destination);

 send_town_info(connection, town); /* Driving successful. */

 send_transporter_arrived(connection->pl, town, id);
}

/******************************************************************************
 Buy goods from the town where the given transporter is located.
******************************************************************************/
static void h_buy_from_town_to_transporter(connection_t *connection,
                                           GPtrArray *arguments)
{
 town_t *town;
 transporter_t *transporter;

 guint transporter_id = (guint)strtol((const gchar *)g_ptr_array_index(arguments, 0), NULL, 0);
 guint what = (guint)strtol((const gchar *)g_ptr_array_index(arguments, 1), NULL, 0);
 glong amount = (glong)strtol((const gchar *)g_ptr_array_index(arguments, 2), NULL, 0);


 if (!is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 0)) || !is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 1))
     || !is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 2)))
 {
  net_send_to_client(connection, "500 Syntax error.");
  return;
 }

 if (!logged_in(connection))
 {
  net_send_to_client(connection, "530 Not logged in.");
  return;
 }

 if ((transporter = transporter_find_by_id(connection->pl, transporter_id)) == NULL)
 {
  net_send_to_client(connection, "551 Transporter doesn't exist.");
  return;
 }

 if ((town = town_find_by_id(transporter->location)) == NULL)
 {
  net_send_to_client(connection, "550 Town doesn't exist.");
  return;
 }

 switch (trade_buy_from_town_to_transporter(transporter, what, amount))
 {
  case 0:
   {
    good_t *good = good_find_by_id(town->goods, what);

    net_send_to_client(connection, "200 Buying successful.");

    send_good_amount_update(connection->pl, town, good);
    send_good_price_update(connection->pl, town, good);

    return;
   }
  break;

  case -1:
   net_send_to_client(connection, "552 Good not available in town.");
   return;
  break;

  case -2:
   net_send_to_client(connection, "553 Not enough goods in town.");
   return;
  break;

  case -3:
   net_send_to_client(connection, 
                      "554 Transporter doesn't have enough free capacity.");
   return;
  break;

  case -4:
   net_send_to_client(connection,
                      "555 Transporter doesn't have enough money.");
   return;
  break;

  /* default:
   net_send_to_client(connection, "501 Buying failed.");
   return;
  break; */
 }
}

/******************************************************************************
 Sell goods from a transporter to the town where the transporter is located.
******************************************************************************/
static void h_sell_from_transporter_to_town(connection_t *connection,
                                            GPtrArray *arguments)
{
 town_t *town;
 transporter_t *transporter;

 guint transporter_id = (guint)strtol((const gchar *)g_ptr_array_index(arguments, 0), NULL, 0);
 guint what = (guint)strtol((const gchar *)g_ptr_array_index(arguments, 1), NULL, 0);
 glong amount = (glong)strtol((const gchar *)g_ptr_array_index(arguments, 2), NULL, 0);


 if (!is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 0)) || !is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 1))
     || !is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 2)))
 {
  net_send_to_client(connection, "500 Syntax error.");
  return;
 }

 if (!logged_in(connection))
 {
  net_send_to_client(connection, "530 Not logged in.");
  return;
 }

 if ((transporter = transporter_find_by_id(connection->pl, transporter_id)) == NULL)
 {
  net_send_to_client(connection, "551 Transporter doesn't exist.");
  return;
 }

 if ((town = town_find_by_id(transporter->location)) == NULL)
 {
  net_send_to_client(connection, "550 Town doesn't exist.");
  return;
 }

 switch (trade_sell_from_transporter_to_town(transporter, what, amount))
 {
  case 0:
   {
    good_t *good = good_find_by_id(town->goods, what);

    net_send_to_client(connection, "200 Selling successful.");

    send_good_amount_update(connection->pl, town, good);
    send_good_price_update(connection->pl, town, good);

    return;
   }
  break;

  case -1:
   net_send_to_client(connection, "552 Good not available in transporter.");
   return;
  break;

  case -2:
   net_send_to_client(connection, "553 Not enough goods in transporter.");
   return;
  break;

  case -3:
   net_send_to_client(connection, "554 Town doesn't have enough money.");
   return;
  break;

  /* default:
   net_send_to_client(connection, "501 Buying failed.");
   return;
  break; */
 }
}

/******************************************************************************
 Set the name of the given transporter.
******************************************************************************/
static void h_transporter_set_name(connection_t *connection, GPtrArray *arguments)
{
 guint transporter_id = (guint)strtol((const gchar *)g_ptr_array_index(arguments, 0), NULL, 0);
 const gchar *name = (const gchar *)g_ptr_array_index(arguments, 1);
 transporter_t *transporter;

 if (!is_number_or_digit((const gchar *)g_ptr_array_index(arguments, 0)))
 {
  net_send_to_client(connection, "500 Syntax error.");
  return;
 }

 if ((transporter = transporter_find_by_id(connection->pl, transporter_id)) == NULL)
 {
  net_send_to_client(connection, "551 Transporter doesn't exist.");
  return;
 }

 /* TODO Check whether transporter name is valid. */

 transporter_set_name(transporter, name);

 net_send_to_client(connection, "200 Transporter name changed successfully.");
}

/******************************************************************************
 Send a help message.
******************************************************************************/
static void h_help(connection_t *connection, GPtrArray *arguments)
{
 net_send_to_client(connection, "210 %s", PROTOCOL_HELP);
}

/******************************************************************************
 Logout from the server.
******************************************************************************/
static void h_logout(connection_t *connection, GPtrArray *arguments)
{
 if (!logged_in(connection))
 {
  net_send_to_client(connection, "530 Not logged in.");
  return;
 }

 net_send_to_client(connection, "200 Logout successful.");

 send_player_logged_out(connection->pl); 

 connection->pl->logged_in = NO;

 connection->pl = NO_PLAYER;
 if ((connection->connect_time = time(NULL)) == (time_t)(-1))
  log_perror("h_logout:time");
}

/******************************************************************************
 Disconnect from the server.
******************************************************************************/
static void h_quit(connection_t *connection, GPtrArray *arguments)
{
 net_send_to_client(connection, "221 Goodbye.");
 send_player_logged_out(connection->pl);
 connection_close(connection);
}

/******************************************************************************
 Save the game.
******************************************************************************/
static void h_save(connection_t *connection, GPtrArray *arguments)
{
 const gchar *filename = (const gchar *)g_ptr_array_index(arguments, 0);

 if (!logged_in_as_admin(connection))
 {
  net_send_to_client(connection, "531 You do not have admin permissions.");
  return;
 }

 net_send_to_client(connection, "500 Savegame support is not yet implemented.");

 /* if (save_game(filename) < 0)
  net_send_to_client(connection, "501 Saving game failed.");
 else
  net_send_to_client(connection, "200 Successfully saved game."); */
}

/******************************************************************************
 Load a game.
******************************************************************************/
static void h_load(connection_t *connection, GPtrArray *arguments)
{
 const gchar *filename = (const gchar *)g_ptr_array_index(arguments, 0);

 if (!logged_in_as_admin(connection))
 {
  net_send_to_client(connection, "531 You do not have admin permissions.");
  return;
 }

 net_send_to_client(connection, "500 Savegame support is not yet implemented.");

 /* if (load_game(filename) < 0)
  net_send_to_client(connection, "501 Loading game failed.");
 else
  net_send_to_client(connection, "200 Successfully loaded game."); */
}

/******************************************************************************
 Drop admin permissions.
******************************************************************************/
static void h_adminlogout(connection_t *connection, GPtrArray *arguments)
{
 if (!logged_in_as_admin(connection))
 {
  net_send_to_client(connection, "531 You do not have admin permissions.");
  return;
 }

 connection->is_admin = NO;
 net_send_to_client(connection, "200 Admin permissions dropped.");

 if ((connection->connect_time = time(NULL)) == (time_t)(-1))
  log_perror("h_adminlogout:time");
}

/******************************************************************************
 Re-read the configuration.
******************************************************************************/
static void h_reconfig(connection_t *connection, GPtrArray *arguments)
{
 if (!logged_in_as_admin(connection))
 {
  net_send_to_client(connection, "531 You do not have admin permissions.");
  return;
 }

 net_send_to_client(connection,
                    "501 The reconfig command doesn't work in this release.");

 /* if (conf_parse() < 0)
  net_send_to_client(connection, "501 Re-reading configuration failed.");
 else
  net_send_to_client(connection, "200 Successfully re-read configuration."); */
}

/******************************************************************************
 Delete a player.
******************************************************************************/
static void h_delplayer(connection_t *connection, GPtrArray *arguments)
{
 const gchar *player_string = (const gchar *)g_ptr_array_index(arguments, 0);
 const gchar *password = (const gchar *)g_ptr_array_index(arguments, 1);
 guint player_id;
 player_t *player;


 if (is_number_or_digit(player_string))
 {
  player_id = (guint)strtol(player_string, NULL, 0);

  if ((player = player_find_by_id(player_id)) == NULL)
  {
   net_send_to_client(connection, "533 Player not found.");
   return;
  }
 }
 else
 {
  if ((player = player_find_by_name(player_string)) == NULL)
  {
   net_send_to_client(connection, "533 Player not found.");
   return;
  }
 }

 if (connection_find_by_player(player) != NULL)
 {
  net_send_to_client(connection,
                     "537 The player to be deleted must be logged out.");
  return;
 }

 if (logged_in_as_admin(connection)
     || player_has_valid_password(player, password))
 {
  net_send_to_client(connection, "200 Player successfully deleted.");

  send_player_remove(player);

  player_free(player); 
  player_remove(player);

  return;
 }

 net_send_to_client(connection, "501 Player deletion failed.");
}

/******************************************************************************
 Pause/continue the game.
******************************************************************************/
static void h_togglepause(connection_t *connection, GPtrArray *arguments)
{
 if (!logged_in_as_admin(connection))
 {
  net_send_to_client(connection, "531 You do not have admin permissions.");
  return;
 }

 if (game_paused == NO)
 {
  game_paused = YES;
  net_send_to_client(connection, "200 Game is paused now.");
  send_pause(connection->pl);
 }
 else
 {
  game_paused = NO;
  net_send_to_client(connection, "200 Game continues.");
  send_continue(connection->pl);
 }
}

/******************************************************************************
 Shutdown the server.
******************************************************************************/
static void h_shutdown(connection_t *connection, GPtrArray *arguments)
{
 if (!logged_in_as_admin(connection))
 {
  net_send_to_client(connection, "531 You do not have admin permissions.");
  return;
 }

 net_send_to_client(connection, "200 Shutting down...");
 send_shutdown(connection); 
 g_main_quit(mainloop);
}

