/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* players.c  --  Functions to handle players. */


#include "common.h"
#include "transporters.h"
#include "towns.h"

#include "players.h"


GList *players = NULL;

static guint player_get_next_free_id(void);

/******************************************************************************
 Create a new player.
******************************************************************************/
player_t *player_new(const gchar *name, const gchar *password,
                     gboolean logged_in, guint age, glong money,
                     GList *transporters, GList *warehouses)
{
 player_t *player = g_malloc(sizeof(player_t));

 player->id = player_get_next_free_id();
 player->logged_in = logged_in;
 g_snprintf(player->name, MAX_LEN_PLAYERNAME+1, "%s", name);
 g_snprintf(player->password, MAX_LEN_PLAYERPASSWORD+1, "%s", password);
 player->age = age;
 player->money = money;
 player->transporters = transporters;
 player->warehouses = warehouses;

 return player;
}

/******************************************************************************
 Free the memory used by a player.
******************************************************************************/
void player_free(player_t *player)
{
 g_free(player);
}

/******************************************************************************
 Free the memory used by all players.
******************************************************************************/
void player_free_all(void)
{
 GList *g;
 player_t *player;

 if ((g = g_list_first(players)) == NULL)
 {
  g_list_free(players);
  return;
 }

 do
 {
  player = g->data;
  player_free(player);
 } while ((g = g_list_next(g)) != NULL);

 g_list_free(players);
}

/******************************************************************************
 Add a player to the list of players.
******************************************************************************/
void player_add(player_t *player)
{
 players = g_list_append(players, player);
}

/******************************************************************************
 Remove a player from the list of players.
******************************************************************************/
void player_remove(player_t *player)
{
 players = g_list_remove(players, player);
}

/******************************************************************************
 The player gets a transporter.
******************************************************************************/
void player_add_transporter(player_t *player, transporter_t *transporter)
{
 player->transporters = g_list_append(player->transporters, transporter);
}

/******************************************************************************
 The player loses a transporter.
******************************************************************************/
void player_remove_transporter(player_t *player, transporter_t *transporter)
{
 player->transporters = g_list_remove(player->transporters, transporter);
}

/******************************************************************************
 Find the player with the given ID.
******************************************************************************/
player_t *player_find_by_id(guint id)
{
 GList *g;
 player_t *player;

 if ((g = g_list_first(players)) == NULL)
  return NULL; /* Player not found. */

 do
 {
  player = g->data;

  if (player->id == id)
   return player;

 } while ((g = g_list_next(g)) != NULL);

 return NULL; /* Player not found. */
}

/******************************************************************************
 Find the player with the given name.
******************************************************************************/
player_t *player_find_by_name(const gchar *playername)
{
 GList *g;
 player_t *player;

 if ((g = g_list_first(players)) == NULL)
  return NULL; /* Player not found. */

 do
 {
  player = g->data;

  if (strncmp(player->name, playername, MAX_LEN_PLAYERNAME) == 0)
   return player;

 } while ((g = g_list_next(g)) != NULL);

 return NULL; /* Player not found. */
}

/******************************************************************************
 Check whether the given player exists.
******************************************************************************/
gboolean player_exists(const gchar *playername)
{
 return (player_find_by_name(playername) != NULL);
}

/******************************************************************************
 Check whether the given player has a valid password.
******************************************************************************/
gboolean player_has_valid_password(player_t *player, const gchar *password)
{
 if (strncmp(player->password, password, MAX_LEN_PLAYERPASSWORD) == 0)
  return YES;
 else
  return NO;
}

/******************************************************************************
 Playernames can only contain letters and spaces.
******************************************************************************/
gboolean player_has_valid_name(const gchar *playername)
{
 size_t i = 0;

 while ((i < strlen(playername)) && (i < MAX_LEN_PLAYERNAME))
 {
  if (!isalpha((gint)*(playername+i)) && (*(playername+i) != ' '))
   return FALSE;

  i++;
 }

 return TRUE;
}

/******************************************************************************
 Return a list of players who have transporters in the same town in which your
 transporter is located.
******************************************************************************/
GList *player_with_transporters_in_same_town(player_t *my_player, town_t *town)
{
 GList *g;
 player_t *player;
 GList *player_list = NULL;
 
 if ((g = g_list_first(players)) == NULL)
  return NULL; /* No player found. */

 do
 {
  player = g->data;

  if (player != my_player)
  {
   GList *g2;
   transporter_t *transporter;

   if ((g2 = g_list_first(player->transporters)) != NULL)
   {

    do
    {
     transporter = g2->data;

     if (transporter->location == town->id)
     {
      player_list = g_list_append(player_list, player);
      break;
     }

    } while ((g2 = g_list_next(g2)) != NULL);

   }
   
  }

 } while ((g = g_list_next(g)) != NULL);

 return player_list;
}

/******************************************************************************
 Return a list which contains all players except the given player.
******************************************************************************/
GList *player_all_except_this_one(player_t *player)
{
 GList *g;
 player_t *player2;
 GList *player_list = NULL;

 if ((g = g_list_first(players)) == NULL)
  return NULL; /* No player found. */

 do
 {
  player2 = g->data;

  if (player2 != player)
   player_list = g_list_append(player_list, player2);

 } while ((g = g_list_next(g)) != NULL);

 return player_list;
}

/******************************************************************************
 Return the next free player ID.
******************************************************************************/
static guint player_get_next_free_id(void)
{
 GList *g;
 guint id = 0;
 player_t *player;

 if ((g = g_list_first(players)) == NULL)
  return 0; /* No player found. */

 do
 {
  player = g->data;

  if (player->id > id)
   id = player->id;

 } while ((g = g_list_next(g)) != NULL);

 return id + 1;
}

