/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* net.c  --  Networking code. */


#include "common.h"
#include "log.h"
#include "protocol.h"
#include "connections.h"
#include "misc.h"
#include "conf.h"
#include "players.h"
#include "defaults.h"
#include "savegame.h"
#include "servercmds.h"
#include "file.h"

#include "net.h"


#define BACKLOG 5


GMainLoop *mainloop;
guint server_event_source_id;
guint auto_save_event_source_id;
guint check_connection_times_event_source_id;

static gboolean check_connection_times(gpointer data);

/******************************************************************************
 Create the main event loop, which manages all available event sources.
******************************************************************************/
void net_main_loop(void)
{
 gint sockfd;
 GIOChannel *server_channel;

 mainloop = g_main_new(FALSE);

 if ((sockfd = net_create_socket(conf.port)) < 0)
 {
  g_warning("net_main_loop: Could not create the server socket.");
  /* TODO Clean-up? */
  exit(EXIT_FAILURE);
 }

 server_channel = g_io_channel_unix_new(sockfd);

 server_event_source_id = g_io_add_watch(server_channel, G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL, connection_accept, NULL);

 /* Check the connection times of the clients once per second. */
 check_connection_times_event_source_id = g_timeout_add(1000, check_connection_times, NULL);

 /* Save the game automatically every conf.autosave_delay seconds.
    If conf.autosave_delay is zero, this functionality is disabled. */
 if (conf.autosave_delay != 0)
 {
  auto_save_event_source_id = g_timeout_add(1000*conf.autosave_delay,
                                            auto_save, NULL);
 }

 g_main_run(mainloop);
}

/******************************************************************************
 Create an internet stream socket and return its descriptor.
******************************************************************************/
gint net_create_socket(gushort port)
{
 gint sockfd;
 gint flags;
 gint reuseaddr = 1;
 struct sockaddr_in server_addr;

 if ((sockfd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
 {
  log_perror("net_create_socket:socket");
  return -1;
 }

 memset(&server_addr, 0, sizeof(struct sockaddr_in));
 server_addr.sin_family = AF_INET;
 server_addr.sin_port = g_htons(port);
 server_addr.sin_addr.s_addr = g_htonl(INADDR_ANY);

 /* Used to allow bind()ing while a previous socket is still in TIME_WAIT. */
 if (setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &reuseaddr,
     (socklen_t)sizeof(reuseaddr)) < 0)
 {
  log_perror("net_create_socket:setsockopt");
 }

 if (bind(sockfd, (struct sockaddr *)&server_addr,
          (socklen_t)sizeof(struct sockaddr_in)) < 0)
 {
  log_perror("net_create_socket:bind");
  if (close(sockfd) < 0)
   log_perror("net_create_socket:close");
  return -1;
 }

 if (listen(sockfd, BACKLOG) < 0)
 {
  log_perror("net_create_socket:listen");
  if (close(sockfd) < 0)
   log_perror("net_create_socket:close");
  return -1;
 }

 if ((flags = fcntl(sockfd, F_GETFL, 0)) < 0)
 {
  log_perror("net_create_socket:fcntl");
  /* if (close(sockfd) < 0)
   log_perror("net_create_socket:close");
  return -1; */
 }

 flags |= O_NONBLOCK; /* Set socket nonblocking. */

 if (fcntl(sockfd, F_SETFL, flags) < 0)
 {
  log_perror("net_create_socket:fcntl");
  /* if (close(sockfd) < 0)
   log_perror("net_create_socket:close");
  return -1; */
 }

 /* Set owner of the socket. */
 if (fcntl(sockfd, F_SETOWN, getpid()) < 0)
 {
  log_perror("net_create_socket:fcntl");
  /* if (close(sockfd) < 0)
   log_perror("net_create_socket:close");
  return -1; */
 }

 return sockfd;
}

/******************************************************************************
 Handle commands received from the client.
******************************************************************************/
gboolean net_handle_client(GIOChannel *source, GIOCondition condition,
                           gpointer data)
{
 connection_t *connection;

 if ((connection = connection_find_by_channel(source)) == NULL)
 {
  /* Error. */
 }

 if (condition == G_IO_IN)
 {
  gchar *message;

  if ((message = net_recv_from_client(connection)) == NULL)
   return FALSE; /* ? */

  if ((game_paused == YES) && (!logged_in_as_admin(connection)))
   net_send_to_client(connection, "450 Game is paused.");
  else
  {
   if ((protocol_parse_message(connection, message)) < 0)
    g_warning("net_handle_client: protocol_parse_message() failed.");
  }  

  g_free(message);
 }

 if (condition == G_IO_PRI)
 {
  g_warning("net_handle_client: G_IO_PRI.");
  /* TODO */
 }

 if (condition == G_IO_ERR)
 {
  g_warning("net_handle_client: G_IO_ERR.");
  /* TODO */
 }

 if (condition == G_IO_HUP) /* Lost connection to client. */
 {
  g_message("Lost connection to player \"%s\" at %s:%d.", connection->pl->name,
            inet_ntoa(connection->addr.sin_addr),
            g_ntohs(connection->addr.sin_port));

  send_player_logged_out(connection->pl);

  connection_close(connection);

  return FALSE;
 }

 if (condition == G_IO_NVAL)
 {
  g_warning("net_handle_client: G_IO_NVAL.");
  /* TODO */
 }

 return TRUE;
}

/******************************************************************************
 Receive a line of data from the given connection.
******************************************************************************/
gchar *net_recv_from_client(connection_t *connection)
{
 player_t *player;
 gchar *message;

 /* FIXME */
 if ((message = file_read_line(connection->channel)) == NULL)
 {
  g_warning("Error reading data from client. Closing connection to client.");
  connection_close(connection);
  return NULL;
 }

 g_strchomp(message);

 if ((player = connection->pl) != NULL)
  g_message("Received from [%u \"%s\" \"%s\" %u]: %s", player->id, player->name,
            inet_ntoa(connection->addr.sin_addr),
            g_ntohs(connection->addr.sin_port), message);
 else
  g_message("Received from [\"%s\" %u]: %s",
            inet_ntoa(connection->addr.sin_addr),
            g_ntohs(connection->addr.sin_port), message);

 return message;
}

/******************************************************************************
 Send a message to a client. Adds trailing \n.
******************************************************************************/
void net_send_to_client(connection_t *connection, const gchar *fmt, ...)
{
 va_list args;
 gchar *tmp, *message;
 player_t *player;

 /* TODO Check whether we can write without blocking. */

 va_start(args, fmt);
 tmp = g_strdup_vprintf(fmt, args);
 va_end(args);

 message = g_strdup_printf("%s\n", tmp);
 g_free(tmp);

 if (file_write_line(connection->channel, message) < 0)
 {
  g_warning("Error sending data to client. Closing connection to client.");
  connection_close(connection);
  g_free(message);
  return;
 }

 g_strchomp(message);

 if ((player = connection->pl) != NULL)
  g_message("Sent to [%u \"%s\" \"%s\" %u]: %s", player->id, player->name,
            inet_ntoa(connection->addr.sin_addr),
            g_ntohs(connection->addr.sin_port), message);
 else
  g_message("Sent to [\"%s\" %u]: %s", inet_ntoa(connection->addr.sin_addr),
            g_ntohs(connection->addr.sin_port), message);

 g_free(message);
}

/******************************************************************************
 Broadcast a message to a list of connected clients.
******************************************************************************/
void net_broadcast_to_connections(GList *connection_list,
                                  const gchar *fmt, ...)
{
 GList *g;
 connection_t *connection;
 gchar *message;
 va_list args;

 va_start(args, fmt);
 message = g_strdup_vprintf(fmt, args);
 va_end(args);

 if ((g = g_list_first(connection_list)) == NULL)
  return; /* No connection found, therefore don't send anything. */

 do
 {
  connection = g->data;
  net_send_to_client(connection, "%s", message);
 } while ((g = g_list_next(g)) != NULL);

 g_free(message);
}

/******************************************************************************
 Broadcast a message to a list of players.
******************************************************************************/
void net_broadcast_to_players(GList *player_list, const gchar *fmt, ...)
{
 GList *g;
 gchar *message;
 va_list args;
 connection_t *connection;
 player_t *player;

 va_start(args, fmt);
 message = g_strdup_vprintf(fmt, args);
 va_end(args);

 if ((g = g_list_first(player_list)) == NULL)
  return; /* No players found, therefore don't send anything. */

 do
 {
  player = g->data;

  if ((connection = connection_find_by_player(player)) != NULL)
   net_send_to_client(connection, "%s", message);

 } while ((g = g_list_next(g)) != NULL);

 g_free(message);
}

/******************************************************************************
 Check if any connection is connected longer than 'conf.login_timeout',
 without having logged in. If yes -> kick that connection.
******************************************************************************/
static gboolean check_connection_times(gpointer data)
{
 GList *g;
 connection_t *connection;

 if ((g = g_list_first(connections)) == NULL)
  return TRUE; /* No connection found. */

 do
 {
  connection = g->data;

  if (((connection->connect_time+(time_t)conf.login_timeout) < time(NULL))
      && !logged_in(connection) && !logged_in_as_admin(connection)
      && (conf.login_timeout != 0))
  {
   send_login_timeout(connection); 
   connection_close(connection);
  }
 } while ((g = g_list_next(g)) != NULL);

 return TRUE;
}

