/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* misc.c  --  Miscellaneous code. */


#include "common.h"
#include "log.h"
#include "savegame.h"
#include "conf.h"
#include "net.h"

#include "misc.h"


gboolean game_paused = NO; /* Set to YES to pause the game. */

static RETSIGTYPE signal_handler_exit(gint signum);
static RETSIGTYPE signal_handler_config(gint signum);

/******************************************************************************
 Makes htsserver a daemon and detaches it from the controlling terminal.
******************************************************************************/
void daemonize(void)
{
 glong i, maxfd;

 /* fork() to allow the parent to exit; this returns control to the shell. */
 switch (fork())
 {
  case 0: break;        /* Child. */
  case -1:              /* Error. */
  {
   log_perror("daemonize:fork");
   exit(EXIT_FAILURE);
  }
  default: exit(EXIT_SUCCESS);    /* Parent: exit. */
 }

 /* Become a process group and session group leader. */
 if (setsid() < 0)
 {
  log_perror("daemonize:setsid");
  exit(EXIT_FAILURE);
 }

 /* fork() again so the parent(the session group leader), can exit; we can */
 /* now never gain a controlling terminal again. */
 switch (fork())
 {
  case 0: break;        /* Child. */
  case -1:              /* Error. */
  {
   log_perror("daemonize:fork");
   exit(EXIT_FAILURE);
  }
  default: exit(EXIT_SUCCESS);    /* Parent: exit. */
 }

 /* chdir("/") to ensure that the process doesn't keep any directory in use. */
 if (chdir("/") < 0)
 {
  log_perror("daemonize:chdir");
  exit(EXIT_FAILURE);
 }

 /* Set umask to 0000, so that we have complete control over the permissions
    of stuff we write. Just to be sure. */
 /* (void) umask(0000); */

 (void) umask(0077); /* Restrictive permissions. */

 switch ((gint)(maxfd = sysconf(_SC_OPEN_MAX)))
 {
  case 0: /* Fallthrough. */
  case 1: maxfd = 3; break;  /* Only close stdin, stdout and stderr. */
  case -1:
  {
   log_perror("daemonize:sysconf");
   maxfd = 3;
  }
 }

 for (i=0; i<maxfd; i++) /* Close all file-descriptors. */
  close(i);


 /* Assign /dev/null to stdin, stdout and stderr. */
 for (i=0; i<3; i++)
 {
  if (open("/dev/null", O_RDWR) < 0)
  {
   log_perror("daemonize:open");
   exit(EXIT_FAILURE);
  }
 }
}

/******************************************************************************
 Install signal handlers for all signals we want to use.
******************************************************************************/
void setup_signal_handling(void)
{
 if (signal(SIGTERM, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGTERM, SIG_IGN);
 if (signal(SIGINT, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGINT, SIG_IGN);
 if (signal(SIGQUIT, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGQUIT, SIG_IGN);
 if (signal(SIGABRT, signal_handler_exit) == SIG_IGN)
  (void) signal(SIGABRT, SIG_IGN);

 if (signal(SIGHUP, signal_handler_config) == SIG_IGN)
  (void) signal(SIGHUP, SIG_IGN);

 (void) signal(SIGPIPE, SIG_IGN);
}

/******************************************************************************
 Signal handler which is used for shutting down the game.
******************************************************************************/
static RETSIGTYPE signal_handler_exit(gint signum)
{
 g_message("Signal \"%s\" (%d) received -> controlled shutdown.",
           g_strsignal(signum), signum);

 /* TODO Save the current game? */

 g_main_quit(mainloop);
}

/******************************************************************************
 Signal handler which is used for re-reading the configfile.
******************************************************************************/
static RETSIGTYPE signal_handler_config(gint signum)
{
 g_message("Signal \"%s\" (%d) received -> re-read config-file.",
           g_strsignal(signum), signum);

 if (conf_parse() < 0)
 {
  g_warning("Error parsing configuration file.");
  exit(EXIT_FAILURE); /* FIXME Recover old configuration. */
 }
}

/******************************************************************************
 Check if the given string contains a number or a digit.
******************************************************************************/
gboolean is_number_or_digit(const gchar *message)
{
 guint i;

 for (i=0; i<strlen(message); i++)
 {
  if (!isdigit(message[i]))
   return FALSE;
 }

 return TRUE;
}

/******************************************************************************
 Split a string into several tokens.
******************************************************************************/
GPtrArray *tokenize(const gchar *message)
{
 GTokenType token;
 GScanner *scanner;
 GPtrArray *ptr_array;

 ptr_array = g_ptr_array_new();
 scanner = g_scanner_new(NULL);

 scanner->config->cset_identifier_first =
 (G_CSET_a_2_z G_CSET_A_2_Z G_CSET_DIGITS);
 scanner->config->scan_identifier_1char = TRUE;
 /* TODO Enable/disable more options? */

 g_scanner_input_text(scanner, message, strlen(message));
 scanner->input_name = "tokenize";

 while (TRUE)
 {
  token = g_scanner_get_next_token(scanner);

  if (token == G_TOKEN_ERROR)
  {
   g_scanner_unexp_token(scanner, token, NULL, NULL, NULL, NULL, TRUE);
   (void) g_ptr_array_free(ptr_array, TRUE);
   g_scanner_destroy(scanner);
   return NULL;
  }

  if (token == G_TOKEN_EOF)
   break;

  /* TODO Check for more return values? */

  g_ptr_array_add(ptr_array, (gpointer)g_strdup(scanner->value.v_string));
 }

 g_scanner_destroy(scanner);

 return ptr_array;
}

