/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* goods.c  --  Functions to handle goods. */


#include "common.h"
#include "log.h"
#include "misc.h"
#include "file.h"
#include "conf.h"

#include "goods.h"


guint number_of_goods;

gchar goodnames[MAX_NUMBER_OF_GOODS][MAX_LEN_GOODNAME+1];


/******************************************************************************
 Create a new good.
******************************************************************************/
good_t *good_new(guint id, glong amount, glong price)
{
 good_t *good = g_malloc(sizeof(good_t));

 good->id = id;
 good->amount = amount;
 good->price = price;

 return good;
}

/******************************************************************************
 Free the memory used by a good.
******************************************************************************/
void good_free(good_t *good)
{
 g_free(good);
}

/******************************************************************************
 Read all goodnames from goodnames.dat.
******************************************************************************/
gint good_read_names(void)
{
 gint fd;
 guint i;
 gchar *line;
 GIOChannel *channel;
 gchar *datafile = g_strdup_printf("%s/goodnames.dat", conf.datadir);

 if ((fd = open(datafile, O_RDONLY)) < 0)
 {
  log_perror("good_read_names:open");
  g_free(datafile);
  return -1;
 }

 g_free(datafile);

 channel = g_io_channel_unix_new(fd);

 if ((line = file_read_line(channel)) == NULL)
 {
  g_warning("good_read_names: file_read_line() failed.");
  return -1;
 }

 if (!is_number_or_digit(g_strchomp(line)))
 {
  g_warning("Corrupt goodnames.dat.");
  g_free(line);
  return -1;
 }

 number_of_goods = (guint)strtol(line, NULL, 0);
 g_free(line);

 /* Read all goodnames, one per line. */
 for (i=0; i<number_of_goods; i++)
 {
  
  if ((line = file_read_line(channel)) == NULL)
  {
   g_warning("good_read_names: file_read_line() failed.");
   return -1;
  }

  g_snprintf((gchar *)&goodnames[i], MAX_LEN_GOODNAME+1, "%s",
             g_strchomp(line));
  g_free(line);
 }

 g_io_channel_close(channel);
 g_io_channel_unref(channel);

 return 0;
}

/******************************************************************************
 Find the good with the given ID.
******************************************************************************/
good_t *good_find_by_id(GList *goods, guint id)
{
 GList *g;
 good_t *good;

 if ((g = g_list_first(goods)) == NULL)
  return NULL; /* Good not found. */

 do
 {
  good = g->data;

  if (good->id == id)
   return good;

 } while ((g = g_list_next(g)) != NULL);

 return NULL; /* Good not found. */
}

