/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* connections.c  --  Functions to handle connections. */


#include "common.h"
#include "log.h"
#include "net.h"
#include "players.h"

#include "connections.h"


GList *connections = NULL;

/******************************************************************************
 Create a new connection.
******************************************************************************/
connection_t *connection_new(void)
{
 connection_t *connection = g_malloc(sizeof(connection_t));

 /* struct sockaddr_in addr; */
 connection->pl = NO_PLAYER;
 connection->is_admin = NO;
 connection->connect_time = 0; /* Bogus. */
 connection->channel = NULL;

 return connection;
}

/******************************************************************************
 Free the memory used by a connection.
******************************************************************************/
void connection_free(connection_t *connection)
{
 g_free(connection);
}

/******************************************************************************
 Add a connection to the list of connections.
******************************************************************************/
void connection_add(connection_t *connection)
{
 connections = g_list_append(connections, connection);
}

/******************************************************************************
 Remove the given connection from the list of connections.
******************************************************************************/
void connection_remove(connection_t *connection)
{
 connections = g_list_remove(connections, connection);
}

/******************************************************************************
 Accepts a new connection from a client.
******************************************************************************/
gboolean connection_accept(GIOChannel *source, GIOCondition condition,
                           gpointer data)
{
 gint client_sockfd;
 GIOChannel *client_channel;
 socklen_t len = (socklen_t)sizeof(struct sockaddr_in);
 connection_t *connection;

 if (condition == G_IO_IN)
 {
  /* TODO Check whether we allow this client to connect (blocklist) etc. */

  connection = connection_new();

  if ((client_sockfd = accept(g_io_channel_unix_get_fd(source),
                              (struct sockaddr *)&connection->addr, &len)) < 0)
  {
   log_perror("connection_accept:accept");
   connection_free(connection);
   return FALSE; /* ? */
  }

  client_channel = g_io_channel_unix_new(client_sockfd);

  connection->event_source_id = g_io_add_watch(client_channel, G_IO_IN |  G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL, net_handle_client, NULL);

  if ((connection->connect_time = time(NULL)) == (time_t)(-1))
   log_perror("connection_accept:time");

  connection->channel = client_channel;

  connection_add(connection);

  net_send_to_client(connection, "220 %s %s, released on %s.", PACKAGE,
                     VERSION, HTSSERVER_RELEASEDATE);
 }


 if (condition == G_IO_PRI)
 {
  g_warning("connection_accept: G_IO_PRI.");
  /* TODO */
 }

 if (condition == G_IO_ERR)
 {
  g_warning("connection_accept: G_IO_ERR.");
  /* TODO */
 }

 if (condition == G_IO_HUP)
 {
  g_warning("connection_accept: G_IO_HUP.");
  /* TODO */
 }

 if (condition == G_IO_NVAL)
 {
  g_warning("connection_accept: G_IO_NVAL.");
  /* TODO */
 }

 return TRUE;
}

/******************************************************************************
 Close a connection.
******************************************************************************/
void connection_close(connection_t *connection)
{
 g_io_channel_close(connection->channel);
 g_io_channel_unref(connection->channel);

 if (g_source_remove(connection->event_source_id) == FALSE)
  g_warning("connection_close: g_source_remove() failed.");

 connection_free(connection);
 connection_remove(connection);
}

/******************************************************************************
 Find the connection with the given GIOChannel.
******************************************************************************/
connection_t *connection_find_by_channel(GIOChannel *channel)
{
 GList *g;
 connection_t *connection;

 if ((g = g_list_first(connections)) == NULL)
  return NULL; /* Connection not found. */

 do
 {
  connection = g->data;

  if (connection->channel == channel)
   return connection;

 } while ((g = g_list_next(g)) != NULL);

 return NULL; /* Connection not found. */
}

/******************************************************************************
 Find the connection which belongs to the given player.
******************************************************************************/
connection_t *connection_find_by_player(player_t *player)
{
 GList *g;
 connection_t *connection;

 if ((g = g_list_first(connections)) == NULL)
  return NULL; /* Connection not found. */

 do
 {
  connection = g->data;

  if (connection->pl == player)
   return connection;

 } while ((g = g_list_next(g)) != NULL);

 return NULL; /* Connection not found. */
}

/******************************************************************************
 Find the connection which belongs to the player with the given playername.
******************************************************************************/
connection_t *connection_find_by_playername(const gchar *playername)
{
 GList *g;
 connection_t *connection;

 if ((g = g_list_first(connections)) == NULL)
  return NULL; /* Connection not found. */

 do
 {
  connection = g->data;

  /* FIXME Currently the player *must* be logged in, else we'll segfault. */
  if (!logged_in(connection))
   return NULL; 

  if (strncmp(connection->pl->name, playername, MAX_LEN_PLAYERNAME) == 0)
   return connection;

 } while ((g = g_list_next(g)) != NULL);

 return NULL; /* Connection not found. */
}

/******************************************************************************
 Return a list which contains all connections except the given connection.
******************************************************************************/
GList *connection_all_except_this_one(connection_t *connection)
{
 GList *g;
 connection_t *connection2;
 GList *connection_list = NULL;

 if ((g = g_list_first(connections)) == NULL)
  return NULL; /* No connection found. */

 do
 {
  connection2 = g->data;

  if (connection2 != connection)
   connection_list = g_list_append(connection_list, connection2);

 } while ((g = g_list_next(g)) != NULL);

 return connection_list;
}

/******************************************************************************
 Check whether the player who belongs to the given connection is logged in.
******************************************************************************/
gboolean logged_in(connection_t *connection)
{
 return (connection->pl != NO_PLAYER);
}

/******************************************************************************
 Check whether the player who belongs to the given connection is logged in
 as admin.
******************************************************************************/
gboolean logged_in_as_admin(connection_t *connection)
{
 return (connection->is_admin != NO);
}

