/******************************************************************************
 htsserver - the server application for the trading game Holsham Traders
 Copyright (C) 1998-2001 Uwe Hermann

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
******************************************************************************/

/* conf.c  --  Functions to handle configuration issues. */


#include "common.h"
#include "defaults.h"
#include "log.h"
#include "cmdline.h"

#include "conf.h"


enum
{
 SYMBOL_PORT           = G_TOKEN_LAST + 1,
 SYMBOL_LOGDEST        = G_TOKEN_LAST + 2,
 SYMBOL_AUTOSAVE_DELAY = G_TOKEN_LAST + 3,
 SYMBOL_LOGIN_TIMEOUT  = G_TOKEN_LAST + 4,
 SYMBOL_LOGFILE        = G_TOKEN_LAST + 5,
 SYMBOL_DATADIR        = G_TOKEN_LAST + 6,
 SYMBOL_SAVEGAMESDIR   = G_TOKEN_LAST + 7,
 SYMBOL_ADMINPASSWORD  = G_TOKEN_LAST + 8
};

static const struct
{
 const gchar *name;
 guint token;
} symbols[] =
{
 { "port",           SYMBOL_PORT },
 { "logdest",        SYMBOL_LOGDEST },
 { "autosave_delay", SYMBOL_AUTOSAVE_DELAY },
 { "login_timeout",  SYMBOL_LOGIN_TIMEOUT },
 { "logfile",        SYMBOL_LOGFILE },
 { "datadir",        SYMBOL_DATADIR },
 { "savegamesdir",   SYMBOL_SAVEGAMESDIR },
 { "adminpassword",  SYMBOL_ADMINPASSWORD },
 { NULL,             0 },
};

static const guint n_symbols = sizeof(symbols) / sizeof(symbols[0]);

static guint conf_parse_symbol(GScanner *scanner);

conf_t conf;


/******************************************************************************
 Configuration initialization.
******************************************************************************/
void conf_init(void)
{
 conf.port           = DEFAULT_PORT;
 conf.logdest        = DEFAULT_LOGDEST;
 conf.autosave_delay = DEFAULT_AUTOSAVE_DELAY;
 conf.login_timeout  = DEFAULT_LOGIN_TIMEOUT;
 conf.configfile     = g_strdup(HTSSERVER_CONFIGFILE); /* !!! */
 conf.logfile        = g_strdup(HTSSERVER_LOGFILE);
 conf.datadir        = g_strdup(HTSSERVER_DATADIR);
 conf.savegamesdir   = g_strdup(HTSSERVER_SAVEGAMESDIR);
 conf.adminpassword  = g_strdup(DEFAULT_ADMINPASSWORD);
}

/******************************************************************************
 Parse configuration file.
******************************************************************************/
gint conf_parse(void)
{
 guint i;
 gint fd;
 guint expected_token;
 GScanner *scanner = g_scanner_new(NULL);

 /* scanner->config->numbers_2_int = TRUE; */
 scanner->config->symbol_2_token = TRUE;

 if ((fd = open(conf.configfile, O_RDONLY)) < 0)
 {
  log_perror("conf_parse:open");
  g_scanner_destroy(scanner);
  return -1;
 }

 g_scanner_input_file(scanner, fd);
 scanner->input_name = "configfile";

 /* Load symbols into the scanner. */
 for (i = 0; i < n_symbols - 1; i++)
  g_scanner_scope_add_symbol(scanner, 0, symbols[i].name,
                             GINT_TO_POINTER(symbols[i].token));

 do
 {
  expected_token = conf_parse_symbol(scanner);
  g_scanner_peek_next_token(scanner);
 }
 while (expected_token == G_TOKEN_NONE && scanner->next_token != G_TOKEN_EOF
        && scanner->next_token != G_TOKEN_ERROR);

 if (expected_token != G_TOKEN_NONE)
 {
  g_scanner_unexp_token(scanner, expected_token, NULL, "symbol", NULL, NULL,
                        TRUE);
  g_scanner_destroy(scanner);
  return -1;
 }

 g_scanner_destroy(scanner);

 if (close(fd) < 0)
  log_perror("conf_parse:close");

 return 0;
}

/******************************************************************************
 ...
******************************************************************************/
static guint conf_parse_symbol(GScanner *scanner)
{
 guint symbol;

 /* Expect a valid symbol. */
 g_scanner_get_next_token(scanner);
 symbol = scanner->token;

 if (symbol < SYMBOL_PORT || symbol > SYMBOL_ADMINPASSWORD)
  return G_TOKEN_SYMBOL;

 /* Expect '='. */
 g_scanner_get_next_token(scanner);

 if (scanner->token != '=')
  return G_TOKEN_EQUAL_SIGN;

 g_scanner_get_next_token(scanner);

 switch (symbol)
 {
  case SYMBOL_PORT:
   if (cmdline.port == NO)
    conf.port = scanner->value.v_int;
  break;

  case SYMBOL_LOGDEST:
   if (cmdline.logdest == NO)
   {
    if ((scanner->token != G_TOKEN_IDENTIFIER))
     return G_TOKEN_IDENTIFIER;

    if (g_strcasecmp(scanner->value.v_string, "file") == 0)
     conf.logdest = LOG_TO_FILE;
    else if (g_strcasecmp(scanner->value.v_string, "syslog") == 0)
     conf.logdest = LOG_TO_SYSLOG;
    else if (g_strcasecmp(scanner->value.v_string, "none") == 0)
     conf.logdest = LOG_TO_NONE;
    else
     return G_TOKEN_IDENTIFIER;
   }
  break;

  case SYMBOL_AUTOSAVE_DELAY:
   if (cmdline.autosave_delay == NO)
    conf.autosave_delay = scanner->value.v_int;
  break;

  case SYMBOL_LOGIN_TIMEOUT:
   if (cmdline.login_timeout == NO)
    conf.login_timeout = scanner->value.v_int;
  break;

  case SYMBOL_LOGFILE:
   if (cmdline.logfile == NO)
   {
    g_free(conf.logfile);
    conf.logfile = g_strdup(scanner->value.v_string);
   }
  break;

  case SYMBOL_DATADIR:
   if (cmdline.datadir == NO)
   {
    g_free(conf.datadir);
    conf.datadir = g_strdup(scanner->value.v_string);
   }
  break;

  case SYMBOL_SAVEGAMESDIR:
   if (cmdline.savegamesdir == NO)
   {
    g_free(conf.savegamesdir);
    conf.savegamesdir = g_strdup(scanner->value.v_string);
   }
  break;

  case SYMBOL_ADMINPASSWORD:
   if (cmdline.adminpassword == NO)
   {
    g_free(conf.adminpassword);
    conf.adminpassword = g_strdup(scanner->value.v_string);
   }
  break;
 }

 return G_TOKEN_NONE;
}

