/* ghtsclient - a GTK+ based client for the trading game Holsham Traders
   Copyright (C) 1999,2000 Piotr Esden-Tempski

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* net.c  --  networking code */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "sock.h"
#include "log.h"
#include "data.h"
#include "messagedialog.h"
#include "logindialog.h"
#include "statusbar.h"
#include "menu.h"

#include "net.h"


#define NET_BUFFER_SIZE   1024


int server_sockfd;
FILE *server_fd;
static gint gdk_input_id;

/*Flags*/
bool connected = NO;

static void recv_data(gpointer data, gint sockfd, GdkInputCondition condition);
static void handle_server_data(char *buf);

static void lost_connection(void);

/*****************************************************************************/
int server_connect(char *servername, int portnr)
{
 struct sockaddr_in serveraddr;

 /* connect to server */
 server_sockfd = create_socket();
 init_sockaddr(&serveraddr, servername, portnr);

 if (connect_to_socket(server_sockfd, &serveraddr))
 {
  ghtslog(GHTSLOG_ERROR, "Couldn't connect to server");
  return -1;
 }

 /* let gdk look if there is any data coming from htsserver */
 gdk_input_id = gdk_input_add(server_sockfd,
                              GDK_INPUT_READ|GDK_INPUT_EXCEPTION,
                              recv_data, NULL);

 if ((server_fd = fdopen(server_sockfd, "r+")) == NULL)
 {
  lerror("server_connect:fdopen");
  close_socket(server_sockfd);
  return -1;
 }

 connected = YES;
 ghts_update_statusbar();
 m_connect();

 ghtslog(GHTSLOG_INFO, "server_connect().");

 return 0;
}

/*****************************************************************************/
void server_disconnect(void)
{
 send_to_server("quit");

 if (fclose(server_fd) == EOF)
  lerror("server_disconnect:fclose");

 gdk_input_remove(gdk_input_id);
 /* close_socket(server_sockfd); */

 connected = NO;
 logged_in = NO;
 ghts_update_statusbar();
 m_disconnect();

 ghtslog(GHTSLOG_INFO, "server_disconnect().");
}

/******************************************************************************
 Receives a message from htsserver. Returns -1 on failure, 0 on success.
******************************************************************************/
int recv_from_server(char *buf, int s)
{
again:
 if (feof(server_fd) != 0)
 {
  lost_connection();
  return -1;
 }
 else
 {
  if (fgets(buf, s, server_fd) == NULL)
  {
   lerror("recv_from_server:fgets");

   if (errno == EINTR)
    goto again;

   lost_connection();
   return -1;
  }
 }

 return 0;
}

/******************************************************************************
 Sends a message to htsserver. Adds trailing \n.
******************************************************************************/
int send_to_server(const char *fmt, ...)
{
 va_list args;

 if (feof(server_fd) != 0)
 {
  lost_connection();
  return -1;
 }
 else
 {
  va_start(args, fmt);
  (void) vfprintf(server_fd, fmt, args);
  va_end(args);

  if ((size_t)fprintf(server_fd, "\n") != strlen("\n"))
  {
   lerror("send_to_server:fprintf");
   return -1;
  }

  if (fflush(server_fd) == EOF)
  {
   lerror("send_to_server:fflush");
   return -1;
  }
 }

 return 0;
}

/******************************************************************************
 Handles a lost connection.
******************************************************************************/
static void lost_connection(void)
{
 ghtslog(GHTSLOG_ERROR, "Lost connection to server.");

 if (fclose(server_fd) == EOF)
  lerror("lost_connection:fclose");

 gdk_input_remove(gdk_input_id);

 m_disconnect();
 connected = NO;
 logged_in = NO;   /* we're not connected, so we can't be logged_in */

 ghts_update_statusbar();
 message_dialog("Error:", "Lost connection.");
}

/******************************************************************************
 This function is called everytime there is data waiting in the socket.
******************************************************************************/
static void recv_data(gpointer data, gint sockfd, GdkInputCondition condition)
{
 char net_buffer[NET_BUFFER_SIZE+1];

 recv_from_server((char *)&net_buffer, NET_BUFFER_SIZE+1);
 handle_server_data((char *)&net_buffer);
}

/******************************************************************************
 Interpret the server-reply and take proper action.
******************************************************************************/
static void handle_server_data(char *buf)
{
 /* FIXME it is only temporary and it should be replaced with nr. handling */
 if (strncmp(buf, "Your login-timeout has expired. Bye.\n", NET_BUFFER_SIZE)==0)
 {
  lost_connection();
  ghts_update_statusbar();
  m_disconnect();

  message_dialog("Message:",
                 "Your login-timeout expired.\nYou have been disconnected.");

 }

 ghtslog(GHTSLOG_DEBUG, "htsserver: %s", buf);
}

