/* ghtsclient - a GTK+ based client for the trading game Holsham Traders
   Copyright (C) 1999,2000 Piotr Esden-Tempski

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* log.c  --  stderr/syslog/file-logging */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <syslog.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif
#endif

#include "defaults.h"
#include "file.h"
#include "conf.h"
#include "portable.h"

#include "log.h"


#define MAX_LEN_LOGMESSAGE 256

static const char *loglevnames[] = { "NONE", "ERROR", "INFO", "DEBUG" };

/******************************************************************************
 Generic log-routine. Can log to stderr, syslog or a file.
 Has four different log-level: none, error, info and debug.
******************************************************************************/
void ghtslog(int log_level, const char *fmt, ...)
{
 FILE *f;
 char some_message[MAX_LEN_LOGMESSAGE] = "";  /* just to satisfy lint */
 va_list args;
 time_t t;
 int syslog_level;
 char *tmp, *current_time;

 
 if (log_level > conf.verbosity)
  return;

 /* stdarg-magic */
 va_start(args, fmt);
 (void) vsnprintf(some_message, MAX_LEN_LOGMESSAGE, fmt, args);
 va_end(args);

 /* strip trailing newline from message */
 if (some_message[strlen(some_message)-1] == '\n')
  some_message[strlen(some_message)-1] = '\0';

 /* get current time */
 if ((t = time(NULL)) == (time_t)(-1))
  perror("time");
 current_time = ctime(&t);
 current_time[strlen(current_time)-1] = '\0';          /* remove the newline */


 switch (conf.logdest)
 {
  /***************************************************************************/
  case GHTSLOG_FILE:

   if ((tmp = ghtsfile(conf.logfile)) == NULL)
   {
    (void) fprintf(stderr, "ghtslog: Couldn't get absolute path to logfile.\n");
    return;
   }

   if ((f = fopen(tmp, "a")) == NULL)
   {
    perror("ghtslog:fopen");
    (void) fprintf(stderr, "Error opening logfile '%s'.\n", conf.logfile);
    free(tmp);
    return; /* exit(EXIT_FAILURE); */
   }

   free(tmp);

   (void) fprintf(f, "%s - %-5s: %s\n", current_time, loglevnames[log_level],
                  some_message);

   if (fclose(f) == EOF)
    perror("ghtslog:fclose");
  break;

  /***************************************************************************/
  case GHTSLOG_SYSLOG:
   switch (log_level)
   {
    case GHTSLOG_ERROR: syslog_level = LOG_ERR; break;
    case GHTSLOG_INFO: syslog_level = LOG_INFO; break;
    case GHTSLOG_DEBUG: syslog_level = LOG_DEBUG; break;
    default: syslog_level = LOG_INFO;  /* just to be sure */
   }
   openlog("ghtsclient", LOG_PID, LOG_USER);
   syslog(LOG_USER|syslog_level, some_message);
   closelog();
  break;

  /***************************************************************************/
  case GHTSLOG_STDERR:

   (void) fprintf(stderr, "%s - %-5s: %s\n", current_time,
                  loglevnames[log_level], some_message);

  break;
 }

}

/******************************************************************************
 Log an error, similar to perror().
******************************************************************************/
void lerror(const char *string)
{
 ghtslog(GHTSLOG_ERROR, "%s: %s(%d)", string, strerror(errno), errno);
}

