/* ghtsclient - a GTK+ based client for the trading game Holsham Traders
   Copyright (C) 1999,2000 Piotr Esden-Tempski

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* file.c  --  file-handling routines */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include "log.h"
#include "conf.h"
#include "defaults.h"
#include "portable.h"

#include "file.h"


#define MAX_LEN_HOME 256
#define MAX_LEN_FILENAME 256
#define DEFAULT_ROOTDIR "~/.ghtsclient"

#define DIRMODE (mode_t)0700


static char *texpand(const char *path);  /* tilde expansion */
static void mkdir_if_not_exists(const char *dirname);

/******************************************************************************
 Get the full(absolute) pathname to a file in ~/.ghtsclient.
 Returns a malloc()ed string(don't forget to free() it). Returns NULL on error.
******************************************************************************/
char *ghtsfile(const char *fmt, ...)
{
 char *root;                       /* expanded root dir */
 char filename[MAX_LEN_FILENAME];  /* formatted filename */
 char *path;                       /* final path of file */
 size_t pathlen;
 va_list args;


 va_start(args, fmt);
 (void) vsnprintf(filename, MAX_LEN_FILENAME, fmt, args);
 va_end(args);

 if ((root = texpand(DEFAULT_ROOTDIR)) == NULL)
  return NULL;

 pathlen = strlen(root) + strlen(filename) + 1;
 if ((path = malloc(pathlen+1)) == NULL)
 {
  lerror("ghtsfile:malloc");
  free(root);
  return NULL;
 }
 else
  (void) snprintf(path, pathlen+1, "%s/%s", root, filename);

 free(root);
 return path;
}

/*****************************************************************************/
static void mkdir_if_not_exists(const char *dirname)
{
 mode_t dirmode = DIRMODE;
 struct stat st;
 int errno_backup;

 if (mkdir(dirname, dirmode) < 0)
 {
  errno_backup = errno;

  if (stat(dirname, &st) < 0)
   lerror("mkdir_if_not_exists:stat");

  if ((errno_backup == EEXIST) && S_ISDIR(st.st_mode))
   return;

  lerror("mkdir_if_not_exists:mkdir");
  exit(EXIT_FAILURE);
 }
}

/******************************************************************************
 Expand tilde(~) characters in 'path' to our home-directory.
 Returns NULL on failure. Returns a malloced() string-pointer to the expanded
 path on success. The returned buffer should be free()d again later.
******************************************************************************/
static char *texpand(const char *path)
{
 char *homedir;
 char *abspath;       /* absolute path */
 size_t abspath_len;  /* length of the absolute path */
 int tilde_count;
 int i;

 tilde_count = 0;
 for (i=0; (path[i] != '\0'); i++)      /* count tilde characters in 'path' */
 {
  if (path[i] == '~')
   tilde_count++;
 }

 if ((homedir = getenv("HOME")) == NULL)
  (void) strcpy(homedir, ".");

 abspath_len = strlen(homedir)*tilde_count + strlen(path) - tilde_count;
 if ((abspath = malloc(abspath_len+1)) == NULL)
 {
  lerror("texpand:malloc");
  return NULL;
 }

 abspath[0] = '\0';  /* clear abspath */
 for (i=0; (path[i] != '\0'); i++)
 {
  if (path[i] == '~')
   strncat(abspath, homedir, MAX_LEN_HOME);
  else
   strncat(abspath, path+i, 1);
 }

 return abspath;
}

/******************************************************************************
 Create the directory ~/.ghtsclient.
******************************************************************************/
void create_dir(void)
{
 char *root;

 if ((root = texpand(DEFAULT_ROOTDIR)) == NULL)
 {
  (void) fprintf(stderr, "create_dirs: Tilde-expansion failed.\n");
  exit(EXIT_FAILURE);
 }
 else
 {
  mkdir_if_not_exists(root);
  free(root);
 }

}

