/* ghtsclient - a GTK+ based client for the trading game Holsham Traders
   Copyright (C) 1999,2000 Piotr Esden-Tempski

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* conf.c  --  configuration code */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "misc.h"       /* strchop() */
#include "defaults.h"
#include "file.h"       /* ghtsfile() */
#include "data.h"
#include "log.h"

#include "conf.h"


#define MAX_LEN_LINE 256
#define MAX_LEN_ENTRY 128


conf_t conf;      /* struct for run-time configuration */

/* config file entry handlers
 * There is one handler for each data type. It takes two arguments:
 *  void *dest		- points to the configuration variable to set
 *  const char *arg	- points to the argument string from the configfile
 */
static int c_unsigned_short(void *dest, const char *arg);
static int c_int(void *dest, const char *arg);
static int c_unsigned_int(void *dest, const char *arg);
static int c_pchar(void *dest, const char *arg);


/* config file entry handler table */
static struct cf_ent
{
 const char *name;                      /* entry name */
 void *var;                             /* config variable */
 int (*h)(void *dest, const char *arg);	/* entry handler */
} cf_ent_htab [] = {
 { "port",                 &conf.port,                 c_unsigned_short },
 { "server_ip",            &conf.server_ip,            c_pchar },
 { "verbosity",            &conf.verbosity,            c_int },
 { "logdest",              &conf.logdest,              c_int },
 { "logfile",              &conf.logfile,              c_pchar },
 { NULL,                   NULL,                       NULL }
};

/******************************************************************************
 Configuration initialization.
******************************************************************************/
void init_conf(void)
{
 conf.port      = PORT_NUMBER;
 conf.verbosity = DEFAULT_VERBOSITY;
 conf.logdest   = DEFAULT_LOGDEST;

 if ((conf.logfile = malloc(strlen(DEFAULT_LOGFILE)+1)) == NULL)
 {
  lerror("init_conf:malloc");
  exit(EXIT_FAILURE);
 }
 else (void) strcpy(conf.logfile, DEFAULT_LOGFILE);

 if ((conf.server_ip = malloc(strlen(SERVER_IP)+1)) == NULL)
 {
  lerror("init_conf:malloc");
  exit(EXIT_FAILURE);
 }
 else (void) strcpy(conf.server_ip, SERVER_IP);
}

/******************************************************************************
 Configfile parser.
******************************************************************************/
int parse_cfile(void)
{
 FILE *f;
 char *configfile;         /* absolute path to config file */
 char lbuf[MAX_LEN_LINE];  /* line buffer */
 char *name;               /* config option name */
 char *arg;                /* config option arg */
 int line;                 /* line count */

 const char *delim = " \t\n\r";     /* delimiters */
 struct cf_ent *ent;


 if ((configfile = ghtsfile(DEFAULT_CONFIGFILE)) == NULL)
 {
  (void) fprintf(stderr, "Couldn't find path to config file '%s'.\n",
                 DEFAULT_CONFIGFILE);
  return -1;
 }

 if ((f = fopen(configfile, "r")) == NULL)
 {
  lerror("parse_cfile:fopen");
  return -1;
 }


#ifdef CFILE_DEBUG
 (void) fprintf(stderr, "-- Reading configuration from '%s'.\n", configfile);
#endif

 for (line = 1; fgets(lbuf, MAX_LEN_LINE, f); line++)
 {
  if ((name = strchop(lbuf, delim)) == NULL)
   continue;

  if (name[0] == '#')      /* skip comment lines */
   continue;

  /* find matching entry */
  for (ent = cf_ent_htab; (ent->name != NULL); ent++)
  {
   if (strncmp(ent->name, name, MAX_LEN_ENTRY) == 0)
   {
    if ((arg = strchop(NULL, delim)) == NULL)
    {
     (void) fprintf(stderr, "%s:%d: Syntax error.\n", DEFAULT_CONFIGFILE, line);
     /* exit(EXIT_FAILURE); */
    }
    else
    {
     (void) ent->h(ent->var, arg);
#ifdef CFILE_DEBUG
     (void) fprintf(stderr, "%s:%d: %s set to '%s'.\n", DEFAULT_CONFIGFILE,
                    line, name, arg);
#endif
     free(arg);
    }
    break;
   }
  }
  free(name);
 }
#ifdef CFILE_DEBUG
 (void) fprintf(stderr, "--\n");
#endif

 return 0;
}


/* Handlers: */

/*****************************************************************************/
static int c_unsigned_short(void *dest, const char *arg)
{
 if (sscanf(arg, "%hu", (unsigned short int *)dest) != 1)
  return 0;
 return 1;
}

/*****************************************************************************/
static int c_int(void *dest, const char *arg)
{
 if (sscanf(arg, "%d", (int *)dest) != 1)
  return 0;
 return 1;
}

/*****************************************************************************/
static int c_unsigned_int(void *dest, const char *arg)
{
 if (sscanf(arg, "%ud", (unsigned int *)dest) != 1)
  return 0;
 return 1;
}

/*****************************************************************************/
static int c_pchar(void *dest, const char *arg)
{
 if (*(char **)dest != NULL)
  free(*(char **)dest);

 if ((*(char **)dest = malloc(strlen(arg)+1)) == NULL)
 {
  lerror("c_pchar:malloc");
  exit(EXIT_FAILURE);
 }
 else (void) strcpy(*(char **)dest, arg);
 return 1;
}

