/* ghtsclient - a GTK+ based client for the trading game Holsham Traders
   Copyright (C) 1999,2000 Piotr Esden-Tempski

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* cmdline.c  --  commandline-options handling */


#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#ifdef HAVE_GETOPT_H        /* getopt_long() */
#include <getopt.h>
#else
#include "getopt.h"          
#endif

#include "log.h"
#include "conf.h"

#include "cmdline.h"


/* The usage message */
#define USAGE "\
ghtsclient "VERSION", Copyright (C) 1999,2000 Piotr Esden-Tempski\n\n\
\
ghtsclient comes with ABSOLUTELY NO WARRANTY. This is free software,\n\
and you are welcome to redistribute it under the terms of\n\
the GNU General Public License. See COPYING for details.\n\n\
\
Usage: ghtsclient [-pvlfhV] [--port=PORT] [--verbosity=LEVEL]\n\
                  [--logdest=DEST] [--logfile=FILENAME] [--help] [--version]\n\
 -p, --port=PORT             connect to the port 'PORT' instead of\n\
                             the default port\n\
 -v, --verbosity=LEVEL       set verbosity level to 'LEVEL'. Possible values\n\
                             range from 0(no verbosity) to 3(maximum verbosity).\n\
 -l, --logdest=DEST          log to DEST which can be one of 'stderr', 'file',\n\
                             or 'syslog'\n\
 -f, --logfile=FILENAME      file to log to; implies --logdest=file\n\
 -h, --help                  show this help and exit\n\
 -V, --version               output version information and exit\n\n"


/*****************************************************************************/
void parse_arguments(int argc, char *argv[])
{
 /* the framework for this code is ripped from the getopt_long() manpage */
 int i, c, tmp;
 char *save;   /* used to temporarily save a char pointer for later recovery */


 while (1)
 {
  int option_index = 0;
  static struct option long_options[] =
  {
   {"port",      required_argument, NULL, (int)'p'},
   {"verbosity", required_argument, NULL, (int)'v'},
   {"logdest",   required_argument, NULL, (int)'l'},
   {"logfile",   required_argument, NULL, (int)'f'},
   {"help",      no_argument,       NULL, (int)'h'},
   {"version",   no_argument,       NULL, (int)'V'},
   {NULL,        0,                 NULL, (int)0}
  };
 
  c = getopt_long(argc, argv, "p:v:l:f:hV", long_options, &option_index);

  if (c == -1) break;  /* end of the options */

  switch (c)
  {
   /**************************************************************************/
   case 'p':  /* port */
    tmp = (int)strlen(optarg);
    
    for (i=0; i<tmp; i++)
    {
     if (!isdigit((int)*(optarg+i)))
     {
      ghtslog(GHTSLOG_ERROR, "The specified port is illegal.");
      exit(EXIT_FAILURE);
     }
    }

    errno = 0;
    conf.port = (unsigned short int)strtol(optarg, NULL, 0);

    if (errno != 0)
    {
     ghtslog(GHTSLOG_ERROR, "The specified port is illegal.");
     exit(EXIT_FAILURE);
    }
   break;

   /**************************************************************************/
   case 'v':  /* verbosity */
    tmp = (int)*optarg;

    if (isdigit(tmp) && (tmp >= (int)'0') && (tmp <= (int)'3')
        && ((int)strlen(optarg) == 1))
     conf.verbosity = (int)strtol(optarg, NULL, 0);
    else
    {
     ghtslog(GHTSLOG_ERROR, "Verbosity-level must be either 0, 1, 2 or 3.");
     exit(EXIT_FAILURE);
    }
   break;

   /**************************************************************************/
   case 'l':  /* logdest */
    if (strncasecmp(optarg, "stderr", strlen("stderr")) == 0)
     conf.logdest = GHTSLOG_STDERR;
    else if (strncasecmp(optarg, "file", strlen("file")) == 0)
     conf.logdest = GHTSLOG_FILE;
    else if (strncasecmp(optarg, "syslog", strlen("syslog")) == 0)
     conf.logdest = GHTSLOG_SYSLOG;
    /* else: logdest remains unchanged */
   break;

   /**************************************************************************/
   case 'f':  /* logfile */
    save = conf.logfile;   /* backup */

    if ((conf.logfile = malloc(strlen(optarg)+1)) == NULL)
    {
     lerror("parse_arguments:malloc");
     (void) fprintf(stderr, "Using default logfile '%s'.\n", save);
     conf.logfile = save;
    }
    else
    {
     /* TODO check if optarg is a valid filename. */

     (void) strcpy(conf.logfile, optarg);

     /* for people who forget to say --logdest=file */
     conf.logdest = GHTSLOG_FILE;
    }
   break;
  
   /**************************************************************************/
   case 'h':  /* print a help message */
    (void) printf(USAGE);
    exit(EXIT_SUCCESS);
   break;

   /**************************************************************************/
   case 'V':  /* print ghtsclient-version */
    (void) printf("%s\n", VERSION);
    exit(EXIT_SUCCESS);
   break;

   /**************************************************************************/
   case '?': exit(EXIT_FAILURE); break;
  }

 }
}

