#ifndef SYNCHRONIZER_H
#define SYNCHRONIZER_H
#include <unistd.h> /* for read */
/* Each client keeps his own copy of the state of the game (SOG). */
/* This means that only the inputs (keyboard) has to be sent over the */
/* network, reducing communications to an accepteble level. Care has */
/* to be taken so that the SOG stays synchronized. If a new client joins */
/* then the server will have to send the complete new SOG, so this will */
/* not initially be implemented. No attempt is made to fix byte order. */

/* This module is not specific to deadline, and it should be possible to */
/* use it for many other games. */

#define SYNCHRONIZER_MS 50
/* The idea is that a human can react in a 10th of a second, so */
/* 50ms gives a nyquist frequency of 10 Hz which should work well. */

void StartCommunications (const char *svr, unsigned short portNo,
  int *downloadSock);
/* If svr is NULL or "" then we run as single player. If svr is "server" */
/* we run as server and set *downloadSock = -1 and we must read everything */
/* from files. Otherwise we run as client */
/* and try to connect to port portNo on svr. If successfull, downloadSock */
/* is a socket throu which the server will send us the SOG (see */
/* downloadNew in Synchronize). Care must be taken so that exactly the */
/* right amount of data is read. */
/* portNo is in host byte order and portNo >= 1024 */

extern int myClientNo; /* 0 is the server, 1 the first client etc. */
/* Use this only for the display etc. */

#define MAX_MSG_SIZE 40 /* Including length */
void Broadcast (void *msg);
/* The first 4 bytes of msg must be the size in bytes (including the size) */
/* You should broadcast BEFORE (not after) changing the SOG, and */
/* wait for the echo of your own message, and then change the SOG */
/* accordingly. */

typedef struct {
  int  clientNo, len;
  char data[0];
} rxMsgHeader;
#define MSG_TO(type,msg) ((type *)&(msg)->len)
#define NEXT_MSG(msg) ( (rxMsgHeader*) ((msg)->len + MSG_TO(char,msg)))
void Synchronize (rxMsgHeader **start, rxMsgHeader **stop,
  int (*downloadNew)(int sock, int numberOfNewClient));
/* If we are the server, we wait for the next SYNCHRONIZER_MS time period. */
/* Then we collect all the incoming messages, pack them together and send */
/* them. If we are a client, we wait for the messages from the server. */
/* You must pass a reference to a pointer in each parameter. Synchronize */
/* will set these pointers. You can then loop throu the messages. */

/* Note that messages we sent with BroadCast will also be returned by */
/* Synchronize some time later, and then we should process them. */
/* Memory will be freed on next call to Synchronize. */

/* If a new client connects, downloadNew will be called (only on the */
/* server). downloadNew must then write the data the client will expect */
/* (see StartComms) to sock (which is blocking). downloadNew should */
/* return non-zero if it accepts the client, and 0 if the lamer is kicked. */

inline int ReadAll (int fd, void *buf, size_t count)
/* Like 'read', but continue after interruptions. */
{
  size_t x, total = 0;
  while (count > 0) {
    x = read (fd, buf, count);
    if (x == 0) break;
    count -= x;
    total += x;
    buf = x + (char*)buf;
  }
  return total;
}

#endif

